/*---------------------------------------------------------+
| PHP Interface for DynaPDF 4.0                            |
+----------------------------------------------------------+
| Copyright (C) 2003-2023                                  |
| Jens Boschulte, DynaForms GmbH.                          |
| All rights reserved.                                     |
+----------------------------------------------------------+
| Please report errors or other potential problems to      |
| support@dynaforms.com.                                   |
| The current version is available at www.dynaforms.com.   |
+----------------------------------------------------------*/

#define DYNAPDF_PHP_EXT_VERSION "2.0.57.13"

/*
   Please note that the public available workspaces for Visual Studio link the dynapdfm.dll since
   DynaPDF Professional doesn't include a static library. The dynapdfm.dll is delivered as 32 bit
   DLL only.
*/

#ifdef _WINDOWS
   #include "config.w32.h"
#elif defined(HAVE_CONFIG_H)
   #include "config.h"
#endif

#define PDF_STATIC
#include <dynapdf.h>
// undefine E_WARNING because it is allready defined in "zend_errors.h"
#undef E_WARNING
#undef E_SYNTAX_ERROR
#undef E_VALUE_ERROR

#if VS_2005_OR_HIGHER == 1
   #pragma warning(disable : 4133)
   #pragma warning(disable : 4267)
   #pragma warning(disable : 4311)
   #pragma warning(disable : 4312)
#endif

#include "php.h"
#include "php_dynapdf.h"
#include "ext/standard/info.h"

#define UNI_REPLACE_CHAR   0xFFFD
#define UNI_MAX_BMP        0xFFFF
#define UNI_MAX_UTF16      0x10FFFF
#define UNI_MAX_UTF32      0x7FFFFFFF
#define UNI_SUR_HIGH_START 0xD7FF
#define UNI_SUR_HIGH_END   0xDC00
#define UNI_SUR_HIGH_ADD   0xD800
#define UNI_SUR_LOW_START  0xDBFF
#define UNI_SUR_LOW_END    0xE000
#define UNI_SUR_LOW_ADD    0xDC00
#define UNI_HALF_SHIFT     10
#define UNI_HALF_BASE      0x10000
#define UNI_HALF_MASK      0x3FF

#ifndef MAX_PATH
   #define MAX_PATH 768
#endif

typedef enum TPointerType
{
   ptInvalid,
   ptBarcode,
   ptColItem,
   ptColorSpace,
   ptDeviceNAttr,
   ptEmbFileNode,
   ptEvents,
   ptField,
   ptFileSpecEx,
   ptFont,
   ptFunction,
   ptImage,
   ptLayer,
   ptMeasure,
   ptMixingHints,
   ptNumberFormat,
   ptOCG,
   ptOCGUINode,
   ptPtData,
   ptRelFileNode,
   ptSignature,
   ptSoftMask,
   pt3DView,
   ptInkList,
   ptStructElem
}TPointerType;

#define NUM_BARCODE_KEYS 11
static const char* BARCODE_KEYS[NUM_BARCODE_KEYS] =
{
   "Caption",
   "ECC",
   "Height",
   "nCodeWordCol",
   "nCodeWordRow",
   "Resolution",
   "Symbology",
   "Version",
   "Width",
   "XSymHeight",
   "XSymWidth"
};

#define NUM_BARCODE2_KEYS 21
static const char* BARCODE2_KEYS[NUM_BARCODE2_KEYS] =
{
   "BarcodeType",
   "BgColor",
   "BorderWidth",
   "Data",
   "DataType",
   "DotSize",
   "Eci",
   "FgColor",
   "FSizeFactor",
   "Option1",
   "Option2",
   "Option3",
   "Options",
   "Orientation",
   "OuterBorder",
   "Primary",
   "Scale",
   "ShowText",
   "SpaceWidth",
   "SymbHeight",
   "TextOffsetY"
};

#define NUM_EXTGSTATE_KEYS 14
static const char* EXTGSTATE_KEYS[NUM_EXTGSTATE_KEYS] =
{
   "AutoStrokeAdjust",
   "BlendMode",
   "FlatnessTol",
   "OverPrintFill",
   "OverPrintStroke",
   "OverPrintMode",
   "RenderingIntent",
   "SmoothnessTol",
   "FillAlpha",
   "StrokeAlpha",
   "AlphaIsShape",
   "TextKnockout",
   "SoftMaskNone",
   "SoftMask"
};

// Input keys only
#define NUM_HEADER_FOOTER_KEYS 11
static const char* HEADER_FOOTER_KEYS[NUM_HEADER_FOOTER_KEYS] =
{
   "FirstPage",
   "Flags",
   "InitColor",
   "InitCS",
   "InitCSHandle",
   "InitCodepage",
   "InitEmbed",
   "InitFont",
   "InitFontSize",
   "LastPage",
   "Margin"
};

// Input keys only
#define NUM_HDR_FTR_KEYS 12
static const char* HDR_FTR_KEYS[NUM_HDR_FTR_KEYS] =
{
   "Color",
   "CS",
   "CSHandle",
   "Codepage",
   "Embed",
   "Font",
   "FontSize",
   "IsHeader",
   "Position",
   "ShiftX",
   "ShiftY",
   "Text"
};

#define NUM_LINE_ANNOT_PARMS 7
static const char* LINE_ANNOT_PARMS[NUM_LINE_ANNOT_PARMS] =
{
   "Caption",
   "CaptionOffsetX",
   "CaptionOffsetY",
   "CaptionPos",
   "LeaderLineLen",
   "LeaderLineExtend",
   "LeaderLineOffset"
};

#define NUM_OCGCONT_USAGE_KEYS 13
static const char* OCGCONT_USAGE_KEYS[NUM_OCGCONT_USAGE_KEYS] =
{
   "ExportState",
   "InfoCreator",
   "InfoSubtype",
   "Language",
   "LangPreferred",
   "PageElement",
   "PrintState",
   "PrintSubtype",
   "UserNamesCount",
   "UserType",
   "ViewState",
   "ZoomMin",
   "ZoomMax"
};

#define NUM_OPTIMIZE_PARAMS_KEYS 13
static const char* OPTIMIZE_PARAMS_KEYS[NUM_OPTIMIZE_PARAMS_KEYS] =
{
   "ExcludeCS",
   "Filter1Bit",
   "FilterColor",
   "FilterGray",
   "JP2KQuality",
   "JPEGQuality",
   "Min1BitRes",
   "MinColorRes",
   "MinGrayRes",
   "MinLineWidth",
   "Res1BitImages",
   "ResColorImages",
   "ResGrayImages"
};

#define NUM_PROFILE_KEYS 5
static const char* PROFILE_KEYS[NUM_PROFILE_KEYS] =
{
   "DefInGray",
   "DefInRGB",
   "DefInCMYK",
   "DeviceProfile",
   "SoftProof"
};

#define NUM_RAWIMAGE_KEYS 11
static const char* RAWIMAGE_KEYS[NUM_RAWIMAGE_KEYS] =
{
   "Buffer",
   "BitsPerComponent",
   "NumComponents",
   "CS",
   "CSHandle",
   "Stride",
   "HasAlpha",
   "IsBGR",
   "MinIsWhite",
   "Width",
   "Height"
};

#define NUM_SIGPARMS_KEYS 11
static const char* SIGPARMS_KEYS[NUM_SIGPARMS_KEYS] =
{
   "PKCS7ObjLen",
   "HashType",
   "ContactInfo",
   "Location",
   "Reason",
   "Signer",
   "Encrypt",
   "OpenPwd",
   "OwnerPwd",
   "KeyLen",
   "Restrict"
};

#define IsUpperCase(c)((c) > 64 && (c) < 91)
#define ToLower(c) (IsUpperCase(c) ? (BYTE)((c) + 32) : (c))

static const UI16 CP_1252[256] =
{
   0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,0x000A,0x000B,0x000C,0x000D,0x000E,0x000F,
   0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001A,0x001B,0x001C,0x001D,0x001E,0x001F,
   0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002A,0x002B,0x002C,0x002D,0x002E,0x002F,
   0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003A,0x003B,0x003C,0x003D,0x003E,0x003F,
   0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004A,0x004B,0x004C,0x004D,0x004E,0x004F,
   0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005A,0x005B,0x005C,0x005D,0x005E,0x005F,
   0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006A,0x006B,0x006C,0x006D,0x006E,0x006F,
   0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007A,0x007B,0x007C,0x007D,0x007E,0x2022,
   0x20AC,0x2022,0x201A,0x0192,0x201E,0x2026,0x2020,0x2021,0x02C6,0x2030,0x0160,0x2039,0x0152,0x2022,0x017D,0x2022,
   0x2022,0x2018,0x2019,0x201C,0x201D,0x2022,0x2013,0x2014,0x02DC,0x2122,0x0161,0x203A,0x0153,0x2022,0x017E,0x0178,
   0x00A0,0x00A1,0x00A2,0x00A3,0x00A4,0x00A5,0x00A6,0x00A7,0x00A8,0x00A9,0x00AA,0x00AB,0x00AC,0x00AD,0x00AE,0x00AF,
   0x00B0,0x00B1,0x00B2,0x00B3,0x00B4,0x00B5,0x00B6,0x00B7,0x00B8,0x00B9,0x00BA,0x00BB,0x00BC,0x00BD,0x00BE,0x00BF,
   0x00C0,0x00C1,0x00C2,0x00C3,0x00C4,0x00C5,0x00C6,0x00C7,0x00C8,0x00C9,0x00CA,0x00CB,0x00CC,0x00CD,0x00CE,0x00CF,
   0x00D0,0x00D1,0x00D2,0x00D3,0x00D4,0x00D5,0x00D6,0x00D7,0x00D8,0x00D9,0x00DA,0x00DB,0x00DC,0x00DD,0x00DE,0x00DF,
   0x00E0,0x00E1,0x00E2,0x00E3,0x00E4,0x00E5,0x00E6,0x00E7,0x00E8,0x00E9,0x00EA,0x00EB,0x00EC,0x00ED,0x00EE,0x00EF,
   0x00F0,0x00F1,0x00F2,0x00F3,0x00F4,0x00F5,0x00F6,0x00F7,0x00F8,0x00F9,0x00FA,0x00FB,0x00FC,0x00FD,0x00FE,0x00FF
};

static const UI32 CRC_TABLE[256] =
{
   0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
   0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
   0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
   0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
   0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
   0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
   0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
   0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
   0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
   0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
   0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
   0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
   0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
   0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
   0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
   0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
   0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
   0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
   0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
   0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
   0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
   0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
   0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
   0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
   0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
   0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
   0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
   0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
   0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
   0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
   0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
   0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

SI32 StrCompEx(const void* Str1, const void* Str2)
{
   BYTE c1;
   BYTE c2;
   SI32 res = 0;
   const BYTE* s1 = (const BYTE*)Str1;
   const BYTE* s2 = (const BYTE*)Str2;
   if (!Str1 || !Str2) return -1;
   while(!res && *s1 && *s2)
   {
      c1 = *s1++;
      c2 = *s2++;
      res = ToLower(c1) - ToLower(c2);
   }
   if (res < 0)
      return -1;
   else if (res > 0)
      return 1;

   if (*s1 == 0 && *s2 != 0)
      return -1; // s1 is shorter than s2
   else if (*s1 != 0 && *s2 == 0)
      return 1;  // s1 is longer than s2
   return res;
}

UI32 StrLenA(const char* Value)
{
   const char* p = Value;
   if (!p) return 0;
   while (*p++) ;
   return (SI32)((p - Value) -1);
}

UI32 StrLenW(const UI16* Value)
{
   const UI16* p = Value;
   if (!p) return 0;
   while (*p++) ;
   return (SI32)((p - Value) -1);
}

SI32 FindKeyEx(const char* Values[], SI32 ValCount, const char* Key)
{
   SI32 i = 0, e = ValCount -1;
   while (i <= e)
   {
      if (!StrCompEx(Values[i], Key)) return i;
      if (!StrCompEx(Values[e], Key)) return e;
      ++i;
      --e;
   }
   return -1;
}

UI32 GetCRC(const void* Ptr, TPointerType PtrType)
{
   UI32 base = 0xFFFFFFFF;
   size_t value = (size_t)Ptr;
   base = (base >> 8) ^ CRC_TABLE[(base ^ (BYTE)PtrType) & 0xFF];
   while (value)
   {
      base    = (base >> 8) ^ CRC_TABLE[(base ^ (BYTE)value) & 0xFF];
      value >>= 8;
   }
   return ~base;
}

UI32 ToUTF8A(const BYTE* Source, UI32 Len, BYTE** Dest)
{
   UI16 c2;
   BYTE* dst;
   UI32  dstLen = Len + Len / 5;
   UI32 i = 0, count, ui, rem, out = 0;
   *Dest = (BYTE*)emalloc(dstLen + 1);
   dst   = *Dest;
   while (i < Len)
   {
      ui = CP_1252[Source[i++]];
      if (ui > UNI_SUR_HIGH_START && ui < UNI_SUR_HIGH_END && i < Len)
      {
         c2 = Source[i++];
         if (c2 > UNI_SUR_LOW_START && c2 < UNI_SUR_LOW_END)
         {
            ui = ((ui - UNI_SUR_HIGH_ADD) << UNI_HALF_SHIFT) + (c2 - UNI_SUR_LOW_ADD) + UNI_HALF_BASE;
         }
      }else if (ui > UNI_SUR_LOW_START && ui < UNI_SUR_LOW_END)
      {
         continue; // skip unpaired low surrogate
      }
      if (ui < 0x80)
         count = 1;
      else if (ui < 0x800)
         count = 2;
      else if (ui < 0x10000)
         count = 3;
      else if (ui < 0x200000)
         count = 4;
      else if (ui < 0x4000000)
         count = 5;
      else if (ui <= 0x7FFFFFFF)
         count = 6;
      else
         continue;
      if (out + count > dstLen)
      {
         rem = Len - i;
         if (rem > 1024)
            dstLen += 256;
         else if (rem > 512)
            dstLen += 128;
         else if (rem > 256)
            dstLen += 64;
         else if (rem > 128)
            dstLen += 32;
         else if (rem > 64)
            dstLen += 16;
         else if (rem > 32)
            dstLen += 8;
         else
            dstLen += 4;
         *Dest = (BYTE*)erealloc(*Dest, dstLen + 1);
         dst   = *Dest + out;
      }
      switch(count)
      {
         case 6: dst[5] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x4000000;
         case 5: dst[4] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x200000;
         case 4: dst[3] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x10000;
         case 3: dst[2] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x800;
         case 2: dst[1] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0xc0;
         case 1: dst[0] = (BYTE)ui;
      }
      dst += count;
      out += count;
   }
   (*Dest)[out] = 0;
   return out;
}

UI32 ToUTF8W(const UI16* Source, UI32 Len, BYTE** Dest)
{
   UI16 c2;
   BYTE* dst;
   UI32  dstLen = Len + Len / 5;
   UI32 i = 0, count, ui, rem, out = 0;
   *Dest = (BYTE*)emalloc(dstLen + 1);
   dst   = *Dest;
   while (i < Len)
   {
      ui = Source[i++];
      if (ui > UNI_SUR_HIGH_START && ui < UNI_SUR_HIGH_END && i < Len)
      {
         c2 = Source[i++];
         if (c2 > UNI_SUR_LOW_START && c2 < UNI_SUR_LOW_END)
         {
            ui = ((ui - UNI_SUR_HIGH_ADD) << UNI_HALF_SHIFT) + (c2 - UNI_SUR_LOW_ADD) + UNI_HALF_BASE;
         }
      }else if (ui > UNI_SUR_LOW_START && ui < UNI_SUR_LOW_END)
      {
         continue; // skip unpaired low surrogate
      }
      if (ui < 0x80)
         count = 1;
      else if (ui < 0x800)
         count = 2;
      else if (ui < 0x10000)
         count = 3;
      else if (ui < 0x200000)
         count = 4;
      else if (ui < 0x4000000)
         count = 5;
      else if (ui <= 0x7FFFFFFF)
         count = 6;
      else
         continue;
      if (out + count > dstLen)
      {
         rem = Len - i;
         if (rem > 1024)
            dstLen += 256;
         else if (rem > 512)
            dstLen += 128;
         else if (rem > 256)
            dstLen += 64;
         else if (rem > 128)
            dstLen += 32;
         else if (rem > 64)
            dstLen += 16;
         else if (rem > 32)
            dstLen += 8;
         else
            dstLen += 4;
         *Dest = (BYTE*)erealloc(*Dest, dstLen + 1);
         dst   = *Dest + out;
      }
      switch(count)
      {
         case 6: dst[5] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x4000000;
         case 5: dst[4] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x200000;
         case 4: dst[3] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x10000;
         case 3: dst[2] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0x800;
         case 2: dst[1] = (BYTE)(0x80 | (ui & 0x3f)); ui = ui >> 6; ui |= 0xc0;
         case 1: dst[0] = (BYTE)ui;
      }
      dst += count;
      out += count;
   }
   (*Dest)[out] = 0;
   return out;
}

zend_object_handlers dynapdf_object_handlers;
zend_class_entry*    dynapdf_ce;

struct TList
{
   UI32        Hash;
   const void* Data;
   void*       Next;
};

typedef enum TCallbackFuncs
{
   cbfAll                     = 0,
   cbfOnErrorProc             = 1,
   cbfOnFontNotFoundProc      = 2,
   cbfOnInitProgressProc      = 4,
   cbfOnPageBreakProc         = 8,
   cbfOnProgressProc          = 16,
   cbfonReplaceICCProfileProc = 32
}TCallbackFuncs;

struct dynapdf_object
{
   zend_object       zo;
   void*             pdf;
   struct TPDFStack* pdfStack;
   struct TList*     first;
   struct TList*     last;
   zval*             onErrorProc;
   zval*             onFontNotFoundProc;
   zval*             onInitProgressProc;
   zval*             onPageBreakProc;
   zval*             onProgressProc;
   zval*             onReplaceICCProfileProc;
};

struct TList* FindPointer(struct dynapdf_object* Object, const void* Ptr)
{
   struct TList* next = Object->first;
   while (next)
   {
      if (next->Data == Ptr) return next;
      next = next->Next;
   }
   return NULL;
}

void FreePointerList(struct dynapdf_object* Object)
{
   if (Object->first)
   {
      struct TList* next;
      struct TList* curr = Object->first;
      next = curr;
      while (next)
      {
         curr = next;
         next = next->Next;
         efree(curr);
      }
      Object->first = NULL;
      Object->last  = NULL;
   }
}

const void* GetPointer(struct dynapdf_object* Object, UI32 Hash, TPointerType PtrType)
{
   struct TList* next = Object->first;
   while (next)
   {
      if (next->Hash == Hash)
      {
         if (GetCRC(next->Data, PtrType) != Hash)
            return NULL;
         else
            return next->Data;
      }
      next = next->Next;
   }
   return NULL;
}

#define DYNAPDF_FETCH_PDFDOC() \
{\
   struct dynapdf_object *obj = (struct dynapdf_object*)zend_object_store_get_object(getThis() TSRMLS_CC); \
   if (obj && obj->pdf) \
      pdf = obj->pdf; \
   else \
   { \
      pdf = NULL; \
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch PDF instance!"); \
      return; \
   } \
}

#define DYNAPDF_FETCH_PDFDOC_EX() \
{\
   obj = (struct dynapdf_object*)zend_object_store_get_object(getThis() TSRMLS_CC); \
   if (!obj) \
   { \
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch PDF instance!"); \
      return; \
   } \
}

#define DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST() \
{\
   struct dynapdf_object *obj = (struct dynapdf_object*)zend_object_store_get_object(getThis() TSRMLS_CC); \
   if (obj && obj->pdf) \
   { \
      FreePointerList(obj); \
      pdf = obj->pdf; \
   }else \
   { \
      pdf = NULL; \
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch PDF instance!"); \
      return; \
   } \
}

PHP_MINFO_FUNCTION(dynapdf)
{
   php_info_print_table_start();
   php_info_print_table_row(2, "DynaPDF Support", "enabled");
   php_info_print_table_row(2, "DynaPDF Version", DYNAPDF_VERSIONSTRING);
   php_info_print_table_row(2, "Extension Version", DYNAPDF_PHP_EXT_VERSION);
   php_info_print_table_row(2, "Extension name:", "lowercase dynapdf");
   php_info_print_table_row(2, "Table class name:", "lowercase dynatbl");
   php_info_print_table_row(2, "The table class is included in the extension:", "dynapdf");
   php_info_print_table_end();
}

/* ----------------------------------------------------------- HELPER FUNCTIONS --------------------------------------------------------------- */

#define ADD_ASSOC_UTF8_STRING(z, k, v) \
   if ((v) && *(v) != 0) add_assoc_string((z), (k), (v), 1)

#define ADD_ASSOC_UTF8_STRINGL(z, k, v, l) \
   if ((v) && *(v) != 0) add_assoc_stringl((z), (k), (v), (l), 1)

#define ADD_ASSOC_BIN_STRING(z, k, v, l) \
   if ((v) && (l) > 0) add_assoc_stringl((z), (k), (v), (l), 1)

// Forward declarations
void AddPointer(zval* retval, struct dynapdf_object* Obj, const char* Key, const void* Ptr, TPointerType PtrType);
void AddString(zval* retval, const char* Key, const char* ValueA, const UI16* ValueW);
void AddStringA(zval* retval, const char* Key, const char* String);
void AddStringEx(zval* retval, const char* Key, const char* ValueA, const UI16* ValueW, UI32 Len);
void AddStringExA(zval* retval, const char* Key, const char* String, UI32 Len);
void AddStringW(zval* retval, const char* Key, const UI16* String);
void AddStringExW(zval* retval, const char* Key, const UI16* String, UI32 Len);
SI32 GetFltRect(HashTable* Array, struct TFltRect* R);
UI32 GetUI32Array(HashTable* Array, UI32* Value, UI32 MaxCount);

void AddBBox(zval* retval, const char* Key, struct TBBox* Value)
{
   zval* node;
   ALLOC_INIT_ZVAL(node);
   array_init(node);
   add_assoc_double(node, "x1", Value->x1);
   add_assoc_double(node, "y1", Value->y1);
   add_assoc_double(node, "x2", Value->x2);
   add_assoc_double(node, "y2", Value->y2);
   add_assoc_zval(retval, (char*)Key, node);
}

void AddCTM(zval* retval, const char* Key, struct TCTM* M)
{
   zval* node;
   ALLOC_INIT_ZVAL(node);
   array_init(node);
   add_assoc_double(node, "a", M->a);
   add_assoc_double(node, "b", M->b);
   add_assoc_double(node, "c", M->c);
   add_assoc_double(node, "d", M->d);
   add_assoc_double(node, "x", M->x);
   add_assoc_double(node, "y", M->y);
   add_assoc_zval(retval, (char*)Key, node);
}

void AddDoubleArray(zval* retval, const char* Key, double* Value, UI32 Count)
{
   if (Count > 0)
   {
      UI32 i;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         add_next_index_double(node, Value[i]);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void AddFileSpec(zval* return_value,  struct dynapdf_object* Obj, struct TPDFFileSpec* F)
{
   ADD_ASSOC_BIN_STRING(return_value, "Buffer", F->Buffer, F->BufSize);
   add_assoc_bool(return_value, "Compressed", F->Compressed);
   if (F->ColItem) AddPointer(return_value, Obj, "ColItem", F->ColItem, ptColItem);
   if (F->Name)
   {
      if (F->NameUnicode)
         AddStringW(return_value, "Name", (const UI16*)F->Name);
      else
         AddStringA(return_value, "Name", F->Name);
   }
   AddStringA(return_value, "FileName", F->FileName);
   add_assoc_bool(return_value, "IsURL", F->IsURL);
   if (F->UF)
   {
      if (F->UFUnicode)
         AddStringW(return_value, "UF", (UI16*)F->UF);
      else
         AddStringA(return_value, "UF", F->UF);
   }
   if (F->Desc)
   {
      if (F->DescUnicode)
         AddStringW(return_value, "Desc", (const UI16*)F->Desc);
      else
         AddStringA(return_value, "Desc", F->Desc);
   }
   add_assoc_long(return_value, "FileSize", F->FileSize);
   AddStringA(return_value, "MIMEType", F->MIMEType);
   AddStringA(return_value, "CreateDate", F->CreateDate);
   AddStringA(return_value, "ModDate", F->ModDate);
   ADD_ASSOC_BIN_STRING(return_value, "CheckSum", F->CheckSum, 16);
}

void AddFileSpecEx(zval* return_value,  struct dynapdf_object* Obj, struct TPDFFileSpecEx* F)
{
   AddStringA(return_value, "AFRelationship", F->AFRelationship);
   if (F->ColItem) AddPointer(return_value, Obj, "ColItem", F->ColItem, ptColItem);
   AddString(return_value, "Description", F->DescriptionA, F->DescriptionW);
   AddStringA(return_value, "DOS", F->DOS);
   if (F->EmbFileNode) AddPointer(return_value, Obj, "EmbFileNode", F->EmbFileNode, ptEmbFileNode);
   AddStringA(return_value, "FileName", F->FileName);
   add_assoc_bool(return_value, "FileNameIsURL", F->FileNameIsURL);
   ADD_ASSOC_BIN_STRING(return_value, "ID1", (char*)F->ID1, F->ID1Len);
   ADD_ASSOC_BIN_STRING(return_value, "ID2", (char*)F->ID2, F->ID2Len);
   add_assoc_bool(return_value, "IsVolatile", F->IsVolatile);
   AddStringA(return_value, "Mac", F->Mac);
   AddStringA(return_value, "Unix", F->Unix);
   if (F->RelFileNode) AddPointer(return_value, Obj, "RelFileNode", F->RelFileNode, ptRelFileNode);
   if (F->Thumb) AddPointer(return_value, Obj, "Thumb", F->Thumb, ptImage);
   AddString(return_value, "UFileName", F->UFileNameA, F->UFileNameW);
}

void AddFloatArray(zval* retval, const char* Key, float* Value, UI32 Count)
{
   if (Count > 0)
   {
      UI32 i;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         add_next_index_double(node, Value[i]);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void AddFltPoint(zval* retval, const char* Key, struct TFltPoint* Value)
{
   zval* node;
   ALLOC_INIT_ZVAL(node);
   array_init(node);
   add_assoc_double(node, "x", Value->x);
   add_assoc_double(node, "y", Value->y);
   add_assoc_zval(retval, (char*)Key, node);
}

void AddFltRect(zval* retval, const char* Key, struct TFltRect* Value)
{
   zval* node;
   ALLOC_INIT_ZVAL(node);
   array_init(node);
   add_assoc_double(node, "Left",   Value->Left);
   add_assoc_double(node, "Bottom", Value->Bottom);
   add_assoc_double(node, "Right",  Value->Right);
   add_assoc_double(node, "Top",    Value->Top);
   add_assoc_zval(retval, (char*)Key, node);
}

void AddKerningArrayA(zval* retval, const char* Key, struct TTextRecordA Values[], UI32 Count)
{
   UI32 i;
   zval* array, *node;
   ALLOC_INIT_ZVAL(array);
   array_init(array);
   for (i = 0; i < Count; i++)
   {
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      add_assoc_double(node, "Advance", Values[i].Advance);
      ADD_ASSOC_BIN_STRING(node, "Text", Values[i].Text, Values[i].Length);
      add_next_index_zval(array, node);
   }
   add_assoc_zval(retval, (char*)Key, array);
}

void AddKerningArrayW(zval* retval, const char* Key, struct TTextRecordW Values[], UI32 Count)
{
   UI32 i;
   zval* array, *node;
   ALLOC_INIT_ZVAL(array);
   array_init(array);
   for (i = 0; i < Count; i++)
   {
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      add_assoc_double(node, "Advance", Values[i].Advance);
      AddStringExW(node, "Text", Values[i].Text, Values[i].Length);
      add_assoc_double(node, "Width", Values[i].Width);
      add_next_index_zval(array, node);
   }
   add_assoc_zval(retval, (char*)Key, array);
}

void AddPDFRect(zval* retval, const char* Key, struct TPDFRect* Value)
{
   zval* node;
   ALLOC_INIT_ZVAL(node);
   array_init(node);
   add_assoc_double(node, "Left",   Value->Left);
   add_assoc_double(node, "Bottom", Value->Bottom);
   add_assoc_double(node, "Right",  Value->Right);
   add_assoc_double(node, "Top",    Value->Top);
   add_assoc_zval(retval, (char*)Key, node);
}

void AddPDFVersionInfo(zval* retval, struct TPDFVersionInfo* Version)
{
   array_init(retval);
   add_assoc_long(retval, "Major",            Version->Major);
   add_assoc_long(retval, "Minor",            Version->Minor);
   AddStringA(retval,     "MainVer",          Version->MainVer);
   AddStringA(retval,     "SubVer",           Version->SubVer);
   AddStringA(retval,     "PDFAConformance",  Version->PDFAConformance);
   add_assoc_long(retval, "PDFAVersion",      Version->PDFAVersion);
   AddStringA(retval,     "PDFEVersion",      Version->PDFEVersion);
   AddStringA(retval,     "PDFVTModDate",     Version->PDFVTModDate);
   AddStringA(retval,     "PDFVTVersion",     Version->PDFVTVersion);
   AddStringA(retval,     "PDFXConformance",  Version->PDFXConformance);
   AddStringA(retval,     "PDFXVersion",      Version->PDFXVersion);
   AddStringA(retval,     "FXConfLevel",      Version->FXConfLevel);
   AddStringA(retval,     "FXDocName",        Version->FXDocName);
   AddStringA(retval,     "FXDocType",        Version->FXDocType);
   AddStringA(retval,     "FXVersion",        Version->FXVersion);
   add_assoc_long(retval, "VersionConst",     Version->VersionConst);
}

void AddPointArray(zval* retval, const char* Key, struct TI32Point* Points, UI32 Count)
{
   if (Count > 0)
   {
      UI32 i;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         add_next_index_long(node, Points[i].x);
         add_next_index_long(node, Points[i].y);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void AddPointer(zval* retval, struct dynapdf_object* Obj, const char* Key, const void* Ptr, TPointerType PtrType)
{
   struct TList* next;
   if ((next = FindPointer(Obj, Ptr)) == NULL)
   {
      next = (struct TList*)emalloc(sizeof(struct TList));
      next->Data = Ptr;
      next->Hash = GetCRC(Ptr, PtrType);
      next->Next = NULL;
      if (!Obj->first)
         Obj->first = next;
      else
         Obj->last->Next = next;
      Obj->last = next;
   }
   add_assoc_long(retval, (char*)Key, next->Hash);
}

UI32 AddPointerEx(struct dynapdf_object* Obj, const void* Ptr, TPointerType PtrType)
{
   struct TList* next;
   if ((next = FindPointer(Obj, Ptr)) == NULL)
   {
      next = (struct TList*)emalloc(sizeof(struct TList));
      next->Data = Ptr;
      next->Hash = GetCRC(Ptr, PtrType);
      next->Next = NULL;
      if (!Obj->first)
         Obj->first = next;
      else
         Obj->last->Next = next;
      Obj->last = next;
   }
   return next->Hash;
}

void AddPointerArray(zval* retval, struct dynapdf_object *Obj, const char* Key, void** Value, UI32 Count, TPointerType PtrType)
{
   if (Count > 0)
   {
      UI32 i, hash;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         hash = AddPointerEx(Obj, Value[i], PtrType);
         add_next_index_long(node, hash);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void AddString(zval* retval, const char* Key, const char* ValueA, const UI16* ValueW)
{
   UI32 len;
   BYTE* buf = NULL;
   if (ValueA)
   {
      len = ToUTF8A(ValueA, StrLenA(ValueA), &buf);
      add_assoc_stringl(retval, (char*)Key, buf, len, 0);
   }else if (ValueW)
   {
      len = ToUTF8W(ValueW, StrLenW(ValueW), &buf);
      add_assoc_stringl(retval, (char*)Key, buf, len, 0);
   }
}

void AddStringA(zval* retval, const char* Key, const char* String)
{
   if (String)
   {
      UI32 len;
      BYTE* buf;
      len = ToUTF8A(String, StrLenA(String), &buf);
      add_assoc_stringl(retval, (char*)Key, buf, len, 0);
   }
}

void AddStringEx(zval* retval, const char* Key, const char* ValueA, const UI16* ValueW, UI32 Len)
{
   BYTE* buf = NULL;
   if (ValueA)
   {
      Len = ToUTF8A(ValueA, Len, &buf);
      add_assoc_stringl(retval, (char*)Key, buf, Len, 0);
   }else if (ValueW)
   {
      Len = ToUTF8W(ValueW, Len, &buf);
      add_assoc_stringl(retval, (char*)Key, buf, Len, 0);
   }
}

void AddStringExA(zval* retval, const char* Key, const char* String, UI32 Len)
{
   if (String)
   {
      BYTE* buf = NULL;
      Len = ToUTF8A(String, Len, &buf);
      add_assoc_stringl(retval, (char*)Key, buf, Len, 0);
   }
}

void AddStringW(zval* retval, const char* Key, const UI16* String)
{
   if (String)
   {
      UI32 len;
      BYTE* buf = NULL;
      len = ToUTF8W(String, StrLenW(String), &buf);
      add_assoc_stringl(retval, (char*)Key, buf, len, 0);
   }
}

void AddStringExW(zval* retval, const char* Key, const UI16* String, UI32 Len)
{
   if (String)
   {
      BYTE* buf = NULL;
      Len = ToUTF8W(String, Len, &buf);
      add_assoc_stringl(retval, (char*)Key, buf, Len, 0);
   }
}

void AddUI16Array(zval* retval, const char* Key, UI16* Value, UI32 Count)
{
   if (Value && Count > 0)
   {
      UI32 i;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         add_next_index_long(node, Value[i]);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void AddUI32Array(zval* retval, const char* Key, UI32* Value, UI32 Count)
{
   if (Value && Count > 0)
   {
      UI32 i;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         add_next_index_long(node, Value[i]);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void AddUTF8StringArray(zval* retval, const char* Key, char** Value, UI32 Count)
{
   if (Count > 0)
   {
      UI32 i;
      zval* node;
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < Count; i++)
      {
         add_next_index_string(node, Value[i], 1);
      }
      add_assoc_zval(retval, (char*)Key, node);
   }
}

void GetBarcodeStruct(HashTable* Array, struct TPDFBarcode* Barcode)
{
   char *key;
   int key_len;
   zval **data;
   ulong index;
   HashPosition pointer;
   memset(Barcode, 0, sizeof(*Barcode));
   Barcode->StructSize = sizeof(*Barcode);
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(BARCODE_KEYS, NUM_BARCODE_KEYS, key))
         {
            case 0:  convert_to_string_ex(data); Barcode->CaptionA     = Z_STRVAL_PP(data);      break;
            case 1:  convert_to_double_ex(data); Barcode->ECC          = (float)Z_DVAL_PP(data); break;
            case 2:  convert_to_double_ex(data); Barcode->Height       = (float)Z_DVAL_PP(data); break;
            case 3:  convert_to_double_ex(data); Barcode->nCodeWordCol = (float)Z_DVAL_PP(data); break;
            case 4:  convert_to_double_ex(data); Barcode->nCodeWordRow = (float)Z_DVAL_PP(data); break;
            case 5:  convert_to_long_ex(data);   Barcode->Resolution   = Z_LVAL_PP(data);        break;
            case 6:  convert_to_string_ex(data); Barcode->Symbology    = Z_STRVAL_PP(data);      break;
            case 7:  convert_to_double_ex(data); Barcode->Version      = (float)Z_DVAL_PP(data); break;
            case 8:  convert_to_double_ex(data); Barcode->Width        = (float)Z_DVAL_PP(data); break;
            case 9:  convert_to_double_ex(data); Barcode->XSymHeight   = (float)Z_DVAL_PP(data); break;
            case 10: convert_to_double_ex(data); Barcode->XSymWidth    = (float)Z_DVAL_PP(data); break;
            default: break;
         }
      }
   }
}

void GetBarcode2Struct(HashTable* Array, struct TPDFBarcode2* Barcode)
{
   char *key;
   int key_len;
   zval **data;
   ulong index;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(BARCODE2_KEYS, NUM_BARCODE2_KEYS, key))
         {
            case 0:  convert_to_long_ex(data);   Barcode->BarcodeType  = Z_LVAL_PP(data);        break;
            case 1:  convert_to_long_ex(data);   Barcode->BgColor      = Z_LVAL_PP(data);        break;
            case 2:  convert_to_long_ex(data);   Barcode->BorderWidth  = Z_LVAL_PP(data);        break;
            case 3:  convert_to_string_ex(data); Barcode->Data         = Z_STRVAL_PP(data);      break;
            case 4:  convert_to_long_ex(data);   Barcode->DataType     = Z_LVAL_PP(data);        break;
            case 5:  convert_to_double_ex(data); Barcode->DotSize      = (float)Z_DVAL_PP(data); break;
            case 6:  convert_to_long_ex(data);   Barcode->Eci          = Z_LVAL_PP(data);        break;
            case 7:  convert_to_long_ex(data);   Barcode->FgColor      = Z_LVAL_PP(data);        break;
            case 8:  convert_to_double_ex(data); Barcode->FSizeFactor  = (float)Z_DVAL_PP(data); break;
            case 9:  convert_to_long_ex(data);   Barcode->Option1      = Z_LVAL_PP(data);        break;
            case 10: convert_to_long_ex(data);   Barcode->Option2      = Z_LVAL_PP(data);        break;
            case 11: convert_to_long_ex(data);   Barcode->Option3      = Z_LVAL_PP(data);        break;
            case 12: convert_to_long_ex(data);   Barcode->Options      = Z_LVAL_PP(data);        break;
            case 13: convert_to_long_ex(data);   Barcode->Orientation  = Z_LVAL_PP(data);        break;
            case 14: convert_to_array_ex(data);  GetFltRect(Z_ARRVAL_PP(data), &Barcode->OuterBorder); break;
            case 15: convert_to_string_ex(data); Barcode->Primary      = Z_STRVAL_PP(data);      break;
            case 16: convert_to_double_ex(data); Barcode->Scale        = (float)Z_DVAL_PP(data); break;
            case 17: convert_to_long_ex(data);   Barcode->ShowText     = Z_LVAL_PP(data);        break;
            case 18: convert_to_long_ex(data);   Barcode->SpaceWidth   = Z_LVAL_PP(data);        break;
            case 19: convert_to_long_ex(data);   Barcode->SymbHeight   = Z_LVAL_PP(data);        break;
            case 20: convert_to_double_ex(data); Barcode->TextOffsetY  = (float)Z_DVAL_PP(data); break;
            default: break;
         }
      }
   }
}

SI32 GetBBoxFromArray(HashTable* Array, struct TPDFRect* BBox)
{
   char *key;
   int key_len;
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   if (zend_hash_num_elements(Array) != 4) return -1;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer), index++)
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         if (key_len > 0)
         {
            switch(key[0])
            {
               case 'L': index = 0;  break;
               case 'B': index = 1;  break;
               case 'R': index = 2;  break;
               case 'T': index = 3;  break;
            }
         }
         convert_to_double_ex(data);
         switch(index)
         {
            case 0:  BBox->Left   = Z_DVAL_PP(data); break;
            case 1:  BBox->Bottom = Z_DVAL_PP(data); break;
            case 2:  BBox->Right  = Z_DVAL_PP(data); break;
            case 3:  BBox->Top    = Z_DVAL_PP(data); break;
            default: return -3;
         }
      }
   }
   return 0;
}

SI32 GetTBBoxFromArray(HashTable* Array, struct TBBox* BBox)
{
   char *key;
   int key_len;
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   if (zend_hash_num_elements(Array) != 4) return -1;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer), index++)
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         if (key_len == 2)
         {
            if (key[0] == 'x')
            {
               if (key[1] == '1')
                  index = 0;
               else
                  index = 2;
            }else if (key[0] == 'y')
            {
               if (key[1] == '1')
                  index = 1;
               else
                  index = 3;
            }else
               return -1;
         } 
         convert_to_double_ex(data);
         switch(index)
         {
            case 0:  BBox->x1 = (float)Z_DVAL_PP(data); break;
            case 1:  BBox->y1 = (float)Z_DVAL_PP(data); break;
            case 2:  BBox->x2 = (float)Z_DVAL_PP(data); break;
            case 3:  BBox->y2 = (float)Z_DVAL_PP(data); break;
            default: return -3;
         }
      }
   }
   return 0;
}

UI32 GetBYTEArray(HashTable* Array, BYTE* Value, UI32 MaxCount)
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_long_ex(data);
      Value[index++] = (BYTE)Z_LVAL_PP(data);
      if (index == MaxCount) break;
   }
   return index;
}

UI32 GetColorants(HashTable* Array, char* Colorants[32])
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_string_ex(data);
      Colorants[index++] = Z_STRVAL_PP(data);
      if (index == 32) break;
   }
   return index;
}

void GetColorProfiles(HashTable* Array, struct TPDFColorProfiles* Profiles)
{
   char *key;
   int key_len;
   zval **data;
   ulong index;
   HashPosition pointer;
   Profiles->StructSize = sizeof(*Profiles);
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         convert_to_string_ex(data);
         switch(FindKeyEx(PROFILE_KEYS, NUM_PROFILE_KEYS, key))
         {
            case 0:  Profiles->DefInGrayA     = Z_STRVAL_PP(data); break;
            case 1:  Profiles->DefInRGBA      = Z_STRVAL_PP(data); break;
            case 2:  Profiles->DefInCMYKA     = Z_STRVAL_PP(data); break;
            case 3:  Profiles->DeviceProfileA = Z_STRVAL_PP(data); break;
            case 4:  Profiles->SoftProofA     = Z_STRVAL_PP(data); break;
            default: break;
         }
      }
   }
}

SI32 GetCTMFromArray(HashTable* Array, struct TCTM* M)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   if (zend_hash_num_elements(Array) != 6) return -1;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         if (key_len == 1)
         {
            switch(key[0])
            {
               case 'a': index = 0; break;
               case 'b': index = 1; break;
               case 'c': index = 2; break;
               case 'd': index = 3; break;
               case 'x': index = 4; break;
               case 'y': index = 5; break;
               default:  return -1;
            }
         }
         convert_to_double_ex(data);
         switch(index)
         {
            case 0:  M->a = Z_DVAL_PP(data); break;
            case 1:  M->b = Z_DVAL_PP(data); break;
            case 2:  M->c = Z_DVAL_PP(data); break;
            case 3:  M->d = Z_DVAL_PP(data); break;
            case 4:  M->x = Z_DVAL_PP(data); break;
            case 5:  M->y = Z_DVAL_PP(data); break;
            default: return -3;
         }
      }
   }
   return 0;
}

UI32 GetDoubleArray(HashTable* Array, double* Value, UI32 MaxCount)
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_long_ex(data);
      Value[index++] = Z_DVAL_PP(data);
      if (index == MaxCount) break;
   }
   return index;
}

SI32 GetExtGState(struct dynapdf_object* Object, HashTable* Array, struct TPDFExtGState* GS)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   SI32 hash;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(EXTGSTATE_KEYS, NUM_EXTGSTATE_KEYS, key))
         {
            case 0:  convert_to_long_ex(data);   GS->AutoStrokeAdjust = Z_LVAL_PP(data);        break;
            case 1:  convert_to_long_ex(data);   GS->BlendMode        = Z_LVAL_PP(data);        break;
            case 2:  convert_to_double_ex(data); GS->FlatnessTol      = (float)Z_DVAL_PP(data); break;
            case 3:  convert_to_long_ex(data);   GS->OverPrintFill    = Z_LVAL_PP(data);        break;
            case 4:  convert_to_long_ex(data);   GS->OverPrintStroke  = Z_LVAL_PP(data);        break;
            case 5:  convert_to_long_ex(data);   GS->OverPrintMode    = Z_LVAL_PP(data);        break;
            case 6:  convert_to_long_ex(data);   GS->RenderingIntent  = Z_LVAL_PP(data);        break;
            case 7:  convert_to_double_ex(data); GS->SmoothnessTol    = (float)Z_DVAL_PP(data); break;
            case 8:  convert_to_double_ex(data); GS->FillAlpha        = (float)Z_DVAL_PP(data); break;
            case 9:  convert_to_double_ex(data); GS->StrokeAlpha      = (float)Z_DVAL_PP(data); break;
            case 10: convert_to_long_ex(data);   GS->AlphaIsShape     = Z_LVAL_PP(data);        break;
            case 11: convert_to_long_ex(data);   GS->TextKnockout     = Z_LVAL_PP(data);        break;
            case 12: convert_to_long_ex(data);   GS->SoftMaskNone     = Z_LVAL_PP(data);        break;
            case 13:
            {
               convert_to_long_ex(data);
               hash = Z_LVAL_PP(data);
               if (hash != 0 && (GS->SoftMask = (PIMG*)GetPointer(Object, hash, ptSoftMask)) == NULL) return -1;
               break;
            }
            default: break;
         }
      }
   }
   return 0;
}

void GetContUsage(struct dynapdf_object* Object, HashTable* Array, struct TPDFOCGContUsage* F)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(OCGCONT_USAGE_KEYS, NUM_OCGCONT_USAGE_KEYS, key))
         {
            case 0:  convert_to_long_ex(data);   F->ExportState   = Z_LVAL_PP(data);        break;
            case 1:  convert_to_string_ex(data); F->InfoCreatorA  = Z_STRVAL_PP(data);      break;
            case 2:  convert_to_string_ex(data); F->InfoSubtype   = Z_STRVAL_PP(data);      break;
            case 3:  convert_to_string_ex(data); F->LanguageA     = Z_STRVAL_PP(data);      break;
            case 4:  convert_to_long_ex(data);   F->LangPreferred = Z_LVAL_PP(data);        break;
            case 5:  convert_to_long_ex(data);   F->PageElement   = Z_LVAL_PP(data);        break;
            case 6:  convert_to_long_ex(data);   F->PrintState    = Z_LVAL_PP(data);        break;
            case 7:  convert_to_string_ex(data); F->PrintSubtype  = Z_STRVAL_PP(data);      break;
            case 8:                                                                         break; // UserNamesCount
            case 9:                                                                         break; // UserType
            case 10: convert_to_long_ex(data);   F->ViewState     = Z_LVAL_PP(data);        break;
            case 11: convert_to_double_ex(data); F->ZoomMin       = (float)Z_DVAL_PP(data); break;
            case 12: convert_to_double_ex(data); F->ZoomMax       = (float)Z_DVAL_PP(data); break;
            default: break;
         }
      }
   }
}

UI32 GetFloatArray(HashTable* Array, float* Value, UI32 MaxCount)
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_double_ex(data);
      Value[index++] = (float)Z_DVAL_PP(data);
      if (index == MaxCount) break;
   }
   return index;
}

SI32 GetFltRect(HashTable* Array, struct TFltRect* R)
{
   char *key;
   int key_len;
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   if (!Array || zend_hash_num_elements(Array) != 4) return -1;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer), index++)
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         if (key_len > 0)
         {
            switch(key[0])
            {
               case 'L': index = 0;  break;
               case 'B': index = 1;  break;
               case 'R': index = 2;  break;
               case 'T': index = 3;  break;
            }
         }
         convert_to_double_ex(data);
         switch(index)
         {
            case 0:  R->Left   = (float)Z_DVAL_PP(data); break;
            case 1:  R->Bottom = (float)Z_DVAL_PP(data); break;
            case 2:  R->Right  = (float)Z_DVAL_PP(data); break;
            case 3:  R->Top    = (float)Z_DVAL_PP(data); break;
            default: return -3;
         }
      }
   }
   return 0;
}

UI32 GetFontSearchOrder(HashTable* Array, TFontBaseType Order[4])
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_long_ex(data);
      Order[index++] = Z_LVAL_PP(data);
      if (index == 4) break;
   }
   return index;
}

void GetHeaderFooterStruct(HashTable* Array, struct TPDFHeaderFooter* Header)
{
   char *key;
   int key_len;
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer), index++)
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(HEADER_FOOTER_KEYS, NUM_HEADER_FOOTER_KEYS, key))
         {
            case 0:  convert_to_long_ex(data);   Header->FirstPage    = Z_LVAL_PP(data);         break;
            case 1:  convert_to_long_ex(data);   Header->Flags        = Z_LVAL_PP(data);         break;
            case 2:  convert_to_long_ex(data);   Header->InitColor    = Z_LVAL_PP(data);         break;
            case 3:  convert_to_long_ex(data);   Header->InitCS       = Z_LVAL_PP(data);         break;
            case 4:  convert_to_long_ex(data);   Header->InitCSHandle = Z_LVAL_PP(data);         break;
            case 5:  convert_to_long_ex(data);   Header->InitCodepage = Z_LVAL_PP(data);         break;
            case 6:  convert_to_long_ex(data);   Header->InitEmbed    = Z_LVAL_PP(data);         break;
            case 7:  convert_to_string_ex(data); Header->InitFontA    = Z_STRVAL_PP(data);       break;
            case 8:  convert_to_double_ex(data); Header->InitFontSize = (float)Z_LVAL_PP(data);  break;
            case 9:  convert_to_long_ex(data);   Header->LastPage     = Z_LVAL_PP(data);         break;
            case 10: convert_to_array_ex(data);  GetFltRect(Z_ARRVAL_PP(data), &Header->Margin); break;
         }
      }
   }
}

void GetHdrFtrStruct(HashTable* Array, struct TPDFHdrFtr* HF)
{
   char *key;
   int key_len;
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer), index++)
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(HDR_FTR_KEYS, NUM_HDR_FTR_KEYS, key))
         {
            case 0:  convert_to_long_ex(data);   HF->Color    = Z_LVAL_PP(data);        break;
            case 1:  convert_to_long_ex(data);   HF->CS       = Z_LVAL_PP(data);        break;
            case 2:  convert_to_long_ex(data);   HF->CSHandle = Z_LVAL_PP(data);        break;
            case 3:  convert_to_long_ex(data);   HF->Codepage = Z_LVAL_PP(data);        break;
            case 4:  convert_to_long_ex(data);   HF->Embed    = Z_LVAL_PP(data);        break;
            case 5:  convert_to_string_ex(data); HF->FontA    = Z_STRVAL_PP(data);      break;
            case 6:  convert_to_double_ex(data); HF->FontSize = (float)Z_LVAL_PP(data); break;
            case 7:  convert_to_long_ex(data);   HF->IsHeader = Z_LVAL_PP(data);        break;
            case 8:  convert_to_long_ex(data);   HF->Position = Z_LVAL_PP(data);        break;
            case 9:  convert_to_double_ex(data); HF->ShiftX   = (float)Z_LVAL_PP(data); break;
            case 10: convert_to_double_ex(data); HF->ShiftY   = (float)Z_LVAL_PP(data); break;
            case 11:
               convert_to_string_ex(data);
               HF->TextA   = Z_STRVAL_PP(data);
               HF->TextLen = Z_STRLEN_PP(data);
               break;
         }
      }
   }
}

UI32 GetHdrFtrArray(HashTable* Array, struct TPDFHdrFtr* HFArray)
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer), index++)
   {
      convert_to_array_ex(data);
      GetHdrFtrStruct(Z_ARRVAL_PP(data), &HFArray[index++]);
      if (index == 6) break;
   }
   return index;
}

void GetLineAnnotParms(HashTable* Array, struct TLineAnnotParms* Parms)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   Parms->StructSize = sizeof(*Parms);
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(LINE_ANNOT_PARMS, NUM_LINE_ANNOT_PARMS, key))
         {
            case 0:  convert_to_long_ex(data);   Parms->Caption          = Z_LVAL_PP(data);        break;
            case 1:  convert_to_double_ex(data); Parms->CaptionOffsetX   = (float)Z_DVAL_PP(data); break;
            case 2:  convert_to_double_ex(data); Parms->CaptionOffsetY   = (float)Z_DVAL_PP(data); break;
            case 3:  convert_to_long_ex(data);   Parms->CaptionPos       = Z_LVAL_PP(data);        break;
            case 4:  convert_to_double_ex(data); Parms->LeaderLineLen    = (float)Z_DVAL_PP(data); break;
            case 5:  convert_to_double_ex(data); Parms->LeaderLineExtend = (float)Z_DVAL_PP(data); break;
            case 6:  convert_to_double_ex(data); Parms->LeaderLineOffset = (float)Z_DVAL_PP(data); break;
            default: break;
         }
      }
   }
}

void GetOptimizeParams(HashTable* Array, struct TOptimizeParams* Parms, UI32* ExcludeCS, UI32 MaxCount)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   Parms->StructSize = sizeof(*Parms);
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(OPTIMIZE_PARAMS_KEYS, NUM_OPTIMIZE_PARAMS_KEYS, key))
         {
            case 0:
               Parms->ExcludeCS      = ExcludeCS;
               Parms->ExcludeCSCount = GetUI32Array(Z_ARRVAL_PP(data), ExcludeCS, MaxCount);
               break;
            case 1:  convert_to_long_ex(data);   Parms->Filter1Bit     = Z_LVAL_PP(data); break;
            case 2:  convert_to_long_ex(data);   Parms->FilterColor    = Z_LVAL_PP(data); break;
            case 3:  convert_to_long_ex(data);   Parms->FilterGray     = Z_LVAL_PP(data); break;
            case 4:  convert_to_long_ex(data);   Parms->JP2KQuality    = Z_LVAL_PP(data); break;
            case 5:  convert_to_long_ex(data);   Parms->JPEGQuality    = Z_LVAL_PP(data); break;
            case 6:  convert_to_long_ex(data);   Parms->Min1BitRes     = Z_LVAL_PP(data); break;
            case 7:  convert_to_long_ex(data);   Parms->MinColorRes    = Z_LVAL_PP(data); break;
            case 8:  convert_to_long_ex(data);   Parms->MinGrayRes     = Z_LVAL_PP(data); break;
            case 9:  convert_to_double_ex(data); Parms->MinLineWidth   = (float)Z_DVAL_PP(data); break;
            case 10: convert_to_long_ex(data);   Parms->Res1BitImages  = Z_LVAL_PP(data); break;
            case 11: convert_to_long_ex(data);   Parms->ResColorImages = Z_LVAL_PP(data); break;
            case 12: convert_to_long_ex(data);   Parms->ResGrayImages  = Z_LVAL_PP(data); break;
            default: break;
         }
      }
   }
}

void GetSigParms(HashTable* Array, struct TPDFSigParms* Parms)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   Parms->StructSize = sizeof(*Parms);
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(SIGPARMS_KEYS, NUM_SIGPARMS_KEYS, key))
         {
            case 0:  convert_to_long_ex(data);   Parms->PKCS7ObjLen  = Z_LVAL_PP(data);   break;
            case 1:  convert_to_long_ex(data);   Parms->HashType     = Z_LVAL_PP(data);   break;
            case 2:  convert_to_string_ex(data); Parms->ContactInfoA = Z_STRVAL_PP(data); break;
            case 3:  convert_to_string_ex(data); Parms->LocationA    = Z_STRVAL_PP(data); break;
            case 4:  convert_to_string_ex(data); Parms->ReasonA      = Z_STRVAL_PP(data); break;
            case 5:  convert_to_string_ex(data); Parms->SignerA      = Z_STRVAL_PP(data); break;
            case 6:  convert_to_long_ex(data);   Parms->Encrypt      = Z_LVAL_PP(data);   break;
            case 7:  convert_to_string_ex(data); Parms->OpenPwd      = Z_STRVAL_PP(data); break;
            case 8:  convert_to_string_ex(data); Parms->OwnerPwd     = Z_STRVAL_PP(data); break;
            case 9:  convert_to_long_ex(data);   Parms->KeyLen       = Z_LVAL_PP(data);   break;
            case 10: convert_to_long_ex(data);   Parms->Restrict     = Z_LVAL_PP(data);   break;
            default: break;
         }
      }
   }
}

UI32 GetStringArray(HashTable* Array, char** List, UI32 MaxCount)
{
   zval **data;
   UI32 index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_string_ex(data);
      List[index++] = Z_STRVAL_PP(data);
      if (index == MaxCount) break;
   }
   return index;
}

void GetTPDFRawImageStruct(HashTable* Array, struct TPDFRawImage* Image)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   memset(Image, 0, sizeof(*Image));
   Image->StructSize = sizeof(*Image);
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         switch(FindKeyEx(RAWIMAGE_KEYS, NUM_RAWIMAGE_KEYS, key))
         {
            case 0:
            {
               convert_to_string_ex(data);
               Image->Buffer  = Z_STRVAL_PP(data);
               Image->BufSize = Z_STRLEN_PP(data);
               break;
            }
            case 1:  convert_to_long_ex(data); Image->BitsPerComponent = Z_LVAL_PP(data); break;
            case 2:  convert_to_long_ex(data); Image->NumComponents    = Z_LVAL_PP(data); break;
            case 3:  convert_to_long_ex(data); Image->CS               = Z_LVAL_PP(data); break;
            case 4:  convert_to_long_ex(data); Image->CSHandle         = Z_LVAL_PP(data); break;
            case 5:  convert_to_long_ex(data); Image->Stride           = Z_LVAL_PP(data); break;
            case 6:  convert_to_long_ex(data); Image->HasAlpha         = Z_LVAL_PP(data); break;
            case 7:  convert_to_long_ex(data); Image->IsBGR            = Z_LVAL_PP(data); break;
            case 8:  convert_to_long_ex(data); Image->MinIsWhite       = Z_LVAL_PP(data); break;
            case 9:  convert_to_long_ex(data); Image->Width            = Z_LVAL_PP(data); break;
            case 10: convert_to_long_ex(data); Image->Height           = Z_LVAL_PP(data); break;
            default: break;
         }
      }
   }
}

SI32 GetTRectLStruct(HashTable* Array, struct TRectL* R)
{
   char *key;
   int key_len;
   ulong index;
   zval **data;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      if (zend_hash_get_current_key_ex(Array, &key, &key_len, &index, 0, &pointer) == HASH_KEY_IS_STRING)
      {
         if (key_len > 0)
         {
            switch(key[0])
            {
               case 'L': index = 0; break;
               case 'T': index = 1; break;
               case 'R': index = 2; break;
               case 'B': index = 3; break;
               default:  return -1;
            }
         }
         convert_to_long_ex(data);
         switch(index)
         {
            case 0:  R->Left   = Z_LVAL_PP(data); break;
            case 1:  R->Top    = Z_LVAL_PP(data); break;
            case 2:  R->Right  = Z_LVAL_PP(data); break;
            case 3:  R->Bottom = Z_LVAL_PP(data); break;
            default: return -2;
         }
      }
   }
   return 0;
}

UI32 GetUI32Array(HashTable* Array, UI32* Value, UI32 MaxCount)
{
   zval **data;
   ulong index = 0;
   HashPosition pointer;
   for(zend_hash_internal_pointer_reset_ex(Array, &pointer); zend_hash_get_current_data_ex(Array, (void**)&data, &pointer) == SUCCESS; zend_hash_move_forward_ex(Array, &pointer))
   {
      convert_to_long_ex(data);
      Value[index++] = Z_LVAL_PP(data);
      if (index == MaxCount) break;
   }
   return index;
}

void ReturnAnnotEx(zval* return_value, struct dynapdf_object *Obj, struct TPDFAnnotationEx* Annot)
{
   add_assoc_long(return_value, "Type",    Annot->Type);
   add_assoc_long(return_value, "Deleted", Annot->Deleted);
   AddPDFRect(return_value, "BBox", &Annot->BBox);
   add_assoc_double(return_value, "BorderWidth", Annot->BorderWidth);
   add_assoc_long(return_value, "BorderColor", Annot->BorderColor);
   add_assoc_long(return_value, "BorderStyle", Annot->BorderStyle);
   add_assoc_long(return_value, "BackColor",   Annot->BackColor);
   add_assoc_long(return_value, "Handle",      Annot->Handle);
   AddString(return_value, "Author",  Annot->AuthorA,  Annot->AuthorW);
   AddString(return_value, "Content", Annot->ContentA, Annot->ContentW);
   AddString(return_value, "Name",    Annot->NameA,    Annot->NameW);
   AddString(return_value, "Subject", Annot->SubjectA, Annot->SubjectW);
   add_assoc_long(return_value, "PageNum",       Annot->PageNum);
   add_assoc_long(return_value, "HighlightMode", Annot->HighlightMode);
   add_assoc_long(return_value, "DestPage",      Annot->DestPage);
   AddPDFRect(return_value, "DestPos", &Annot->DestPos);
   add_assoc_long(return_value, "DestType",      Annot->DestType);
   AddStringA(return_value, "DestFile", Annot->DestFile);
   add_assoc_long(return_value, "Icon",          Annot->Icon);
   AddStringA(return_value, "StampName", Annot->StampName);
   add_assoc_long(return_value, "AnnotFlags",    Annot->AnnotFlags);
   AddStringA(return_value, "CreateDate", Annot->CreateDate);
   AddStringA(return_value, "ModDate",    Annot->ModDate);
   add_assoc_bool(return_value, "Grouped",       Annot->Grouped);
   if (Annot->OC > -1) add_assoc_long(return_value, "OC",   Annot->OC);
   add_assoc_bool(return_value, "Open",          Annot->Open);
   add_assoc_long(return_value, "Parent",        Annot->Parent);
   add_assoc_long(return_value, "PopUp",         Annot->PopUp);
   AddStringA(return_value, "State",      Annot->State);
   AddStringA(return_value, "StateModel", Annot->StateModel);
   add_assoc_long(return_value, "EmbeddedFile",  Annot->EmbeddedFile);
   AddStringA(return_value, "Subtype",    Annot->Subtype);
   add_assoc_long(return_value, "PageIndex",     Annot->PageIndex);
   add_assoc_bool(return_value, "MarkupAnnot",   Annot->MarkupAnnot);
   add_assoc_double(return_value, "Opacity", Annot->Opacity);
   AddFloatArray(return_value, "QuadPoints", Annot->QuadPoints, Annot->QuadPointsCount);
   AddFloatArray(return_value, "DashPattern", Annot->DashPattern, Annot->DashPatternCount);
   AddStringA(return_value, "Intent", Annot->Intent);
   add_assoc_long(return_value, "LE1",           Annot->LE1);
   add_assoc_long(return_value, "LE2",           Annot->LE2);
   AddFloatArray(return_value, "Vertices", Annot->Vertices, Annot->VerticesCount);
   add_assoc_bool(return_value, "Caption",       Annot->Caption);
   add_assoc_double(return_value, "CaptionOffsetX", Annot->CaptionOffsetX);
   add_assoc_double(return_value, "CaptionOffsetY", Annot->CaptionOffsetY);
   add_assoc_long(return_value, "CaptionPos",    Annot->CaptionPos);
   add_assoc_double(return_value, "LeaderLineLen", Annot->LeaderLineLen);
   add_assoc_double(return_value, "LeaderLineExtend", Annot->LeaderLineExtend);
   add_assoc_double(return_value, "LeaderLineOffset", Annot->LeaderLineOffset);
   add_assoc_long(return_value, "BorderEffect",           Annot->BorderEffect);
   AddStringA(return_value, "RichStyle", Annot->RichStyle);
   AddStringA(return_value, "RichText", Annot->RichText);
   AddPointerArray(return_value, Obj, "InkList", (void**)Annot->InkList, Annot->InkListCount, ptInkList);
}

void ReturnColorSpaceStruct(zval* return_value, struct dynapdf_object *Obj, struct TPDFColorSpaceObj* CS)
{
   add_assoc_long(return_value, "Type",      CS->Type);
   add_assoc_long(return_value, "Alternate", CS->Alternate);
   if (CS->IAlternate) AddPointer(return_value, Obj, "IAlternate", CS->IAlternate, ptColorSpace);
   ADD_ASSOC_BIN_STRING(return_value, "Buffer", CS->Buffer, CS->BufSize);
   if (CS->BlackPoint) AddFloatArray(return_value, "BlackPoint", CS->BlackPoint, 3);
   if (CS->WhitePoint) AddFloatArray(return_value, "WhitePoint", CS->WhitePoint, 3);
   if (CS->Gamma)      AddFloatArray(return_value, "Gamma", CS->Gamma, CS->NumInComponents);
   if (CS->Range)
   {
      if (CS->Type == esLab)
         AddFloatArray(return_value, "Range", CS->Range, 4);
      else
         AddFloatArray(return_value, "Range", CS->Range, CS->NumInComponents << 1);
   }
   if (CS->Matrix) AddFloatArray(return_value, "Matrix", CS->Matrix, 9);
   add_assoc_long(return_value, "NumInComponents",  CS->NumInComponents);
   add_assoc_long(return_value, "NumOutComponents", CS->NumOutComponents);
   add_assoc_long(return_value, "NumColors", CS->NumColors);
   AddUTF8StringArray(return_value, "Colorants", (char**)CS->Colorants, CS->ColorantsCount);
   ADD_ASSOC_BIN_STRING(return_value, "Metadata", CS->Metadata, CS->MetadataSize);
   if (CS->IFunction)      AddPointer(return_value, Obj, "IFunction", CS->IFunction, ptFunction);
   if (CS->IAttributes)    AddPointer(return_value, Obj, "IAttributes", CS->IAttributes, ptDeviceNAttr);
   if (CS->IColorSpaceObj) AddPointer(return_value, Obj, "IColorSpaceObj", CS->IColorSpaceObj, ptColorSpace);
}

void ReturnField(zval* return_value, struct TPDFField* Field)
{
   add_assoc_long(return_value, "FieldType", Field->FieldType);
   add_assoc_bool(return_value, "Deleted", Field->Deleted);
   AddPDFRect(return_value, "BBox", &Field->BBox);
   add_assoc_long(return_value, "Handle", Field->Handle);
   if (Field->FieldName)
   {
      if (StrLenA(Field->FieldName) != Field->FieldNameLen)
         AddStringExW(return_value, "FieldName", (const UI16*)Field->FieldName, Field->FieldNameLen);
      else
         AddStringExA(return_value, "FieldName", Field->FieldName, Field->FieldNameLen);
   }
   add_assoc_long(return_value, "BackCS",      Field->BackCS);
   add_assoc_long(return_value, "TextCS",      Field->TextCS);
   add_assoc_long(return_value, "BackColor",   Field->BackColor);
   add_assoc_long(return_value, "BorderColor", Field->BorderColor);
   add_assoc_long(return_value, "TextColor",   Field->TextColor);
   add_assoc_bool(return_value, "Checked",     Field->Checked);
   add_assoc_long(return_value, "Parent",      Field->Parent);
   add_assoc_long(return_value, "KidCount",    Field->KidCount);
   AddStringA(return_value, "Font", Field->Font);
   add_assoc_double(return_value, "FontSize",  Field->FontSize);
   if (Field->Value)
   {
      if (Field->UniVal)
         AddStringExW(return_value, "Value", (const UI16*)Field->Value, Field->ValLen);
      else
         AddStringExA(return_value, "Value", Field->Value, Field->ValLen);
   }
   if (Field->ToolTip)
   {
      if (Field->UniToolTip)
         AddStringExW(return_value, "ToolTip", (const UI16*)Field->ToolTip, Field->ToolTipLen);
      else
         AddStringExA(return_value, "ToolTip", Field->ToolTip, Field->ToolTipLen);
   }
}

void ReturnFieldEx(zval* return_value, struct dynapdf_object *Obj, struct TPDFFieldEx* Field)
{
   add_assoc_bool(return_value, "Deleted",       Field->Deleted);
   AddPDFRect(return_value, "BBox", &Field->BBox);
   add_assoc_long(return_value, "FieldType",     Field->FieldType);
   add_assoc_long(return_value, "GroupType",     Field->GroupType);
   add_assoc_long(return_value, "Handle",        Field->Handle);
   add_assoc_long(return_value, "BackColor",     Field->BackColor);
   add_assoc_long(return_value, "BackColorSP",   Field->BackColorSP);
   add_assoc_long(return_value, "BorderColor",   Field->BorderColor);
   add_assoc_long(return_value, "BorderColorSP", Field->BorderColorSP);
   add_assoc_long(return_value, "BorderStyle",   Field->BorderStyle);
   add_assoc_double(return_value, "BorderWidth", Field->BorderWidth);
   add_assoc_double(return_value, "CharSpacing", Field->CharSpacing);
   add_assoc_bool(return_value, "Checked", Field->Checked);
   add_assoc_long(return_value, "CheckBoxChar",  Field->CheckBoxChar);
   add_assoc_long(return_value, "DefState",      Field->DefState);
   AddString(return_value, "DefValue", Field->DefValueA, Field->DefValueW);
   if (Field->IEditFont) AddPointer(return_value, Obj, "IEditFont", Field->IEditFont, ptFont);
   AddStringA(return_value, "EditFont", Field->EditFont);
   add_assoc_long(return_value, "ExpValCount", Field->ExpValCount);
   AddString(return_value, "ExpValue", Field->ExpValueA, Field->ExpValueW);
   add_assoc_long(return_value, "FieldFlags", Field->FieldFlags);
   if (Field->IFieldFont) AddPointer(return_value, Obj, "IFieldFont", Field->IFieldFont, ptFont);
   AddStringA(return_value, "FieldFont", Field->FieldFont);
   add_assoc_double(return_value, "FontSize", Field->FontSize);
   AddString(return_value, "FieldName", Field->FieldNameA, Field->FieldNameW);
   add_assoc_long(return_value, "HighlightMode", Field->HighlightMode);
   add_assoc_bool(return_value, "IsCalcField",   Field->IsCalcField);
   AddString(return_value, "MapName", Field->MapNameA, Field->MapNameW);
   add_assoc_long(return_value, "MaxLen", Field->MaxLen);
   AddPointerArray(return_value, Obj, "Kids", (void**)Field->Kids, Field->KidCount, ptField);
   if (Field->OC > -1) add_assoc_long(return_value, "OC",   Field->OC);
   if (Field->Parent) AddPointer(return_value, Obj, "Parent", Field->Parent, ptField);
   add_assoc_long(return_value, "PageNum",     Field->PageNum);
   add_assoc_long(return_value, "Rotate",      Field->Rotate);
   add_assoc_long(return_value, "TextAlign",   Field->TextAlign);
   add_assoc_long(return_value, "TextColor",   Field->TextColor);
   add_assoc_long(return_value, "TextColorSP", Field->TextColorSP);
   add_assoc_double(return_value, "TextScaling", Field->TextScaling);
   AddString(return_value, "ToolTip", Field->ToolTipA, Field->ToolTipW);
   AddString(return_value, "UniqueName", Field->UniqueNameA, Field->UniqueNameW);
   AddString(return_value, "Value", Field->ValueA, Field->ValueW);
   add_assoc_double(return_value, "WordSpacing", Field->WordSpacing);
   add_assoc_long(return_value, "PageIndex", Field->PageIndex);
   if (Field->IBarcode)   AddPointer(return_value, Obj, "IBarcode", Field->IBarcode, ptBarcode);
   if (Field->ISignature) AddPointer(return_value, Obj, "ISignature", Field->ISignature, ptSignature);
   add_assoc_long(return_value, "Action", Field->Action);
   add_assoc_long(return_value, "ActionType", Field->ActionType);
   if (Field->Events) AddPointer(return_value, Obj, "Events", Field->Events, ptEvents);
   add_assoc_long(return_value, "FmtTextColor", Field->FmtTextColor);
   AddString(return_value, "FmtText", Field->FmtTextA, Field->FmtTextW);
}

void ReturnFontInfo(zval* return_value, struct TPDFFontInfo* F)
{
   add_assoc_double(return_value, "Ascent", F->Ascent);
   add_assoc_double(return_value, "AvgWidth", F->AvgWidth);
   add_assoc_long(return_value, "BaseEncoding", F->BaseEncoding);
   AddStringA(return_value, "BaseFont", F->BaseFont);
   add_assoc_double(return_value, "CapHeight", F->CapHeight);
   AddStringExA(return_value, "CharSet", F->CharSet, F->CharSetSize);
   AddStringA(return_value, "CIDOrdering", F->CIDOrdering);
   AddStringA(return_value, "CIDRegistry", F->CIDRegistry);
   AddStringExA(return_value, "CIDSet", F->CIDSet, F->CIDSetSize);
   add_assoc_long(return_value, "CIDSupplement", F->CIDSupplement);
   AddStringExA(return_value, "CIDToGIDMap", F->CIDToGIDMap, F->CIDToGIDMapSize);
   AddStringExA(return_value, "CMapBuf", F->CMapBuf, F->CMapBufSize);
   AddStringA(return_value, "CMapName", F->CMapName);
   add_assoc_double(return_value, "Descent", F->Descent);
   AddUI16Array(return_value, "Encoding", F->Encoding, 256);
   add_assoc_long(return_value, "FirstChar", F->FirstChar);
   add_assoc_long(return_value, "Flags", F->Flags);
   AddBBox(return_value, "FontBBox", &F->FontBBox);
   AddStringExA(return_value, "FontBuffer", F->FontBuffer, F->FontBufSize);
   AddString(return_value, "FontFamily", F->FontFamilyA, F->FontFamilyW);
   AddString(return_value, "FontFilePath", F->FontFilePathA, F->FontFilePathW);
   add_assoc_long(return_value, "FontFileType", F->FontFileType);
   AddStringA(return_value, "FontName", F->FontName);
   AddStringA(return_value, "FontStretch", F->FontStretch);
   add_assoc_long(return_value, "FontType", F->FontType);
   add_assoc_double(return_value, "FontWeight", F->FontWeight);
   AddString(return_value, "FullName", F->FullNameA, F->FullNameW);
   add_assoc_bool(return_value, "HaveEncoding", F->HaveEncoding);
   AddFloatArray(return_value, "HorzWidths", F->HorzWidths, F->HorzWidthsCount);
   add_assoc_bool(return_value, "Imported", F->Imported);
   add_assoc_double(return_value, "ItalicAngle", F->ItalicAngle);
   AddStringA(return_value, "Lang", F->Lang);
   add_assoc_long(return_value, "LastChar", F->LastChar);
   add_assoc_double(return_value, "Leading", F->Leading);
   add_assoc_long(return_value, "Length1", F->Length1);
   add_assoc_long(return_value, "Length2", F->Length2);
   add_assoc_long(return_value, "Length3", F->Length3);
   add_assoc_double(return_value, "MaxWidth", F->MaxWidth);
   AddStringExA(return_value, "Metadata", F->Metadata, F->MetadataSize);
   add_assoc_double(return_value, "MisWidth", F->MisWidth);
   AddStringExA(return_value, "Panose", F->Panose, 12);
   AddString(return_value, "PostScriptName", F->PostScriptNameA, F->PostScriptNameW);
   add_assoc_double(return_value, "SpaceWidth", F->SpaceWidth);
   add_assoc_double(return_value, "StemH", F->StemH);
   add_assoc_double(return_value, "StemV", F->StemV);
   AddStringExA(return_value, "ToUnicode", F->ToUnicode, F->ToUnicodeSize);
   AddFltPoint(return_value, "VertDefPos", &F->VertDefPos);
   AddFloatArray(return_value, "VertWidths", (float*)F->VertWidths, F->VertWidthsCount * 3);
   add_assoc_long(return_value, "WMode", F->WMode);
   add_assoc_double(return_value, "XHeight", F->XHeight);
}

void ReturnFontMetrics(zval* return_value, struct TPDFFontMetrics* M)
{
   add_assoc_double(return_value, "Ascent", M->Ascent);
   add_assoc_double(return_value, "AvgWidth", M->AvgWidth);
   add_assoc_double(return_value, "CapHeight", M->CapHeight);
   add_assoc_double(return_value, "DefWidth", M->DefWidth);
   add_assoc_double(return_value, "Descent", M->Descent);
   add_assoc_long(return_value, "FirstChar", M->FirstChar);
   add_assoc_double(return_value, "FixedWidth", M->FixedWidth);
   add_assoc_long(return_value, "FontStyle", M->FontStyle);
   AddBBox(return_value, "FontBBox", &M->FontBBox);
   add_assoc_bool(return_value, "IsFixedPitch", M->IsFixedPitch);
   add_assoc_bool(return_value, "IsSymbolFont", M->IsSymbolFont);
   add_assoc_bool(return_value, "IsStdFont", M->IsStdFont);
   add_assoc_double(return_value, "ItalicAngle", M->ItalicAngle);
   add_assoc_long(return_value, "LastChar", M->LastChar);
   add_assoc_double(return_value, "LineGap", M->LineGap);
   add_assoc_long(return_value, "MacStyle", M->MacStyle);
   add_assoc_long(return_value, "RealStyle", M->RealStyle);

   add_assoc_double(return_value, "StemV", M->StemV);
   add_assoc_double(return_value, "StrikeoutPos", M->StrikeoutPos);
   add_assoc_double(return_value, "StrikeoutWidth", M->StrikeoutWidth);
   add_assoc_double(return_value, "UnderlinePos", M->UnderlinePos);
   add_assoc_double(return_value, "UnderlineWidth", M->UnderlineWidth);
   add_assoc_long(return_value, "UnicodeRange1", M->UnicodeRange1);
   add_assoc_long(return_value, "UnicodeRange2", M->UnicodeRange2);
   add_assoc_long(return_value, "UnicodeRange3", M->UnicodeRange3);
   add_assoc_long(return_value, "UnicodeRange4", M->UnicodeRange4);
   add_assoc_long(return_value, "Weight", M->Weight);
   add_assoc_long(return_value, "WidthClass", M->WidthClass);
   add_assoc_double(return_value, "WinDescent", M->WinDescent);
   add_assoc_long(return_value, "WinStyle", M->WinStyle);
   add_assoc_double(return_value, "XHeight", M->XHeight);
   add_assoc_double(return_value, "TypoAscender", M->TypoAscender);
   add_assoc_double(return_value, "TypoLeading", M->TypoLeading);
}

void ReturnFontObj(zval* return_value, struct TPDFFontObj* F)
{
   add_assoc_double(return_value, "Ascent", F->Ascent);
   AddStringA(return_value, "BaseFont", F->BaseFont);
   add_assoc_double(return_value, "CapHeight", F->CapHeight);
   add_assoc_double(return_value, "Descent",   F->Descent);
   AddUI16Array(return_value, "Encoding", F->Encoding, 256);
   add_assoc_long(return_value, "FirstChar",   F->FirstChar);
   add_assoc_long(return_value, "Flags",       F->Flags);
   if (F->FontFamilyUni)
      AddStringW(return_value, "FontFamily", (const UI16*)F->FontFamily);
   else
      AddStringA(return_value, "FontFamily", F->FontFamily);
   AddStringA(return_value, "FontName", F->FontName);
   add_assoc_long(return_value, "FontType", F->FontType);
   add_assoc_double(return_value, "ItalicAngle", F->ItalicAngle);
   add_assoc_long(return_value, "LastChar", F->LastChar);
   add_assoc_double(return_value, "SpaceWidth", F->SpaceWidth);
   AddFloatArray(return_value, "Widths", F->Widths, F->WidthsCount);
   add_assoc_double(return_value, "XHeight", F->XHeight);
   add_assoc_double(return_value, "DefWidth", F->DefWidth);
   if (F->Length1 > 0)
      ADD_ASSOC_BIN_STRING(return_value, "FontFile", F->FontFile, F->Length1 + F->Length2 + F->Length3);
   else
      AddStringA(return_value, "FontFile", F->FontFile);
   add_assoc_long(return_value, "Length1", F->Length1);
   add_assoc_long(return_value, "Length2", F->Length2);
   add_assoc_long(return_value, "Length3", F->Length3);
   add_assoc_long(return_value, "FontFileType", F->FontFileType);
}

void Return_ImageObj(zval* return_value, struct dynapdf_object *Obj, struct TPDFImage* Image)
{
   if (Image->BufSize > 0) ADD_ASSOC_BIN_STRING(return_value, "Buffer", Image->Buffer, Image->BufSize);
   add_assoc_long(return_value, "Filter", Image->Filter);
   add_assoc_long(return_value, "OrgFilter", Image->OrgFilter);
   if (Image->JBIG2GlobalsSize > 0) ADD_ASSOC_BIN_STRING(return_value, "JBIG2Globals", Image->JBIG2Globals, Image->JBIG2GlobalsSize);
   add_assoc_long(return_value, "BitsPerPixel", Image->BitsPerPixel);
   add_assoc_long(return_value, "ColorSpace", Image->ColorSpace);
   add_assoc_long(return_value, "NumComponents", Image->NumComponents);
   add_assoc_bool(return_value, "MinIsWhite", Image->MinIsWhite);
   if (Image->IColorSpaceObj) AddPointer(return_value, Obj, "IColorSpaceObj", Image->IColorSpaceObj, ptColorSpace);
   if (Image->ColorCount > 0) ADD_ASSOC_BIN_STRING(return_value, "ColorTable", Image->ColorTable, Image->ColorCount);
   add_assoc_long(return_value, "Width", Image->Width);
   add_assoc_long(return_value, "Height", Image->Height);
   add_assoc_long(return_value, "ScanLineLength", Image->ScanLineLength);
   add_assoc_bool(return_value, "InlineImage", Image->InlineImage);
   add_assoc_bool(return_value, "Interpolate", Image->Interpolate);
   add_assoc_bool(return_value, "Transparent", Image->Transparent);
   if (Image->ColorMask) ADD_ASSOC_BIN_STRING(return_value, "ColorMask", Image->ColorMask, Image->NumComponents << 1);
   if (Image->IMaskImage) AddPointer(return_value, Obj, "IMaskImage", Image->IMaskImage, ptImage);
   if (Image->ISoftMask)  AddPointer(return_value, Obj, "ISoftMask", Image->ISoftMask, ptImage);
   if (Image->Decode)     AddFloatArray(return_value, "Decode", Image->Decode, Image->NumComponents << 1);
   add_assoc_long(return_value, "Intent", Image->Intent);
   if (Image->SMaskInData != PDF_MAX_INT)add_assoc_long(return_value, "SMaskInData", Image->SMaskInData);
   if (Image->OC) AddPointer(return_value, Obj, "OC", Image->OC, ptOCG);
   if (Image->MetadataSize > 0) ADD_ASSOC_BIN_STRING(return_value, "Metadata", Image->Metadata, Image->MetadataSize);
   add_assoc_double(return_value, "ResolutionX", Image->ResolutionX);
   add_assoc_double(return_value, "ResolutionY", Image->ResolutionY);
   if (Image->Measure) AddPointer(return_value, Obj, "Measure", Image->Measure, ptMeasure);
   if (Image->PtData)  AddPointer(return_value, Obj, "PtData",  Image->Measure, ptPtData);
   add_assoc_double(return_value, "DestX1", Image->DestX1);
   add_assoc_double(return_value, "DestY1", Image->DestY1);
   add_assoc_double(return_value, "DestX2", Image->DestX2);
   add_assoc_double(return_value, "DestY2", Image->DestY2);
   add_assoc_double(return_value, "DestX3", Image->DestX3);
   add_assoc_double(return_value, "DestY3", Image->DestY3);
   add_assoc_double(return_value, "DestX4", Image->DestX4);
   add_assoc_double(return_value, "DestY4", Image->DestY4);
   add_assoc_double(return_value, "DestWidth", Image->DestWidth);
   add_assoc_double(return_value, "DestHeight", Image->DestHeight);
}

void ReturnOCG(zval* return_value, struct TPDFOCG* F)
{
   add_assoc_long(return_value, "Handle", F->Handle);
   add_assoc_long(return_value, "Intent", F->Intent);
   AddString(return_value, "Name", F->NameA, F->NameW);
   add_assoc_bool(return_value, "HaveContUsage", F->HaveContUsage);
   if (F->HaveContUsage)
   {
      add_assoc_long(return_value, "AppEvents", F->AppEvents);
      add_assoc_long(return_value, "Categories", F->Categories);
   }
}

void ReturnOCGContUsage(zval* return_value, struct TPDFOCGContUsage* F)
{
   if (F->ExportState != PDF_MAX_INT)   add_assoc_long(return_value, "ExportState",    F->ExportState);
   AddString(return_value, "InfoCreator", F->InfoCreatorA, F->InfoCreatorW);
   AddStringA(return_value, "InfoSubtype", F->InfoSubtype);
   AddString(return_value, "Language", F->LanguageA, F->LanguageW);
   if (F->LangPreferred != PDF_MAX_INT) add_assoc_long(return_value, "LangPreferred",  F->LangPreferred);
   if (F->PageElement != peNone)        add_assoc_long(return_value, "PageElement",    F->PageElement);
   if (F->PrintState != PDF_MAX_INT)    add_assoc_long(return_value, "PrintState",     F->PrintState);
   AddStringA(return_value, "PrintSubtype", F->PrintSubtype);
   if (F->UserNamesCount > 0)           add_assoc_long(return_value, "UserNamesCount", F->UserNamesCount);
   if (F->UserType != utNotSet)         add_assoc_long(return_value, "UserType",       F->UserType);
   if (F->ViewState != PDF_MAX_INT)     add_assoc_long(return_value, "ViewState",      F->ViewState);
   if (F->ZoomMin > 0.0f)               add_assoc_double(return_value, "ZoomMin",      F->ZoomMin);
   if (F->ZoomMax > 0.0f)               add_assoc_double(return_value, "ZoomMax",      F->ZoomMax);
}

void ReturnStringA(zval* retval, const char* String)
{
   if (String)
   {
      UI32 len;
      BYTE* buf;
      len = ToUTF8A(String, StrLenA(String), &buf);
      ZVAL_STRINGL(retval, buf, len, 0);
   }
}

void ReturnStringExA(zval* retval, const char* String, UI32 Len)
{
   if (String)
   {
      BYTE* buf = NULL;
      Len = ToUTF8A(String, Len, &buf);
      ZVAL_STRINGL(retval, buf, Len, 0);
   }
}

void ReturnStringW(zval* retval, const UI16* String)
{
   if (String)
   {
      UI32 len;
      BYTE* buf = NULL;
      len = ToUTF8W(String, StrLenW(String), &buf);
      ZVAL_STRINGL(retval, buf, len, 0);
   }
}

void ReturnStringExW(zval* retval, const UI16* String, UI32 Len)
{
   if (String)
   {
      BYTE* buf = NULL;
      Len = ToUTF8W(String, Len, &buf);
      ZVAL_STRINGL(retval, buf, Len, 0);
   }
}

/* ----------------------------------------------------------- CALLBACK FUNCTIONS ------------------------------------------------------------- */

static SI32 PDF_CALL OnFontNotFoundProc(const void* Data, const void* PDFFont, const char* FontName, SI32 Style, SI32 StdFontIndex, LBOOL IsSymbolFont)
{
   TSRMLS_FETCH();
   UI32 fontPtr = 0;
   zval retval, *params[5], param1, param2, param3, param4, param5;
   struct dynapdf_object *obj = (struct dynapdf_object*)Data;

   params[0] = &param1;
   INIT_PZVAL(params[0]);
   params[1] = &param2;
   INIT_PZVAL(params[1]);
   params[2] = &param3;
   INIT_PZVAL(params[2]);
   params[3] = &param4;
   INIT_PZVAL(params[3]);
   params[4] = &param5;
   INIT_PZVAL(params[4]);

   if (PDFFont) fontPtr = AddPointerEx(obj, PDFFont, ptFont);

   ZVAL_LONG(params[0], fontPtr);
   ZVAL_STRING(params[1], (char*)FontName, 0);
   ZVAL_LONG(params[2], Style);
   ZVAL_LONG(params[3], StdFontIndex);
   ZVAL_BOOL(params[4], IsSymbolFont);

   if (call_user_function(CG(function_table), (zval**)NULL, obj->onFontNotFoundProc, &retval, 5, params TSRMLS_CC) == SUCCESS)
   {
      if (Z_TYPE(retval) == IS_LONG)
         return Z_LVAL(retval);
      else
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "The OnFontNotFound callback function must return an integer value!");
         return -1;
      }
   }else
   {
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "Failed to execute the OnFontNotFound callback function!");
      return -1;
   }
}

static void PDF_CALL OnInitProgressProc(const void* Data, TProgType ProgType, SI32 MaxCount)
{
   TSRMLS_FETCH();
   zval retval, *params[2], param1, param2;
   struct dynapdf_object *obj = (struct dynapdf_object*)Data;

   params[0] = &param1;
   INIT_PZVAL(params[0]);
   params[1] = &param2;
   INIT_PZVAL(params[1]);

   ZVAL_LONG(params[0], ProgType);
   ZVAL_LONG(params[1], MaxCount);

   if (call_user_function(CG(function_table), (zval**)NULL, obj->onPageBreakProc, &retval, 2, params TSRMLS_CC) != SUCCESS)
   {
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "Failed to execute the OnInitProgressProc callback function!");
   }
}

static SI32 PDF_CALL OnPageBreakProc(const void* Data, double LastPosX, double LastPosY, LBOOL PageBreak)
{
   TSRMLS_FETCH();
   zval retval, *params[3], param1, param2, param3;
   struct dynapdf_object *obj = (struct dynapdf_object*)Data;

   params[0] = &param1;
   INIT_PZVAL(params[0]);
   params[1] = &param2;
   INIT_PZVAL(params[1]);
   params[2] = &param3;
   INIT_PZVAL(params[2]);

   ZVAL_DOUBLE(params[0], LastPosX);
   ZVAL_DOUBLE(params[1], LastPosY);
   ZVAL_BOOL(params[2], PageBreak);

   if (call_user_function(CG(function_table), (zval**)NULL, obj->onPageBreakProc, &retval, 3, params TSRMLS_CC) == SUCCESS)
   {
      if (Z_TYPE(retval) == IS_LONG)
         return Z_LVAL(retval);
      else
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "The OnPageBreakProc callback function must return an integer value!");
         return -1;
      }
   }else
   {
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "Failed to execute the OnPageBreakProc callback function!");
      return -1;
   }
}

static SI32 PDF_CALL OnProgressProc(const void* Data, SI32 ActivePage)
{
   TSRMLS_FETCH();
   zval retval, *params[1], param1;
   struct dynapdf_object *obj = (struct dynapdf_object*)Data;

   params[0] = &param1;
   INIT_PZVAL(params[0]);

   ZVAL_LONG(params[0], ActivePage);

   if (call_user_function(CG(function_table), (zval**)NULL, obj->onPageBreakProc, &retval, 1, params TSRMLS_CC) == SUCCESS)
   {
      if (Z_TYPE(retval) == IS_LONG)
         return Z_LVAL(retval);
      else
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "The OnPageProgessProc callback function must return an integer value!");
         return -1;
      }
   }else
   {
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "Failed to execute the OnPageProgessProc callback function!");
      return -1;
   }
}

static SI32 PDF_CALL OnReplaceICCProfileProc(const void* Data, TICCProfileType Type, SI32 ColorSpace)
{
   TSRMLS_FETCH();
   zval retval, *params[2], param1, param2;
   struct dynapdf_object *obj = (struct dynapdf_object*)Data;

   params[0] = &param1;
   INIT_PZVAL(params[0]);
   params[1] = &param2;
   INIT_PZVAL(params[1]);

   ZVAL_LONG(params[0], Type);
   ZVAL_LONG(params[1], ColorSpace);

   if (call_user_function(CG(function_table), (zval**)NULL, obj->onReplaceICCProfileProc, &retval, 2, params TSRMLS_CC) == SUCCESS)
   {
      if (Z_TYPE(retval) == IS_LONG)
         return Z_LVAL(retval);
      else
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "The OnReplaceICCProfile callback function must return an integer value!");
         return -1;
      }
   }else
   {
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "Failed to execute the OnReplaceICCProfile callback function!");
      return -1;
   }
}

static SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   TSRMLS_FETCH();
   (void)Data;
   (void)ErrCode;
   (void)ErrType;
   php_error_docref(NULL TSRMLS_CC, E_WARNING, "%s", ErrMessage);
   return 0; // any other return value breaks processing!
}

static SI32 PDF_CALL UserErrorProc(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   TSRMLS_FETCH();
   zval retval, *params[3], param1, param2, param3;
   struct dynapdf_object *obj = (struct dynapdf_object*)Data;

   params[0] = &param1;
   INIT_PZVAL(params[0]);
   params[1] = &param2;
   INIT_PZVAL(params[1]);
   params[2] = &param3;
   INIT_PZVAL(params[2]);

   // Notice: We don't need to create a copy of the error message since no destructor of the zval variable will be executed.
   ZVAL_LONG(params[0], ErrCode);
   ZVAL_STRING(params[1], (char*)ErrMessage, 0);
   ZVAL_LONG(params[2], ErrType);

   if (call_user_function(CG(function_table), (zval**)NULL, obj->onErrorProc, &retval, 3, params TSRMLS_CC) == SUCCESS)
   {
      if (Z_TYPE(retval) == IS_LONG)
         return Z_LVAL(retval);
      else
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "The OnErrorProc callback function must return an integer value!");
         return -1;
      }
   }else
   {
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "%s", ErrMessage);
      php_error_docref(NULL TSRMLS_CC, E_WARNING, "Failed to execute the OnErrorProc callback function!");
      return -1;
   }
}

/* -------------------------------------------------------------------------------------------------------------------------------------------- */

PHP_METHOD(dynapdf, __construct)
{
   struct dynapdf_object* obj;
   obj = (struct dynapdf_object*)zend_object_store_get_object(getThis() TSRMLS_CC);
   if (obj)
   {
      obj->pdf = pdfNewPDF();
      if (!obj->pdf) php_error_docref(NULL TSRMLS_CC, E_ERROR, "Out of memory!");
      pdfSetOnErrorProc(obj->pdf, NULL, PDFError);
      pdfSetGStateFlags(obj->pdf, gfAnsiStringIsUTF8, false);
   }else
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Out of memory!");
}

static void dynapdf_objects_dtor(void *object, zend_object_handle handle TSRMLS_DC)
{
   zend_objects_destroy_object(object, handle TSRMLS_CC);
}

static void dynapdf_free_storage(void* object TSRMLS_DC)
{
   struct dynapdf_object *obj = (struct dynapdf_object*)object;
   pdfDeletePDF(obj->pdf);
   FreePointerList(obj);
   if (obj->pdfStack)                efree(obj->pdfStack);
   if (obj->onErrorProc)             zval_ptr_dtor(&obj->onErrorProc);
   if (obj->onFontNotFoundProc)      zval_ptr_dtor(&obj->onFontNotFoundProc);
   if (obj->onInitProgressProc)      zval_ptr_dtor(&obj->onInitProgressProc);
   if (obj->onPageBreakProc)         zval_ptr_dtor(&obj->onPageBreakProc);
   if (obj->onProgressProc)          zval_ptr_dtor(&obj->onProgressProc);
   if (obj->onReplaceICCProfileProc) zval_ptr_dtor(&obj->onReplaceICCProfileProc);

   zend_object_std_dtor(&obj->zo TSRMLS_CC);
   efree(obj);
}

static zend_object_value dynapdf_create_handler(zend_class_entry *ce TSRMLS_DC)
{
   struct dynapdf_object *obj;
   zend_object_value retval;
   obj = (struct dynapdf_object*)ecalloc(1, sizeof(struct dynapdf_object));

   zend_object_std_init(&obj->zo, ce TSRMLS_CC);
#if PHP_VERSION_ID < 50399
    zend_hash_copy(obj->zo.properties, &ce->default_properties, (copy_ctor_func_t)zval_add_ref, NULL, sizeof(zval*));
#else
    object_properties_init((zend_object*)obj, ce);
#endif

   retval.handle   = zend_objects_store_put(obj, dynapdf_objects_dtor, dynapdf_free_storage, NULL TSRMLS_CC);
   retval.handlers = &dynapdf_object_handlers;

   return retval;
}

//PDF_EXP LBOOL  PDF_CALL pdfActivateAltFontList(const PPDF* IPDF, SI32 Handle, LBOOL EnableSysFonts);
PHP_METHOD(dynapdf, ActivateAltFontList)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, EnableSysFonts;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Handle         = Z_LVAL_PP(argv[0]);
   EnableSysFonts = Z_LVAL_PP(argv[1]);

   rc = pdfActivateAltFontList(pdf, Handle, EnableSysFonts);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfAddActionToObj()
   const void* IPDF, TObjType ObjType, TObjEvent Event, UI32 ActHandle, UI32 ObjHandle); */
PHP_METHOD(dynapdf, AddActionToObj)
{
   zval **argv[4];
   PPDF* pdf;
   TObjType ObjType;
   TObjEvent Event;
   UI32 ActHandle;
   UI32 ObjHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   ObjType   = Z_LVAL_PP(argv[0]);
   Event     = Z_LVAL_PP(argv[1]);
   ActHandle = Z_LVAL_PP(argv[2]);
   ObjHandle = Z_LVAL_PP(argv[3]);

   rc = pdfAddActionToObj(pdf, ObjType, Event, ActHandle, ObjHandle);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfAddAnnotToPage(const PPDF* IPDF, UI32 PageNum, UI32 Handle);
PHP_METHOD(dynapdf, AddAnnotToPage)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 PageNum, Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   PageNum = Z_LVAL_PP(argv[0]);
   Handle  = Z_LVAL_PP(argv[1]);

   rc = pdfAddAnnotToPage(pdf, PageNum, Handle);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfAddArticle()
   const void* IPDF, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, AddArticle)
{
   zval **argv[4];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);

   rc = pdfAddArticle(pdf, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfAddBookmark()
   const void* IPDF, const char* Title, SI32 Parent, UI32 DestPage, LBOOL Open); */
PHP_METHOD(dynapdf, AddBookmark)
{
   zval **argv[4];
   PPDF* pdf;
   char *Title;
   SI32 Parent;
   UI32 DestPage;
   SI32 Open;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   Title    = Z_STRVAL_PP(argv[0]);
   Parent   = Z_LVAL_PP(argv[1]);
   DestPage = Z_LVAL_PP(argv[2]);
   Open     = Z_LVAL_PP(argv[3]);

   rc = pdfAddBookmarkA(pdf, Title, Parent, DestPage, Open);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfAddBookmarkExA(const void* IPDF, const char* Title, SI32 Parent, UI32 NamedDest, LBOOL Open);
PHP_METHOD(dynapdf, AddBookmarkEx)
{
   zval **argv[4];
   PPDF* pdf;
   char *Title;
   SI32 Parent;
   UI32 NamedDest;
   SI32 Open;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   Title     = Z_STRVAL_PP(argv[0]);
   Parent    = Z_LVAL_PP(argv[1]);
   NamedDest = Z_LVAL_PP(argv[2]);
   Open      = Z_LVAL_PP(argv[3]);

   rc = pdfAddBookmarkExA(pdf, Title, Parent, NamedDest, Open);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfAddBookmarkEx2A(const void* IPDF, const char* Title, SI32 Parent, const void* NamedDest, LBOOL Unicode, LBOOL Open);
PHP_METHOD(dynapdf, AddBookmarkEx2)
{
   zval **argv[4];
   PPDF* pdf;
   char *Title;
   SI32  Parent;
   char* NamedDest;
   SI32 Open;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   Title     = Z_STRVAL_PP(argv[0]);
   Parent    = Z_LVAL_PP(argv[1]);
   NamedDest = Z_STRVAL_PP(argv[2]);
   Open      = Z_LVAL_PP(argv[3]);

   rc = pdfAddBookmarkEx2A(pdf, Title, Parent, NamedDest, false, Open);
   RETURN_LONG(rc);
}


/* {{{ proto SI32 pdfAddButtonImage()
   const void* IPDF, UI32 BtnHandle, TButtonState State, const char* Caption, const char* ImgFile); */
PHP_METHOD(dynapdf, AddButtonImage)
{
   zval **argv[4];
   PPDF* pdf;
   UI32 BtnHandle;
   TButtonState State;
   char *Caption;
   char* ImgFile;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   BtnHandle = Z_LVAL_PP(argv[0]);
   State     = Z_LVAL_PP(argv[1]);
   Caption   = Z_STRVAL_PP(argv[2]);
   ImgFile   = Z_STRVAL_PP(argv[3]);

   rc = pdfAddButtonImage(pdf, BtnHandle, State, Caption, ImgFile);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfAddButtonImageEx()
   const void* IPDF, UI32 BtnHandle, TButtonState State, const char* Caption, void* hBitmap); */
PHP_METHOD(dynapdf, AddButtonImageEx)
{
   php_error_docref(NULL TSRMLS_CC, E_ERROR, "This function is not implemented!");
   RETURN_BOOL(0);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfAddButtonImageEx2(const PPDF* IPDF, UI32 BtnHandle, TButtonState State, const char* Caption, const void* Buffer, UI32 BufSize);
PHP_METHOD(dynapdf, AddButtonImageEx2)
{
   zval **argv[4];
   PPDF* pdf;
   UI32 BtnHandle;
   TButtonState State;
   char *Caption;
   char* Buffer;
   SI32 BufSize;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   BtnHandle = Z_LVAL_PP(argv[0]);
   State     = Z_LVAL_PP(argv[1]);
   Caption   = Z_STRVAL_PP(argv[2]);
   Buffer    = Z_STRVAL_PP(argv[3]);
   BufSize   = Z_STRLEN_PP(argv[3]);

   rc = pdfAddButtonImageEx2(pdf, BtnHandle, State, Caption, Buffer, BufSize);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfAddContinueText()
   const void* IPDF, const char* AText); */
PHP_METHOD(dynapdf, AddContinueText)
{
   SI32 rc;
   zval **arg1;
   PPDF* pdf;
   char *AText;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   AText = Z_STRVAL_PP(arg1);

   rc = pdfAddContinueTextA(pdf, AText);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfAddDeviceNProcessColorants(const void* IPDF, UI32 DeviceNCS, const char** Colorants, UI32 NumColorants, TExtColorSpace ProcessCS, SI32 Handle);
PHP_METHOD(dynapdf, AddDeviceNProcessColorants)
{
   SI32 rc;
   zval **argv[4];
   PPDF* pdf;
   HashTable *arr_hash;
   SI32 ProcessCS, Handle, DeviceNCS, NumColorants;
   char* Colorants[32];
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();

   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   DeviceNCS = Z_LVAL_PP(argv[0]);
   arr_hash  = Z_ARRVAL_PP(argv[1]);
   if ((NumColorants = GetColorants(arr_hash, Colorants)) == 0) WRONG_PARAM_COUNT;
   ProcessCS = Z_LVAL_PP(argv[2]);
   Handle    = Z_LVAL_PP(argv[3]);

   rc = pdfAddDeviceNProcessColorants(pdf, DeviceNCS, (const char**)Colorants, NumColorants, ProcessCS, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfAddDeviceNSeparations(const void* IPDF, UI32 DeviceNCS, const char** Colorants, UI32* SeparationCS, UI32 NumColorants);
PHP_METHOD(dynapdf, AddDeviceNSeparations)
{
   SI32 rc;
   zval **argv[3];
   PPDF* pdf;
   HashTable *arr_hash1, *arr_hash2;
   UI32 SeparationCS[32], NumColorants;
   SI32 DeviceNCS;
   char* Colorants[32];
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();

   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   convert_to_array_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   DeviceNCS = Z_LVAL_PP(argv[0]);
   arr_hash1 = Z_ARRVAL_PP(argv[1]);
   arr_hash2 = Z_ARRVAL_PP(argv[2]);
   if ((NumColorants = GetColorants(arr_hash1, Colorants)) == 0) WRONG_PARAM_COUNT;
   if (GetUI32Array(arr_hash2, SeparationCS, 32) != NumColorants) WRONG_PARAM_COUNT;

   rc = pdfAddDeviceNSeparations(pdf, DeviceNCS, (const char**)Colorants, SeparationCS, NumColorants);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfAddFieldToFormAction()
   const void* IPDF, UI32 Action, UI32 AField, LBOOL Include); */
PHP_METHOD(dynapdf, AddFieldToFormAction)
{
   zval **argv[3];

   PPDF* pdf;
   UI32 Action, AField;
   SI32 Include;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   Action  = Z_LVAL_PP(argv[0]);
   AField  = Z_LVAL_PP(argv[1]);
   Include = Z_LVAL_PP(argv[2]);

   rc = pdfAddFieldToFormAction(pdf, Action, AField, Include);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfAddFieldToHideAction()
   const void* IPDF, UI32 HideAct, UI32 AField); */
PHP_METHOD(dynapdf, AddFieldToHideAction)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 HideAct, AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   HideAct = Z_LVAL_PP(argv[0]);
   AField  = Z_LVAL_PP(argv[1]);

   rc = pdfAddFieldToHideAction(pdf, HideAct, AField);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfAddFileComment()
   const void* IPDF, const char* AText); */
PHP_METHOD(dynapdf, AddFileComment)
{
   SI32 rc;
   zval **arg1;

   PPDF* pdf;
   char *AText;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   AText = Z_STRVAL_PP(arg1);

   rc = pdfAddFileCommentA(pdf, AText);
   RETURN_BOOL(rc);
}
/* }}} */



/* {{{ proto SI32 pdfAddFontSearchPath()
   const void* IPDF, const char* APath, SI32 Recursive); */
PHP_METHOD(dynapdf, AddFontSearchPath)
{
   zval **argv[2];
   PPDF* pdf;
   char *APath;
   SI32 Recursive;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   APath     = Z_STRVAL_PP(argv[0]);
   Recursive = Z_LVAL_PP(argv[1]);

   rc = pdfAddFontSearchPathA(pdf, APath, Recursive);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfAddHeaderFooter(const PPDF* IPDF, struct TPDFHeaderFooter* Value, struct TPDFHdrFtr* HFArray, UI32 Count);
PHP_METHOD(dynapdf, AddHeaderFooter)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 rc, numValues;
   HashTable* arrValue, *arrHFArray;
   struct TPDFHeaderFooter Value;
   struct TPDFHdrFtr HFArray[6];
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   arrValue   = Z_ARRVAL_PP(argv[0]);
   arrHFArray = Z_ARRVAL_PP(argv[1]);

   Value.StructSize = sizeof(Value);
   pdfInitHeaderFooter(&Value, HFArray, 6);
   
   GetHeaderFooterStruct(arrValue, &Value);
   numValues = GetHdrFtrArray(arrHFArray, HFArray);

   rc = pdfAddHeaderFooter(pdf, &Value, HFArray, numValues);
   RETURN_BOOL(rc);
}

// PDF_EXP LBOOL  PDF_CALL pdfAddInkList(const PPDF* IPDF, UI32 InkAnnot, struct TFltPoint* Points, UI32 NumPoints);
PHP_METHOD(dynapdf, AddInkList)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   SI32 rc, InkAnnot, NumPoints;
   HashTable *arr_hash;
   struct TFltPoint* Points = NULL;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   InkAnnot = Z_LVAL_PP(argv[0]);
   arr_hash = Z_ARRVAL_PP(argv[1]);

   NumPoints = zend_hash_num_elements(arr_hash) >> 1;
   if (NumPoints > 0)
   {
      Points = emalloc(NumPoints * sizeof(struct TFltPoint));
      GetFloatArray(arr_hash, (float*)Points, NumPoints << 1);
   }
   rc = pdfAddInkList(obj->pdf, InkAnnot, Points, NumPoints);
   efree(Points);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfAddJavaScript()
   const void* IPDF, const char* Name, const char* Script); */
PHP_METHOD(dynapdf, AddJavaScript)
{
   zval **argv[2];
   PPDF* pdf;
   char *Name, *Script;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Name   = Z_STRVAL_PP(argv[0]);
   Script = Z_STRVAL_PP(argv[1]);

   rc = pdfAddJavaScriptA(pdf, Name, Script);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LGRP*  PDF_CALL pdfAddLayerToDisplTreeA(const void* IPDF, LGRP* Parent, SI32 Layer, const char* Title);
PHP_METHOD(dynapdf, AddLayerToDisplTree)
{
   zval **argv[3];
   struct dynapdf_object *obj;
   LGRP* rc, *Parent = NULL;
   char *Title;
   UI32 hash;
   SI32 Layer;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   hash  = Z_LVAL_PP(argv[0]);
   Layer = Z_LVAL_PP(argv[1]);
   Title = Z_STRVAL_PP(argv[2]);

   if (hash && (Parent = (LGRP*)GetPointer(obj, hash, ptLayer)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Parent must be 0 or a valid safe pointer!");
      RETURN_NULL();
   }

   if ((rc = pdfAddLayerToDisplTreeA(obj->pdf, Parent, Layer, Title)) != NULL)
   {
      hash = AddPointerEx(obj, rc, ptLayer);
      RETURN_LONG(hash);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfAddMaskImage(const void* IPDF, UI32 BaseImage, const void* Buffer, UI32 BufSize, SI32 Stride, UI32 BitsPerPixel, UI32 Width, UI32 Height);
PHP_METHOD(dynapdf, AddMaskImage)
{
   zval **argv[6];
   PPDF* pdf;
   char* Buffer;
   SI32 rc, BaseImage, BufSize, Stride, BitsPerPixel, Width, Height;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   BaseImage    = Z_LVAL_PP(argv[0]);
   Buffer       = Z_STRVAL_PP(argv[1]);
   BufSize      = Z_STRLEN_PP(argv[1]);
   Stride       = Z_LVAL_PP(argv[2]);
   BitsPerPixel = Z_LVAL_PP(argv[3]);
   Width        = Z_LVAL_PP(argv[4]);
   Height       = Z_LVAL_PP(argv[5]);

   rc = pdfAddMaskImage(pdf, BaseImage, Buffer, BufSize, Stride, BitsPerPixel, Width, Height);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfAddObjectToLayer(const void* IPDF, UI32 OCG, TOCObject ObjType, UI32 Handle);
PHP_METHOD(dynapdf, AddObjectToLayer)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 rc, OCG, ObjType, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   OCG     = Z_LVAL_PP(argv[0]);
   ObjType = Z_LVAL_PP(argv[1]);
   Handle  = Z_LVAL_PP(argv[2]);

   rc = pdfAddObjectToLayer(pdf, OCG, ObjType, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfAddOCGToAppEvent(const PPDF* IPDF, UI32 Handle, TOCAppEvent Events, TOCGUsageCategory Categories);
PHP_METHOD(dynapdf, AddOCGToAppEvent)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 rc, Handle, Events, Categories;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Handle     = Z_LVAL_PP(argv[0]);
   Events     = Z_LVAL_PP(argv[1]);
   Categories = Z_LVAL_PP(argv[2]);

   rc = pdfAddOCGToAppEvent(pdf, Handle, Events, Categories);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfAddOutputIntentA(const void* IPDF, const char* ICCFile);
PHP_METHOD(dynapdf, AddOutputIntent)
{
   zval **arg1;
   PPDF* pdf;
   char *ICCFile;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);

   ICCFile = Z_STRVAL_PP(arg1);

   rc = pdfAddOutputIntentA(pdf, ICCFile);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfAddPageLabelA(const void* IPDF, UI32 StartRange, TPageLabelFormat Format, const char* Prefix, SI32 FirstPageNum);
PHP_METHOD(dynapdf, AddPageLabel)
{
   zval **argv[4];
   PPDF* pdf;
   char *Prefix;
   SI32 rc, StartRange, Format, FirstPageNum;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   StartRange   = Z_LVAL_PP(argv[0]);
   Format       = Z_LVAL_PP(argv[1]);
   Prefix       = Z_STRVAL_PP(argv[2]);
   FirstPageNum = Z_LVAL_PP(argv[3]);

   rc = pdfAddPageLabelA(pdf, StartRange, Format, Prefix, FirstPageNum);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfAddValToChoiceField()
   const void* IPDF, UI32 Field, const char* ExpValue, const char* Value, LBOOL Selected); */
PHP_METHOD(dynapdf, AddValToChoiceField)
{
   zval **argv[4];
   PPDF* pdf;
   UI32 Field;
   char *ExpValue, *Value;
   SI32 Selected;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   Field     = Z_LVAL_PP(argv[0]);
   ExpValue  = Z_STRVAL_PP(argv[1]);
   Value     = Z_STRVAL_PP(argv[2]);
   Selected  = Z_LVAL_PP(argv[3]);

   rc = pdfAddValToChoiceFieldA(pdf, Field, ExpValue, Value, Selected);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfAppend()
   const void* IPDF); */
PHP_METHOD(dynapdf, Append)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfAppend(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfApplyAppEvent(const PPDF* IPDF, TOCAppEvent Event, LBOOL SaveResult);
PHP_METHOD(dynapdf, ApplyAppEvent)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Event, SaveResult;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Event      = Z_LVAL_PP(argv[0]);
   SaveResult = Z_LVAL_PP(argv[1]);

   rc = pdfApplyAppEvent(pdf, Event, SaveResult);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfApplyPattern()
   const void* IPDF, SI32 PattHandle, TColorMode ColorMode, UI32 Color); */
PHP_METHOD(dynapdf, ApplyPattern)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 PattHandle;
   TColorMode ColorMode;
   UI32 Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   PattHandle = Z_LVAL_PP(argv[0]);
   ColorMode  = Z_LVAL_PP(argv[1]);
   Color      = Z_LVAL_PP(argv[2]);

   rc = pdfApplyPattern(pdf, PattHandle, ColorMode, Color);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfApplyShading()
   const void* IPDF, SI32 ShadHandle); */
PHP_METHOD(dynapdf, ApplyShading)
{
   zval **arg1;
   PPDF* pdf;
   SI32 ShadHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   ShadHandle = Z_LVAL_PP(arg1);

   rc = pdfApplyShading(pdf, ShadHandle);
   RETURN_BOOL(rc);
}
/* }}} */

// PDF_EXP LBOOL  PDF_CALL pdfAssociateEmbFile(const void* IPDF, TAFDestObject DestObject, SI32 DestHandle, TAFRelationship Relationship, UI32 EmbFile);
PHP_METHOD(dynapdf, AssociateEmbFile)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, DestObject, DestHandle, Relationship, EmbFile;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   DestObject   = Z_LVAL_PP(argv[0]);
   DestHandle   = Z_LVAL_PP(argv[1]);
   Relationship = Z_LVAL_PP(argv[2]);
   EmbFile      = Z_LVAL_PP(argv[3]);

   rc = pdfAssociateEmbFile(pdf, DestObject, DestHandle, Relationship, EmbFile);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfAttachFileA(const void* IPDF, const char* FilePath, const char* Description, LBOOL Compress);
PHP_METHOD(dynapdf, AttachFile)
{
   zval **argv[3];
   PPDF* pdf;
   char* FilePath, *Description;
   SI32 rc, Compress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   FilePath    = Z_STRVAL_PP(argv[0]);
   Description = Z_STRVAL_PP(argv[1]);
   Compress    = Z_LVAL_PP(argv[2]);

   rc = pdfAttachFileA(pdf, FilePath, Description, Compress);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfAttachFileExA(const void* IPDF, const void* Buffer, UI32 BufSize, const char* FileName, const char* Description, LBOOL Compress);
PHP_METHOD(dynapdf, AttachFileEx)
{
   zval **argv[4];
   PPDF* pdf;
   char* Buffer, *FileName, *Description;
   SI32 rc, BufSize, Compress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Buffer      = Z_STRVAL_PP(argv[0]);
   BufSize     = Z_STRLEN_PP(argv[0]);
   FileName    = Z_STRVAL_PP(argv[1]);
   Description = Z_STRVAL_PP(argv[2]);
   Compress    = Z_LVAL_PP(argv[3]);

   rc = pdfAttachFileExA(pdf, Buffer, BufSize, FileName, Description, Compress);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfAutoTemplate()
   const void* IPDF, UI32 Templ, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, AutoTemplate)
{
   zval **argv[4];
   PPDF* pdf;
   UI32 Templ;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   Templ  = Z_LVAL_PP(argv[0]);
   PosX   = Z_DVAL_PP(argv[1]);
   PosY   = Z_DVAL_PP(argv[2]);
   Width  = Z_DVAL_PP(argv[3]);
   Height = Z_DVAL_PP(argv[4]);

   rc = pdfAutoTemplate(pdf, Templ, PosX, PosY, Width, Height);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfBeginClipPath()
   const void* IPDF); */
PHP_METHOD(dynapdf, BeginClipPath)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfBeginClipPath(pdf);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfBeginContinueText()
   const void* IPDF, double PosX, double PosY); */
PHP_METHOD(dynapdf, BeginContinueText)
{
   zval **argv[2];
   PPDF* pdf;
   double PosX, PosY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);

   rc = pdfBeginContinueText(pdf, PosX, PosY);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfBeginLayer(const void* IPDF, UI32 OCG);
PHP_METHOD(dynapdf, BeginLayer)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, OCG;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);

   OCG = Z_LVAL_PP(arg1);

   rc = pdfBeginLayer(pdf, OCG);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfBeginPageTemplate(const void* IPDF, const char* Name, LBOOL UseAutoTemplates);
PHP_METHOD(dynapdf, BeginPageTemplate)
{
   zval **argv[2];
   PPDF* pdf;
   char* Name;
   SI32 rc, UseAutoTemplates;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Name             = Z_STRVAL_PP(argv[0]);
   UseAutoTemplates = Z_LVAL_PP(argv[1]);

   rc = pdfBeginPageTemplate(pdf, Name, UseAutoTemplates);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfBeginPattern()
   const void* IPDF, TPatternType PatternType, TTilingType TilingType, double Width, double Height); */
PHP_METHOD(dynapdf, BeginPattern)
{
   zval **argv[4];
   PPDF* pdf;
   TPatternType PatternType;
   TTilingType TilingType;
   double Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   PatternType = Z_LVAL_PP(argv[0]);
   TilingType  = Z_LVAL_PP(argv[1]);
   Width       = Z_DVAL_PP(argv[2]);
   Height      = Z_DVAL_PP(argv[3]);

   rc = pdfBeginPattern(pdf, PatternType, TilingType, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfBeginTemplate()
   const void* IPDF, double Width, double Height); */
PHP_METHOD(dynapdf, BeginTemplate)
{
   zval **argv[2];
   PPDF* pdf;
   double Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   Width       = Z_DVAL_PP(argv[0]);
   Height      = Z_DVAL_PP(argv[1]);

   rc = pdfBeginTemplate(pdf, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfBeginTemplateEx(const void* IPDF, struct TPDFRect* BBox, struct TCTM* Matrix);
PHP_METHOD(dynapdf, BeginTemplateEx)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr_hash1, *arr_hash2;
   SI32 rc;
   struct TPDFRect BBox = {0.0, 0.0, 0.0, 0.0};
   struct TCTM Matrix = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   arr_hash1 = Z_ARRVAL_PP(argv[0]);
   arr_hash2 = Z_ARRVAL_PP(argv[1]);

   if (GetBBoxFromArray(arr_hash1, &BBox))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"BBox\" must contain values for Left, Bottom, Right, Top!");
      RETURN_LONG(-1);
   }
   if (GetCTMFromArray(arr_hash2, &Matrix))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"Matrix\" must be an array of six numbers (a, b, c, d, x, y)!");
      RETURN_LONG(-1);
   }

   rc = pdfBeginTemplateEx(pdf, &BBox, &Matrix);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfBeginTransparencyGroup(const void* IPDF, double x1, double y1, double x2, double y2, LBOOL Isolated, LBOOL Knockout, TExtColorSpace CS, SI32 CSHandle);
PHP_METHOD(dynapdf, BeginTransparencyGroup)
{
   zval **argv[8];
   PPDF* pdf;
   double x1, y1, x2, y2;
   SI32 rc, Isolated, Knockout, CS, CSHandle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);

   x1       = Z_DVAL_PP(argv[0]);
   y1       = Z_DVAL_PP(argv[1]);
   x2       = Z_DVAL_PP(argv[2]);
   y2       = Z_DVAL_PP(argv[3]);
   Isolated = Z_LVAL_PP(argv[4]);
   Knockout = Z_LVAL_PP(argv[5]);
   CS       = Z_LVAL_PP(argv[6]);
   CSHandle = Z_LVAL_PP(argv[7]);

   rc = pdfBeginTransparencyGroup(pdf, x1, y1, x2, y2, Isolated, Knockout, CS, CSHandle);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfBezier_1_2_3()
   const void* IPDF, double x1, double y1, double x2, double y2, double x3, double y3); */
PHP_METHOD(dynapdf, Bezier_1_2_3)
{
   zval **argv[6];
   PPDF* pdf;
   double x1, y1, x2, y2, x3, y3;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   x1 = Z_DVAL_PP(argv[0]);
   y1 = Z_DVAL_PP(argv[1]);
   x2 = Z_DVAL_PP(argv[2]);
   y2 = Z_DVAL_PP(argv[3]);
   x3 = Z_DVAL_PP(argv[4]);
   y3 = Z_DVAL_PP(argv[5]);

   rc = pdfBezier_1_2_3(pdf, x1, y1, x2, y2, x3, y3);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfBezier_1_3()
   const void* IPDF, double x1, double y1, double x3, double y3); */
PHP_METHOD(dynapdf, Bezier_1_3)
{
   zval **argv[4];
   PPDF* pdf;
   double x1, y1, x3, y3;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   x1 = Z_DVAL_PP(argv[0]);
   y1 = Z_DVAL_PP(argv[1]);
   x3 = Z_DVAL_PP(argv[2]);
   y3 = Z_DVAL_PP(argv[3]);

   rc = pdfBezier_1_3(pdf, x1, y1, x3, y3);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfBezier_2_3()
   const void* IPDF, double x2, double y2, double x3, double y3); */
PHP_METHOD(dynapdf, Bezier_2_3)
{
   zval **argv[4];
   PPDF* pdf;
   double x2, y2, x3, y3;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   x2 = Z_DVAL_PP(argv[0]);
   y2 = Z_DVAL_PP(argv[1]);
   x3 = Z_DVAL_PP(argv[2]);
   y3 = Z_DVAL_PP(argv[3]);

   rc = pdfBezier_2_3(pdf, x2, y2, x3, y3);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL rasCalcPagePixelSize(IPGE* PagePtr, TPDFPageScale DefScale, float Scale, UI32 FrameWidth, UI32 FrameHeight, TRasterFlags Flags, UI32* Width, UI32* Height);
PHP_METHOD(dynapdf, CalcPagePixelSize)
{
   zval **argv[6];
   PPDF* pdf;
   float Scale;
   IPGE* pageObject;
   SI32 PageNum, DefScale, FrameWidth, FrameHeight, Flags;
   UI32 Width = 0, Height = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   PageNum     = Z_LVAL_PP(argv[0]);
   DefScale    = Z_LVAL_PP(argv[1]);
   Scale       = (float)Z_DVAL_PP(argv[2]);
   FrameWidth  = Z_LVAL_PP(argv[3]);
   FrameHeight = Z_LVAL_PP(argv[4]);
   Flags       = Z_LVAL_PP(argv[5]);

   if ((pageObject = pdfGetPageObject(pdf, PageNum)) == NULL) RETURN_NULL();

   array_init(return_value);
   rasCalcPagePixelSize(pageObject, DefScale, Scale, FrameWidth, FrameHeight, Flags, &Width, &Height);

   add_assoc_double(return_value, "Width", Width);
   add_assoc_double(return_value, "Height", Height);
}

/* {{{ proto double pdfCalcWidthHeight()
   const void* IPDF, double OrgWidth, double OrgHeight, double ScaledWidth, double ScaledHeight); */
PHP_METHOD(dynapdf, CalcWidthHeight)
{
   zval **argv[4];
   PPDF* pdf;
   double OrgWidth, OrgHeight, ScaledWidth, ScaledHeight;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   OrgWidth     = Z_DVAL_PP(argv[0]);
   OrgHeight    = Z_DVAL_PP(argv[1]);
   ScaledWidth  = Z_DVAL_PP(argv[2]);
   ScaledHeight = Z_DVAL_PP(argv[3]);

   drc = pdfCalcWidthHeight(pdf, OrgWidth, OrgHeight, ScaledWidth, ScaledHeight);
   RETURN_DOUBLE(drc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfCaretAnnotA(const PPDF* IPDF, double PosX, double PosY, double Width, double Height, UI32 Color, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Content);
PHP_METHOD(dynapdf, CaretAnnot)
{
   zval **argv[9];
   PPDF* pdf;
   char* Author, *Subject, *Content;
   double PosX, PosY, Width, Height;
   SI32 rc, Color, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_string_ex(argv[7]);
   convert_to_string_ex(argv[8]);

   PosX    = Z_DVAL_PP(argv[0]);
   PosY    = Z_DVAL_PP(argv[1]);
   Width   = Z_DVAL_PP(argv[2]);
   Height  = Z_DVAL_PP(argv[3]);
   Color   = Z_LVAL_PP(argv[4]);
   CS      = Z_LVAL_PP(argv[5]);
   Author  = Z_STRVAL_PP(argv[6]);
   Subject = Z_STRVAL_PP(argv[7]);
   Content = Z_STRVAL_PP(argv[8]);

   rc = pdfCaretAnnotA(pdf, PosX, PosY, Width, Height, Color, CS, Author, Subject, Content);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfChangeAnnotNameA(const void* IPDF, UI32 Handle, const char* Name);
PHP_METHOD(dynapdf, ChangeAnnotName)
{
   zval **argv[2];
   PPDF* pdf;
   char* Name;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Name   = Z_STRVAL_PP(argv[1]);

   rc = pdfChangeAnnotNameA(pdf, Handle, Name);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfChangeAnnotPos(const void* IPDF, UI32 Handle, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, ChangeAnnotPos)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   Handle = Z_LVAL_PP(argv[0]);
   PosX   = Z_DVAL_PP(argv[1]);
   PosY   = Z_DVAL_PP(argv[2]);
   Width  = Z_DVAL_PP(argv[3]);
   Height = Z_DVAL_PP(argv[4]);

   rc = pdfChangeAnnotPos(pdf, Handle, PosX, PosY, Width, Height);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfChangeBookmark()
   const void* IPDF, SI32 ABmk, const char* Title, SI32 Parent, UI32 DestPage, SI32 Open); */
PHP_METHOD(dynapdf, ChangeBookmark)
{
   zval **argv[5];
   PPDF* pdf;
   SI32 ABmk;
   char* Title;
   SI32 Parent, DestPage, Open;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   ABmk     = Z_LVAL_PP(argv[0]);
   Title    = Z_STRVAL_PP(argv[1]);
   Parent   = Z_LVAL_PP(argv[2]);
   DestPage = Z_LVAL_PP(argv[3]);
   Open     = Z_LVAL_PP(argv[4]);

   rc = pdfChangeBookmarkA(pdf, ABmk, Title, Parent, DestPage, Open);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfChangeFont()
   const void* IPDF, SI32 AHandle); */
PHP_METHOD(dynapdf, ChangeFont)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AHandle     = Z_LVAL_PP(arg1);

   rc = pdfChangeFont(pdf, AHandle);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfChangeFontEx(const PPDF* IPDF, SI32 Handle, double FontSize, TFStyle Style);
PHP_METHOD(dynapdf, ChangeFontEx)
{
   zval **argv[3];
   PPDF* pdf;
   double FontSize;
   SI32 Handle, Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Handle   = Z_LVAL_PP(argv[0]);
   FontSize = Z_DVAL_PP(argv[1]);
   Style    = Z_LVAL_PP(argv[2]);

   rc = pdfChangeFontEx(pdf, Handle, FontSize, Style);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfChangeFontSize()
   const void* IPDF, double Size); */
PHP_METHOD(dynapdf, ChangeFontSize)
{
   zval **arg1;
   PPDF* pdf;
   double Size;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Size   = Z_DVAL_PP(arg1);

   rc = pdfChangeFontSize(pdf, Size);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfChangeFontStyle()
   const void* IPDF, TFStyle Style); */
PHP_METHOD(dynapdf, ChangeFontStyle)
{
   zval **arg1;
   PPDF* pdf;
   TFStyle Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Style = Z_LVAL_PP(arg1);

   rc = pdfChangeFontStyle(pdf, Style);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfChangeFontStyleEx(const void* IPDF, TFStyle Style);
PHP_METHOD(dynapdf, ChangeFontStyleEx)
{
   zval **arg1;
   PPDF* pdf;
   TFStyle Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Style = Z_LVAL_PP(arg1);

   rc = pdfChangeFontStyleEx(pdf, Style);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfChangeJavaScript()
   const void* IPDF, UI32 AHandle, const char* NewScript); */
PHP_METHOD(dynapdf, ChangeJavaScript)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AHandle;
   char* NewScript;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   AHandle   = Z_LVAL_PP(argv[0]);
   NewScript = Z_STRVAL_PP(argv[1]);

   rc = pdfChangeJavaScriptA(pdf, AHandle, NewScript);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfChangeJavaScriptAction()
   const void* IPDF, UI32 Handle, const char* NewScript); */
PHP_METHOD(dynapdf, ChangeJavaScriptAction)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Handle;
   char* NewScript;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Handle    = Z_LVAL_PP(argv[0]);
   NewScript = Z_STRVAL_PP(argv[1]);

   rc = pdfChangeJavaScriptActionA(pdf, Handle, NewScript);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfChangeJavaScriptNameA(const void* IPDF, UI32 Handle, const char* Name);
PHP_METHOD(dynapdf, ChangeJavaScriptName)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Handle;
   char* Name;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Handle = Z_LVAL_PP(argv[0]);
   Name   = Z_STRVAL_PP(argv[1]);

   rc = pdfChangeJavaScriptNameA(pdf, Handle, Name);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfChangeLinkAnnot(const void* IPDF, UI32 Handle, const char* URL);
PHP_METHOD(dynapdf, ChangeLinkAnnot)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Handle;
   char* URL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if (argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Handle = Z_LVAL_PP(argv[0]);
   URL    = Z_STRVAL_PP(argv[1]);

   rc = pdfChangeLinkAnnot(pdf, Handle, URL);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfChangeNamedDestA(const PPDF* IPDF, TObjType ObjType, UI32 Handle, const char* NewDestName);
PHP_METHOD(dynapdf, ChangeNamedDest)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 ObjType, Handle;
   char* NewDestName;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if (argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   ObjType     = Z_LVAL_PP(argv[0]);
   Handle      = Z_LVAL_PP(argv[1]);
   NewDestName = Z_STRVAL_PP(argv[2]);

   rc = pdfChangeNamedDestA(pdf, ObjType, Handle, NewDestName);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfChangeOCGNameA(const PPDF* IPDF, UI32 Handle, const char* Value);
PHP_METHOD(dynapdf, ChangeOCGName)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Handle;
   char* Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if (argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Handle = Z_LVAL_PP(argv[0]);
   Value  = Z_STRVAL_PP(argv[1]);

   rc = pdfChangeOCGNameA(pdf, Handle, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfChangeSeparationColor(const void* IPDF, UI32 CSHandle, UI32 NewColor, TExtColorSpace Alternate, SI32 AltHandle);
PHP_METHOD(dynapdf, ChangeSeparationColor)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, CSHandle, NewColor, Alternate, AltHandle;
   int argc = ZEND_NUM_ARGS();

   if (argc < 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   if (argc > 3)
   {
      convert_to_long_ex(argv[3]);
      AltHandle = Z_LVAL_PP(argv[3]);
   }else
      AltHandle = -1;

   CSHandle  = Z_LVAL_PP(argv[0]);
   NewColor  = Z_LVAL_PP(argv[1]);
   Alternate = Z_LVAL_PP(argv[2]);

   rc = pdfChangeSeparationColor(pdf, CSHandle, NewColor, Alternate, AltHandle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCheckCollection(const void* IPDF);
PHP_METHOD(dynapdf, CheckCollection)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCheckCollection(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCheckConformance(const void* IPDF, TConformanceType Type, TCheckOptions Options, const void* UserData, TOnFontNotFoundProc* OnFontNotFound, TOnReplaceICCProfile* OnReplaceICCProfile);
PHP_METHOD(dynapdf, CheckConformance)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   char* fontNotFound = NULL, *replaceICCProfile = NULL;
   TOnFontNotFoundProc* fontProcPtr = NULL;
   TOnReplaceICCProfile* iccProcPtr = NULL;
   SI32 rc, Type, Options;
   int argc = ZEND_NUM_ARGS();

   if (argc < 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Type    = Z_LVAL_PP(argv[0]);
   Options = Z_LVAL_PP(argv[1]);

#if PHP_VERSION_ID < 50327
   if (argc > 2 && !ZVAL_IS_NULL(*argv[2]) && !zend_is_callable(*argv[2], 0, &fontNotFound))
#else
   if (argc > 2 && !ZVAL_IS_NULL(*argv[2]) && !zend_is_callable(*argv[2], 0, &fontNotFound TSRMLS_CC))
#endif
   {
      php_error_docref1(NULL TSRMLS_CC, fontNotFound, E_WARNING, "The third argument must be NULL or a callback function!");
      efree(fontNotFound);
      RETURN_LONG(-1);
   }
#if PHP_VERSION_ID < 50327
   if (argc > 3 && !ZVAL_IS_NULL(*argv[3]) && !zend_is_callable(*argv[3], 0, &replaceICCProfile))
#else
   if (argc > 3 && !ZVAL_IS_NULL(*argv[3]) && !zend_is_callable(*argv[3], 0, &replaceICCProfile TSRMLS_CC))
#endif
   {
      php_error_docref1(NULL TSRMLS_CC, replaceICCProfile, E_WARNING, "The fourth argument must be absent, NULL, or a callback function!");
      if (fontNotFound) efree(fontNotFound);
      efree(replaceICCProfile);
      RETURN_LONG(-1);
   }
   if (fontNotFound)
   {
      if (!obj->onFontNotFoundProc)
      {
         MAKE_STD_ZVAL(obj->onFontNotFoundProc);
      }else
         zval_dtor(obj->onFontNotFoundProc);
      ZVAL_STRING(obj->onFontNotFoundProc, fontNotFound, 0);
      fontProcPtr = OnFontNotFoundProc;
   }else if (!obj->onFontNotFoundProc)
   {
      zval_ptr_dtor(&obj->onFontNotFoundProc);
      obj->onFontNotFoundProc = NULL;
   }
   if (replaceICCProfile)
   {
      if (!obj->onReplaceICCProfileProc)
      {
         MAKE_STD_ZVAL(obj->onReplaceICCProfileProc);
      }else
         zval_dtor(obj->onReplaceICCProfileProc);
      ZVAL_STRING(obj->onReplaceICCProfileProc, replaceICCProfile, 0);
      iccProcPtr = OnReplaceICCProfileProc;
   }else if (!obj->onReplaceICCProfileProc)
   {
      zval_ptr_dtor(&obj->onReplaceICCProfileProc);
      obj->onReplaceICCProfileProc = NULL;
   }
   rc = pdfCheckConformance(obj->pdf, Type, Options, obj, fontProcPtr, iccProcPtr);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCheckFieldNames()
   const void* IPDF); */
PHP_METHOD(dynapdf, CheckFieldNames)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCheckFieldNames(pdf);
   RETURN_LONG(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfCircleAnnotA(const void* IPDF, double PosX, double PosY, double Width, double Height, double LineWidth, UI32 FillColor, UI32 StrokeColor, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Comment);
PHP_METHOD(dynapdf, CircleAnnot)
{
   zval **argv[11];
   PPDF* pdf;
   char* Author, *Subject, *Comment;
   double PosX, PosY, Width, Height, LineWidth;
   SI32 rc, FillColor, StrokeColor, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 11 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_string_ex(argv[8]);
   convert_to_string_ex(argv[9]);
   convert_to_string_ex(argv[10]);

   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   Width       = Z_DVAL_PP(argv[2]);
   Height      = Z_DVAL_PP(argv[3]);
   LineWidth   = Z_DVAL_PP(argv[4]);
   FillColor   = Z_LVAL_PP(argv[5]);
   StrokeColor = Z_LVAL_PP(argv[6]);
   CS          = Z_LVAL_PP(argv[7]);
   Author      = Z_STRVAL_PP(argv[8]);
   Subject     = Z_STRVAL_PP(argv[9]);
   Comment     = Z_STRVAL_PP(argv[10]);

   rc = pdfCircleAnnotA(pdf, PosX, PosY, Width, Height, LineWidth, FillColor, StrokeColor, CS, Author, Subject, Comment);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfClearAutoTemplates()
   const void* IPDF); */
PHP_METHOD(dynapdf, ClearAutoTemplates)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfClearAutoTemplates(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfClearErrorLog(const void* IPDF);
PHP_METHOD(dynapdf, ClearErrorLog)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   pdfClearErrorLog(pdf);
}

/* {{{ proto SI32 pdfClearHostFonts()
   const void* IPDF); */
PHP_METHOD(dynapdf, ClearHostFonts)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfClearHostFonts(pdf);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfClipPath()
   const void* IPDF, TClippingMode ClipMode, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, ClipPath)
{
   zval **argv[2];
   PPDF* pdf;
   TClippingMode ClipMode;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   ClipMode   = Z_LVAL_PP(argv[0]);
   FillMode   = Z_LVAL_PP(argv[1]);

   rc = pdfClipPath(pdf, ClipMode, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfCloseAndSignFile(const void* IPDF, const char* CertFile, const char* Password, const char* Reason, const char* Location);
PHP_METHOD(dynapdf, CloseAndSignFile)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc;
   char* CertFile, *Password, *Reason, *Location;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);

   CertFile = Z_STRVAL_PP(argv[0]);
   Password = Z_STRVAL_PP(argv[1]);
   Reason   = Z_STRVAL_PP(argv[2]);
   Location = Z_STRVAL_PP(argv[3]);

   rc = pdfCloseAndSignFile(pdf, CertFile, Password, Reason, Location);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCloseAndSignFileEx(const void* IPDF, const char* OpenPwd, const char* OwnerPwd, TKeyLen KeyLen, TRestrictions Restrict, const char* CertFile, const char* Password, const char* Reason, const char* Location);
PHP_METHOD(dynapdf, CloseAndSignFileEx)
{
   zval **argv[8];
   PPDF* pdf;
   SI32 rc, KeyLen, Restrict;
   char* OpenPwd, *OwnerPwd, *CertFile, *Password, *Reason, *Location = NULL;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_string_ex(argv[7]);

   OpenPwd  = Z_STRVAL_PP(argv[0]);
   OwnerPwd = Z_STRVAL_PP(argv[1]);
   KeyLen   = Z_LVAL_PP(argv[2]);
   Restrict = Z_LVAL_PP(argv[3]);
   CertFile = Z_STRVAL_PP(argv[4]);
   Password = Z_STRVAL_PP(argv[5]);
   Reason   = Z_STRVAL_PP(argv[6]);
   Location = Z_STRVAL_PP(argv[7]);

   rc = pdfCloseAndSignFileEx(pdf, OpenPwd, OwnerPwd, KeyLen, Restrict, CertFile, Password, Reason, Location);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCloseAndSignFileExt(const void* IPDF, struct TPDFSigParms* SigParms);
PHP_METHOD(dynapdf, CloseAndSignFileExt)
{
   zval **arg1;
   PPDF* pdf;
   HashTable *arr_hash;
   struct TPDFSigParms SigParms;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   convert_to_array_ex(arg1);

   arr_hash = Z_ARRVAL_PP(arg1);

   memset(&SigParms, 0, sizeof(SigParms));
   GetSigParms(arr_hash, &SigParms);

   if ((rc = pdfCloseAndSignFileExt(pdf, &SigParms)) != 0)
   {
      array_init(return_value);
      ADD_ASSOC_BIN_STRING(return_value, "Range1", (char*)SigParms.Range1, SigParms.Range1Len);
      ADD_ASSOC_BIN_STRING(return_value, "Range2", (char*)SigParms.Range2, SigParms.Range2Len);
   }
}

/* {{{ proto SI32 pdfCloseFile()
   const void* IPDF); */
PHP_METHOD(dynapdf, CloseFile)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   rc = pdfCloseFile(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCloseFileEx()
   const void* IPDF, const char* OpenPwd, const char* OwnerPwd, TKeyLen KeyLen, TRestrictions Restrict); */
PHP_METHOD(dynapdf, CloseFileEx)
{
   zval **argv[4];

   PPDF* pdf;
   char *OpenPwd, *OwnerPwd;
   TKeyLen KeyLen;
   TRestrictions Restrict;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   OpenPwd  = Z_STRVAL_PP(argv[0]);
   OwnerPwd = Z_STRVAL_PP(argv[1]);
   KeyLen   = Z_LVAL_PP(argv[2]);
   Restrict = Z_LVAL_PP(argv[3]);

   rc = pdfCloseFileEx(pdf, OpenPwd, OwnerPwd, KeyLen, Restrict);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfCloseImage(const void* IPDF);
PHP_METHOD(dynapdf, CloseImage)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCloseImage(pdf);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfCloseImportFile()
   const void* IPDF); */
PHP_METHOD(dynapdf, CloseImportFile)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCloseImportFile(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfCloseImportFileEx(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, CloseImportFileEx)
{
   zval **argv1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &argv1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv1);
   Handle = Z_LVAL_PP(argv1);

   rc = pdfCloseImportFileEx(pdf, Handle);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfClosePath()
   const void* IPDF, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, ClosePath)
{
   zval **argv1;
   PPDF* pdf;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &argv1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv1);
   FillMode = Z_LVAL_PP(argv1);

   rc = pdfClosePath(pdf, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfCloseTag(const void* IPDF);
PHP_METHOD(dynapdf, CloseTag)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCloseTag(pdf);
   RETURN_BOOL(rc);
}

// CMYK -> PHP specific
PHP_METHOD(dynapdf, CMYK)
{
   zval **argv[4];
   PPDF* pdf;
   BYTE c, m, y, k;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   c = (BYTE)Z_LVAL_PP(argv[0]);
   m = (BYTE)Z_LVAL_PP(argv[1]);
   y = (BYTE)Z_LVAL_PP(argv[2]);
   k = (BYTE)Z_LVAL_PP(argv[3]);

   rc = PDF_CMYK(c, m, y, k);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfComputeBBox(const void* IPDF, struct TPDFRect ADDR BBox, TCompBBoxFlags Flags);
PHP_METHOD(dynapdf, ComputeBBox)
{
   zval **argv1;
   PPDF* pdf;
   struct TPDFRect BBox = {0.0, 0.0, 0.0, 0.0};
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &argv1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv1);
   Flags = Z_LVAL_PP(argv1);

   array_init(return_value);
   if ((rc = pdfComputeBBox(pdf, &BBox, Flags)) != 0)
   {
      add_assoc_double(return_value, "Left",   BBox.Left);
      add_assoc_double(return_value, "Bottom", BBox.Bottom);
      add_assoc_double(return_value, "Right",  BBox.Right);
      add_assoc_double(return_value, "Top",    BBox.Top);
   }
}

//PDF_EXP UI32   PDF_CALL pdfConvColor(const double* Color, UI32 NumComps, TExtColorSpace SourceCS, const void* IColorSpace, TExtColorSpace DestCS);
PHP_METHOD(dynapdf, ConvColor)
{
   zval **argv[4];
   struct dynapdf_object *obj;
   double Color[32];
   HashTable *arr_hash;
   const void* IColorSpace = NULL;
   SI32 rc, csPtr, NumComps, SourceCS, DestCS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_array_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   arr_hash = Z_ARRVAL_PP(argv[0]);
   SourceCS = Z_LVAL_PP(argv[1]);
   csPtr    = Z_LVAL_PP(argv[2]);
   DestCS   = Z_LVAL_PP(argv[3]);

   if (csPtr && (IColorSpace = GetPointer(obj, csPtr, ptColorSpace)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IColorSpace must be 0 or a valid safe pointer!");
      RETURN_LONG(0);
   }

   NumComps = GetDoubleArray(arr_hash, Color, 32);

   rc = pdfConvColor(Color, NumComps, SourceCS, IColorSpace, DestCS);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfConvertColors(const void* IPDF, TColorConvFlags Flags, const float* Add);
PHP_METHOD(dynapdf, ConvertColors)
{
   zval **argv[2];
   PPDF* pdf;
   float Add = 0.0f;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc < 1 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);

   Flags = Z_LVAL_PP(argv[0]);
   if (argc > 1)
   {
      convert_to_double_ex(argv[1]);
      Add = (float)Z_DVAL_PP(argv[1]);
   }
   rc = pdfConvertColors(pdf, Flags, &Add);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfConvertEMFSpoolA(const void* IPDF, const char* SpoolFile, double LeftMargin, double TopMargin, TSpoolConvFlags Flags);
PHP_METHOD(dynapdf, ConvertEMFSpool)
{
   zval **argv[4];
   PPDF* pdf;
   char* SpoolFile;
   double LeftMargin, TopMargin;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc < 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   SpoolFile  = Z_STRVAL_PP(argv[0]);
   LeftMargin = Z_DVAL_PP(argv[1]);
   TopMargin  = Z_DVAL_PP(argv[2]);
   Flags      = Z_LVAL_PP(argv[3]);

   rc = pdfConvertEMFSpoolA(pdf, SpoolFile, LeftMargin, TopMargin, Flags);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCopyChoiceValues(const PPDF* IPDF, UI32 Source, UI32 Dest, LBOOL Share);
PHP_METHOD(dynapdf, CopyChoiceValues)
{
   zval **argv[3];
   PPDF* pdf;
   UI32 Source, Dest;
   LBOOL Share;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Source = Z_LVAL_PP(argv[0]);
   Dest   = Z_LVAL_PP(argv[1]);
   Share  = Z_LVAL_PP(argv[2]);

   rc = pdfCopyChoiceValues(pdf, Source, Dest, Share);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreate3DAnnot(const void* IPDF, double PosX, double PosY, double Width, double Height, const char* Author, const char* Name, const char* U3DFile, const char* Image);
PHP_METHOD(dynapdf, Create3DAnnot)
{
   zval **argv[8];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char* Author, *Name, *U3DFile, *Image;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_string_ex(argv[7]);

   PosX    = Z_DVAL_PP(argv[0]);
   PosY    = Z_DVAL_PP(argv[1]);
   Width   = Z_DVAL_PP(argv[2]);
   Height  = Z_DVAL_PP(argv[3]);
   Author  = Z_STRVAL_PP(argv[4]);
   Name    = Z_STRVAL_PP(argv[5]);
   U3DFile = Z_STRVAL_PP(argv[6]);
   Image   = Z_STRVAL_PP(argv[7]);

   rc = pdfCreate3DAnnot(pdf, PosX, PosY, Width, Height, Author, Name, U3DFile, Image);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreate3DBackground(const void* IPDF, const void* IView, UI32 BackColor);
PHP_METHOD(dynapdf, Create3DBackground)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   const void* IView;
   SI32 rc, viewPtr, BackColor;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   viewPtr   = Z_LVAL_PP(argv[0]);
   BackColor = Z_LVAL_PP(argv[1]);

   if (!viewPtr || (IView = GetPointer(obj, viewPtr, pt3DView)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IView must be a valid safe pointer!");
      RETURN_LONG(0);
   }

   rc = pdfCreate3DBackground(obj->pdf, IView, BackColor);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreate3DGotoViewAction(const void* IPDF, UI32 Base3DAnnot, const void* IView, T3DNamedAction Named);
PHP_METHOD(dynapdf, Create3DGotoViewAction)
{
   zval **argv[3];
   struct dynapdf_object *obj;
   const void* IView;
   SI32 rc, viewPtr, Base3DAnnot, Named;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Base3DAnnot = Z_LVAL_PP(argv[0]);
   viewPtr     = Z_LVAL_PP(argv[1]);
   Named       = Z_LVAL_PP(argv[2]);

   if (!viewPtr || (IView = GetPointer(obj, viewPtr, pt3DView)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IView must be a valid safe pointer!");
      RETURN_LONG(0);
   }

   rc = pdfCreate3DGotoViewAction(obj->pdf, Base3DAnnot, IView, Named);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreate3DProjection(const void* IPDF, const void* IView, T3DProjType ProjType, T3DScaleType ScaleType, double Diameter, double FOV);
PHP_METHOD(dynapdf, Create3DProjection)
{
   zval **argv[5];
   struct dynapdf_object *obj;
   const void* IView;
   double Diameter, FOV;
   SI32 rc, viewPtr, ProjType, ScaleType;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   viewPtr   = Z_LVAL_PP(argv[0]);
   ProjType  = Z_LVAL_PP(argv[1]);
   ScaleType = Z_LVAL_PP(argv[2]);
   Diameter  = Z_DVAL_PP(argv[3]);
   FOV       = Z_DVAL_PP(argv[4]);

   if (!viewPtr || (IView = GetPointer(obj, viewPtr, pt3DView)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IView contains an invalid value!");
      RETURN_LONG(0);
   }

   rc = pdfCreate3DProjection(obj->pdf, IView, ProjType, ScaleType, Diameter, FOV);
   RETURN_BOOL(rc);
}

//PDF_EXP void*  PDF_CALL pdfCreate3DView(const void* IPDF, UI32 Base3DAnnot, const char* Name, LBOOL SetAsDefault, double* Matrix, double CamDistance, T3DRenderingMode RM, T3DLightingSheme LS);
PHP_METHOD(dynapdf, Create3DView)
{
   zval **argv[7];
   HashTable *arr_hash;
   struct dynapdf_object *obj;
   void* IView;
   double Matrix[12], *matrixPtr, CamDistance;
   const char* Name;
   SI32 viewHash = 0, Base3DAnnot, SetAsDefault, RM, LS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_array_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);

   Base3DAnnot  = Z_LVAL_PP(argv[0]);
   Name         = Z_STRVAL_PP(argv[1]);
   SetAsDefault = Z_LVAL_PP(argv[2]);
   arr_hash     = Z_ARRVAL_PP(argv[3]);
   CamDistance  = Z_DVAL_PP(argv[4]);
   RM           = Z_LVAL_PP(argv[5]);
   LS           = Z_LVAL_PP(argv[6]);

   if (GetDoubleArray(arr_hash, Matrix, 12) != 12)
      matrixPtr = NULL;
   else
      matrixPtr = Matrix;

   IView = pdfCreate3DView(obj->pdf, Base3DAnnot, Name, SetAsDefault, matrixPtr, CamDistance, RM, LS);

   if (IView) viewHash = AddPointerEx(obj, IView, pt3DView);

   RETURN_LONG(viewHash);
}

//PDF_EXP SI32   PDF_CALL pdfCreateAltFontList(const PPDF* IPDF);
PHP_METHOD(dynapdf, CreateAltFontList)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCreateAltFontList(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateAnnotAP(const void* IPDF, UI32 Annot);
PHP_METHOD(dynapdf, CreateAnnotAP)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Annot;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Annot = Z_LVAL_PP(arg1);

   rc = pdfCreateAnnotAP(pdf, Annot);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateArticleThread()
   const void* IPDF, const char* ThreadName); */
PHP_METHOD(dynapdf, CreateArticleThread)
{
   zval **arg1;
   PPDF* pdf;
   char* ThreadName;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   ThreadName = Z_STRVAL_PP(arg1);

   rc = pdfCreateArticleThreadA(pdf, ThreadName);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfCreateAxialShading()
   const void* IPDF, double sX, double sY, double eX, double eY, double SCenter, UI32 SColor, UI32 EColor, LBOOL Extend1, LBOOL Extend2); */
PHP_METHOD(dynapdf, CreateAxialShading)
{
   zval **argv[9];
   PPDF* pdf;
   double sX, sY, eX, eY, SCenter;
   SI32 SColor, EColor, Extend1, Extend2;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_long_ex(argv[8]);
   sX      = Z_DVAL_PP(argv[0]);
   sY      = Z_DVAL_PP(argv[1]);
   eX      = Z_DVAL_PP(argv[2]);
   eY      = Z_DVAL_PP(argv[3]);
   SCenter = Z_DVAL_PP(argv[4]);
   SColor  = Z_LVAL_PP(argv[5]);
   EColor  = Z_LVAL_PP(argv[6]);
   Extend1 = Z_LVAL_PP(argv[7]);
   Extend2 = Z_LVAL_PP(argv[8]);

   rc = pdfCreateAxialShading(pdf, sX, sY, eX, eY, SCenter, SColor, EColor, Extend1, Extend2);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateBarcodeField(const void* IPDF, const char* Name, SI32 Parent, double PosX, double PosY, double Width, double Height, struct TPDFBarcode* Barcode);
PHP_METHOD(dynapdf, CreateBarcodeField)
{
   zval **argv[7];
   PPDF* pdf;
   char* Name;
   HashTable *array;
   struct TPDFBarcode Barcode;
   double PosX, PosY, Width, Height;
   SI32 rc, Parent;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_array_ex(argv[6]);

   Name   = Z_STRVAL_PP(argv[0]);
   Parent = Z_LVAL_PP(argv[1]);
   PosX   = Z_DVAL_PP(argv[2]);
   PosY   = Z_DVAL_PP(argv[3]);
   Width  = Z_DVAL_PP(argv[4]);
   Height = Z_LVAL_PP(argv[5]);
   array  = Z_ARRVAL_PP(argv[6]);
   GetBarcodeStruct(array, &Barcode);
   rc = pdfCreateBarcodeField(pdf, Name, Parent, PosX, PosY, Width, Height, &Barcode);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateButton()
   const void* IPDF, const char* Name, const char* Caption, SI32 Parent, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, CreateButton)
{
   zval **argv[7];
   PPDF* pdf;
   char *Name, *Caption;
   SI32 Parent;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   Name    = Z_STRVAL_PP(argv[0]);
   Caption = Z_STRVAL_PP(argv[1]);
   Parent  = Z_LVAL_PP(argv[2]);
   PosX    = Z_DVAL_PP(argv[3]);
   PosY    = Z_DVAL_PP(argv[4]);
   Width   = Z_DVAL_PP(argv[5]);
   Height  = Z_DVAL_PP(argv[6]);

   rc = pdfCreateButtonA(pdf, Name, Caption, Parent, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfCreateCheckBox()
   const void* IPDF, const char* Name, const char* ExpValue, LBOOL Checked, SI32 Parent, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, CreateCheckBox)
{
   zval **argv[8];
   PPDF* pdf;
   char *Name, *ExpValue;
   SI32 Checked, Parent;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_double_ex(argv[7]);
   Name     = Z_STRVAL_PP(argv[0]);
   ExpValue = Z_STRVAL_PP(argv[1]);
   Checked  = Z_LVAL_PP(argv[2]);
   Parent   = Z_LVAL_PP(argv[3]);
   PosX     = Z_DVAL_PP(argv[4]);
   PosY     = Z_DVAL_PP(argv[5]);
   Width    = Z_DVAL_PP(argv[6]);
   Height   = Z_DVAL_PP(argv[7]);

   rc = pdfCreateCheckBox(pdf, Name, ExpValue, Checked, Parent, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCreateComboBox()
   const void* IPDF, const char* Name, LBOOL Sort, SI32 Parent, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, CreateComboBox)
{
   zval **argv[7];
   PPDF* pdf;
   char *Name;
   SI32 Sort, Parent;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   Name     = Z_STRVAL_PP(argv[0]);
   Sort     = Z_LVAL_PP(argv[1]);
   Parent   = Z_LVAL_PP(argv[2]);
   PosX     = Z_DVAL_PP(argv[3]);
   PosY     = Z_DVAL_PP(argv[4]);
   Width    = Z_DVAL_PP(argv[5]);
   Height   = Z_DVAL_PP(argv[6]);

   rc = pdfCreateComboBox(pdf, Name, Sort, Parent, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateCIEColorSpace(const void* IPDF, TExtColorSpace CS, float* WhitePoint, float* BlackPoint, float* Gamma, float* Matrix);
PHP_METHOD(dynapdf, CreateCIEColorSpace)
{
   zval **argv[5];
   PPDF* pdf;
   float WhitePoint[3], BlackPoint[3], Gamma[3], Matrix[9], *blackPointPtr = NULL, *gammaPtr = NULL, *matrixPtr = NULL;
   HashTable *hash_wp, *hash_bp, *hash_gm, *hash_mtx;
   SI32 rc, CS, cmps;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   convert_to_array_ex(argv[2]);
   convert_to_array_ex(argv[3]);
   convert_to_array_ex(argv[4]);

   CS       = Z_LVAL_PP(argv[0]);
   hash_wp  = Z_ARRVAL_PP(argv[1]);
   hash_bp  = Z_ARRVAL_PP(argv[2]);
   hash_gm  = Z_ARRVAL_PP(argv[3]);
   hash_mtx = Z_ARRVAL_PP(argv[4]);

   if (GetFloatArray(hash_wp, WhitePoint, 3) != 3)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter WhitePoint must contain 3 numbers!");
      RETURN_LONG(-1);
   }
   if (GetFloatArray(hash_bp, BlackPoint, 3) == 3)
      blackPointPtr = BlackPoint;
   rc = GetFloatArray(hash_gm, Gamma, 3);
   switch(CS)
   {
      case esCalGray: cmps = 1;  break;
      case esCalRGB:  cmps = 3;  break;
      case esLab:     cmps = 4;  break;
      default:        cmps = -1; break;
   }
   if (rc > 0 && rc == cmps)
      gammaPtr = Gamma;
   if (GetFloatArray(hash_mtx, Matrix, 9) == 9)
      matrixPtr = Matrix;
   rc = pdfCreateCIEColorSpace(pdf, CS, WhitePoint, blackPointPtr, gammaPtr, matrixPtr);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreateColItemDate(const void* IPDF, UI32 EmbFile, const char* Key, long Date, const char* Prefix);
PHP_METHOD(dynapdf, CreateColItemDate)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, EmbFile;
   char* Key, *Prefix;
   long Date;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_string_ex(argv[3]);

   EmbFile = Z_LVAL_PP(argv[0]);
   Key     = Z_STRVAL_PP(argv[1]);
   Date    = Z_LVAL_PP(argv[2]);
   Prefix  = Z_STRVAL_PP(argv[3]);

   rc = pdfCreateColItemDate(pdf, EmbFile, Key, Date, Prefix);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreateColItemNumber(const void* IPDF, UI32 EmbFile, const char* Key, double Value, const char* Prefix);
PHP_METHOD(dynapdf, CreateColItemNumber)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, EmbFile;
   char* Key, *Prefix;
   double Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_string_ex(argv[3]);

   EmbFile = Z_LVAL_PP(argv[0]);
   Key     = Z_STRVAL_PP(argv[1]);
   Value   = Z_DVAL_PP(argv[2]);
   Prefix  = Z_STRVAL_PP(argv[3]);

   rc = pdfCreateColItemNumber(pdf, EmbFile, Key, Value, Prefix);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreateColItemStringA(const void* IPDF, UI32 EmbFile, const char* Key, const char* Value, const char* Prefix);
PHP_METHOD(dynapdf, CreateColItemString)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, EmbFile;
   char* Key, *Prefix;
   char* Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);

   EmbFile = Z_LVAL_PP(argv[0]);
   Key     = Z_STRVAL_PP(argv[1]);
   Value   = Z_STRVAL_PP(argv[2]);
   Prefix  = Z_STRVAL_PP(argv[3]);

   rc = pdfCreateColItemStringA(pdf, EmbFile, Key, Value, Prefix);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreateCollection(const void* IPDF, TColView View);
PHP_METHOD(dynapdf, CreateCollection)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, View;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);

   View = Z_LVAL_PP(arg1);

   rc = pdfCreateCollection(pdf, View);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateCollectionFieldA(const void* IPDF, TColColumnType ColType, SI32 Column, const char* Name, const char* Key, LBOOL Visible, LBOOL Editable);
PHP_METHOD(dynapdf, CreateCollectionField)
{
   zval **argv[6];
   PPDF* pdf;
   SI32 rc, ColType, Column, Visible, Editable;
   char* Name, *Key;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   ColType  = Z_LVAL_PP(argv[0]);
   Column   = Z_LVAL_PP(argv[1]);
   Name     = Z_STRVAL_PP(argv[2]);
   Key      = Z_STRVAL_PP(argv[3]);
   Visible  = Z_LVAL_PP(argv[4]);
   Editable = Z_LVAL_PP(argv[5]);

   rc = pdfCreateCollectionFieldA(pdf, ColType, Column, Name, Key, Visible, Editable);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateDeviceNColorSpace(const void* IPDF, const char** Colorants, UI32 NumColorants, const char* PostScriptFunc, TExtColorSpace Alternate, SI32 Handle);
PHP_METHOD(dynapdf, CreateDeviceNColorSpace)
{
   zval **argv[4];
   PPDF* pdf;
   HashTable* arr_hash;
   char* Colorants[32], *PostScriptFunc;
   SI32 rc, NumColorants, Alternate, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   arr_hash       = Z_ARRVAL_PP(argv[0]);
   PostScriptFunc = Z_STRVAL_PP(argv[1]);
   Alternate      = Z_LVAL_PP(argv[2]);
   Handle         = Z_LVAL_PP(argv[3]);

   NumColorants = GetColorants(arr_hash, Colorants);

   rc = pdfCreateDeviceNColorSpace(pdf, (const char**)Colorants, NumColorants, PostScriptFunc, Alternate, Handle);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateExtGState(const void* IPDF, struct TPDFExtGState* GS);
PHP_METHOD(dynapdf, CreateExtGState)
{
   zval **arg1;
   HashTable* arr_hash;
   struct dynapdf_object *obj;
   struct TPDFExtGState gs;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_array_ex(arg1);

   arr_hash = Z_ARRVAL_PP(arg1);

   pdfInitExtGState(&gs);
   if (GetExtGState(obj, arr_hash, &gs) < 0)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter SoftMask must be a valid object returned by CreateSoftMask or NULL or absent!");
      RETURN_LONG(-1);
   }
   rc = pdfCreateExtGState(obj->pdf, &gs);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateGoToAction()
   const void* IPDF, TDestType DestType, UI32 PageNum, double a, double b, double c, double d); */
PHP_METHOD(dynapdf, CreateGoToAction)
{
   zval **argv[6];
   PPDF* pdf;
   TDestType DestType;
   SI32 PageNum;
   double a, b, c, d;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   DestType = Z_LVAL_PP(argv[0]);
   PageNum  = Z_LVAL_PP(argv[1]);
   a = Z_DVAL_PP(argv[2]);
   b = Z_DVAL_PP(argv[3]);
   c = Z_DVAL_PP(argv[4]);
   d = Z_DVAL_PP(argv[5]);

   rc = pdfCreateGoToAction(pdf, DestType, PageNum, a, b, c, d);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateGoToActionEx(const void* IPDF, UI32 NamedDest);
PHP_METHOD(dynapdf, CreateGoToActionEx)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, NamedDest;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   NamedDest = Z_LVAL_PP(arg1);

   rc = pdfCreateGoToActionEx(pdf, NamedDest);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateGoToEActionA(const void* IPDF, TEmbFileLocation Location, const char* Source, UI32 SrcPage, const char* Target, const char* DestName, UI32 DestPage, LBOOL NewWindow);
PHP_METHOD(dynapdf, CreateGoToEAction)
{
   zval **argv[7];
   PPDF* pdf;
   char *Source, *Target, *DestName;
   SI32 rc, Location, SrcPage, DestPage, NewWindow;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);

   Location  = Z_LVAL_PP(argv[0]);
   Source    = Z_STRVAL_PP(argv[1]);
   SrcPage   = Z_LVAL_PP(argv[2]);
   Target    = Z_STRVAL_PP(argv[3]);
   DestName  = Z_STRVAL_PP(argv[4]);
   DestPage  = Z_LVAL_PP(argv[5]);
   NewWindow = Z_LVAL_PP(argv[6]);

   rc = pdfCreateGoToEActionA(pdf, Location, Source, SrcPage, Target, DestName, DestPage, NewWindow);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateGoToRAction()
   const void* IPDF, const char* FileName, UI32 PageNum); */
PHP_METHOD(dynapdf, CreateGoToRAction)
{
   zval **argv[2];
   PPDF* pdf;
   char *FileName;
   SI32 PageNum;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   FileName = Z_STRVAL_PP(argv[0]);
   PageNum  = Z_LVAL_PP(argv[1]);

   rc = pdfCreateGoToRAction(pdf, FileName, PageNum);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateGoToRActionExA(const void* IPDF, const char* FileName, char* DestName, LBOOL NewWindow);
PHP_METHOD(dynapdf, CreateGoToRActionEx)
{
   zval **argv[3];
   PPDF* pdf;
   char *FileName, *DestName;
   SI32 rc, NewWindow;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   FileName  = Z_STRVAL_PP(argv[0]);
   DestName  = Z_STRVAL_PP(argv[1]);
   NewWindow = Z_LVAL_PP(argv[2]);

   rc = pdfCreateGoToRActionExA(pdf, FileName, DestName, NewWindow);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateGroupField()
   const void* IPDF, const char* Name, SI32 Parent); */
PHP_METHOD(dynapdf, CreateGroupField)
{
   zval **argv[2];
   PPDF* pdf;
   char *Name;
   SI32 Parent;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Name = Z_STRVAL_PP(argv[0]);
   Parent  = Z_LVAL_PP(argv[1]);

   rc = pdfCreateGroupField(pdf, Name, Parent);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCreateHideAction()
   const void* IPDF, UI32 AField, LBOOL Hide); */
PHP_METHOD(dynapdf, CreateHideAction)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Hide;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField = Z_LVAL_PP(argv[0]);
   Hide   = Z_LVAL_PP(argv[1]);

   rc = pdfCreateHideAction(pdf, AField, Hide);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateICCBasedColorSpaceA(const void* IPDF, const char* ICCProfile);
PHP_METHOD(dynapdf, CreateICCBasedColorSpace)
{
   zval **arg1;

   PPDF* pdf;
   char *ICCProfile;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   ICCProfile = Z_STRVAL_PP(arg1);

   rc = pdfCreateICCBasedColorSpaceA(pdf, ICCProfile);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateICCBasedColorSpaceEx(const void* IPDF, const void* Buffer, UI32 BufSize);
PHP_METHOD(dynapdf, CreateICCBasedColorSpaceEx)
{
   zval **arg1;

   PPDF* pdf;
   char *Buffer;
   SI32 rc, BufSize;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Buffer  = Z_STRVAL_PP(arg1);
   BufSize = Z_STRLEN_PP(arg1);

   rc = pdfCreateICCBasedColorSpaceEx(pdf, Buffer, BufSize);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreateImageA(const void* IPDF, const char* FileName, TImageFormat Format);
PHP_METHOD(dynapdf, CreateImage)
{
   zval **argv[2];
   PPDF* pdf;
   char* FileName;
   SI32 rc, Format;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   FileName = Z_STRVAL_PP(argv[0]);
   Format   = Z_LVAL_PP(argv[1]);

   rc = pdfCreateImageA(pdf, FileName, Format);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfCreateImpDataAction()
   const void* IPDF, const char* DataFile); */
PHP_METHOD(dynapdf, CreateImpDataAction)
{
   zval **arg1;

   PPDF* pdf;
   char *DataFile;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   DataFile = Z_STRVAL_PP(arg1);

   rc = pdfCreateImpDataActionA(pdf, DataFile);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateIndexedColorSpace(const void* IPDF, TExtColorSpace Base, SI32 Handle, const void* ColorTable, UI32 NumColors);
PHP_METHOD(dynapdf, CreateIndexedColorSpace)
{
   zval **argv[2];
   PPDF* pdf;
   char* ColorTable;
   SI32 rc, Base, Handle, NumColors, len;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Base       = Z_LVAL_PP(argv[0]);
   Handle     = Z_LVAL_PP(argv[1]);
   ColorTable = Z_STRVAL_PP(argv[2]);
   len        = Z_STRLEN_PP(argv[2]);
   NumColors  = Z_LVAL_PP(argv[3]);
   if (len < NumColors)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The length of color table must be NumColors * NumComponents of the underlying base color space!");
      RETURN_LONG(-1);
   }
   rc = pdfCreateIndexedColorSpace(pdf, Base, Handle, ColorTable, NumColors);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateJSAction()
   const void* IPDF, const char* Script); */
PHP_METHOD(dynapdf, CreateJSAction)
{
   zval **arg1;
   PPDF* pdf;
   char *Script;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Script = Z_STRVAL_PP(arg1);

   rc = pdfCreateJSActionA(pdf, Script);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfCreateLaunchAction()
   const void* IPDF, TFileOP OP, const char* FileName, const char* DefDir, const char* Param, LBOOL NewWindow); */
PHP_METHOD(dynapdf, CreateLaunchAction)
{
   zval **argv[4];
   PPDF* pdf;
   TFileOP OP;
   char *FileName, *DefDir, *Param;
   SI32 NewWindow;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   OP        = Z_LVAL_PP(argv[0]);
   FileName  = Z_STRVAL_PP(argv[1]);
   DefDir    = Z_STRVAL_PP(argv[2]);
   Param     = Z_STRVAL_PP(argv[3]);
   NewWindow = Z_LVAL_PP(argv[4]);

   rc = pdfCreateLaunchAction(pdf, OP, FileName, DefDir, Param, NewWindow);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateLaunchActionExA(const void* IPDF, const char* FileName, LBOOL NewWindow);
PHP_METHOD(dynapdf, CreateLaunchActionEx)
{
   zval **argv[2];
   PPDF* pdf;
   char *FileName;
   SI32 NewWindow;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   FileName  = Z_STRVAL_PP(argv[0]);
   NewWindow = Z_LVAL_PP(argv[1]);

   rc = pdfCreateLaunchActionExA(pdf, FileName, NewWindow);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateListBox()
   const void* IPDF, const char* Name, LBOOL Sort, SI32 Parent, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, CreateListBox)
{
   zval **argv[7];
   PPDF* pdf;
   char *Name;
   SI32 Sort, Parent;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   Name     = Z_STRVAL_PP(argv[0]);
   Sort     = Z_LVAL_PP(argv[1]);
   Parent   = Z_LVAL_PP(argv[2]);
   PosX     = Z_DVAL_PP(argv[3]);
   PosY     = Z_DVAL_PP(argv[4]);
   Width    = Z_DVAL_PP(argv[5]);
   Height   = Z_DVAL_PP(argv[6]);

   rc = pdfCreateListBox(pdf, Name, Sort, Parent, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCreateNamedAction()
   const void* IPDF, TNamedAction Action); */
PHP_METHOD(dynapdf, CreateNamedAction)
{
   zval **arg1;
   PPDF* pdf;
   TNamedAction Action;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Action = Z_LVAL_PP(arg1);

   rc = pdfCreateNamedAction(pdf, Action);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateNamedDestA(const void* IPDF, const char* Name, UI32 DestPage, TDestType DestType, double a, double b, double c, double d);
PHP_METHOD(dynapdf, CreateNamedDest)
{
   zval **argv[7];
   PPDF* pdf;
   char* Name;
   double a, b, c, d;
   SI32 rc, DestPage, DestType;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);

   Name     = Z_STRVAL_PP(argv[0]);
   DestPage = Z_LVAL_PP(argv[1]);
   DestType = Z_LVAL_PP(argv[2]);
   a        = Z_DVAL_PP(argv[3]);
   b        = Z_DVAL_PP(argv[4]);
   c        = Z_DVAL_PP(argv[5]);
   d        = Z_DVAL_PP(argv[6]);

   rc = pdfCreateNamedDestA(pdf, Name, DestPage, DestType, a, b, c, d);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateNewPDF()
   const void* IPDF, const char* OutPDF); */
PHP_METHOD(dynapdf, CreateNewPDF)
{
   zval **arg1;
   char *filename = (char*)NULL;
   PPDF* pdf;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc > 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }
   DYNAPDF_FETCH_PDFDOC();

   if(argc > 0)
   {
      convert_to_string_ex(arg1);
      filename = Z_STRVAL_PP(arg1);
   }
   rc = pdfCreateNewPDFA(pdf, filename);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateOCGA(const void* IPDF, const char* Name, LBOOL DisplayInUI, LBOOL Visible, TOCGIntent Intent);
PHP_METHOD(dynapdf, CreateOCG)
{
   zval **argv[4];
   PPDF* pdf;
   char* Name;
   SI32 rc, DisplayInUI, Visible, Intent;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Name        = Z_STRVAL_PP(argv[0]);
   DisplayInUI = Z_LVAL_PP(argv[1]);
   Visible     = Z_LVAL_PP(argv[2]);
   Intent      = Z_LVAL_PP(argv[3]);

   rc = pdfCreateOCGA(pdf, Name, DisplayInUI, Visible, Intent);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateOCMD(const void* IPDF, TOCVisibility Visibility, UI32* OCGs, UI32 Count);
PHP_METHOD(dynapdf, CreateOCMD)
{
   zval **argv[2];
   PPDF* pdf;
   UI32* OCGs;
   HashTable *arr_hash;
   SI32 rc, Visibility, Count;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   Visibility = Z_LVAL_PP(argv[0]);
   arr_hash   = Z_ARRVAL_PP(argv[1]);
   Count      = zend_hash_num_elements(arr_hash);

   if (!Count) RETURN_LONG(-1);

   OCGs = (UI32*)emalloc(Count * sizeof(UI32));

   Count = GetUI32Array(arr_hash, OCGs, Count);

   rc = pdfCreateOCMD(pdf, Visibility, OCGs, Count);

   efree(OCGs);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateRadialShading()
   const void* IPDF, double sX, double sY, double R1, double eX, double eY, double R2, double SCenter, UI32 SColor, UI32 EColor, LBOOL Extend1, LBOOL Extend2); */
PHP_METHOD(dynapdf, CreateRadialShading)
{
   zval **argv[11];
   PPDF* pdf;
   double sX, sY, R1, eX, eY, R2, SCenter;
   SI32 SColor, EColor, Extend1, Extend2;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 11 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_long_ex(argv[8]);
   convert_to_long_ex(argv[9]);
   convert_to_long_ex(argv[10]);
   sX      = Z_DVAL_PP(argv[0]);
   sY      = Z_DVAL_PP(argv[1]);
   R1      = Z_DVAL_PP(argv[2]);
   eX      = Z_DVAL_PP(argv[3]);
   eY      = Z_DVAL_PP(argv[4]);
   R2      = Z_DVAL_PP(argv[5]);
   SCenter = Z_DVAL_PP(argv[6]);
   SColor  = Z_LVAL_PP(argv[7]);
   EColor  = Z_LVAL_PP(argv[8]);
   Extend1 = Z_LVAL_PP(argv[9]);
   Extend2 = Z_LVAL_PP(argv[10]);

   rc = pdfCreateRadialShading(pdf, sX, sY, R1, eX, eY, R2, SCenter, SColor, EColor, Extend1, Extend2);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCreateRadioButton()
   const void* IPDF, const char* Name, const char* ExpValue, SI32 Checked, SI32 Parent, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, CreateRadioButton)
{
   zval **argv[8];
   PPDF* pdf;
   char *Name, *ExpValue;
   SI32 Checked, Parent;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_double_ex(argv[7]);
   Name     = Z_STRVAL_PP(argv[0]);
   ExpValue = Z_STRVAL_PP(argv[1]);
   Checked  = Z_LVAL_PP(argv[2]);
   Parent   = Z_LVAL_PP(argv[3]);
   PosX     = Z_DVAL_PP(argv[4]);
   PosY     = Z_DVAL_PP(argv[5]);
   Width    = Z_DVAL_PP(argv[6]);
   Height   = Z_DVAL_PP(argv[7]);

   rc = pdfCreateRadioButton(pdf, Name, ExpValue, Checked, Parent, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfCreateRectilinearMeasureA(const PPDF* IPDF, UI32 Viewport, double Scale, const char* UnitDescription, const char* LabelDistance, const char* LabelArea);
PHP_METHOD(dynapdf, CreateRectilinearMeasure)
{
   zval **argv[5];
   PPDF* pdf;
   UI32 Viewport;
   double Scale;
   const char* UnitDescription, *LabelDistance, *LabelArea;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_string_ex(argv[4]);

   Viewport        = Z_LVAL_PP(argv[0]);
   Scale           = Z_DVAL_PP(argv[1]);
   UnitDescription = Z_STRVAL_PP(argv[2]);
   LabelDistance   = Z_STRVAL_PP(argv[3]);
   LabelArea       = Z_STRVAL_PP(argv[4]);

   rc = pdfCreateRectilinearMeasureA(pdf, Viewport, Scale, UnitDescription, LabelDistance, LabelArea);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfCreateResetAction()
   const void* IPDF); */
PHP_METHOD(dynapdf, CreateResetAction)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfCreateResetAction(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfCreateSeparationCS(const void* IPDF, const char* Colorant, TExtColorSpace Alternate, SI32 Handle, UI32 Color);
PHP_METHOD(dynapdf, CreateSeparationCS)
{
   zval **argv[4];
   PPDF* pdf;
   char* Colorant;
   SI32 rc, Alternate, Handle, Color;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Colorant  = Z_STRVAL_PP(argv[0]);
   Alternate = Z_LVAL_PP(argv[1]);
   Handle    = Z_LVAL_PP(argv[2]);
   Color     = Z_LVAL_PP(argv[3]);

   rc = pdfCreateSeparationCS(pdf, Colorant, Alternate, Handle, Color);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateSetOCGStateAction(const void* IPDF, UI32* On, UI32 OnCount, UI32* Off, UI32 OffCount, UI32* Toggle, UI32 ToggleCount, LBOOL PreserveRB);
PHP_METHOD(dynapdf, CreateSetOCGStateAction)
{
   zval **argv[4];
   PPDF* pdf;
   HashTable* on_hash, *off_hash, *tog_hash;
   UI32* On = NULL, *Off = NULL, *Toggle = NULL;
   SI32 rc, OnCount, OffCount, ToggleCount, PreserveRB;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   convert_to_array_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   on_hash     = Z_ARRVAL_PP(argv[0]);
   off_hash    = Z_ARRVAL_PP(argv[1]);
   tog_hash    = Z_ARRVAL_PP(argv[2]);
   PreserveRB  = Z_LVAL_PP(argv[3]);

   OnCount     = zend_hash_num_elements(on_hash);
   OffCount    = zend_hash_num_elements(off_hash);
   ToggleCount = zend_hash_num_elements(tog_hash);

   if (OnCount > 0)
   {
      On = (UI32*)emalloc(OnCount * sizeof(UI32));
      GetUI32Array(on_hash, On, OnCount);
   }
   if (OffCount > 0)
   {
      Off = (UI32*)emalloc(OffCount * sizeof(UI32));
      GetUI32Array(off_hash, Off, OffCount);
   }
   if (ToggleCount > 0)
   {
      Toggle = (UI32*)emalloc(ToggleCount * sizeof(UI32));
      GetUI32Array(tog_hash, Toggle, ToggleCount);
   }

   rc = pdfCreateSetOCGStateAction(pdf, On, OnCount, Off, OffCount, Toggle, ToggleCount, PreserveRB);

   if (On)     efree(On);
   if (Off)    efree(Off);
   if (Toggle) efree(Toggle);

   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateSigField(const void* IPDF, const char* Name, SI32 Parent, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, CreateSigField)
{
   zval **argv[6];
   PPDF* pdf;
   char* Name;
   SI32 rc, Parent;
   double PosX, PosY, Width, Height;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);

   Name   = Z_STRVAL_PP(argv[0]);
   Parent = Z_LVAL_PP(argv[1]);
   PosX   = Z_DVAL_PP(argv[2]);
   PosY   = Z_DVAL_PP(argv[3]);
   Width  = Z_DVAL_PP(argv[4]);
   Height = Z_DVAL_PP(argv[5]);

   rc = pdfCreateSigField(pdf, Name, Parent, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateSigFieldAP(const void* IPDF, UI32 SigField);
PHP_METHOD(dynapdf, CreateSigFieldAP)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, SigField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);

   SigField = Z_LVAL_PP(arg1);

   rc = pdfCreateSigFieldAP(pdf, SigField);
   RETURN_LONG(rc);
}

//PDF_EXP void*  PDF_CALL pdfCreateSoftMask(const void* IPDF, UI32 TranspGroup, TSoftMaskType MaskType, UI32 BackColor);
PHP_METHOD(dynapdf, CreateSoftMask)
{
   zval **argv[3];
   void* ptr;
   struct dynapdf_object *obj;
   SI32 hash = 0, TranspGroup, MaskType, BackColor;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   TranspGroup = Z_LVAL_PP(argv[0]);
   MaskType    = Z_LVAL_PP(argv[1]);
   BackColor   = Z_LVAL_PP(argv[2]);

   ptr = pdfCreateSoftMask(obj->pdf, TranspGroup, MaskType, BackColor);

   if (ptr) hash = AddPointerEx(obj, ptr, ptSoftMask);
   RETURN_LONG(hash);
}

/* {{{ proto SI32 pdfCreateStdPattern()
   const void* IPDF, TStdPattern Pattern, double LineWidth, double Distance, UI32 LineColor, UI32 BackColor); */
PHP_METHOD(dynapdf, CreateStdPattern)
{
   zval **argv[5];
   PPDF* pdf;
   TStdPattern Pattern;
   double LineWidth, Distance;
   SI32 LineColor, BackColor;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   Pattern   = Z_LVAL_PP(argv[0]);
   LineWidth = Z_DVAL_PP(argv[1]);
   Distance  = Z_DVAL_PP(argv[2]);
   LineColor = Z_LVAL_PP(argv[3]);
   BackColor = Z_LVAL_PP(argv[4]);

   rc = pdfCreateStdPattern(pdf, Pattern, LineWidth, Distance, LineColor, BackColor);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfCreateStructureTree(const void* IPDF);
PHP_METHOD(dynapdf, CreateStructureTree)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   rc = pdfCreateStructureTree(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfCreateStructureTreeEx(const PPDF* IPDF, TPDFBaseTag RootTag);
PHP_METHOD(dynapdf, CreateStructureTreeEx)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, RootTag;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);

   RootTag = Z_LVAL_PP(arg1);

   rc = pdfCreateStructureTreeEx(pdf, RootTag);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfCreateSubmitAction()
   const void* IPDF, TSubmitFlags Flags, const char* URL); */
PHP_METHOD(dynapdf, CreateSubmitAction)
{
   zval **argv[2];
   PPDF* pdf;
   TSubmitFlags Flags;
   char* URL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Flags = Z_LVAL_PP(argv[0]);
   URL   = Z_STRVAL_PP(argv[1]);

   rc = pdfCreateSubmitAction(pdf, Flags, URL);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCreateTextField()
   const void* IPDF, const char* Name, SI32 Parent, LBOOL Multiline, SI32 MaxLen, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, CreateTextField)
{
   zval **argv[8];
   PPDF* pdf;
   char *Name;
   SI32 Parent, Multiline, MaxLen;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_double_ex(argv[7]);
   Name     = Z_STRVAL_PP(argv[0]);
   Parent   = Z_LVAL_PP(argv[1]);
   Multiline= Z_LVAL_PP(argv[2]);
   MaxLen   = Z_LVAL_PP(argv[3]);
   PosX     = Z_DVAL_PP(argv[4]);
   PosY     = Z_DVAL_PP(argv[5]);
   Width    = Z_DVAL_PP(argv[6]);
   Height   = Z_DVAL_PP(argv[7]);

   rc = pdfCreateTextField(pdf, Name, Parent, Multiline, MaxLen, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfCreateURIAction()
   const void* IPDF, const char* URL); */
PHP_METHOD(dynapdf, CreateURIAction)
{
   zval **arg1;
   PPDF* pdf;
   char *URL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   URL = Z_STRVAL_PP(arg1);

   rc = pdfCreateURIAction(pdf, URL);
   RETURN_LONG(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfCreateViewportA(const PPDF* IPDF, const char* Name, double X1, double Y1, double X2, double Y2);
PHP_METHOD(dynapdf, CreateViewport)
{
   zval **argv[5];
   PPDF* pdf;
   char* Name;
   double X1, Y1, X2, Y2;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   Name = Z_STRVAL_PP(argv[0]);
   X1   = Z_DVAL_PP(argv[1]);
   Y1   = Z_DVAL_PP(argv[2]);
   X2   = Z_DVAL_PP(argv[3]);
   Y2   = Z_DVAL_PP(argv[4]);

   rc = pdfCreateViewportA(pdf, Name, X1, Y1, X2, Y2);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfCreateXFAStreamA(const PPDF* IPDF, const char* Name, const void* Buffer, UI32 BufSize);
PHP_METHOD(dynapdf, CreateXFAStream)
{
   zval** argv[2];
   PPDF* pdf;
   char* Name, *Buffer;
   UI32 BufSize;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if (argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Name    = Z_STRVAL_PP(argv[0]);
   Buffer  = Z_STRVAL_PP(argv[1]);
   BufSize = Z_STRLEN_PP(argv[1]);

   rc = pdfCreateXFAStreamA(pdf, Name, Buffer, BufSize);
   RETURN_LONG(rc);
}


/* {{{ proto SI32 pdfDecryptPDF()
   const void* IPDF, const char* FileName, TPwdType PwdType, const char* Password); */
PHP_METHOD(dynapdf, DecryptPDF)
{
   zval **argv[3];
   PPDF* pdf;
   char *FileName;
   TPwdType PwdType;
   char *Password;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   FileName = Z_STRVAL_PP(argv[0]);
   PwdType  = Z_LVAL_PP(argv[1]);
   Password = Z_STRVAL_PP(argv[2]);

   rc = pdfDecryptPDFA(pdf, FileName, PwdType, Password);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfDeleteAcroForm(const void* IPDF);
PHP_METHOD(dynapdf, DeleteAcroForm)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   pdfDeleteAcroForm(pdf);
}

/* {{{ proto SI32 pdfDeleteActionFromObj()
   const void* IPDF, TObjType ObjType, UI32 ActHandle, UI32 ObjHandle); */
PHP_METHOD(dynapdf, DeleteActionFromObj)
{
   zval **argv[3];
   PPDF* pdf;
   TObjType ObjType;
   SI32 ActHandle, ObjHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   ObjType   = Z_LVAL_PP(argv[0]);
   ActHandle = Z_LVAL_PP(argv[1]);
   ObjHandle = Z_LVAL_PP(argv[2]);

   rc = pdfDeleteActionFromObj(pdf, ObjType, ActHandle, ObjHandle);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfDeleteActionFromObjEx()
   const void* IPDF, TObjType ObjType, UI32 ObjHandle, UI32 ActIndex); */
PHP_METHOD(dynapdf, DeleteActionFromObjEx)
{
   zval **argv[3];
   PPDF* pdf;
   TObjType ObjType;
   SI32 ObjHandle, ActIndex;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   ObjType   = Z_LVAL_PP(argv[0]);
   ObjHandle = Z_LVAL_PP(argv[1]);
   ActIndex  = Z_LVAL_PP(argv[2]);

   rc = pdfDeleteActionFromObjEx(pdf, ObjType, ObjHandle, ActIndex);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfDeleteAltFontList(const PPDF* IPDF, SI32 Handle);
PHP_METHOD(dynapdf, DeleteAltFontList)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle  = Z_LVAL_PP(arg1);

   rc = pdfDeleteAltFontList(pdf, Handle);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfDeleteAnnotation()
   const void* IPDF, UI32 Handle); */
PHP_METHOD(dynapdf, DeleteAnnotation)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle  = Z_LVAL_PP(arg1);

   rc = pdfDeleteAnnotation(pdf, Handle);
   RETURN_BOOL(rc);
}
/* }}} */

// PDF_EXP LBOOL  PDF_CALL pdfDeleteAnnotationFromPage(const PPDF* IPDF, UI32 PageNum, UI32 Handle);
PHP_METHOD(dynapdf, DeleteAnnotationFromPage)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 PageNum, Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   PageNum = Z_LVAL_PP(argv[0]);
   Handle  = Z_LVAL_PP(argv[1]);

   rc = pdfDeleteAnnotationFromPage(pdf, PageNum, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfDeleteAppEvents(const PPDF* IPDF, LBOOL ApplyEvent, TOCAppEvent Event);
PHP_METHOD(dynapdf, DeleteAppEvents)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Event, ApplyEvent;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   ApplyEvent = Z_LVAL_PP(argv[0]);
   Event      = Z_LVAL_PP(argv[1]);

   rc = pdfDeleteAppEvents(pdf, ApplyEvent, Event);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfDeleteBookmark()
   const void* IPDF, UI32 ABmk); */
PHP_METHOD(dynapdf, DeleteBookmark)
{
   zval **arg1;
   PPDF* pdf;
   SI32 ABmk;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   ABmk  = Z_LVAL_PP(arg1);

   rc = pdfDeleteBookmark(pdf, ABmk);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfDeleteEmbeddedFile(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, DeleteEmbeddedFile)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfDeleteEmbeddedFile(pdf, Handle);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfDeleteField()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, DeleteField)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfDeleteField(pdf, AField);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfDeleteFieldEx()
   const void* IPDF, const char* Name); */
PHP_METHOD(dynapdf, DeleteFieldEx)
{
   zval **arg1;
   PPDF* pdf;
   char *Name;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Name = Z_STRVAL_PP(arg1);

   rc = pdfDeleteFieldEx(pdf, Name);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfDeleteJavaScripts(const void* IPDF, LBOOL DelJavaScriptActions);
PHP_METHOD(dynapdf, DeleteJavaScripts)
{
   zval **arg1;
   PPDF* pdf;
   SI32 DelJavaScriptActions;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   DelJavaScriptActions = Z_LVAL_PP(arg1);

   pdfDeleteJavaScripts(pdf, DelJavaScriptActions);
}

//PDF_EXP LBOOL  PDF_CALL pdfDeleteNamedDestA(const PPDF* IPDF, const char* Name);
PHP_METHOD(dynapdf, DeleteNamedDest)
{
   zval **arg1;
   PPDF* pdf;
   char* Name = NULL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   DYNAPDF_FETCH_PDFDOC();
   
   if (argc > 0 && zend_get_parameters_array_ex(argc, &arg1) == SUCCESS)
   {
      convert_to_string_ex(arg1);
      Name = Z_STRVAL_PP(arg1);
   }

   rc = pdfDeleteNamedDestA(pdf, Name);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfDeleteNamedDestByIndex(const PPDF* IPDF, SI32 Index);
PHP_METHOD(dynapdf, DeleteNamedDestByIndex)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Index = -1;
   int argc = ZEND_NUM_ARGS();

   DYNAPDF_FETCH_PDFDOC();
   
   if (argc > 0 && zend_get_parameters_array_ex(argc, &arg1) == SUCCESS)
   {
      convert_to_long_ex(arg1);
      Index = (SI32)Z_LVAL_PP(arg1);
   }

   rc = pdfDeleteNamedDestByIndex(pdf, Index);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfDeleteOCGFromAppEvent(const PPDF* IPDF, UI32 Handle, TOCAppEvent Events, TOCGUsageCategory Categories, LBOOL DelCategoryOnly);
PHP_METHOD(dynapdf, DeleteOCGFromAppEvent)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, Handle, Events, Categories, DelCategoryOnly;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Handle          = Z_LVAL_PP(argv[0]);
   Events          = Z_LVAL_PP(argv[1]);
   Categories      = Z_LVAL_PP(argv[2]);
   DelCategoryOnly = Z_LVAL_PP(argv[3]);

   rc = pdfDeleteOCGFromAppEvent(pdf, Handle, Events, Categories, DelCategoryOnly);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfDeleteOCGFromDisplayTree(const PPDF* IPDF, UI32 OCGHandle, LBOOL Recursive);
PHP_METHOD(dynapdf, DeleteOCGFromDisplayTree)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, OCGHandle, Recursive;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   OCGHandle = Z_LVAL_PP(argv[0]);
   Recursive = Z_LVAL_PP(argv[1]);

   rc = pdfDeleteOCGFromDisplayTree(pdf, OCGHandle, Recursive);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfDeleteOCUINode(const PPDF* IPDF, IOCN* Node);
PHP_METHOD(dynapdf, DeleteOCUINode)
{
   zval **arg1;
   SI32 hash, rc;
   IOCN* Node = NULL;
   struct dynapdf_object *obj;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }
   hash = Z_LVAL_PP(arg1);

   DYNAPDF_FETCH_PDFDOC_EX();
   if (hash)
   {
      if ((Node = (IOCN*)GetPointer(obj, hash, ptOCGUINode)) == NULL)
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Node contains an invalid value!");
         RETURN_LONG(-1);
      }
   }
   rc = pdfDeleteOCUINode(obj->pdf, Node);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfDeleteOutputIntent(const void* IPDF, SI32 Index);
PHP_METHOD(dynapdf, DeleteOutputIntent)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Index;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   rc = pdfDeleteOutputIntent(pdf, Index);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfDeletePage()
   const void* IPDF, UI32 PageNum); */
PHP_METHOD(dynapdf, DeletePage)
{
   zval **arg1;
   PPDF* pdf;
   SI32 PageNum;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum = Z_LVAL_PP(arg1);

   rc = pdfDeletePage(pdf, PageNum);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfDeletePageLabels(const void* IPDF);
PHP_METHOD(dynapdf, DeletePageLabels)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   pdfDeletePageLabels(pdf);
}

//PDF_EXP LBOOL  PDF_CALL pdfDeleteSeparationInfo(const void* IPDF, LBOOL AllPages);
PHP_METHOD(dynapdf, DeleteSeparationInfo)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, AllPages;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AllPages  = Z_LVAL_PP(arg1);

   rc = pdfDeleteSeparationInfo(pdf, AllPages);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfDeleteTemplate()
   const void* IPDF, UI32 Handle); */
PHP_METHOD(dynapdf, DeleteTemplate)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle  = Z_LVAL_PP(arg1);

   rc = pdfDeleteTemplate(pdf, Handle);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfDeleteTemplateEx()
   const void* IPDF, UI32 Index); */
PHP_METHOD(dynapdf, DeleteTemplateEx)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Index;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index  = Z_LVAL_PP(arg1);

   rc = pdfDeleteTemplateEx(pdf, Index);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfDeleteXFAForm(const void* IPDF);
PHP_METHOD(dynapdf, DeleteXFAForm)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   pdfDeleteXFAForm(pdf);
}

// PHP specific
PHP_METHOD(dynapdf, DisableUTF8Support)
{
   PPDF* pdf;
   UI32 flags;
   DYNAPDF_FETCH_PDFDOC();
   flags = pdfGetGStateFlags(pdf);
   flags &= ~gfAnsiStringIsUTF8;
   pdfSetGStateFlags(pdf, flags, 1);
}

/* {{{ proto SI32 pdfDrawArc()
   const void* IPDF, double PosX, double PosY, double Radius, double StartAngle, double EndAngle); */
PHP_METHOD(dynapdf, DrawArc)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Radius, StartAngle, EndAngle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Radius  = Z_DVAL_PP(argv[2]);
   StartAngle = Z_DVAL_PP(argv[3]);
   EndAngle = Z_DVAL_PP(argv[4]);

   rc = pdfDrawArc(pdf, PosX, PosY, Radius, StartAngle, EndAngle);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfDrawArcEx()
   const void* IPDF, double PosX, double PosY, double Width, double Height, double StartAngle, double EndAngle); */
PHP_METHOD(dynapdf, DrawArcEx)
{
   zval **argv[6];
   PPDF* pdf;
   double PosX, PosY, Width, Height, StartAngle, EndAngle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   StartAngle = Z_DVAL_PP(argv[4]);
   EndAngle = Z_DVAL_PP(argv[5]);

   rc = pdfDrawArcEx(pdf, PosX, PosY, Width, Height, StartAngle, EndAngle);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfDrawChord()
   const void* IPDF, double PosX, double PosY, double Width, double Height, double StartAngle, double EndAngle, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, DrawChord)
{
   zval **argv[7];
   PPDF* pdf;
   double PosX, PosY, Width, Height, StartAngle, EndAngle;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   StartAngle = Z_DVAL_PP(argv[4]);
   EndAngle = Z_DVAL_PP(argv[5]);
   FillMode = Z_LVAL_PP(argv[6]);

   rc = pdfDrawChord(pdf, PosX, PosY, Width, Height, StartAngle, EndAngle, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfDrawCircle()
   const void* IPDF, double PosX, double PosY, double Radius, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, DrawCircle)
{
   zval **argv[4];
   PPDF* pdf;
   double PosX, PosY, Radius;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Radius = Z_DVAL_PP(argv[2]);
   FillMode = Z_LVAL_PP(argv[3]);

   rc = pdfDrawCircle(pdf, PosX, PosY, Radius, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfDrawNGon(const PPDF* IPDF, double PosX, double PosY, double Radius, double Alpha, UI32 NumSides, TPathFillMode FillMode);
PHP_METHOD(dynapdf, DrawNGon)
{
   zval **argv[6];
   PPDF* pdf;
   UI32 NumSides;
   TPathFillMode FillMode;
   double PosX, PosY, Radius, Alpha;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   PosX     = Z_DVAL_PP(argv[0]);
   PosY     = Z_DVAL_PP(argv[1]);
   Radius   = Z_DVAL_PP(argv[2]);
   Alpha    = Z_DVAL_PP(argv[3]);
   NumSides = Z_LVAL_PP(argv[4]);
   FillMode = Z_LVAL_PP(argv[5]);

   rc = pdfDrawNGon(pdf, PosX, PosY, Radius, Alpha, NumSides, FillMode);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfDrawPie()
   const void* IPDF, double PosX, double PosY, double Width, double Height, double StartAngle, double EndAngle, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, DrawPie)
{
   zval **argv[7];
   PPDF* pdf;
   double PosX, PosY, Width, Height, StartAngle, EndAngle;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   StartAngle = Z_DVAL_PP(argv[4]);
   EndAngle = Z_DVAL_PP(argv[5]);
   FillMode = Z_LVAL_PP(argv[6]);

   rc = pdfDrawPie(pdf, PosX, PosY, Width, Height, StartAngle, EndAngle, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfEditPage()
   const void* IPDF, SI32 PageNum); */
PHP_METHOD(dynapdf, EditPage)
{
   zval **arg1;
   PPDF* pdf;
   SI32 PageNum;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum  = Z_LVAL_PP(arg1);

   rc = pdfEditPage(pdf, PageNum);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfEditTemplate()
   const void* IPDF, UI32 Index); */
PHP_METHOD(dynapdf, EditTemplate)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Index;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index  = Z_LVAL_PP(arg1);

   rc = pdfEditTemplate(pdf, Index);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfEditTemplate2()
   const void* IPDF, UI32 Handle); */
PHP_METHOD(dynapdf, EditTemplate2)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle  = Z_LVAL_PP(arg1);

   rc = pdfEditTemplate2(pdf, Handle);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfEllipse()
   const void* IPDF, double PosX, double PosY, double Width, double Height, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, Ellipse)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   FillMode = Z_LVAL_PP(argv[4]);

   rc = pdfEllipse(pdf, PosX, PosY, Width, Height, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

// PHP specific
PHP_METHOD(dynapdf, EnableUTF8Support)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   pdfSetGStateFlags(pdf, gfAnsiStringIsUTF8, 0);
}

/* {{{ proto SI32 pdfEncryptPDF()
   const void* IPDF, const char* FileName, const char* OpenPwd, const char* OwnerPwd, TKeyLen KeyLen, TRestrictions Restrict); */
PHP_METHOD(dynapdf, EncryptPDF)
{
   zval **argv[5];

   PPDF* pdf;
   char *FileName, *OpenPwd, *OwnerPwd;
   TKeyLen KeyLen;
   TRestrictions Restrict;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   FileName = Z_STRVAL_PP(argv[0]);
   OpenPwd  = Z_STRVAL_PP(argv[1]);
   OwnerPwd = Z_STRVAL_PP(argv[2]);
   KeyLen   = Z_LVAL_PP(argv[3]);
   Restrict = Z_LVAL_PP(argv[4]);

   rc = pdfEncryptPDFA(pdf, FileName, OpenPwd, OwnerPwd, KeyLen, Restrict);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfEndContinueText()
   const void* IPDF); */
PHP_METHOD(dynapdf, EndContinueText)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfEndContinueText(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfEndLayer(const void* IPDF);
PHP_METHOD(dynapdf, EndLayer)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfEndLayer(pdf);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfEndPage()
   const void* IPDF); */
PHP_METHOD(dynapdf, EndPage)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfEndPage(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfEndPattern()
   const void* IPDF); */
PHP_METHOD(dynapdf, EndPattern)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfEndPattern(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfEndTemplate()
   const void* IPDF); */
PHP_METHOD(dynapdf, EndTemplate)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfEndTemplate(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

struct TCallbackData
{
   struct dynapdf_object* obj;
   zval*  return_value;
};

/*******************************************************************************/
/** Callback function for EnumDocFonts                                        **/
/*******************************************************************************/
SI32 PDF_CALL EnumDocFontsProc(const void* Data, const void* PDFFont, TFontType Type, const char* BaseFont, const char* FontName, LBOOL Embedded, LBOOL IsFormFont, UI32 Flags)
{
   zval* node;
   struct TCallbackData* data = (struct TCallbackData*)Data;

   ALLOC_INIT_ZVAL(node);
   array_init(node);

   AddStringA(node, "BaseFont", BaseFont);
   AddStringA(node, "FontName", FontName);
   if (PDFFont) AddPointer(node, data->obj,"PDFFont", PDFFont, ptFont);
   add_assoc_long(node, "Embedded", Embedded);
   add_assoc_long(node, "IsFormFont", IsFormFont);
   add_assoc_long(node, "Flags", Flags);
   add_next_index_zval(data->return_value, node);
   return 0;
}

//PDF_EXP SI32   PDF_CALL pdfEnumDocFonts(const void* IPDF, const void* Data, TEnumFontProc2* EnumProc);
PHP_METHOD(dynapdf, EnumDocFonts)
{
   struct dynapdf_object *obj;
   struct TCallbackData data;
   DYNAPDF_FETCH_PDFDOC_EX();

   array_init(return_value);
   data.obj          = obj;
   data.return_value = return_value;
   pdfEnumDocFonts(obj->pdf, &data, EnumDocFontsProc);
}

/*******************************************************************************/
/** Callback function for EnumHostFonts                                       **/
/*******************************************************************************/
SI32 PDF_CALL EnumHostFontsProc(const void* return_value, const UI16* FamilyName, const char* PostScriptName, SI32 Style)
{
   zval* node, *pretval = (zval*)return_value;

   ALLOC_INIT_ZVAL(node);
   array_init(node);

   AddStringW(node, "FamilyName", FamilyName);
   AddStringA(node, "PostScriptName", PostScriptName);
   add_assoc_long(node, "Style", Style);
   add_next_index_zval(pretval, node);
   return 0;
}

/* {{{ proto SI32 pdfEnumHostFonts()
   const void* IPDF, const void* Data, TEnumFontProc* EnumProc); */
PHP_METHOD(dynapdf, EnumHostFonts)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   array_init(return_value);
   pdfEnumHostFonts(pdf, return_value, EnumHostFontsProc);
}
/* }}} */

/*******************************************************************************/
/** Callback function for EnumHostFontsEx                                     **/
/*******************************************************************************/
SI32 PDF_CALL EnumHostFontsExProc(const void* return_value, const UI16* FamilyName, const char* PostScriptName, SI32 Style, TFontBaseType BaseType, TEnumFontProcFlags Flags, const char* FilePath)
{
   zval* node, *pretval = (zval*)return_value;

   ALLOC_INIT_ZVAL(node);
   array_init(node);

   AddStringW(node, "FamilyName", FamilyName);
   AddStringA(node, "PostScriptName", PostScriptName);
   add_assoc_long(node, "Style", Style);
   add_assoc_long(node, "BaseType", BaseType);
   add_assoc_bool(node, "Embeddable", (Flags & efpEmbeddable) != 0);
   add_assoc_bool(node, "Editable", (Flags & efpEditable) != 0);
   if (Flags & efpUnicodePath)
      AddStringW(node, "FilePath", (const UI16*)FilePath);
   else
      AddStringA(node, "FilePath", FilePath);
   add_next_index_zval(pretval, node);
   return 0;
}

//PDF_EXP SI32   PDF_CALL pdfEnumHostFontsEx(const void* IPDF, const void* Data, TEnumFontProcEx* EnumProc);
PHP_METHOD(dynapdf, EnumHostFontsEx)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();

   array_init(return_value);
   pdfEnumHostFontsEx(pdf, return_value, EnumHostFontsExProc);
}

/* {{{ proto SI32 pdfExchangeBookmarks()
   const void* IPDF, SI32 Bmk1, SI32 Bmk2); */
PHP_METHOD(dynapdf, ExchangeBookmarks)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Bmk1, Bmk2;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Bmk1 = Z_LVAL_PP(argv[0]);
   Bmk2 = Z_LVAL_PP(argv[1]);

   rc = pdfExchangeBookmarks(pdf, Bmk1, Bmk2);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfExchangePages(const void* IPDF, UI32 First, UI32 Second);
PHP_METHOD(dynapdf, ExchangePages)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 First, Second;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   First  = Z_LVAL_PP(argv[0]);
   Second = Z_LVAL_PP(argv[1]);

   rc = pdfExchangePages(pdf, First, Second);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfExtractText(const PPDF* IPDF, UI32 PageNum, TTextExtractionFlags Flags, struct TFltRect* Area, UI16** Text, UI32* TextLen);
PHP_METHOD(dynapdf, ExtractText)
{
   zval **argv[3];
   PPDF* pdf;
   UI32 PageNum;
   SI32 Flags;
   UI16* text = NULL;
   UI32 textLen = 0;
   struct TFltRect Area, *Area_Ptr = NULL;
   int argc = ZEND_NUM_ARGS();
   if(argc < 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }
   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   if (argc == 3)
   {
      HashTable *arr_hash;
      convert_to_array_ex(argv[2]);
      arr_hash = Z_ARRVAL_PP(argv[2]);
      if (!GetFltRect(arr_hash, &Area))
         Area_Ptr = &Area;
   }
   PageNum = Z_LVAL_PP(argv[0]);
   Flags   = Z_LVAL_PP(argv[1]);

   if (pdfExtractText(pdf, PageNum, Flags, Area_Ptr, &text, &textLen))
   {
      UI32 len;
      BYTE* buf = NULL;
      len = ToUTF8W(text, textLen, &buf);
      RETURN_STRINGL(buf, len, 0);
   }else
   {
      RETURN_NULL();
   }
}

//PDF_EXP SI32   PDF_CALL pdfFileAttachAnnotA(const void* IPDF, double PosX, double PosY, TFileAttachIcon Icon, const char* Author, const char* Desc, const char* AFile, LBOOL Compress);
PHP_METHOD(dynapdf, FileAttachAnnot)
{
   zval **argv[7];
   PPDF* pdf;
   double PosX, PosY;
   char *Author, *Desc, *AFile;
   SI32 rc, Icon, Compress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_long_ex(argv[6]);

   PosX     = Z_DVAL_PP(argv[0]);
   PosY     = Z_DVAL_PP(argv[1]);
   Icon     = Z_LVAL_PP(argv[2]);
   Author   = Z_STRVAL_PP(argv[3]);
   Desc     = Z_STRVAL_PP(argv[4]);
   AFile    = Z_STRVAL_PP(argv[5]);
   Compress = Z_LVAL_PP(argv[6]);

   rc = pdfFileAttachAnnotA(pdf, PosX, PosY, Icon, Author, Desc, AFile, Compress);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfFileAttachAnnotExA(const PPDF* IPDF, double PosX, double PosY, TFileAttachIcon Icon, const char* FileName, const char* Author, const char* Desc, const void* Buffer, UI32 BufSize, LBOOL Compress);
PHP_METHOD(dynapdf, FileAttachAnnotEx)
{
   zval **argv[8];
   PPDF* pdf;
   double PosX, PosY;
   char *FileName, *Author, *Desc, *Buffer;
   SI32 rc, Icon, BufSize, Compress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_long_ex(argv[8]);

   PosX     = Z_DVAL_PP(argv[0]);
   PosY     = Z_DVAL_PP(argv[1]);
   Icon     = Z_LVAL_PP(argv[2]);
   FileName = Z_STRVAL_PP(argv[3]);
   Author   = Z_STRVAL_PP(argv[4]);
   Desc     = Z_STRVAL_PP(argv[5]);
   Buffer   = Z_STRVAL_PP(argv[6]);
   BufSize  = Z_STRLEN_PP(argv[6]);
   Compress = Z_LVAL_PP(argv[7]);

   rc = pdfFileAttachAnnotExA(pdf, PosX, PosY, Icon, FileName, Author, Desc, Buffer, BufSize, Compress);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfFileLink()
   const void* IPDF, double PosX, double PosY, double Width, double Height, const char* AFilePath); */
PHP_METHOD(dynapdf, FileLink)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char *AFilePath;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   AFilePath = Z_STRVAL_PP(argv[4]);

   rc = pdfFileLinkA(pdf, PosX, PosY, Width, Height, AFilePath);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfFindBookmark()
   const void* IPDF, SI32 DestPage, const char* Title); */
PHP_METHOD(dynapdf, FindBookmark)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 DestPage;
   char *Title;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   DestPage = Z_LVAL_PP(argv[0]);
   Title    = Z_STRVAL_PP(argv[1]);

   rc = pdfFindBookmarkA(pdf, DestPage, Title);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfFindEmbeddedFileA(const PPDF* IPDF, const char* Name);
PHP_METHOD(dynapdf, FindEmbeddedFile)
{
   zval **arg1;
   PPDF* pdf;
   char *Name;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Name = Z_STRVAL_PP(arg1);

   rc = pdfFindEmbeddedFileA(pdf, Name);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfFindFieldA(const void* IPDF, const char* Name);
PHP_METHOD(dynapdf, FindField)
{
   zval **arg1;
   PPDF* pdf;
   char *Name;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Name = Z_STRVAL_PP(arg1);

   rc = pdfFindFieldA(pdf, Name);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfFinishSignature(const void* IPDF, const void* PKCS7Obj, UI32 Length);
PHP_METHOD(dynapdf, FinishSignature)
{
   zval **arg1;
   PPDF* pdf;
   char *PKCS7Obj;
   SI32 rc, Length;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   PKCS7Obj = Z_STRVAL_PP(arg1);
   Length   = Z_STRLEN_PP(arg1);

   rc = pdfFinishSignature(pdf, PKCS7Obj, Length);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfFindLinkAnnot(const void* IPDF, const char* URL);
PHP_METHOD(dynapdf, FindLinkAnnot)
{
   zval **arg1;
   PPDF* pdf;
   char *URL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   URL = Z_STRVAL_PP(arg1);

   rc = pdfFindLinkAnnot(pdf, URL);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfFindNextBookmark()
   const void* IPDF); */
PHP_METHOD(dynapdf, FindNextBookmark)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfFindNextBookmark(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfFlattenAnnotOrField(const PPDF* IPDF, UI32 Index, TAnnotFlattenFlags Flags);
PHP_METHOD(dynapdf, FlattenAnnotOrField)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Index, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Index = Z_LVAL_PP(argv[0]);
   Flags = Z_LVAL_PP(argv[1]);

   rc = pdfFlattenAnnotOrField(pdf, Index, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfFlattenAnnots(const void* IPDF, TAnnotFlattenFlags Flags);
PHP_METHOD(dynapdf, FlattenAnnots)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags = Z_LVAL_PP(arg1);

   rc = pdfFlattenAnnots(pdf, Flags);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfFlattenForm()
   const void* IPDF); */
PHP_METHOD(dynapdf, FlattenForm)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfFlattenForm(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfFlushPageContent(const void* IPDF, struct TPDFStack* Stack);
PHP_METHOD(dynapdf, FlushPageContent)
{
   SI32 rc;
   struct dynapdf_object *obj;
   struct TPDFStack* Stack;

   DYNAPDF_FETCH_PDFDOC_EX();

   Stack = obj->pdfStack;

   if (!Stack)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "You must call InitStack() before this function can be called!");
      RETURN_NULL();
   }

   rc = pdfFlushPageContent(obj->pdf, Stack);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfFlushPages(const void* IPDF, TFlushPageFlags Flags);
PHP_METHOD(dynapdf, FlushPages)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags = Z_LVAL_PP(arg1);

   rc = pdfFlushPages(pdf, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfFlushPagesEx(const PPDF* IPDF, TFlushPageFlags Flags, UI32 LastPage);
PHP_METHOD(dynapdf, FlushPagesEx)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Flags, LastPage;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Flags    = Z_LVAL_PP(argv[0]);
   LastPage = Z_LVAL_PP(argv[1]);

   rc = pdfFlushPagesEx(pdf, Flags, LastPage);
   RETURN_BOOL(rc);
}

//PDF_EXP void   PDF_CALL pdfFreeImageBuffer(const void* IPDF);
PHP_METHOD(dynapdf, FreeImageBuffer)
{
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   pdfFreeImageBuffer(pdf);
}

//PDF_EXP LBOOL  PDF_CALL pdfFreeImageObj(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, FreeImageObj)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfFreeImageObj(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfFreeImageObjEx(const void* IPDF, const void* ImagePtr);
PHP_METHOD(dynapdf, FreeImageObjEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   PIMG* ImagePtr;
   SI32 rc, ptrHash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   ptrHash = Z_LVAL_PP(arg1);

   if ((ImagePtr = (PIMG*)GetPointer(obj, ptrHash, ptImage)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter ImagePtr contains an invalid value!");
      RETURN_BOOL(0);
   }else
   {
      rc = pdfFreeImageObjEx(obj->pdf, ImagePtr);
      RETURN_BOOL(rc);
   }
}

/* {{{ proto SI32 pdfFreePDF()
   const void* IPDF); */
PHP_METHOD(dynapdf, FreePDF)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC_FREE_PTR_LIST();
   rc = pdfFreePDF(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfFreeTextAnnot()
   const void* IPDF, double PosX, double PosY, double Width, double Height, const char* Author, const char* AText, TTextAlign Align); */
PHP_METHOD(dynapdf, FreeTextAnnot)
{
   zval **argv[7];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char *Author, *AText;
   TTextAlign Align;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   Author = Z_STRVAL_PP(argv[4]);
   AText  = Z_STRVAL_PP(argv[5]);
   Align  = Z_LVAL_PP(argv[6]);

   rc = pdfFreeTextAnnotA(pdf, PosX, PosY, Width, Height, Author, AText, Align);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfFreeUniBuf()
   const void* IPDF); */
PHP_METHOD(dynapdf, FreeUniBuf)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfFreeUniBuf(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetActionCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetActionCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetActionCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfGetActionHandle(const PPDF* IPDF, TObjType ObjType, UI32 ObjHandle, UI32 ActIndex)
PHP_METHOD(dynapdf, GetActionHandle)
{
   zval **argv[3];
   PPDF* pdf;
   TObjType ObjType;
   SI32 ObjHandle, ActIndex;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   ObjType   = Z_LVAL_PP(argv[0]);
   ObjHandle = Z_LVAL_PP(argv[1]);
   ActIndex  = Z_LVAL_PP(argv[2]);

   rc = pdfGetActionHandle(pdf, ObjType, ObjHandle, ActIndex);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetActionType()
   const void* IPDF, UI32 ActHandle); */
PHP_METHOD(dynapdf, GetActionType)
{
   zval **arg1;
   PPDF* pdf;
   SI32 ActHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   ActHandle  = Z_LVAL_PP(arg1);

   rc = pdfGetActionType(pdf, ActHandle);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetActionTypeEx()
   const void* IPDF, TObjType ObjType, UI32 ObjHandle, UI32 ActIndex); */
PHP_METHOD(dynapdf, GetActionTypeEx)
{
   zval **argv[3];
   PPDF* pdf;
   TObjType ObjType;
   SI32 ObjHandle, ActIndex;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   ObjType   = Z_LVAL_PP(argv[0]);
   ObjHandle = Z_LVAL_PP(argv[1]);
   ActIndex  = Z_LVAL_PP(argv[2]);

   rc = pdfGetActionTypeEx(pdf, ObjType, ObjHandle, ActIndex);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetActiveFont()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetActiveFont)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetActiveFont(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetAllocBy()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetAllocBy)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetAllocBy(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetAnnotBBox(const void* IPDF, UI32 Handle, struct TPDFRect ADDR BBox);
PHP_METHOD(dynapdf, GetAnnotBBox)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFRect BBox = {0.0, 0.0, 0.0, 0.0};
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if ((rc = pdfGetAnnotBBox(pdf, Handle, &BBox)) != 0)
   {
      array_init(return_value);
      add_assoc_double(return_value, "Left",   BBox.Left);
      add_assoc_double(return_value, "Bottom", BBox.Bottom);
      add_assoc_double(return_value, "Right",  BBox.Right);
      add_assoc_double(return_value, "Top",    BBox.Top);
   }
}

/* {{{ proto SI32 pdfGetAnnotCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetAnnotCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetAnnotCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetAnnotEx(const void* IPDF, UI32 Handle, struct TPDFAnnotationEx ADDR Annot);
PHP_METHOD(dynapdf, GetAnnotEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFAnnotationEx Annot;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if ((rc = pdfGetAnnotEx(obj->pdf, Handle, &Annot)) != 0)
   {
      array_init(return_value);
      ReturnAnnotEx(return_value, obj, &Annot);
   }
}

/* {{{ proto SI32 pdfGetAnnotFlags()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetAnnotFlags)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetAnnotFlags(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP char*  PDF_CALL pdfGetAnnotLink(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, GetAnnotLink)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   char* link;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle  = Z_LVAL_PP(arg1);
   link = pdfGetAnnotLink(pdf, Handle);
   if (link) ReturnStringA(return_value, link);
}

/* {{{ proto SI32 pdfGetAnnotType()
   const void* IPDF, UI32 Handle); */
PHP_METHOD(dynapdf, GetAnnotType)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle  = Z_LVAL_PP(arg1);
   rc = pdfGetAnnotType(pdf, Handle);
   RETURN_LONG(rc);
}
/* }}} */

// PHP specific
PHP_METHOD(dynapdf, GetAPIVersion)
{
   BYTE c;
   BYTE* src = (BYTE*)DYNAPDF_PHP_EXT_VERSION;
   SI32 number = 0;
   while (*src != 0)
   {
      c = *src++;
      if (c == '.') c = *src++;
      number = number * 10 + c - '0';
   }
   RETURN_LONG(number);
}

/* {{{ proto double pdfGetAscent()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetAscent)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetAscent(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetBarcodeDict(const PBCD* IBarcode, struct TPDFBarcode* Barcode);
PHP_METHOD(dynapdf, GetBarcodeDict)
{
   zval **arg1;
   struct dynapdf_object *obj;
   PBCD* IBarcode;
   struct TPDFBarcode Barcode;
   SI32 rc, hash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   hash = Z_LVAL_PP(arg1);

   if ((IBarcode = (PBCD*)GetPointer(obj, hash, ptBarcode)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IBarcode contains an invalid value!");
      RETURN_NULL();
   }
   Barcode.StructSize = sizeof(Barcode);
   if ((rc = pdfGetBarcodeDict(IBarcode, &Barcode)) != 0)
   {
      array_init(return_value);
      AddString(return_value, "Caption", Barcode.CaptionA, Barcode.CaptionW);
      add_assoc_double(return_value, "ECC",          Barcode.ECC);
      add_assoc_double(return_value, "Height",       Barcode.Height);
      add_assoc_double(return_value, "nCodeWordCol", Barcode.nCodeWordCol);
      add_assoc_double(return_value, "nCodeWordRow", Barcode.nCodeWordRow);
      add_assoc_long(return_value, "Resolution", Barcode.Resolution);
      AddStringA(return_value, "Symbology", Barcode.Symbology);
      add_assoc_double(return_value, "Version",    Barcode.Version);
      add_assoc_double(return_value, "Width",      Barcode.Width);
      add_assoc_double(return_value, "XSymHeight", Barcode.XSymHeight);
      add_assoc_double(return_value, "XSymWidth",  Barcode.XSymWidth);
   }
}

/* {{{ proto SI32 pdfGetBBox()
   const void* IPDF, TPageBoundary Boundary, struct TPDFRect * BBox); */
PHP_METHOD(dynapdf, GetBBox)
{
   zval **arg1;
   PPDF* pdf;
   TPageBoundary Boundary;
   struct TPDFRect BBox = {0.0, 0.0, 0.0, 0.0};
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Boundary = Z_LVAL_PP(arg1);

   if ((rc = pdfGetBBox(pdf, Boundary, &BBox)) != 0)
   {
      array_init(return_value);
      add_assoc_double(return_value, "Left",   BBox.Left);
      add_assoc_double(return_value, "Bottom", BBox.Bottom);
      add_assoc_double(return_value, "Right",  BBox.Right);
      add_assoc_double(return_value, "Top",    BBox.Top);
   }
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetBidiMode(const void* IPDF);
PHP_METHOD(dynapdf, GetBidiMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetBidiMode(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetBookmark(const void* IPDF, SI32 Handle, struct TBookmark ADDR Bmk);
PHP_METHOD(dynapdf, GetBookmark)
{
   zval **arg1;
   PPDF* pdf;
   struct TBookmark Bmk;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if ((rc = pdfGetBookmark(pdf, Handle, &Bmk)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Color",    Bmk.Color);
      add_assoc_long(return_value, "DestPage", Bmk.DestPage);
      AddPDFRect(return_value, "DestPos", &Bmk.DestPos);
      add_assoc_long(return_value, "DestType", Bmk.DestType);
      add_assoc_bool(return_value, "Open",     Bmk.Open);
      add_assoc_long(return_value, "Parent",   Bmk.Parent);
      add_assoc_long(return_value, "Style",    Bmk.Style);
      if (Bmk.Title)
      {
         if (Bmk.Unicode)
            AddStringExW(return_value, "Title", (const UI16*)Bmk.Title, Bmk.TitleLen);
         else
            AddStringExA(return_value, "Title", Bmk.Title, Bmk.TitleLen);
      }
   }
}

/* {{{ proto SI32 pdfGetBookmarkCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetBookmarkCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetBookmarkCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetBookmarkEx(const PPDF* IPDF, UI32 Handle, struct TPDFBookmark* Bmk);
PHP_METHOD(dynapdf, GetBookmarkEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFBookmark Bmk;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   Bmk.StructSize = sizeof(Bmk);
   if ((rc = pdfGetBookmarkEx(obj->pdf, Handle, &Bmk)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value,  "Action",     Bmk.Action);
      add_assoc_long(return_value,  "Color",      Bmk.Color);
      add_assoc_long(return_value,  "DestPage",   Bmk.DestPage);
      AddPDFRect(return_value,      "DestPos",   &Bmk.DestPos);
      add_assoc_long(return_value,  "DestType",   Bmk.DestType);
      AddString(return_value,       "DestName",   Bmk.NamedDestA, Bmk.NamedDestW);
      add_assoc_bool(return_value,  "Open",       Bmk.Open);
      add_assoc_long(return_value,  "Parent",     Bmk.Parent);
      AddPointer(return_value, obj, "StructElem", Bmk.StructElem, ptStructElem);
      add_assoc_long(return_value,  "Style",      Bmk.Style);
      AddStringEx(return_value,     "Title",      Bmk.TitleA, Bmk.TitleW, Bmk.TitleLen);
   }
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfGetBorderStyle()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetBorderStyle)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetBorderStyle(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto  char *pdfGetBuffer()
   SI32 pdfGetBuffer(const void* IPDF); */
PHP_METHOD(dynapdf, GetBuffer)
{
   PPDF* pdf;
   char *buffer;
   UI32  bufSize = 0;

   DYNAPDF_FETCH_PDFDOC();
   buffer = pdfGetBuffer(pdf, &bufSize);
   if (buffer) RETURN_STRINGL(buffer, bufSize, 1);
}
/* }}} */

// PHP Specific
PHP_METHOD(dynapdf, GetBufSize)
{
   PPDF* pdf;
   UI32  bufSize = 0;

   DYNAPDF_FETCH_PDFDOC();
   // GetBuffer() can be called arbitrary often
   pdfGetBuffer(pdf, &bufSize);
   RETURN_LONG(bufSize);
}

/* {{{ proto double pdfGetCapHeight()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetCapHeight)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetCapHeight(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto double pdfGetCharacterSpacing()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetCharacterSpacing)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetCharacterSpacing(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto SI32 pdfGetCheckBoxChar()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetCheckBoxChar)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetCheckBoxChar(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetCheckBoxCharEx(const void* IPDF, UI32 AField);
PHP_METHOD(dynapdf, GetCheckBoxCharEx)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, AField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField = Z_LVAL_PP(arg1);

   rc = pdfGetCheckBoxCharEx(pdf, AField);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetCheckBoxDefState(const void* IPDF, UI32 AField);
PHP_METHOD(dynapdf, GetCheckBoxDefState)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, AField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField = Z_LVAL_PP(arg1);

   rc = pdfGetCheckBoxDefState(pdf, AField);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetCMap(const void* IPDF, UI32 Index, struct TPDFCMap* CMap);
PHP_METHOD(dynapdf, GetCMap)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFCMap CMap;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   memset(&CMap, 0, sizeof(CMap));
   CMap.StructSize = sizeof(CMap);
   if ((rc = pdfGetCMap(pdf, Index, &CMap)) != 0)
   {
      array_init(return_value);
      if (CMap.BaseCMap) AddStringA(return_value, "BaseCMap", CMap.BaseCMap);
      add_assoc_long(return_value,   "CIDCount",       CMap.CIDCount);
      if (CMap.CMapName) AddStringA(return_value, "CMapName", CMap.CMapName);
      add_assoc_long(return_value,   "CMapType",       CMap.CMapType);
      add_assoc_double(return_value, "CMapVersion",    CMap.CMapVersion);
      if (CMap.DSCBaseCMap) AddStringA(return_value, "DSCBaseCMap", CMap.DSCBaseCMap);
      add_assoc_double(return_value, "DSCCMapVersion", CMap.DSCCMapVersion);
      if (CMap.DSCResName) AddStringA(return_value, "DSCResName", CMap.DSCResName);
      if (CMap.DSCTitle)   AddStringA(return_value, "DSCTitle", CMap.DSCTitle);
      AddString(return_value, "FileName", CMap.FileNameA, CMap.FileNameW);
      AddString(return_value, "FilePath", CMap.FilePathA, CMap.FilePathW);
      if (CMap.Ordering) AddStringA(return_value, "Ordering", CMap.Ordering);
      if (CMap.Registry) AddStringA(return_value, "Registry", CMap.Registry);
      add_assoc_long(return_value, "Supplement",  CMap.Supplement);
      add_assoc_long(return_value, "WritingMode", CMap.WritingMode);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetCMapCount(const void* IPDF);
PHP_METHOD(dynapdf, GetCMapCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetCMapCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetCollectionInfo(const PPDF* IPDF, struct TPDFCollectionInfo* Value);
PHP_METHOD(dynapdf, GetCollectionInfo)
{
   SI32 rc;
   PPDF* pdf;
   struct TPDFCollectionInfo Value;
   DYNAPDF_FETCH_PDFDOC();
   Value.StructSize = sizeof(Value);
   if ((rc = pdfGetCollectionInfo(pdf, &Value)) != 0)
   {
      add_assoc_long(return_value,   "InitalFile", Value.InitalFile);
      add_assoc_long(return_value,   "InitalView", Value.InitalView);
      AddStringA(return_value,       "SortBy",     Value.SortBy);
      add_assoc_bool(return_value,   "SortDesc",   Value.SortDesc);
      add_assoc_long(return_value,   "SplitInfo",  Value.SplitInfo);
      add_assoc_double(return_value, "SplitPos",   Value.SplitPos);
   }
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetColorSpace()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetColorSpace)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetColorSpace(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetColorSpaceCount(const void* IPDF);
PHP_METHOD(dynapdf, GetColorSpaceCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetColorSpaceCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetColorSpaceObj(const void* IPDF, UI32 Handle, struct TPDFColorSpaceObj* CS);
PHP_METHOD(dynapdf, GetColorSpaceObj)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFColorSpaceObj CS;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if ((rc = pdfGetColorSpaceObj(obj->pdf, Handle, &CS)) != 0)
   {
      array_init(return_value);
      ReturnColorSpaceStruct(return_value, obj, &CS);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetColorSpaceObjEx(const void* IColorSpace, struct TPDFColorSpaceObj* CS);
PHP_METHOD(dynapdf, GetColorSpaceObjEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFColorSpaceObj CS;
   const void* IColorSpace;
   SI32 rc, ptrHash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   ptrHash = Z_LVAL_PP(arg1);

   if ((IColorSpace = GetPointer(obj, ptrHash, ptColorSpace)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IColorSpace contains an invalid value!");
      RETURN_NULL();
   }

   if ((rc = pdfGetColorSpaceObjEx(IColorSpace, &CS)) != 0)
   {
      array_init(return_value);
      ReturnColorSpaceStruct(return_value, obj, &CS);
   }
}

/* {{{ proto SI32 pdfGetCompressionFilter()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetCompressionFilter)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetCompressionFilter(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetCompressionLevel()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetCompressionLevel)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetCompressionLevel(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetContent(const void* IPDF, char* ADDR Buffer);
PHP_METHOD(dynapdf, GetContent)
{
   SI32 rc;
   PPDF* pdf;
   char* buffer = NULL;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetContent(pdf, &buffer);
   if (rc > 0) RETURN_STRINGL(buffer, rc, 1);
}

/* {{{ proto SI32 pdfGetDefBitsPerPixel()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetDefBitsPerPixel)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetDefBitsPerPixel(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto double pdfGetDescent()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetDescent)
{
   PPDF* pdf;
   double drc;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetDescent(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetDeviceNAttributes(void* IAttributes, struct TDeviceNAttributes* Attributes);
PHP_METHOD(dynapdf, GetDeviceNAttributes)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TDeviceNAttributes Attributes;
   void* IAttributes;
   SI32 rc, ptrHash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   ptrHash = Z_LVAL_PP(arg1);

   if ((IAttributes = (void*)GetPointer(obj, ptrHash, ptDeviceNAttr)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IColorSpace contains an invalid value!");
      RETURN_NULL();
   }

   if ((rc = pdfGetDeviceNAttributes(IAttributes, &Attributes)) != 0)
   {
      array_init(return_value);
      if (Attributes.IProcessColorSpace) AddPointer(return_value, obj, "IProcessColorSpace", Attributes.IProcessColorSpace, ptColorSpace);
      AddUTF8StringArray(return_value, "ProcessColorants", (char**)Attributes.ProcessColorants, Attributes.ProcessColorantsCount);
      AddPointerArray(return_value, obj, "Separations", Attributes.Separations, Attributes.SeparationsCount, ptColorSpace);
      if (Attributes.IMixingHints) AddPointer(return_value, obj, "IMixingHints", Attributes.IMixingHints, ptMixingHints);
   }
}

/* {{{ proto SI32 pdfGetDocInfo()
   const void* IPDF, TDocumentInfo DInfo, unsigned short* * Value); */
PHP_METHOD(dynapdf, GetDocInfo)
{
   zval **arg1;
   PPDF* pdf;
   UI16* value = NULL;
   SI32 rc, DInfo;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   DInfo = Z_LVAL_PP(arg1);
   rc = pdfGetDocInfo(pdf, DInfo, &value);
   if (rc > 0) ReturnStringExW(return_value, value, rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetDocInfoCount(const void* IPDF);
PHP_METHOD(dynapdf, GetDocInfoCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetDocInfoCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetDocInfoEx(const void* IPDF, UI32 Index, TDocumentInfo ADDR DInfo, char* ADDR Key, char* ADDR Value, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetDocInfoEx)
{
   zval **arg1;
   PPDF* pdf;
   char* key = NULL, *value = NULL;
   TDocumentInfo DInfo = diAuthor;
   SI32 rc, Index, unicode;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);
   rc = pdfGetDocInfoEx(pdf, Index, &DInfo, &key, &value, &unicode);
   if (rc > 0 || key)
   {
      array_init(return_value);
      if (key) AddStringA(return_value, "Key", key);
      add_assoc_long(return_value, "DInfo", DInfo);
      if (unicode)
         AddStringExW(return_value, "Value", (const UI16*)value, rc);
      else
         AddStringExA(return_value, "Value", value, rc);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetDocUsesTransparency(const void* IPDF, UI32 Flags);
PHP_METHOD(dynapdf, GetDocUsesTransparency)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags = Z_LVAL_PP(arg1);
   rc = pdfGetDocUsesTransparency(pdf, Flags);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfGetDrawDirection()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetDrawDirection)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetDrawDirection(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP char*  PDF_CALL pdfGetDynaPDFVersion(void);
PHP_METHOD(dynapdf, GetDynaPDFVersion)
{
   char* rc;
   rc = pdfGetDynaPDFVersion();
   RETURN_STRING(rc, 1);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetEmbeddedFile(const void* IPDF, UI32 Handle, struct TPDFFileSpec* FileSpec, LBOOL Decompress);
PHP_METHOD(dynapdf, GetEmbeddedFile)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   struct TPDFFileSpec FileSpec;
   SI32 Handle, Decompress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle     = Z_LVAL_PP(argv[0]);
   Decompress = Z_LVAL_PP(argv[1]);

   if (pdfGetEmbeddedFile(obj->pdf, Handle, &FileSpec, Decompress) != 0)
   {
      array_init(return_value);
      AddFileSpec(return_value, obj, &FileSpec);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetEmbeddedFileCount(const void* IPDF);
PHP_METHOD(dynapdf, GetEmbeddedFileCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetEmbeddedFileCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetEmbeddedFileNode(const IEFN* IEF, struct TPDFEmbFileNode* F, LBOOL Decompress);
PHP_METHOD(dynapdf, GetEmbeddedFileNode)
{
   zval **argv[2];
   IEFN* IEF;
   struct dynapdf_object *obj;
   struct TPDFEmbFileNode F;
   UI32 hash, Decompress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   hash       = Z_LVAL_PP(argv[0]);
   Decompress = Z_LVAL_PP(argv[1]);

   if (!hash || (IEF = (IEFN*)GetPointer(obj, hash, ptEmbFileNode)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IEF must be a valid safe pointer!");
      RETURN_NULL();
   }

   F.StructSize = sizeof(F);
   if (pdfGetEmbeddedFileNode(IEF, &F, Decompress) != 0)
   {
      zval* node;
      array_init(return_value);
      AddStringA(return_value, "Name", F.Name);

      ALLOC_INIT_ZVAL(node);
      array_init(node);
         ADD_ASSOC_BIN_STRING(node, "Buffer", F.EF.Buffer, F.EF.BufSize);
         add_assoc_bool(node, "Compressed", F.EF.Compressed);
         if (F.EF.ColItem) AddPointer(node, obj, "ColItem", F.EF.ColItem, ptColItem);
         if (F.EF.Name)
         {
            if (F.EF.NameUnicode)
               AddStringW(node, "Name", (const UI16*)F.EF.Name);
            else
               AddStringA(node, "Name", F.EF.Name);
         }
         AddStringA(node, "FileName", F.EF.FileName);
         add_assoc_bool(node, "IsURL", F.EF.IsURL);
         if (F.EF.UF)
         {
            if (F.EF.UFUnicode)
               AddStringW(node, "UF", (UI16*)F.EF.UF);
            else
               AddStringA(node, "UF", F.EF.UF);
         }
         if (F.EF.Desc)
         {
            if (F.EF.DescUnicode)
               AddStringW(node, "Desc", (const UI16*)F.EF.Desc);
            else
               AddStringA(node, "Desc", F.EF.Desc);
         }
         add_assoc_long(node, "FileSize", F.EF.FileSize);
         AddStringA(node, "MIMEType", F.EF.MIMEType);
         AddStringA(node, "CreateDate", F.EF.CreateDate);
         AddStringA(node, "ModDate", F.EF.ModDate);
         ADD_ASSOC_BIN_STRING(node, "CheckSum", F.EF.CheckSum, 16);
      add_assoc_zval(return_value, "EF", node);
      if (F.NextNode) AddPointer(return_value, obj, "NextNode", F.NextNode, ptEmbFileNode);
   }
}

/* {{{ proto double pdfGetEMFPatternDistance()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetEMFPatternDistance)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetEMFPatternDistance(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetErrLogMessage(const void* IPDF, UI32 Index, struct TPDFError* Err);
PHP_METHOD(dynapdf, GetErrLogMessage)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFError Err;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   Err.StructSize = sizeof(Err);
   if ((rc = pdfGetErrLogMessage(pdf, Index, &Err)) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "Message", Err.Message);
      add_assoc_long(return_value, "ObjNum", Err.ObjNum);
      add_assoc_long(return_value, "Offset", Err.Offset);
      AddStringA(return_value, "SrcFile", Err.SrcFile);
      add_assoc_long(return_value, "SrcLine", Err.SrcLine);
      add_assoc_long(return_value, "ErrCode", Err.ErrCode);
      add_assoc_long(return_value, "ErrType", Err.ErrType);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetErrLogMessageCount(const void* IPDF);
PHP_METHOD(dynapdf, GetErrLogMessageCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetErrLogMessageCount(pdf);
   RETURN_LONG(rc);
}

/* {{{ proto  char *pdfGetErrorMessage()
   char *pdfGetErrorMessage(const void* IPDF); */
PHP_METHOD(dynapdf, GetErrorMessage)
{
   PPDF* pdf;
   char* ErrMsg;
   DYNAPDF_FETCH_PDFDOC();
   ErrMsg = pdfGetErrorMessage(pdf);
   ReturnStringA(return_value, ErrMsg);
}
/* }}} */

/* {{{ proto SI32 pdfGetErrorMode()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetErrorMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetErrorMode(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetField(const void* IPDF, UI32 Handle, struct TPDFField* Field);
PHP_METHOD(dynapdf, GetField)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFField Field;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if ((rc = pdfGetField(pdf, Handle, &Field)) != 0)
   {
      array_init(return_value);
      ReturnField(return_value, &Field);
   }
}

/* {{{ proto  ulong pdfGetFieldBackColor()
   ulong pdfGetFieldBackColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetFieldBackColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFieldBackColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto  ulong pdfGetFieldBorderColor()
   ulong pdfGetFieldBorderColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetFieldBorderColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFieldBorderColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetFieldBorderStyle()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, GetFieldBorderStyle)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldBorderStyle(pdf, AField);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP double PDF_CALL pdfGetFieldBorderWidth(const void* IPDF, UI32 AField);
PHP_METHOD(dynapdf, GetFieldBorderWidth)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   drc = pdfGetFieldBorderWidth(pdf, AField);
   RETURN_DOUBLE(drc);
}

//PDF_EXP SI32   PDF_CALL pdfGetFieldCalcOrder(const PPDF* IPDF, UI32** Out);
PHP_METHOD(dynapdf, GetFieldCalcOrder)
{
   PPDF* pdf;
   SI32 n;
   UI32* Out;

   DYNAPDF_FETCH_PDFDOC();

   if ((n = pdfGetFieldCalcOrder(pdf, &Out)) > 0)
   {
      SI32 i;
      array_init(return_value);
      for (i = 0; i < n; i++)
         add_next_index_long(return_value, Out[i]);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetFieldChoiceValue(const void* IPDF, UI32 AField, UI32 ValIndex, struct TPDFChoiceValue* Value);
PHP_METHOD(dynapdf, GetFieldChoiceValue)
{
   zval **argv[2];
   PPDF* pdf;
   struct TPDFChoiceValue Value;
   SI32 rc, AField, ValIndex;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   AField   = Z_LVAL_PP(argv[0]);
   ValIndex = Z_LVAL_PP(argv[1]);

   Value.StructSize = sizeof(Value);

   if ((rc = pdfGetFieldChoiceValue(pdf, AField, ValIndex, &Value)) != 0)
   {
      array_init(return_value);
      AddStringEx(return_value, "ExpValue", Value.ExpValueA, Value.ExpValueW, Value.ExpValueLen);
      AddStringEx(return_value, "Value", Value.ValueA, Value.ValueW, Value.ValueLen);
      add_assoc_bool(return_value, "Selected", Value.Selected);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetFieldColor(const void* IPDF, UI32 AField, TFieldColor ColorType, SI32 ADDR ColorSpace, UI32 ADDR Color);
PHP_METHOD(dynapdf, GetFieldColor)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 Color;
   SI32 rc, AField, ColorType, ColorSpace;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   AField    = Z_LVAL_PP(argv[0]);
   ColorType = Z_LVAL_PP(argv[1]);

   if ((rc = pdfGetFieldColor(pdf, AField, ColorType, &ColorSpace, &Color)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "ColorSpace", ColorSpace);
      add_assoc_long(return_value, "Color",      Color);
   }
}

/* {{{ proto SI32 pdfGetFieldCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetFieldCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFieldCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetFieldEx(const void* IPDF, UI32 Handle, struct TPDFFieldEx* Field);
PHP_METHOD(dynapdf, GetFieldEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFFieldEx Field;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   Field.StructSize = sizeof(Field);

   if ((rc = pdfGetFieldEx(obj->pdf, Handle, &Field)) != 0)
   {
      array_init(return_value);
      ReturnFieldEx(return_value, obj, &Field);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetFieldEx2(const IFLD* IField, struct TPDFFieldEx* Field);
PHP_METHOD(dynapdf, GetFieldEx2)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFFieldEx Field;
   IFLD* IField;
   SI32 rc, hash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   hash = Z_LVAL_PP(arg1);

   if ((IField = (IFLD*)GetPointer(obj, hash, ptField)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IField contains an invalid value!");
      RETURN_NULL();
   }

   Field.StructSize = sizeof(Field);

   if ((rc = pdfGetFieldEx2(IField, &Field)) != 0)
   {
      array_init(return_value);
      ReturnFieldEx(return_value, obj, &Field);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetFieldExpValue(const void* IPDF, UI32 AField, char* ADDR Value);
PHP_METHOD(dynapdf, GetFieldExpValue)
{
   zval **arg1;
   PPDF* pdf;
   char* Value = NULL;
   SI32 rc, AField, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   if ((rc = pdfGetFieldExpValue(pdf, AField, &Value)) > 0)
   {
      ReturnStringExA(return_value, Value, rc);
   }
}

/* {{{ proto SI32 pdfGetFieldExpValCount()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, GetFieldExpValCount)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldExpValCount(pdf, AField);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetFieldExpValueEx(const void* IPDF, UI32 AField, UI32 ValIndex, char* ADDR Value, char* ADDR ExpValue, LBOOL ADDR Selected);
PHP_METHOD(dynapdf, GetFieldExpValueEx)
{
   zval **argv[2];
   PPDF* pdf;
   char* Value = NULL, *ExpValue = NULL;
   SI32 rc, AField, ValIndex, Selected = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField   = Z_LVAL_PP(argv[0]);
   ValIndex = Z_LVAL_PP(argv[1]);

   if ((rc = pdfGetFieldExpValueEx(pdf, AField, ValIndex, &Value, &ExpValue, &Selected)) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "Value", Value);
      AddStringA(return_value, "ExpValue", ExpValue);
      add_assoc_bool(return_value, "Selected", Selected);
   }
}

/* {{{ proto SI32 pdfGetFieldFlags()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, GetFieldFlags)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldFlags(pdf, AField);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetFieldGroupType(const void* IPDF, UI32 AField);
PHP_METHOD(dynapdf, GetFieldGroupType)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldGroupType(pdf, AField);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetFieldHighlightMode(const void* IPDF, UI32 AField);
PHP_METHOD(dynapdf, GetFieldHighlightMode)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldHighlightMode(pdf, AField);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetFieldIndex()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, GetFieldIndex)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, AField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldIndex(pdf, AField);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetFieldMapName(const void* IPDF, UI32 AField, void* ADDR Value, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetFieldMapName)
{
   zval **arg1;
   PPDF* pdf;
   void* Value = NULL;
   SI32 rc, AField, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   if ((rc = pdfGetFieldMapName(pdf, AField, &Value, &Unicode)) > 0)
   {
      if (Unicode)
         ReturnStringExW(return_value, (const UI16*)Value, rc);
      else
         ReturnStringExA(return_value, (const char*)Value, rc);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetFieldName(const void* IPDF, UI32 AField, char* ADDR Name);
PHP_METHOD(dynapdf, GetFieldName)
{
   zval **arg1;
   PPDF* pdf;
   char* Name = NULL;
   SI32 rc, AField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   if ((rc = pdfGetFieldName(pdf, AField, &Name)) > 0)
   {
      if (StrLenA(Name) != (UI32)rc)
         ReturnStringExW(return_value, (const UI16*)Name, rc);
      else
         ReturnStringExA(return_value, Name, rc);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetFieldOrientation(const void* IPDF, UI32 AField);
PHP_METHOD(dynapdf, GetFieldOrientation)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldOrientation(pdf, AField);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetFieldTextAlign()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, GetFieldTextAlign)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   rc = pdfGetFieldTextAlign(pdf, AField);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto  ulong pdfGetFieldTextColor()
   ulong pdfGetFieldTextColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetFieldTextColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFieldTextColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetFieldToolTip(const void* IPDF, UI32 AField, void* ADDR Value, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetFieldToolTip)
{
   zval **arg1;
   PPDF* pdf;
   void* Value = NULL;
   SI32 rc, AField, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField  = Z_LVAL_PP(arg1);

   if ((rc = pdfGetFieldToolTip(pdf, AField, &Value, &Unicode)) > 0)
   {
      if (Unicode)
         ReturnStringExW(return_value, (const UI16*)Value, rc);
      else
         ReturnStringExA(return_value, (const char*)Value, rc);
   }
}

/* {{{ proto SI32 pdfGetFieldType()
   const void* IPDF, UI32 AField); */
PHP_METHOD(dynapdf, GetFieldType)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField = Z_LVAL_PP(arg1);

   rc = pdfGetFieldType(pdf, AField);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetFileSpec(const IFSR* IFS, struct TPDFFileSpecEx* F);
PHP_METHOD(dynapdf, GetFileSpec)
{
   zval **arg1;
   IFSR* IFS;
   struct dynapdf_object *obj;
   struct TPDFFileSpecEx F;
   UI32 hash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   hash = Z_LVAL_PP(arg1);

   if (!hash || (IFS = (IFSR*)GetPointer(obj, hash, ptFileSpecEx)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFS must be a valid safe pointer!");
      RETURN_NULL();
   }

   F.StructSize = sizeof(F);
   if (pdfGetFileSpec(IFS, &F) != 0)
   {
      array_init(return_value);
      AddFileSpecEx(return_value, obj, &F);
   }
}

/* {{{ proto  ulong pdfGetFillColor()
   ulong pdfGetFillColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetFillColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFillColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL fntGetFont(const PFNT* IFont, struct TPDFFontObj* F);
PHP_METHOD(dynapdf, GetFont)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFFontObj F;
   PFNT* IFont;
   SI32 rc, hash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   hash = Z_LVAL_PP(arg1);

   if ((IFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFont contains an invalid value!");
      RETURN_NULL();
   }

   if ((rc = fntGetFont(IFont, &F)) != 0)
   {
      array_init(return_value);
      ReturnFontObj(return_value, &F);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetFontCount(const void* IPDF);
PHP_METHOD(dynapdf, GetFontCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFontCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetFontEx(const void* IPDF, UI32 Handle, struct TPDFFontObj* F);
PHP_METHOD(dynapdf, GetFontEx)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFFontObj F;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if ((rc = pdfGetFontEx(pdf, Handle, &F)) != 0)
   {
      array_init(return_value);
      ReturnFontObj(return_value, &F);
   }
}

//PDF_EXP LBOOL  PDF_CALL fntGetFontInfo(const PFNT* IFont, struct TPDFFontInfo* F);
PHP_METHOD(dynapdf, GetFontInfo)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFFontInfo F;
   PFNT* IFont;
   SI32 rc, hash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   hash = Z_LVAL_PP(arg1);

   if ((IFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFont contains an invalid value!");
      RETURN_NULL();
   }
   F.StructSize = sizeof(F);
   if ((rc = fntGetFontInfo(IFont, &F)) != 0)
   {
      array_init(return_value);
      ReturnFontInfo(return_value, &F);
   }
}

// PDF_EXP LBOOL  PDF_CALL pdfGetFontInfoEx(const void* IPDF, UI32 Handle, struct TPDFFontInfo* F);
PHP_METHOD(dynapdf, GetFontInfoEx)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFFontInfo F;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   F.StructSize = sizeof(F);
   if ((rc = pdfGetFontInfoEx(pdf, Handle, &F)) != 0)
   {
      array_init(return_value);
      ReturnFontInfo(return_value, &F);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetFontMetrics(const PPDF* IPDF, UI32 Handle, struct TPDFFontMetrics* Metrics);
PHP_METHOD(dynapdf, GetFontMetrics)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFFontMetrics Metrics;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   Metrics.StructSize = sizeof(Metrics);
   if ((rc = pdfGetFontMetrics(pdf, Handle, &Metrics)) != 0)
   {
      array_init(return_value);
      ReturnFontMetrics(return_value, &Metrics);
   }
}

/* {{{ proto SI32 pdfGetFontOrigin()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetFontOrigin)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFontOrigin(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfGetFontSearchOrder(const void* IPDF, TFontBaseType Order[4]);
PHP_METHOD(dynapdf, GetFontSearchOrder)
{
   PPDF* pdf;
   TFontBaseType Order[4];
   DYNAPDF_FETCH_PDFDOC();
   pdfGetFontSearchOrder(pdf, Order);
   array_init(return_value);
   add_next_index_long(return_value, Order[0]);
   add_next_index_long(return_value, Order[1]);
   add_next_index_long(return_value, Order[2]);
   add_next_index_long(return_value, Order[3]);
}

/* {{{ proto SI32 pdfGetFontSelMode()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetFontSelMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFontSelMode(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP double PDF_CALL pdfGetFontSize(const PPDF* IPDF);
PHP_METHOD(dynapdf, GetFontSize)
{
   double rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFontSize(pdf);
   RETURN_DOUBLE(rc);
}

/* {{{ proto SI32 pdfGetFontWeight()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetFontWeight)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetFontWeight(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto double pdfGetFTextHeight()
   const void* IPDF, TTextAlign Align, const char* AText); */
PHP_METHOD(dynapdf, GetFTextHeight)
{
   zval **argv[2];
   PPDF* pdf;
   TTextAlign Align;
   char *AText;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Align = Z_LVAL_PP(argv[0]);
   AText = Z_STRVAL_PP(argv[1]);

   drc = pdfGetFTextHeightA(pdf, Align, AText);
   RETURN_DOUBLE(drc);
}
/* }}} */


/* {{{ proto double pdfGetFTextHeightEx()
   const void* IPDF, double Width, TTextAlign Align, const char* AText); */
PHP_METHOD(dynapdf, GetFTextHeightEx)
{
   zval **argv[3];
   PPDF* pdf;
   double Width;
   TTextAlign Align;
   char *AText;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   Width = Z_DVAL_PP(argv[0]);
   Align = Z_LVAL_PP(argv[1]);
   AText = Z_STRVAL_PP(argv[2]);

   drc = pdfGetFTextHeightExA(pdf, Width, Align, AText);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetGlyphIndex(const PPDF* IPDF, UI32 Index);
PHP_METHOD(dynapdf, GetGlyphIndex)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   rc = pdfGetGlyphIndex(pdf, Index);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetGlyphOutline(const PPDF* IPDF, UI32 Index, struct TPDFGlyphOutline* Outline);
PHP_METHOD(dynapdf, GetGlyphOutline)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   if ((rc = pdfGetGlyphOutline(pdf, Index, NULL)) > -1)
   {
      struct TPDFGlyphOutline g;
      memset(&g, 0, sizeof(g));
      g.Outline = (struct TI32Point*)emalloc(rc * sizeof(struct TI32Point));
      rc = pdfGetGlyphOutline(pdf, Index, &g);

      add_assoc_double(return_value, "AdvanceX", g.AdvanceX);
      add_assoc_double(return_value, "AdvanceY", g.AdvanceY);
      add_assoc_double(return_value, "OriginX",  g.OriginX);
      add_assoc_double(return_value, "OriginY",  g.OriginY);
      add_assoc_long(return_value,   "Lsb",      g.Lsb);
      add_assoc_long(return_value,   "Tsb",      g.Tsb);
      add_assoc_bool(return_value,   "HaveBBox", g.HaveBBox);
      if (g.HaveBBox)
      {
         zval* node;
         ALLOC_INIT_ZVAL(node);
         array_init(node);
         add_assoc_long(node, "MinX", g.BBox.MinX);
         add_assoc_long(node, "MinY", g.BBox.MinY);
         add_assoc_long(node, "MaxX", g.BBox.MaxX);
         add_assoc_long(node, "MaxY", g.BBox.MaxY);
         add_assoc_zval(return_value, "BBox", node);
      }
      AddPointArray(return_value, "Outline", g.Outline, g.Size);
      efree(g.Outline);
   }else
      RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetGoToAction(const PPDF* IPDF, UI32 Handle, struct TPDFGoToAction* Action);
PHP_METHOD(dynapdf, GetGoToAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFGoToAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetGoToAction(obj->pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "DestPage", Action.DestPage);
      if (Action.DestPos) AddFloatArray(return_value, "DestPos", Action.DestPos, 4);
      add_assoc_long(return_value, "DestType", Action.DestType);
      if (Action.DestFile) AddPointer(return_value, obj, "DestFile", Action.DestFile, ptFileSpecEx);
      AddString(return_value, "DestName", Action.DestNameA, Action.DestNameW);
      add_assoc_long(return_value, "NewWindow", Action.NewWindow);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetGoToRAction(const PPDF* IPDF, UI32 Handle, struct TPDFGoToAction* Action);
PHP_METHOD(dynapdf, GetGoToRAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFGoToAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetGoToRAction(obj->pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "DestPage", Action.DestPage);
      if (Action.DestPos) AddFloatArray(return_value, "DestPos", Action.DestPos, 4);
      add_assoc_long(return_value, "DestType", Action.DestType);
      if (Action.DestFile) AddPointer(return_value, obj, "DestFile", Action.DestFile, ptFileSpecEx);
      AddString(return_value, "DestName", Action.DestNameA, Action.DestNameW);
      add_assoc_long(return_value, "NewWindow", Action.NewWindow);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
   }
}

//PDF_EXP UI32   PDF_CALL pdfGetGStateFlags(const void* IPDF);
PHP_METHOD(dynapdf, GetGStateFlags)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetGStateFlags(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetHideAction(const PPDF* IPDF, UI32 Handle, struct TPDFHideAction* Action);
PHP_METHOD(dynapdf, GetHideAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFHideAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetHideAction(obj->pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      AddPointerArray(return_value, obj, "Fields", (void**)Action.Fields, Action.FieldsCount, ptField);
      add_assoc_bool(return_value, "Hide", Action.Hide);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
   }
}

/* {{{ proto  ulong pdfGetIconColor()
   ulong pdfGetIconColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetIconColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetIconColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP char*  PDF_CALL pdfGetImageBuffer(const void* IPDF, UI32 ADDR BufSize);
PHP_METHOD(dynapdf, GetImageBuffer)
{
   char* rc;
   PPDF* pdf;
   UI32 bufSize = 0;
   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetImageBuffer(pdf, &bufSize)) != NULL)
   {
      RETURN_STRINGL(rc, bufSize, 1);
   }
}

// GetImageBufSize() -> PHP specific

PHP_METHOD(dynapdf, GetImageBufSize)
{
   PPDF* pdf;
   UI32 bufSize = 0;
   DYNAPDF_FETCH_PDFDOC();
   pdfGetImageBuffer(pdf, &bufSize);
   RETURN_LONG(bufSize);
}

/* {{{ proto SI32 pdfGetImageCount()
   const void* IPDF, const char* FileName); */
PHP_METHOD(dynapdf, GetImageCount)
{
   zval **arg1;
   PPDF* pdf;
   char *FileName;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   FileName = Z_STRVAL_PP(arg1);

   rc = pdfGetImageCount(pdf, FileName);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetImageCountEx()
   const void* IPDF, const void* Buffer, UI32 BufSize); */
PHP_METHOD(dynapdf, GetImageCountEx)
{
   zval **arg1;
   PPDF* pdf;
   char* Buffer;
   SI32 BufSize;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Buffer  = Z_STRVAL_PP(arg1);
   BufSize = Z_STRLEN_PP(arg1);

   rc = pdfGetImageCountEx(pdf, Buffer, BufSize);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetImageHeight()
   const void* IPDF, UI32 AHandle); */
PHP_METHOD(dynapdf, GetImageHeight)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AHandle  = Z_LVAL_PP(arg1);

   rc = pdfGetImageHeight(pdf, AHandle);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetImageObj(const void* IPDF, UI32 Handle, TParseFlags Flags, struct TPDFImage* Image);
PHP_METHOD(dynapdf, GetImageObj)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   SI32 Handle, Flags;
   struct TPDFImage image;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Flags  = Z_LVAL_PP(argv[1]);

   if (pdfGetImageObj(obj->pdf, Handle, Flags, &image) != 0)
      Return_ImageObj(return_value, obj, &image);
   else
      RETURN_NULL();
}

//PDF_EXP SI32   PDF_CALL pdfGetImageObjCount(const void* IPDF);
PHP_METHOD(dynapdf, GetImageObjCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetImageObjCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetImageObjEx(const void* IPDF, const void* ImagePtr, TParseFlags Flags, struct TPDFImage* Image);
PHP_METHOD(dynapdf, GetImageObjEx)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   PIMG* ImagePtr;
   SI32 ptrHash, Flags;
   struct TPDFImage image;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   ptrHash = Z_LVAL_PP(argv[0]);
   Flags   = Z_LVAL_PP(argv[1]);

   if ((ImagePtr = (PIMG*)GetPointer(obj, ptrHash, ptImage)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter ImagePtr contains an invalid value!");
      RETURN_NULL();
   }else
   {
      if (pdfGetImageObjEx(obj->pdf, ImagePtr, Flags, &image) != 0)
         Return_ImageObj(return_value, obj, &image);
      else
         RETURN_NULL();
   }
}

/* {{{ proto SI32 pdfGetImageWidth()
   const void* IPDF, UI32 AHandle); */
PHP_METHOD(dynapdf, GetImageWidth)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AHandle  = Z_LVAL_PP(arg1);

   rc = pdfGetImageWidth(pdf, AHandle);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetImportDataAction(const PPDF* IPDF, UI32 Handle, struct TPDFImportDataAction* Action);
PHP_METHOD(dynapdf, GetImportDataAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFImportDataAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetImportDataAction(obj->pdf, Handle, &Action) != 0)
   {
      zval* node;
      array_init(return_value);
      ALLOC_INIT_ZVAL(node);
      array_init(node);
      AddFileSpecEx(node, obj, &Action.Data);
      add_assoc_zval(return_value, "Data", node);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
   }
}

/* {{{ proto  ulong pdfGetImportFlags()
   ulong pdfGetImportFlags(const void* IPDF); */
PHP_METHOD(dynapdf, GetImportFlags)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetImportFlags(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto  ulong pdfGetImportFlags2()
   ulong pdfGetImportFlags2(const void* IPDF); */
PHP_METHOD(dynapdf, GetImportFlags2)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetImportFlags2(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetInBBox()
   const void* IPDF, UI32 PageNum, TPageBoundary Boundary, struct TPDFRect * BBox); */
PHP_METHOD(dynapdf, GetInBBox)
{
   zval **argv[2];
   PPDF* pdf;
   TPageBoundary Boundary;
   UI32 PageNum;
   struct TPDFRect BBox;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   PageNum  = Z_LVAL_PP(argv[0]);
   Boundary = Z_LVAL_PP(argv[1]);

   array_init(return_value);
   if ((rc = pdfGetInBBox(pdf, PageNum, Boundary, &BBox)) != 0)
   {
      add_assoc_double(return_value, "Left",   BBox.Left);
      add_assoc_double(return_value, "Bottom", BBox.Bottom);
      add_assoc_double(return_value, "Right",  BBox.Right);
      add_assoc_double(return_value, "Top",    BBox.Top);
   }
}
/* }}} */

/* {{{ proto SI32 pdfGetInDocInfo()
   const void* IPDF, TDocumentInfo DInfo, unsigned short* * Value); */
PHP_METHOD(dynapdf, GetInDocInfo)
{
   zval **arg1;
   PPDF* pdf;
   UI16* value = NULL;
   SI32 rc, DInfo;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   DInfo = Z_LVAL_PP(arg1);
   rc = pdfGetInDocInfo(pdf, DInfo, &value);
   if (rc > 0) ReturnStringExW(return_value, value, rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetInDocInfoCount(const void* IPDF);
PHP_METHOD(dynapdf, GetInDocInfoCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInDocInfoCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetInDocInfoEx(const void* IPDF, UI32 Index, TDocumentInfo ADDR DInfo, char* ADDR Key, char* ADDR Value, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetInDocInfoEx)
{
   zval **arg1;
   PPDF* pdf;
   char* key = NULL, *value = NULL;
   TDocumentInfo DInfo = diAuthor;
   SI32 rc, Index, unicode;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);
   rc = pdfGetInDocInfoEx(pdf, Index, &DInfo, &key, &value, &unicode);
   if (rc > 0 || key)
   {
      array_init(return_value);
      if (key) AddStringA(return_value, "Key", key);
      add_assoc_long(return_value, "DInfo", DInfo);
      if (unicode)
         AddStringExW(return_value, "Value", (const UI16*)value, rc);
      else
         AddStringExA(return_value, "Value", value, rc);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetInEncryptionFlags(const void* IPDF);
PHP_METHOD(dynapdf, GetInEncryptionFlags)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInEncryptionFlags(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetInFieldCount(const void* IPDF);
PHP_METHOD(dynapdf, GetInFieldCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInFieldCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32  PDF_CALL pdfGetInIsCollection(const void* IPDF);
PHP_METHOD(dynapdf, GetInIsCollection)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInIsCollection(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32  PDF_CALL pdfGetInIsEncrypted(const void* IPDF);
PHP_METHOD(dynapdf, GetInIsEncrypted)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInIsEncrypted(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32  PDF_CALL pdfGetInIsSigned(const void* IPDF);
PHP_METHOD(dynapdf, GetInIsSigned)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInIsSigned(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetInIsTaggedPDF(const PPDF* IPDF);
PHP_METHOD(dynapdf, GetInIsTaggedPDF)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInIsTaggedPDF(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32  PDF_CALL pdfGetInIsTrapped(const void* IPDF);
PHP_METHOD(dynapdf, GetInIsTrapped)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInIsTrapped(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32  PDF_CALL pdfGetInIsXFAForm(const void* IPDF);
PHP_METHOD(dynapdf, GetInIsXFAForm)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInIsXFAForm(pdf);
   RETURN_LONG(rc);
}

// PDF_EXP LBOOL  PDF_CALL pdfGetInkList(const ILST* List, float** Points, UI32* Count);
PHP_METHOD(dynapdf, GetInkList)
{
   zval **arg1;
   SI32 hash;
   const ILST* List;
   UI32 Count = 0;
   float* Points = NULL;
   struct dynapdf_object *obj;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   hash = Z_LVAL_PP(arg1);

   if ((List = (ILST*)GetPointer(obj, hash, ptInkList)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter List contains an invalid value!");
      RETURN_NULL();
   }

   if (pdfGetInkList(List, &Points, &Count) != 0)
   {
      if (Count > 0)
      {
         UI32 i;
         array_init(return_value);
         for (i = 0; i < Count; i++)
         {
            add_next_index_double(return_value, Points[i]);
         }
      }
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetInMetadata(const void* IPDF, SI32 PageNum, BYTE** Buffer, UI32* BufSize);
PHP_METHOD(dynapdf, GetInMetadata)
{
   zval **arg1;
   PPDF* pdf;
   SI32 PageNum;
   BYTE* buffer = NULL;
   UI32 bufSize = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum = Z_LVAL_PP(arg1);
   if (pdfGetInMetadata(pdf, PageNum, &buffer, &bufSize) != 0)
   {
      if (bufSize > 0) RETURN_STRINGL(buffer, bufSize, 1);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetInOrientation(const void* IPDF, SI32 PageNum);
PHP_METHOD(dynapdf, GetInOrientation)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, PageNum;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum = Z_LVAL_PP(arg1);
   rc = pdfGetInOrientation(pdf, PageNum);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetInPageCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetInPageCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInPageCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetInPDFVersion()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetInPDFVersion)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInPDFVersion(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetPDFVersionEx(const PPDF* IPDF, struct TPDFVersionInfo* Value);
PHP_METHOD(dynapdf, GetInPDFVersionEx)
{
   PPDF* pdf;
   struct TPDFVersionInfo ver;
   DYNAPDF_FETCH_PDFDOC();
   ver.StructSize = sizeof(ver);
   if (pdfGetInPDFVersionEx(pdf, &ver) != 0)
   {
      AddPDFVersionInfo(return_value, &ver);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetInPrintSettings(const void* IPDF, struct TPDFPrintSettings* Settings);
PHP_METHOD(dynapdf, GetInPrintSettings)
{
   SI32 rc;
   PPDF* pdf;
   struct TPDFPrintSettings Settings;
   DYNAPDF_FETCH_PDFDOC();

   if ((rc = pdfGetInPrintSettings(pdf, &Settings)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "DuplexMode",        Settings.DuplexMode);
      add_assoc_long(return_value, "NumCopies",         Settings.NumCopies);
      add_assoc_long(return_value, "PickTrayByPDFSize", Settings.PickTrayByPDFSize);
      AddUI32Array(return_value, "PrintRanges", Settings.PrintRanges, Settings.PrintRangesCount);
      add_assoc_long(return_value, "PrintScaling",      Settings.PrintScaling);
   }
}

//PDF_EXP SI32  PDF_CALL pdfGetInRepairMode(const void* IPDF);
PHP_METHOD(dynapdf, GetInRepairMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetInRepairMode(pdf);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetIsFixedPitch()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetIsFixedPitch)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetIsFixedPitch(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetIsTaggingEnabled(const void* IPDF);
PHP_METHOD(dynapdf, GetIsTaggingEnabled)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetIsTaggingEnabled(pdf);
   RETURN_BOOL(rc);
}

/* {{{ proto double pdfGetItalicAngle()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetItalicAngle)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetItalicAngle(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP char*  PDF_CALL pdfGetJavaScript(const void* IPDF, UI32 Handle, UI32 ADDR Len, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetJavaScript)
{
   zval **arg1;
   PPDF* pdf;
   char* rc;
   UI32 Handle, Len = 0, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);
   if ((rc = pdfGetJavaScript(pdf, Handle, &Len, &Unicode)) != NULL)
   {
      if (Unicode)
         ReturnStringExW(return_value, (const UI16*)rc, Len);
      else
         ReturnStringExA(return_value, rc, Len);
   }
}

//PDF_EXP char*  PDF_CALL pdfGetJavaScriptAction(const void* IPDF, UI32 Handle, UI32 ADDR Len, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetJavaScriptAction)
{
   zval **arg1;
   PPDF* pdf;
   char* rc;
   UI32 Handle, Len = 0, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);
   if ((rc = pdfGetJavaScriptAction(pdf, Handle, &Len, &Unicode)) != NULL)
   {
      if (Unicode)
         ReturnStringExW(return_value, (const UI16*)rc, Len);
      else
         ReturnStringExA(return_value, rc, Len);
   }
}

//PDF_EXP char*  PDF_CALL pdfGetJavaScriptAction2(const void* IPDF, TObjType ObjType, UI32 ObjHandle, UI32 ActIndex, UI32 ADDR Len, LBOOL ADDR Unicode, TObjEvent ADDR Event);
PHP_METHOD(dynapdf, GetJavaScriptAction2)
{
   zval **argv[3];
   PPDF* pdf;
   char* rc;
   TObjEvent Event = 0;
   UI32 ObjType, ObjHandle, ActIndex, Len = 0, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   ObjType   = Z_LVAL_PP(argv[0]);
   ObjHandle = Z_LVAL_PP(argv[1]);
   ActIndex  = Z_LVAL_PP(argv[2]);

   if ((rc = pdfGetJavaScriptAction2(pdf, ObjType, ObjHandle, ActIndex, &Len, &Unicode, &Event)) != NULL)
   {
      array_init(return_value);
      if (Unicode)
         AddStringExW(return_value, "result", (const UI16*)rc, Len);
      else
         AddStringExA(return_value, "result", rc, Len);
      add_assoc_long(return_value, "Event", Event);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetJavaScriptActionEx(const PPDF* IPDF, UI32 Handle, struct TPDFJavaScriptAction* Action);
PHP_METHOD(dynapdf, GetJavaScriptActionEx)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFJavaScriptAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   if (pdfGetJavaScriptActionEx(pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      AddStringEx(return_value, "Script", Action.ScriptA, Action.ScriptW, Action.ScriptLen);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
   }
}

/* {{{ proto SI32 pdfGetJavaScriptCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetJavaScriptCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetJavaScriptCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP char*  PDF_CALL pdfGetJavaScriptEx(const void* IPDF, const char* Name, UI32 ADDR Len, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetJavaScriptEx)
{
   zval **arg1;
   PPDF* pdf;
   char* rc;
   char* Name;
   UI32 Len = 0, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Name = Z_STRVAL_PP(arg1);

   if ((rc = pdfGetJavaScriptEx(pdf, Name, &Len, &Unicode)) != NULL)
   {
      if (Unicode)
         ReturnStringExW(return_value, (const UI16*)rc, Len);
      else
         ReturnStringExA(return_value, rc, Len);
   }
}

//PDF_EXP char*  PDF_CALL pdfGetJavaScriptName(const void* IPDF, UI32 Handle, UI32 ADDR Len, LBOOL ADDR Unicode);
PHP_METHOD(dynapdf, GetJavaScriptName)
{
   zval **arg1;
   PPDF* pdf;
   char* rc;
   UI32 Handle, Len = 0, Unicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);
   if ((rc = pdfGetJavaScriptName(pdf, Handle, &Len, &Unicode)) != NULL)
   {
      if (Unicode)
         ReturnStringExW(return_value, (const UI16*)rc, Len);
      else
         ReturnStringExA(return_value, rc, Len);
   }
}

/* {{{ proto SI32 pdfGetJPEGQuality()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetJPEGQuality)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetJPEGQuality(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto  char *pdfGetLanguage()
   char *pdfGetLanguage(const void* IPDF); */
PHP_METHOD(dynapdf, GetLanguage)
{
   char *Lang;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();

   Lang = pdfGetLanguage(pdf);
   ReturnStringA(return_value, Lang);
}
/* }}} */

/* {{{ proto double pdfGetLastTextPosX()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLastTextPosX)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetLastTextPosX(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto double pdfGetLastTextPosY()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLastTextPosY)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetLastTextPosY(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetLayerConfig(const PPDF* IPDF, SI32 Index, struct TPDFOCLayerConfig* Config);
PHP_METHOD(dynapdf, GetLayerConfig)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFOCLayerConfig Config;
   UI32 Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Index = Z_LVAL_PP(arg1);

   Config.StructSize = sizeof(Config);
   if (pdfGetLayerConfig(obj->pdf, Index, &Config) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Intent", Config.Intent);
      add_assoc_bool(return_value, "IsDefault", Config.IsDefault);
      AddStringEx(return_value, "Name", Config.NameA, Config.NameW, Config.NameLen);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetLayerConfigCount(const PPDF* IPDF);
PHP_METHOD(dynapdf, GetLayerConfigCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetLayerConfigCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetLaunchAction(const PPDF* IPDF, UI32 Handle, struct TPDFLaunchAction* Action);
PHP_METHOD(dynapdf, GetLaunchAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFLaunchAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetLaunchAction(obj->pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "AppName", Action.AppName);
      AddStringA(return_value, "DefDir", Action.DefDir);
      if (Action.File) AddPointer(return_value, obj, "File", Action.File, ptFileSpecEx);
      add_assoc_long(return_value, "NewWindow", Action.NewWindow);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
      AddStringA(return_value, "Operation", Action.Operation);
      AddStringA(return_value, "Parameter", Action.Parameter);
   }
}

/* {{{ proto double pdfGetLeading()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLeading)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetLeading(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto SI32 pdfGetLineCapStyle()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLineCapStyle)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetLineCapStyle(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetLineJoinStyle()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLineJoinStyle)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetLineJoinStyle(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto double pdfGetLineWidth()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLineWidth)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetLineWidth(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto SI32 pdfGetLinkHighlightMode()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetLinkHighlightMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetLinkHighlightMode(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetLogMetafileSizeA(const void* IPDF, const char* FileName, struct TRectL* R);
PHP_METHOD(dynapdf, GetLogMetafileSize)
{
   zval **arg1;
   PPDF* pdf;
   char* FileName;
   struct TRectL R;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   FileName = Z_STRVAL_PP(arg1);

   if ((rc = pdfGetLogMetafileSizeA(pdf, FileName, &R)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Left",   R.Left);
      add_assoc_long(return_value, "Top",    R.Top);
      add_assoc_long(return_value, "Right",  R.Right);
      add_assoc_long(return_value, "Bottom", R.Bottom);
   }
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetLogMetafileSizeEx(const void* IPDF, const void* Buffer, UI32 BufSize, struct TRectL* R);
PHP_METHOD(dynapdf, GetLogMetafileSizeEx)
{
   zval **arg1;
   PPDF* pdf;
   char* Buffer;
   struct TRectL R;
   SI32 rc, BufSize;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Buffer  = Z_STRVAL_PP(arg1);
   BufSize = Z_STRLEN_PP(arg1);

   if ((rc = pdfGetLogMetafileSizeEx(pdf, Buffer, BufSize, &R)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Left",   R.Left);
      add_assoc_long(return_value, "Top",    R.Top);
      add_assoc_long(return_value, "Right",  R.Right);
      add_assoc_long(return_value, "Bottom", R.Bottom);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetMatrix(const void* IPDF, struct TCTM* M);
PHP_METHOD(dynapdf, GetMatrix)
{
   SI32 rc;
   PPDF* pdf;
   struct TCTM M;
   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetMatrix(pdf, &M)) != 0)
   {
      array_init(return_value);
      add_assoc_double(return_value, "a",   M.a);
      add_assoc_double(return_value, "b",   M.b);
      add_assoc_double(return_value, "c",   M.c);
      add_assoc_double(return_value, "d",   M.d);
      add_assoc_double(return_value, "x",   M.x);
      add_assoc_double(return_value, "y",   M.y);
   }
}

/* {{{ proto SI32 pdfGetMaxFieldLen()
   const void* IPDF, UI32 TxtField); */
PHP_METHOD(dynapdf, GetMaxFieldLen)
{
   zval **arg1;
   PPDF* pdf;
   SI32 TxtField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   TxtField  = Z_LVAL_PP(arg1);

   rc = pdfGetMaxFieldLen(pdf, TxtField);
   RETURN_LONG(rc);
}
/* }}} */

// PDF_EXP LBOOL  PDF_CALL pdfGetMeasureObj(const IMSR* Measure, struct TPDFMeasure* Value);
PHP_METHOD(dynapdf, GetMeasureObj)
{
   zval **arg1;
   SI32 hash;
   struct dynapdf_object *obj;
   IMSR* Measure;
   struct TPDFMeasure Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   hash = Z_LVAL_PP(arg1);

   if ((Measure = (IMSR*)GetPointer(obj, hash, ptMeasure)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Measure contains an invalid value!");
      RETURN_NULL();
   }

   Value.StructSize = sizeof(Value);
   if (pdfGetMeasureObj(Measure, &Value) != 0)
   {
      array_init(return_value);

      add_assoc_bool(return_value, "IsRectilinear", Value.IsRectilinear);
      if (Value.IsRectilinear)
      {
         AddPointerArray(return_value, obj, "Angles", Value.Angles, Value.AnglesCount, ptNumberFormat);
         AddPointerArray(return_value, obj, "Area", Value.Area, Value.AreaCount, ptNumberFormat);
         add_assoc_double(return_value, "CXY", Value.CXY);
         AddPointerArray(return_value, obj, "Distance", Value.Distance, Value.DistanceCount, ptNumberFormat);
         add_assoc_double(return_value, "OriginX", Value.OriginX);
         add_assoc_double(return_value, "OriginY", Value.OriginY);
         AddString(return_value, "R", Value.RA, Value.RW);
         AddPointerArray(return_value, obj, "Slope", Value.Slope, Value.SlopeCount, ptNumberFormat);
         AddPointerArray(return_value, obj, "X", Value.X, Value.XCount, ptNumberFormat);
         AddPointerArray(return_value, obj, "Y", Value.Y, Value.YCount, ptNumberFormat);
      }else
      {
         AddDoubleArray(return_value, "Bounds", Value.Bounds, Value.BoundCount);
         if (Value.DCS_IsSet)
         {
            add_assoc_bool(return_value, "DCS_IsSet", Value.DCS_IsSet);
            add_assoc_bool(return_value, "DCS_Projected", Value.DCS_Projected);
            add_assoc_long(return_value, "DCS_EPSG", Value.DCS_EPSG);
            AddStringA(return_value, "DCS_WKT", Value.DCS_WKT);
         }
         add_assoc_bool(return_value, "GCS_Projected", Value.GCS_Projected);
         add_assoc_long(return_value, "GCS_EPSG", Value.GCS_EPSG);
         AddStringA(return_value, "GCS_WKT", Value.GCS_WKT);
         AddDoubleArray(return_value, "GPTS", Value.GPTS, Value.GPTSCount);
         AddDoubleArray(return_value, "LPTS", Value.LPTS, Value.LPTSCount);
         AddStringA(return_value, "PDU1", Value.PDU1);
         AddStringA(return_value, "PDU2", Value.PDU2);
         AddStringA(return_value, "PDU3", Value.PDU3);
      }
   }
}

/* {{{ proto  ulong pdfGetMetaConvFlags()
   ulong pdfGetMetaConvFlags(const void* IPDF); */
PHP_METHOD(dynapdf, GetMetaConvFlags)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetMetaConvFlags(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetMetadata(const void* IPDF, TMetadataObj ObjType, SI32 Handle, BYTE** Buffer, UI32* BufSize);
PHP_METHOD(dynapdf, GetMetadata)
{
   zval **argv[2];
   PPDF* pdf;
   BYTE* buffer = NULL;
   UI32 bufSize = 0;
   SI32 ObjType, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   ObjType = Z_LVAL_PP(argv[0]);
   Handle  = Z_LVAL_PP(argv[0]);

   if (pdfGetMetadata(pdf, ObjType, Handle, &buffer, &bufSize) != 0)
   {
      if (bufSize > 0) RETURN_STRINGL(buffer, bufSize, 1);
   }
}

//PDF_EXP UI32*  PDF_CALL pdfGetMissingGlyphs(const void* IPDF, UI32 ADDR Count);
PHP_METHOD(dynapdf, GetMissingGlyphs)
{
   UI32* rc, i, Count = 0;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetMissingGlyphs(pdf, &Count)) != NULL)
   {
      array_init(return_value);
      for (i = 0; i < Count; i++)
      {
         add_next_index_long(return_value, rc[i]);
      }
   }
}

/* {{{ proto double pdfGetMiterLimit()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetMiterLimit)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetMiterLimit(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetMovieAction(const PPDF* IPDF, UI32 Handle, struct TPDFMovieAction* Action);
PHP_METHOD(dynapdf, GetMovieAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFMovieAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetMovieAction(obj->pdf, Handle, &Action) != 0)
   {
      zval* node;
      array_init(return_value);
      add_assoc_long(return_value, "Annot", Action.Annot);

      ALLOC_INIT_ZVAL(node);
      array_init(node);
      add_next_index_double(node, Action.FWPosition[0]);
      add_next_index_double(node, Action.FWPosition[1]);
      add_assoc_zval(return_value, "FWPosition", node);

      ALLOC_INIT_ZVAL(node);
      array_init(node);
      add_next_index_long(node, Action.FWScale[0]);
      add_next_index_long(node, Action.FWScale[1]);
      add_assoc_zval(return_value, "FWScale", node);
      AddStringA(return_value, "Mode", Action.Mode);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
      AddStringA(return_value, "Operation", Action.Operation);
      add_assoc_double(return_value, "Rate", Action.Rate);
      add_assoc_bool(return_value, "ShowControls", Action.ShowControls);
      add_assoc_bool(return_value, "Synchronous", Action.Synchronous);
      AddString(return_value, "Title", Action.TitleA, Action.TitleW);
      add_assoc_double(return_value, "Volume", Action.Volume);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetNamedAction(const PPDF* IPDF, UI32 Handle, struct TPDFNamedAction* Action);
PHP_METHOD(dynapdf, GetNamedAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFNamedAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetNamedAction(obj->pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      ADD_ASSOC_UTF8_STRING(return_value, "Name", (char*)Action.Name);
      add_assoc_long(return_value, "NewWindow", Action.NewWindow);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
      add_assoc_long(return_value, "Type", Action.Type);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetNamedDest(const void* IPDF, UI32 Index, struct TPDFNamedDest* Dest);
PHP_METHOD(dynapdf, GetNamedDest)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFNamedDest Dest;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   Dest.StructSize = sizeof(Dest);
   if ((rc = pdfGetNamedDest(pdf, Index, &Dest)) != 0)
   {
      array_init(return_value);
      AddStringEx(return_value, "Name",     Dest.NameA,     Dest.NameW,     Dest.NameLen);
      AddStringEx(return_value, "DestFile", Dest.DestFileA, Dest.DestFileW, Dest.DestFileLen);
      add_assoc_long(return_value, "DestPage", Dest.DestPage);
      AddPDFRect(return_value, "DestPos", &Dest.DestPos);
      add_assoc_long(return_value, "DestType", Dest.DestType);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetNamedDestCount(const void* IPDF);
PHP_METHOD(dynapdf, GetNamedDestCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetNamedDestCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetNeedAppearance(const void* IPDF);
PHP_METHOD(dynapdf, GetNeedAppearance)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetNeedAppearance(pdf);
   RETURN_BOOL(rc);
}

// PDF_EXP LBOOL  PDF_CALL pdfGetNumberFormatObj(INFM* NumberFmt, struct TPDFNumberFormat* Value);
PHP_METHOD(dynapdf, GetNumberFormatObj)
{
   zval **arg1;
   SI32 hash;
   struct dynapdf_object *obj;
   INFM* NumberFmt;
   struct TPDFNumberFormat Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   hash = Z_LVAL_PP(arg1);

   if ((NumberFmt = (INFM*)GetPointer(obj, hash, ptNumberFormat)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter NumberFmt contains an invalid value!");
      RETURN_NULL();
   }

   Value.StructSize = sizeof(Value);
   if (pdfGetNumberFormatObj(NumberFmt, &Value) != 0)
   {
      array_init(return_value);

      add_assoc_double(return_value, "C", Value.C);
      add_assoc_long(return_value, "D", Value.D);
      add_assoc_long(return_value, "F", Value.F);
      add_assoc_bool(return_value, "FD", Value.FD);
      add_assoc_long(return_value, "O", Value.O);
      AddString(return_value, "PS", Value.PSA, Value.PSW);
      AddString(return_value, "RD", Value.RDA, Value.RDW);
      AddString(return_value, "RT", Value.RTA, Value.RTW);
      AddString(return_value, "SS", Value.SSA, Value.SSW);
      AddString(return_value, "U", Value.UA, Value.UW);
   }
}

/* {{{ proto SI32 pdfGetObjActionCount()
   const void* IPDF, TObjType ObjType, UI32 ObjHandle); */
PHP_METHOD(dynapdf, GetObjActionCount)
{
   zval **argv[2];
   PPDF* pdf;
   TObjType ObjType;
   SI32 ObjHandle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   ObjType   = Z_LVAL_PP(argv[0]);
   ObjHandle = Z_LVAL_PP(argv[0]);

   rc = pdfGetObjActionCount(pdf, ObjType, ObjHandle);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetObjActions(const PPDF* IPDF, TObjType ObjType, UI32 ObjHandle, struct TPDFObjActions* Actions);
PHP_METHOD(dynapdf, GetObjActions)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   struct TPDFObjActions Action;
   UI32 ObjType, ObjHandle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   ObjType   = Z_LVAL_PP(argv[0]);
   ObjHandle = Z_LVAL_PP(argv[1]);

   Action.StructSize = sizeof(Action);
   if (pdfGetObjActions(obj->pdf, ObjType, ObjHandle, &Action) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Action", Action.Action);
      add_assoc_long(return_value, "ActionType", Action.ActionType);
      if (Action.Events) AddPointer(return_value, obj, "Events", Action.Events, ptEvents);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetObjEvent(const IEVT* IEvent, struct TPDFObjEvent* Event);
PHP_METHOD(dynapdf, GetObjEvent)
{
   zval **arg1;
   IEVT* IEvent;
   struct dynapdf_object *obj;
   struct TPDFObjEvent Event;
   UI32 hash;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   hash = Z_LVAL_PP(arg1);

   if (!hash || (IEvent = (IEVT*)GetPointer(obj, hash, ptEvents)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IEvent must be a valid safe pointer!");
      RETURN_NULL();
   }

   Event.StructSize = sizeof(Event);
   if (pdfGetObjEvent(IEvent, &Event) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Action", Event.Action);
      add_assoc_long(return_value, "ActionType", Event.ActionType);
      add_assoc_long(return_value, "Event", Event.Event);
      if (Event.Next) AddPointer(return_value, obj, "Next", Event.Next, ptEvents);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetOCG(const PPDF* IPDF, UI32 Handle, struct TPDFOCG* Out);
PHP_METHOD(dynapdf, GetOCG)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFOCG Out;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   Out.StructSize = sizeof(Out);
   if ((rc = pdfGetOCG(pdf, Handle, &Out)) != 0)
   {
      array_init(return_value);
      ReturnOCG(return_value, &Out);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetOCGContUsage(const PPDF* IPDF, UI32 Handle, struct TPDFOCGContUsage* Out);
PHP_METHOD(dynapdf, GetOCGContUsage)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFOCGContUsage Out;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   Out.StructSize = sizeof(Out);
   if ((rc = pdfGetOCGContUsage(pdf, Handle, &Out)) != 0)
   {
      array_init(return_value);
      ReturnOCGContUsage(return_value, &Out);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetOCGCount(const PPDF* IPDF);
PHP_METHOD(dynapdf, GetOCGCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetOCGCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetOCGUsageUserName(const PPDF* IPDF, UI32 Handle, UI32 Index, char* ADDR NameA, UI16* ADDR NameW);
PHP_METHOD(dynapdf, GetOCGUsageUserName)
{
   zval **arg[2];
   PPDF* pdf;
   char* NameA = NULL;
   UI16* NameW = NULL;
   SI32 rc, Handle, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, arg) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg[0]);
   convert_to_long_ex(arg[1]);

   Handle = Z_LVAL_PP(arg[0]);
   Index  = Z_LVAL_PP(arg[1]);

   if ((rc = pdfGetOCGUsageUserName(pdf, Handle, Index, &NameA, &NameW)) != 0)
   {
      if (NameA)
         ReturnStringA(return_value, NameA);
      else
         ReturnStringW(return_value, NameW);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetOCHandle(const IOCG* OC);
PHP_METHOD(dynapdf, GetOCHandle)
{
   IOCG* OC;
   zval **arg1;
   SI32 rc, hash;
   struct dynapdf_object *obj;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }
   hash = Z_LVAL_PP(arg1);

   DYNAPDF_FETCH_PDFDOC_EX();

   if ((OC = (IOCG*)GetPointer(obj, hash, ptOCG)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter OC contains an invalid value!");
      RETURN_LONG(-1);
   }
   rc = pdfGetOCHandle(OC);
   RETURN_LONG(rc);
}

//PDF_EXP IOCN*  PDF_CALL pdfGetOCUINode(const PPDF* IPDF, const IOCN* Node, struct TPDFOCUINode* OutNode);
PHP_METHOD(dynapdf, GetOCUINode)
{
   IOCN* Node, *retval;
   zval **arg1;
   SI32 hash;
   struct dynapdf_object *obj;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }
   hash = Z_LVAL_PP(arg1);

   DYNAPDF_FETCH_PDFDOC_EX();
   if (hash)
   {
      struct TPDFOCUINode OutNode;
      OutNode.StructSize = sizeof(OutNode);
      if ((Node = (IOCN*)GetPointer(obj, hash, ptOCGUINode)) == NULL)
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter OC contains an invalid value!");
         RETURN_LONG(-1);
      }
      retval = pdfGetOCUINode(obj->pdf, Node, &OutNode);
      array_init(return_value);
      AddStringEx(return_value, "Label", OutNode.LabelA, OutNode.LabelW, OutNode.LabelLength);
      add_assoc_bool(return_value, "NewNode", OutNode.NewNode);
      AddPointer(return_value, obj, "NextChild", OutNode.NextChild, ptOCGUINode);
      add_assoc_long(return_value, "OCG", OutNode.OCG);
      if (retval)
         AddPointer(return_value, obj, "Next", retval, ptOCGUINode);
      else
         RETURN_LONG(0);
   }else
   {
      retval = pdfGetOCUINode(obj->pdf, NULL, NULL);
      if (retval)
      {
         array_init(return_value);
         AddPointer(return_value, obj, "Next", retval, ptOCGUINode);
      }else
         RETURN_LONG(0);
   }
}

/* {{{ proto double pdfGetOpacity()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetOpacity)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetOpacity(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto SI32 pdfGetOrientation()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetOrientation)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetOrientation(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetOutputIntent(const void* IPDF, UI32 Index, struct TPDFOutputIntent* Intent);
PHP_METHOD(dynapdf, GetOutputIntent)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFOutputIntent Intent;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   Intent.StructSize = sizeof(Intent);
   if ((rc = pdfGetOutputIntent(pdf, Index, &Intent)) != 0)
   {
      array_init(return_value);
      ADD_ASSOC_BIN_STRING(return_value, "Buffer", Intent.Buffer, Intent.BufSize);
      AddString(return_value, "Info", Intent.InfoA, Intent.InfoW);
      add_assoc_long(return_value, "NumComponents", Intent.NumComponents);
      AddString(return_value, "OutputCondition",   Intent.OutputConditionA,   Intent.OutputConditionW);
      AddString(return_value, "OutputConditionID", Intent.OutputConditionIDA, Intent.OutputConditionIDW);
      AddString(return_value, "RegistryName",      Intent.RegistryNameA,      Intent.RegistryNameW);
      AddStringA(return_value, "SubType",          Intent.SubType);
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetOutputIntentCount(const void* IPDF);
PHP_METHOD(dynapdf, GetOutputIntentCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetOutputIntentCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfGetPageAnnotCount(const void* IPDF);
PHP_METHOD(dynapdf, GetPageAnnotCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageAnnotCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetPageAnnotEx(const void* IPDF, UI32 Index, struct TPDFAnnotationEx ADDR Annot);
PHP_METHOD(dynapdf, GetPageAnnotEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFAnnotationEx Annot;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   if ((rc = pdfGetPageAnnotEx(obj->pdf, Index, &Annot)) != 0)
   {
      array_init(return_value);
      ReturnAnnotEx(return_value, obj, &Annot);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetPageBBox(IPGE* PagePtr, TPageBoundary Boundary, struct TFltRect* BBox);
PHP_METHOD(dynapdf, GetPageBBox)
{
   zval **argv[2];
   PPDF* pdf;
   IPGE* PagePtr;
   TPageBoundary Boundary;
   struct TFltRect BBox = {0.0f, 0.0f, 0.0f, 0.0f};
   SI32 rc, PageNum;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   PageNum  = Z_LVAL_PP(argv[0]);
   Boundary = Z_LVAL_PP(argv[1]);

   if ((PagePtr = pdfGetPageObject(pdf, PageNum)) == NULL) RETURN_NULL();

   if ((rc = pdfGetPageBBox(PagePtr, Boundary, &BBox)) != 0)
   {
      array_init(return_value);
      add_assoc_double(return_value, "Left",   BBox.Left);
      add_assoc_double(return_value, "Bottom", BBox.Bottom);
      add_assoc_double(return_value, "Right",  BBox.Right);
      add_assoc_double(return_value, "Top",    BBox.Top);
   }
}

/* {{{ proto SI32 pdfGetPageCoords()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageCoords)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageCoords(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetPageCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetPageField(const void* IPDF, UI32 Index, struct TPDFField* Field);
PHP_METHOD(dynapdf, GetPageField)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFField Field;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   if ((rc = pdfGetPageField(pdf, Index, &Field)) != 0)
   {
      array_init(return_value);
      ReturnField(return_value, &Field);
   }
}

/* {{{ proto SI32 pdfGetPageFieldCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageFieldCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageFieldCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetPageFieldEx(const void* IPDF, UI32 Index, struct TPDFFieldEx* Field);
PHP_METHOD(dynapdf, GetPageFieldEx)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFFieldEx Field;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   Field.StructSize = sizeof(Field);

   if ((rc = pdfGetPageFieldEx(obj->pdf, Index, &Field)) != 0)
   {
      array_init(return_value);
      ReturnFieldEx(return_value, obj, &Field);
   }
}

/* {{{ proto double pdfGetPageHeight()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageHeight)
{
   PPDF* pdf;
   double drc;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetPageHeight(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetPageLabel(const void* IPDF, UI32 Index, struct TPDFPageLabel* Label);
PHP_METHOD(dynapdf, GetPageLabel)
{
   zval **arg1;
   PPDF* pdf;
   struct TPDFPageLabel Label;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   if ((rc = pdfGetPageLabel(pdf, Index, &Label)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "StartRange",   Label.StartRange);
      add_assoc_long(return_value, "Format",       Label.Format);
      add_assoc_long(return_value, "FirstPageNum", Label.FirstPageNum);
      if (Label.Prefix)
      {
         if (Label.PrefixUni)
            AddStringExW(return_value, "Prefix", (const UI16*)Label.Prefix, Label.PrefixLen);
         else
            AddStringExA(return_value, "Prefix", Label.Prefix, Label.PrefixLen);
      }
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetPageLabelCount(const void* IPDF);
PHP_METHOD(dynapdf, GetPageLabelCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageLabelCount(pdf);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetPageLayout()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageLayout)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageLayout(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetPageMode()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageMode(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetPageNum()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageNum)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPageMode(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetPageOrientation(IPGE* PagePtr);
PHP_METHOD(dynapdf, GetPageOrientation)
{
   zval **arg1;
   PPDF* pdf;
   IPGE* PagePtr;
   SI32 rc, PageNum;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum = Z_LVAL_PP(arg1);

   if ((PagePtr = pdfGetPageObject(pdf, PageNum)) == NULL) RETURN_LONG(0);
   rc = pdfGetPageOrientation(PagePtr);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetPageText(const void* IPDF, struct TPDFStack* Stack);
PHP_METHOD(dynapdf, GetPageText)
{
   SI32 rc;
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFStack* Stack;
   int argc = ZEND_NUM_ARGS();

   if(argc > 0 && (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      RETURN_NULL();
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   Stack = obj->pdfStack;

   if (!Stack)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "You must call InitStack() before this function can be called!");
      RETURN_NULL();
   }

   if (argc > 0)
   {
      convert_to_long_ex(arg1);
      Stack->ConvColors = 1;
      Stack->DestSpace  = Z_LVAL_PP(arg1);
   }else
      Stack->ConvColors = 0;

   rc = pdfGetPageText(obj->pdf, Stack);

   if (!rc && !Stack->TextLen) RETURN_NULL();

   array_init(return_value);
   AddCTM(return_value, "ctm", &Stack->ctm);
   AddCTM(return_value, "tm",  &Stack->tm);
   add_assoc_double(return_value, "x",               Stack->x);
   add_assoc_double(return_value, "y",               Stack->y);
   add_assoc_double(return_value, "FontSize",        Stack->FontSize);
   add_assoc_double(return_value, "CharSP",          Stack->CharSP);
   add_assoc_double(return_value, "WordSP",          Stack->WordSP);
   add_assoc_double(return_value, "HScale",          Stack->HScale);
   add_assoc_double(return_value, "TextRise",        Stack->TextRise);
   add_assoc_double(return_value, "Leading",         Stack->Leading);
   add_assoc_double(return_value, "LineWidth",       Stack->LineWidth);
   add_assoc_long(return_value,   "DrawMode",        Stack->DrawMode);
   add_assoc_long(return_value,   "FillCS",          Stack->FillCS);
   add_assoc_long(return_value,   "StrokeCS",        Stack->StrokeCS);
   add_assoc_long(return_value,   "FillColor",       Stack->FillColor);
   add_assoc_long(return_value,   "StrokeColor",     Stack->StrokeColor);
   add_assoc_bool(return_value,   "CIDFont",         Stack->CIDFont);
   ADD_ASSOC_BIN_STRING(return_value, "Text",        Stack->Text, Stack->TextLen);
   AddKerningArrayA(return_value, "RawKern",         Stack->RawKern, Stack->KerningCount);
   AddKerningArrayW(return_value, "Kerning",         Stack->Kerning, Stack->KerningCount);
   add_assoc_double(return_value, "TextWidth",       Stack->TextWidth);
   AddPointer(return_value, obj,  "IFont",           Stack->IFont, ptFont);
   add_assoc_bool(return_value,   "Embedded",        Stack->Embedded);
   add_assoc_double(return_value, "SpaceWidth",      Stack->SpaceWidth);
   add_assoc_bool(return_value,   "ConvColors",      Stack->ConvColors);
   add_assoc_long(return_value,   "DestSpace",       Stack->DestSpace);
   add_assoc_long(return_value,   "DeleteKerningAt", Stack->DeleteKerningAt);
   add_assoc_long(return_value,   "FontFlags",       Stack->FontFlags);
}

/* {{{ proto double pdfGetPageWidth()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPageWidth)
{
   PPDF* pdf;
   double drc;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetPageWidth(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto SI32 pdfGetPDFVersion()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetPDFVersion)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetPDFVersion(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetPDFVersionEx(const PPDF* IPDF, struct TPDFVersionInfo* Value);
PHP_METHOD(dynapdf, GetPDFVersionEx)
{
   PPDF* pdf;
   struct TPDFVersionInfo ver;
   DYNAPDF_FETCH_PDFDOC();
   ver.StructSize = sizeof(ver);
   if (pdfGetPDFVersionEx(pdf, &ver) != 0)
   {
      AddPDFVersionInfo(return_value, &ver);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetPrintSettings(const void* IPDF, struct TPDFPrintSettings* Settings);
PHP_METHOD(dynapdf, GetPrintSettings)
{
   SI32 rc;
   PPDF* pdf;
   struct TPDFPrintSettings Settings;
   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetPrintSettings(pdf, &Settings)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "DuplexMode",        Settings.DuplexMode);
      add_assoc_long(return_value, "NumCopies",         Settings.NumCopies);
      add_assoc_long(return_value, "PickTrayByPDFSize", Settings.PickTrayByPDFSize);
      AddUI32Array(return_value,   "PrintRanges",       Settings.PrintRanges, Settings.PrintRangesCount);
      add_assoc_long(return_value, "PrintScaling",      Settings.PrintScaling);
   }
}

// PDF_EXP LBOOL  PDF_CALL pdfGetPtDataArray(const IPTD* PtData, UI32 Index, char** DataType, float** Values, UI32* ValCount);
PHP_METHOD(dynapdf, GetPtDataArray)
{
   zval **argv[2];
   SI32 hash;
   struct dynapdf_object *obj;
   IPTD* PtData;
   char* DataType = NULL;
   float* Values = NULL;
   UI32 Index, ValCount = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   hash  = Z_LVAL_PP(argv[0]);
   Index = Z_LVAL_PP(argv[1]);

   if ((PtData = (IPTD*)GetPointer(obj, hash, ptPtData)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter NumberFmt contains an invalid value!");
      RETURN_NULL();
   }

   if (pdfGetPtDataArray(PtData, Index, &DataType, &Values, &ValCount) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "DataType", DataType);
      AddFloatArray(return_value, "Values", Values, ValCount);
   }
}

// PDF_EXP LBOOL  PDF_CALL pdfGetPtDataObj(const IPTD* PtData, char** Subtype, UI32* NumArrays);
PHP_METHOD(dynapdf, GetPtDataObj)
{
   zval **arg1;
   SI32 hash;
   struct dynapdf_object *obj;
   IPTD* PtData;
   char* Subtype = NULL;
   UI32 NumArrays = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   hash = Z_LVAL_PP(arg1);

   if ((PtData = (IPTD*)GetPointer(obj, hash, ptPtData)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter NumberFmt contains an invalid value!");
      RETURN_NULL();
   }

   if (pdfGetPtDataObj(PtData, &Subtype, &NumArrays) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "Subtype", Subtype);
      add_assoc_long(return_value, "NumArrays", NumArrays);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfGetRelFileNode(const IRFN* IRF, struct TPDFRelFileNode* F, LBOOL Decompress);
PHP_METHOD(dynapdf, GetRelFileNode)
{
   zval **argv[2];
   IRFN* IRF;
   struct dynapdf_object *obj;
   struct TPDFRelFileNode F;
   UI32 hash, Decompress;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   hash       = Z_LVAL_PP(argv[0]);
   Decompress = Z_LVAL_PP(argv[1]);

   if (!hash || (IRF = (IEVT*)GetPointer(obj, hash, ptEvents)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IRF must be a valid safe pointer!");
      RETURN_NULL();
   }

   F.StructSize = sizeof(F);
   if (pdfGetRelFileNode(IRF, &F, Decompress) != 0)
   {
      zval* node;
      array_init(return_value);
      AddString(return_value, "Name", F.NameA, F.NameW);

      ALLOC_INIT_ZVAL(node);
      array_init(node);
      AddFileSpec(node, obj, &F.EF);
      add_assoc_zval(return_value, "EF", node);
      if (F.NextNode) AddPointer(return_value, obj, "NextNode", F.NextNode, ptRelFileNode);
   }
}

// PDF_EXP LBOOL  PDF_CALL pdfGetResetAction(const PPDF* IPDF, UI32 Handle, struct TPDFResetFormAction* Value);
PHP_METHOD(dynapdf, GetResetAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   SI32 Handle;
   struct TPDFResetFormAction Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Value.StructSize = sizeof(Value);
   if (pdfGetResetAction(obj->pdf, Handle, &Value) != 0)
   {
      array_init(return_value);
      AddPointerArray(return_value, obj, "Fields", (void**)Value.Fields, Value.FieldsCount, ptField);
      add_assoc_bool(return_value, "Include", Value.Include);
      add_assoc_long(return_value, "NextAction", Value.NextAction);
      add_assoc_long(return_value, "NextActionType", Value.NextActionType);
   }
}

/* {{{ proto SI32 pdfGetResolution()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetResolution)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetResolution(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetSaveNewImageFormat()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetSaveNewImageFormat)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetSaveNewImageFormat(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetSeparationInfo(const void* IPDF, char* ADDR Colorant, TExtColorSpace ADDR CS);
PHP_METHOD(dynapdf, GetSeparationInfo)
{
   SI32 rc;
   PPDF* pdf;
   char* Colorant = NULL;
   TExtColorSpace CS;
   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetSeparationInfo(pdf, &Colorant, &CS)) != 0)
   {
      array_init(return_value);
      ADD_ASSOC_UTF8_STRING(return_value, "Colorant", Colorant);
      add_assoc_long(return_value, "CS", CS);
   }
}

//PDF_EXP double PDF_CALL fntGetSpaceWidth(const void* IFont, double FontSize);
PHP_METHOD(dynapdf, GetSpaceWidth)
{
   zval **argv[2];
   SI32 hash;
   struct dynapdf_object *obj;
   PFNT* IFont;
   double rc, FontSize;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   hash     = Z_LVAL_PP(argv[0]);
   FontSize = Z_LVAL_PP(argv[1]);

   if ((IFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFont contains an invalid value!");
      RETURN_NULL();
   }

   rc = fntGetSpaceWidth(IFont, FontSize);
   RETURN_DOUBLE(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetSigDict(const PSIG* ISignature, struct TPDFSigDict* SigDict);
PHP_METHOD(dynapdf, GetSigDict)
{
   zval **arg1;
   SI32 rc, hash;
   struct dynapdf_object *obj;
   PSIG* ISignature;
   struct TPDFSigDict SigDict;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   convert_to_long_ex(arg1);
   hash = Z_LVAL_PP(arg1);

   if ((ISignature = (PSIG*)GetPointer(obj, hash, ptSignature)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter ISignature contains an invalid value!");
      RETURN_NULL();
   }

   SigDict.StructSize = sizeof(SigDict);

   if ((rc = pdfGetSigDict(ISignature, &SigDict)) != 0)
   {
      array_init(return_value);
      AddUI32Array(return_value, "ByteRange", (UI32*)SigDict.ByteRange, SigDict.ByteRangeCount);
      ADD_ASSOC_BIN_STRING(return_value, "Cert", (char*)SigDict.Cert, SigDict.CertLen);
      if (SigDict.Changes) AddUI32Array(return_value, "Changes", (UI32*)SigDict.Changes, 3);
      AddString(return_value, "ContactInfo", SigDict.ContactInfoA, SigDict.ContactInfoW);
      ADD_ASSOC_BIN_STRING(return_value, "Contents", (char*)SigDict.Contents, SigDict.ContentsSize);
      AddStringA(return_value, "Filter", SigDict.Filter);
      AddString(return_value, "Location", SigDict.LocationA, SigDict.LocationW);
      AddStringA(return_value, "SignTime", SigDict.SignTime);
      AddString(return_value, "Name", SigDict.NameA, SigDict.NameW);
      add_assoc_long(return_value, "PropAuthTime", SigDict.PropAuthTime);
      AddStringA(return_value, "PropAuthType", SigDict.PropAuthType);
      AddString(return_value, "Reason", SigDict.ReasonA, SigDict.ReasonW);
      add_assoc_long(return_value, "Revision", SigDict.Revision);
      AddStringA(return_value, "SubFilter", SigDict.SubFilter);
      add_assoc_long(return_value, "Version", SigDict.Version);
   }
}

/* {{{ proto  ulong pdfGetStrokeColor()
   ulong pdfGetStrokeColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetStrokeColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetStrokeColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

// PDF_EXP LBOOL  PDF_CALL pdfGetSubmitAction(const PPDF* IPDF, UI32 Handle, struct TPDFSubmitFormAction* Value);
PHP_METHOD(dynapdf, GetSubmitAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   SI32 Handle;
   struct TPDFSubmitFormAction Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Value.StructSize = sizeof(Value);
   if (pdfGetSubmitAction(obj->pdf, Handle, &Value) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "CharSet", Value.CharSet);
      AddPointerArray(return_value, obj, "Fields", (void**)Value.Fields, Value.FieldsCount, ptField);
      add_assoc_long(return_value, "Flags", Value.Flags);
      AddStringA(return_value, "URL", Value.URL);
      add_assoc_long(return_value, "NextAction", Value.NextAction);
      add_assoc_long(return_value, "NextActionType", Value.NextActionType);
   }
}

// PDF_EXP SI32 PDF_CALL pdfGetSysFontInfo(const PPDF* IPDF, UI32 Handle, struct TPDFSysFont* Out);
PHP_METHOD(dynapdf, GetSysFontInfo)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   struct TPDFSysFont Out;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   Out.StructSize = sizeof(Out);
   Out.FamilyName = NULL;

   rc = pdfGetSysFontInfo(pdf, Handle, &Out);
   if (Out.FamilyName)
   {
      array_init(return_value);

      add_assoc_long(return_value, "BaseType", Out.BaseType);
      AddStringA(return_value, "CIDOrdering", Out.CIDOrdering);
      AddStringA(return_value, "CIDRegistry", Out.CIDRegistry);
      add_assoc_long(return_value, "CIDSupplement", Out.CIDSupplement);
      add_assoc_long(return_value, "DataOffset", Out.DataOffset);
      AddStringW(return_value, "FamilyName", Out.FamilyName);
      AddString(return_value, "FilePath", Out.FilePathA, Out.FilePathW);
      add_assoc_long(return_value, "FileSize", Out.FileSize);
      add_assoc_long(return_value, "Flags", Out.Flags);
      AddStringW(return_value, "FullName", Out.FullName);
      add_assoc_long(return_value, "Length1", Out.Length1);
      add_assoc_long(return_value, "Length2", Out.Length2);
      AddString(return_value, "PostScriptName", Out.PostScriptNameA, Out.PostScriptNameW);
      add_assoc_long(return_value, "Index", Out.Index);
      add_assoc_bool(return_value, "IsFixedPitch", Out.IsFixedPitch);
      add_assoc_long(return_value, "Style", Out.Style);
      add_assoc_long(return_value, "UnicodeRange1", Out.UnicodeRange1);
      add_assoc_long(return_value, "UnicodeRange2", Out.UnicodeRange2);
      add_assoc_long(return_value, "UnicodeRange3", Out.UnicodeRange3);
      add_assoc_long(return_value, "UnicodeRange4", Out.UnicodeRange4);
      add_assoc_long(return_value, "Next", rc);
   }
}

/* {{{ proto SI32 pdfGetTabLen()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetTabLen)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTabLen(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetTemplCount()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetTemplCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTemplCount(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetTemplHandle(const void* IPDF);
PHP_METHOD(dynapdf, GetTemplHandle)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTemplHandle(pdf);
   RETURN_LONG(rc);
}

/* {{{ proto double pdfGetTemplHeight()
   const void* IPDF,SI32 Handle); */
PHP_METHOD(dynapdf, GetTemplHeight)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   drc = pdfGetTemplHeight(pdf, Handle);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto double pdfGetTemplWidth()
   const void* IPDF,SI32 Handle); */
PHP_METHOD(dynapdf, GetTemplWidth)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Handle;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   drc = pdfGetTemplWidth(pdf, Handle);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto SI32 pdfGetTextDrawMode()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetTextDrawMode)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTextDrawMode(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetTextFieldValue(const void* IPDF, UI32 AField, char* ADDR Value, LBOOL ADDR ValIsUnicode, char* ADDR DefValue, LBOOL ADDR DefValIsUnicode);
PHP_METHOD(dynapdf, GetTextFieldValue)
{
   zval **arg1;
   PPDF* pdf;
   char* Value = NULL, *DefValue = NULL;
   SI32 rc, AField, ValIsUnicode = 0, DefValIsUnicode = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AField = Z_LVAL_PP(arg1);

   if ((rc = pdfGetTextFieldValue(pdf, AField, &Value, &ValIsUnicode, &DefValue, &DefValIsUnicode)) != 0)
   {
      array_init(return_value);
      add_assoc_bool(return_value, "result", 1);
      if (Value)
      {
         if (ValIsUnicode)
            AddStringW(return_value, "Value", (const UI16*)Value);
         else
            AddStringA(return_value, "Value", Value);
      }
      if (DefValue)
      {
         if (DefValIsUnicode)
            AddStringW(return_value, "DefValue", (const UI16*)DefValue);
         else
            AddStringA(return_value, "DefValue", DefValue);
      }
   }
}

/* {{{ proto SI32 pdfGetTextRect()
   const void* IPDF, double * PosX, double * PosY, double * Width, double * Height); */
PHP_METHOD(dynapdf, GetTextRect)
{
   SI32 rc;
   PPDF* pdf;
   double PosX=0.0, PosY=0.0, Width=0.0, Height=0.0;
   DYNAPDF_FETCH_PDFDOC();

   if ((rc = pdfGetTextRect(pdf, &PosX, &PosY, &Width, &Height)) != 0)
   {
      array_init(return_value);
      add_assoc_double(return_value, "PosX",   PosX);
      add_assoc_double(return_value, "PosY",   PosY);
      add_assoc_double(return_value, "Width",  Width);
      add_assoc_double(return_value, "Height", Height);
   }
}
/* }}} */

/* {{{ proto double pdfGetTextRise()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetTextRise)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetTextRise(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto double pdfGetTextScaling()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetTextScaling)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetTextScaling(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

/* {{{ proto double pdfGetTextWidth()
   const void* IPDF, const char* AText); */
PHP_METHOD(dynapdf, GetTextWidth)
{
   zval **arg1;
   PPDF* pdf;
   char* AText;
   double drc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   AText = Z_STRVAL_PP(arg1);

   drc = pdfGetTextWidthA(pdf, AText);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP double PDF_CALL fntGetTextWidth(const PFNT* IFont, const BYTE* Text, UI32 Len, float CharSpacing, float WordSpacing, float TextScale);
PHP_METHOD(dynapdf, GetTextWidthF)
{
   zval **argv[5];
   SI32 hash, Len;
   struct dynapdf_object *obj;
   BYTE* Text;
   PFNT* IFont;
   float CharSpacing, WordSpacing, TextScale;
   double rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   hash        = Z_LVAL_PP(argv[0]);
   Text        = (BYTE*)Z_STRVAL_PP(argv[1]);
   Len         = Z_STRLEN_PP(argv[1]);
   CharSpacing = (float)Z_DVAL_PP(argv[2]);
   WordSpacing = (float)Z_DVAL_PP(argv[3]);
   TextScale   = (float)Z_DVAL_PP(argv[4]);

   if ((IFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFont contains an invalid value!");
      RETURN_DOUBLE(0.0);
   }

   rc = fntGetTextWidth(IFont, Text, Len, CharSpacing, WordSpacing, TextScale);
   RETURN_DOUBLE(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetTrapped(const void* IPDF);
PHP_METHOD(dynapdf, GetTrapped)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTrapped(pdf);
   RETURN_BOOL(rc);
}

/* {{{ proto  ulong pdfGetTransparentColor()
   ulong pdfGetTransparentColor(const void* IPDF); */
PHP_METHOD(dynapdf, GetTransparentColor)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTransparentColor(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP double PDF_CALL pdfGetTypoLeading(const PPDF* IPDF);
PHP_METHOD(dynapdf, GetTypoLeading)
{
   double rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetTypoLeading(pdf);
   RETURN_DOUBLE(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfGetURIAction(const PPDF* IPDF, UI32 Handle, struct TPDFURIAction* Action);
PHP_METHOD(dynapdf, GetURIAction)
{
   zval **arg1;
   struct dynapdf_object *obj;
   struct TPDFURIAction Action;
   UI32 Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   Action.StructSize = sizeof(Action);
   if (pdfGetURIAction(obj->pdf, Handle, &Action) != 0)
   {
      array_init(return_value);
      AddStringA(return_value, "BaseURL", Action.BaseURL);
      add_assoc_bool(return_value, "IsMap", Action.IsMap);
      AddStringA(return_value, "URI", Action.URI);
      add_assoc_long(return_value, "NextAction", Action.NextAction);
      add_assoc_long(return_value, "NextActionType", Action.NextActionType);
   }
}

/* {{{ proto SI32 pdfGetUseExactPwd()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUseExactPwd)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUseExactPwd(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetUseGlobalImpFiles()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUseGlobalImpFiles)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUseGlobalImpFiles(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetUserRights()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUserRights)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUserRights(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP float  PDF_CALL pdfGetUserUnit(const void* IPDF);
PHP_METHOD(dynapdf, GetUserUnit)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetUserUnit(pdf);
   RETURN_DOUBLE(drc);
}

/* {{{ proto SI32 pdfGetUseStdFonts()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUseStdFonts)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUseStdFonts(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetUseSystemFonts()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUseSystemFonts)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUseSystemFonts(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfGetUsesTransparency(const PPDF* IPDF, SI32 PageNum);
PHP_METHOD(dynapdf, GetUsesTransparency)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, PageNum;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum = Z_LVAL_PP(arg1);

   rc = pdfGetUsesTransparency(pdf, PageNum);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetUseTransparency()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUseTransparency)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUseTransparency(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfGetUseVisibleCoords()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetUseVisibleCoords)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetUseVisibleCoords(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetViewerPreferences(const void* IPDF, SI32 ADDR Preference, SI32 ADDR AddVal);
PHP_METHOD(dynapdf, GetViewerPreferences)
{
   SI32 rc, Preference = 0, AddVal = 0;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetViewerPreferences(pdf, &Preference, &AddVal)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Preference", Preference);
      add_assoc_long(return_value, "AddVal",     AddVal);
   }
}

// PDF_EXP LBOOL  PDF_CALL pdfGetViewport(const PPDF* IPDF, UI32 PageNum, UI32 Index, struct TPDFViewport* VP);
PHP_METHOD(dynapdf, GetViewport)
{
   zval **argv[2];
   UI32 PageNum, Index;
   struct TPDFViewport VP;
   struct dynapdf_object *obj;

   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   PageNum = Z_LVAL_PP(argv[0]);
   Index   = Z_LVAL_PP(argv[1]);

   VP.StructSize = sizeof(VP);
   if (pdfGetViewport(obj->pdf, PageNum, Index, &VP) != 0)
   {
      array_init(return_value);
      AddFltRect(return_value, "BBox", &VP.BBox);
      AddPointer(return_value, obj, "Measure", VP.Measure, ptMeasure);
      AddString(return_value, "Name", VP.NameA, VP.NameW);
      AddPointer(return_value, obj, "PtData", VP.PtData, ptPtData);
   }
}

// PDF_EXP SI32   PDF_CALL pdfGetViewportCount(const PPDF* IPDF, UI32 PageNum);
PHP_METHOD(dynapdf, GetViewportCount)
{
   zval **arg1;
   PPDF* pdf;
   UI32 PageNum;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum = Z_LVAL_PP(arg1);

   rc = pdfGetViewportCount(pdf, PageNum);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfGetWMFDefExtent()
   const void* IPDF, UI32 * Width, UI32 * Height); */
PHP_METHOD(dynapdf, GetWMFDefExtent)
{
   PPDF* pdf;
   UI32 Width=0, Height=0;
   SI32 rc;

   DYNAPDF_FETCH_PDFDOC();
   if ((rc = pdfGetWMFDefExtent(pdf, &Width, &Height)) != 0)
   {
      array_init(return_value);
      add_assoc_double(return_value, "Width",   Width);
      add_assoc_double(return_value, "Height", Height);
   }
}
/* }}} */

/* {{{ proto SI32 pdfGetWMFPixelPerInch()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetWMFPixelPerInch)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetWMFPixelPerInch(pdf);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto double pdfGetWordSpacing()
   const void* IPDF); */
PHP_METHOD(dynapdf, GetWordSpacing)
{
   double drc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   drc = pdfGetWordSpacing(pdf);
   RETURN_DOUBLE(drc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfGetXFAStream(const PPDF* IPDF, UI32 Index, struct TPDFXFAStream* Out);
PHP_METHOD(dynapdf, GetXFAStream)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 Index;
   SI32 compress = 0;

   int argc = ZEND_NUM_ARGS();

   if(argc < 1 || argc > 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);

   Index = Z_LVAL_PP(argv[0]);

   if (argc > 1)
   {
      convert_to_long_ex(argv[1]);
      compress = Z_LVAL_PP(argv[1]);
   }

   if (compress)
      pdfGetXFAStream(pdf, Index, NULL);
   else
   {
      struct TPDFXFAStream strm;
      strm.StructSize = sizeof(strm);
      if (pdfGetXFAStream(pdf, Index, &strm) != 0)
      {
         array_init(return_value);
         ADD_ASSOC_BIN_STRING(return_value, "Buffer", (BYTE*)strm.Buffer, strm.BufSize);
         AddString(return_value, "Name", strm.NameA, strm.NameW);
      }
   }
}

//PDF_EXP SI32   PDF_CALL pdfGetXFAStreamCount(const PPDF* IPDF);
PHP_METHOD(dynapdf, GetXFAStreamCount)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfGetXFAStreamCount(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfHaveOpenDoc(const void* IPDF);
PHP_METHOD(dynapdf, HaveOpenDoc)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfHaveOpenDoc(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfHaveOpenPage(const void* IPDF);
PHP_METHOD(dynapdf, HaveOpenPage)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfHaveOpenPage(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfHighlightAnnotA(const void* IPDF, TAnnotType SubType, double PosX, double PosY, double Width, double Height, UI32 Color, const char* Author, const char* Subject, const char* Comment);
PHP_METHOD(dynapdf, HighlightAnnot)
{
   zval **argv[9];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char* Author, *Subject, *Comment;
   SI32 rc, SubType, Color;
   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_string_ex(argv[7]);
   convert_to_string_ex(argv[8]);

   SubType = Z_LVAL_PP(argv[0]);
   PosX    = Z_DVAL_PP(argv[1]);
   PosY    = Z_DVAL_PP(argv[2]);
   Width   = Z_DVAL_PP(argv[3]);
   Height  = Z_DVAL_PP(argv[4]);
   Color   = Z_LVAL_PP(argv[5]);
   Author  = Z_STRVAL_PP(argv[6]);
   Subject = Z_STRVAL_PP(argv[7]);
   Comment = Z_STRVAL_PP(argv[8]);

   rc = pdfHighlightAnnotA(pdf, SubType, PosX, PosY, Width, Height, Color, Author, Subject, Comment);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfImportBookmarks(const void* IPDF);
PHP_METHOD(dynapdf, ImportBookmarks)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfImportBookmarks(pdf);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfImportCatalogObjects(const void* IPDF);
PHP_METHOD(dynapdf, ImportCatalogObjects)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfImportCatalogObjects(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfImportDocInfo(const void* IPDF);
PHP_METHOD(dynapdf, ImportDocInfo)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfImportDocInfo(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfImportEncryptionSettings(const PPDF* IPDF);
PHP_METHOD(dynapdf, ImportEncryptionSettings)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfImportEncryptionSettings(pdf);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfImportOCProperties(const PPDF* IPDF);
PHP_METHOD(dynapdf, ImportOCProperties)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfImportOCProperties(pdf);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfImportPage()
   const void* IPDF, UI32 PageNum); */
PHP_METHOD(dynapdf, ImportPage)
{
   zval **arg1;
   PPDF* pdf;
   SI32 PageNum;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageNum  = Z_LVAL_PP(arg1);

   rc = pdfImportPage(pdf, PageNum);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfImportPageEx()
   const void* IPDF, UI32 PageNum, double ScaleX, double ScaleY); */
PHP_METHOD(dynapdf, ImportPageEx)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 PageNum;
   double ScaleX, ScaleY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   PageNum  = Z_LVAL_PP(argv[0]);
   ScaleX   = Z_DVAL_PP(argv[1]);
   ScaleY   = Z_DVAL_PP(argv[2]);

   rc = pdfImportPageEx(pdf, PageNum, ScaleX, ScaleY);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfImportPDFFile()
   const void* IPDF, UI32 DestPage, double ScaleX, double ScaleY); */
PHP_METHOD(dynapdf, ImportPDFFile)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 DestPage;
   double ScaleX, ScaleY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   DestPage = Z_LVAL_PP(argv[0]);
   ScaleX   = Z_DVAL_PP(argv[1]);
   ScaleY   = Z_DVAL_PP(argv[2]);

   rc = pdfImportPDFFile(pdf, DestPage, ScaleX, ScaleY);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfInitColorManagement(const void* IPDF, struct TPDFColorProfiles* Profiles, TPDFColorSpace DestSpace, TPDFInitCMFlags Flags);
PHP_METHOD(dynapdf, InitColorManagement)
{
   zval **argv[3];
   PPDF* pdf;
   HashTable *arr_hash;
   struct TPDFColorProfiles Profiles;
   SI32 rc, DestSpace, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   arr_hash  = Z_ARRVAL_PP(argv[0]);
   DestSpace = Z_LVAL_PP(argv[1]);
   Flags     = Z_LVAL_PP(argv[2]);

   memset(&Profiles, 0, sizeof(Profiles));
   GetColorProfiles(arr_hash, &Profiles);

   rc = pdfInitColorManagement(pdf, &Profiles, DestSpace, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfInitStack(const void* IPDF, struct TPDFStack* Stack);
PHP_METHOD(dynapdf, InitStack)
{
   SI32 rc;
   struct dynapdf_object *obj;
   DYNAPDF_FETCH_PDFDOC_EX();

   if (!obj->pdfStack) obj->pdfStack = (struct TPDFStack*)emalloc(sizeof(struct TPDFStack));

   rc = pdfInitStack(obj->pdf, obj->pdfStack);
   RETURN_BOOL(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfInkAnnotA(const PPDF* IPDF, struct TFltPoint* Points, UI32 NumPoints, double LineWidth, UI32 Color, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Content);
PHP_METHOD(dynapdf, InkAnnot)
{
   zval **argv[7];
   PPDF* pdf;
   HashTable *arr_hash;
   struct TFltPoint* Points = NULL;
   SI32 rc, NumPoints;
   double LineWidth;
   UI32 Color;
   TPDFColorSpace CS;
   char* Author, *Subject, *Content;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_string_ex(argv[6]);

   arr_hash  = Z_ARRVAL_PP(argv[0]);
   NumPoints = zend_hash_num_elements(arr_hash) >> 1;
   LineWidth = Z_DVAL_PP(argv[1]);
   Color     = Z_LVAL_PP(argv[2]);
   CS        = Z_LVAL_PP(argv[3]);
   Author    = Z_STRVAL_PP(argv[4]);
   Subject   = Z_STRVAL_PP(argv[5]);
   Content   = Z_STRVAL_PP(argv[6]);

   if (NumPoints > 0)
   {
      Points = emalloc(NumPoints * sizeof(struct TFltPoint));
      NumPoints = GetFloatArray(arr_hash, (float*)Points, NumPoints << 1) >> 1;
   }
   rc = pdfInkAnnotA(pdf, Points, NumPoints, LineWidth, Color, CS, Author, Subject, Content);

   efree(Points);

   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfInsertBarcode(const PPDF* IPDF, double PosX, double PosY, double Width, double Height, TCellAlign HAlign, TCellAlign VAlign, struct TPDFBarcode2* Barcode);
PHP_METHOD(dynapdf, InsertBarcode)
{
   zval **argv[7];
   PPDF* pdf;
   HashTable* hash;
   SI32 HAlign, VAlign;
   struct TPDFBarcode2 Barcode;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_array_ex(argv[6]);

   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   HAlign = Z_LVAL_PP(argv[4]);
   VAlign = Z_LVAL_PP(argv[5]);
   hash   = Z_ARRVAL_PP(argv[6]);

   Barcode.StructSize = sizeof(Barcode);
   if (!pdfInitBarcode2(&Barcode))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Failed to initialize the barcode structure!");
      RETURN_NULL();
   }
   GetBarcode2Struct(hash, &Barcode);

   rc = pdfInsertBarcode(pdf, PosX, PosY, Width, Height, HAlign, VAlign, &Barcode);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfInsertBMPFromBuffer()
   const void* IPDF, double PosX, double PosY, double ScaleWidth, double ScaleHeight, const void* Buffer); */
PHP_METHOD(dynapdf, InsertBMPFromBuffer)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   char *Buffer;
   UI32 BufSize;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);

   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   ScaleWidth  = Z_DVAL_PP(argv[2]);
   ScaleHeight = Z_DVAL_PP(argv[3]);
   Buffer      = Z_STRVAL_PP(argv[4]);
   BufSize     = Z_STRLEN_PP(argv[4]);

   rc = pdfInsertImageFromBuffer(pdf, PosX, PosY, ScaleWidth, ScaleHeight, Buffer, BufSize, 0);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfInsertBMPFromHandle()
   const void* IPDF, double PosX, double PosY, double ScaleWidth, double ScaleHeight, void* hBitmap); */
PHP_METHOD(dynapdf, InsertBMPFromHandle)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   void *hBitmap;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   ScaleWidth  = Z_DVAL_PP(argv[2]);
   ScaleHeight = Z_DVAL_PP(argv[3]);
   hBitmap     = (void*)Z_LVAL_PP(argv[4]);

   rc = pdfInsertBMPFromHandle(pdf, PosX, PosY, ScaleWidth, ScaleHeight, hBitmap);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfInsertBookmarkA(const void* IPDF, const char* Title, SI32 Parent, UI32 DestPage, LBOOL Open, LBOOL AddChildren);
PHP_METHOD(dynapdf, InsertBookmark)
{
   zval **argv[5];
   PPDF* pdf;
   char* Title;
   SI32 rc, Parent, DestPage, Open, AddChildren;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   Title       = Z_STRVAL_PP(argv[0]);
   Parent      = Z_LVAL_PP(argv[1]);
   DestPage    = Z_LVAL_PP(argv[2]);
   Open        = Z_LVAL_PP(argv[3]);
   AddChildren = Z_LVAL_PP(argv[4]);

   rc = pdfInsertBookmarkA(pdf, Title, Parent, DestPage, Open, AddChildren);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfInsertBookmarkExA(const void* IPDF, const char* Title, SI32 Parent, UI32 NamedDest, LBOOL Open, LBOOL AddChildren);
PHP_METHOD(dynapdf, InsertBookmarkEx)
{
   zval **argv[5];
   PPDF* pdf;
   char* Title;
   SI32 rc, Parent, NamedDest, Open, AddChildren;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   Title       = Z_STRVAL_PP(argv[0]);
   Parent      = Z_LVAL_PP(argv[1]);
   NamedDest   = Z_LVAL_PP(argv[2]);
   Open        = Z_LVAL_PP(argv[3]);
   AddChildren = Z_LVAL_PP(argv[4]);

   rc = pdfInsertBookmarkExA(pdf, Title, Parent, NamedDest, Open, AddChildren);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfInsertImage()
   const void* IPDF, double PosX, double PosY, double ScaleWidth, double ScaleHeight, const char* AFile); */
PHP_METHOD(dynapdf, InsertImage)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   char *AFile;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   ScaleWidth  = Z_DVAL_PP(argv[2]);
   ScaleHeight = Z_DVAL_PP(argv[3]);
   AFile       = Z_STRVAL_PP(argv[4]);

   rc = pdfInsertImage(pdf, PosX, PosY, ScaleWidth, ScaleHeight, AFile);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfInsertImageEx()
   const void* IPDF, double PosX, double PosY, double ScaleWidth, double ScaleHeight, const char* AFile, UI32 Index); */
PHP_METHOD(dynapdf, InsertImageEx)
{
   zval **argv[6];
   PPDF* pdf;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   char *AFile;
   SI32 Index;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   ScaleWidth  = Z_DVAL_PP(argv[2]);
   ScaleHeight = Z_DVAL_PP(argv[3]);
   AFile       = Z_STRVAL_PP(argv[4]);
   Index       = Z_LVAL_PP(argv[5]);

   rc = pdfInsertImageEx(pdf, PosX, PosY, ScaleWidth, ScaleHeight, AFile, Index);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfInsertImageFromBuffer()
   const void* IPDF, double PosX, double PosY, double ScaleWidth, double ScaleHeight, const void* Buffer, UI32 BufSize, UI32 Index); */
PHP_METHOD(dynapdf, InsertImageFromBuffer)
{
   zval **argv[6];
   PPDF* pdf;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   char *Buffer;
   SI32 BufSize, Index;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   ScaleWidth  = Z_DVAL_PP(argv[2]);
   ScaleHeight = Z_DVAL_PP(argv[3]);
   Buffer      = Z_STRVAL_PP(argv[4]);
   BufSize     = Z_STRLEN_PP(argv[4]);
   Index       = Z_LVAL_PP(argv[5]);

   rc = pdfInsertImageFromBuffer(pdf, PosX, PosY, ScaleWidth, ScaleHeight, Buffer, BufSize, Index);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfInsertMetafileA(const void* IPDF, const char* FileName, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, InsertMetafile)
{
   zval **argv[5];
   PPDF* pdf;
   char *FileName;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   FileName = Z_STRVAL_PP(argv[0]);
   PosX   = Z_DVAL_PP(argv[1]);
   PosY   = Z_DVAL_PP(argv[2]);
   Width  = Z_DVAL_PP(argv[3]);
   Height = Z_DVAL_PP(argv[4]);

   rc = pdfInsertMetafile(pdf, FileName, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfInsertMetafileEx(const void* IPDF, const void* Buffer, UI32 BufSize, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, InsertMetafileEx)
{
   zval **argv[5];
   PPDF* pdf;
   char *Buffer;
   SI32 BufSize;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   Buffer  = Z_STRVAL_PP(argv[0]);
   BufSize = Z_STRLEN_PP(argv[0]);
   PosX    = Z_DVAL_PP(argv[1]);
   PosY    = Z_DVAL_PP(argv[2]);
   Width   = Z_DVAL_PP(argv[3]);
   Height  = Z_DVAL_PP(argv[4]);

   rc = pdfInsertMetafileEx(pdf, Buffer, BufSize, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfInsertMetafileExtA(const void* IPDF, const char* FileName, struct TRectL* View, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, InsertMetafileExt)
{
   zval **argv[6];
   PPDF* pdf;
   HashTable *arr_hash;
   struct TRectL View = {0, 0, 0, 0}, *pView = NULL;
   char *FileName;
   double PosX, PosY;
   double Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);

   FileName = Z_STRVAL_PP(argv[0]);
   arr_hash = Z_ARRVAL_PP(argv[1]);
   PosX     = Z_DVAL_PP(argv[2]);
   PosY     = Z_DVAL_PP(argv[3]);
   Width    = Z_DVAL_PP(argv[4]);
   Height   = Z_DVAL_PP(argv[5]);

   if (GetTRectLStruct(arr_hash, &View))
      pView = &View;

   rc = pdfInsertMetafileExtA(pdf, FileName, pView, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfInsertMetafileExtEx(const void* IPDF, const void* Buffer, UI32 BufSize, struct TRectL* View, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, InsertMetafileExtEx)
{
   zval **argv[6];
   PPDF* pdf;
   char *Buffer;
   SI32 BufSize;
   HashTable *arr_hash;
   struct TRectL View = {0, 0, 0, 0}, *pView = NULL;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_array_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);

   Buffer   = Z_STRVAL_PP(argv[0]);
   BufSize  = Z_STRLEN_PP(argv[0]);
   arr_hash = Z_ARRVAL_PP(argv[1]);
   PosX     = Z_DVAL_PP(argv[2]);
   PosY     = Z_DVAL_PP(argv[3]);
   Width    = Z_DVAL_PP(argv[4]);
   Height   = Z_DVAL_PP(argv[5]);

   if (GetTRectLStruct(arr_hash, &View))
      pView = &View;

   rc = pdfInsertMetafileExtEx(pdf, Buffer, BufSize, pView, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfInsertRawImage()
   const void* IPDF, const void* Buffer, UI32 BitsPerPixel, UI32 ColorCount, UI32 ImgWidth, UI32 ImgHeight, double PosX, double PosY, double ScaleWidth, double ScaleHeight); */
PHP_METHOD(dynapdf, InsertRawImage)
{
   zval **argv[9];
   PPDF* pdf;
   char *Buffer;
   SI32 rc, BitsPerPixel, ColorCount, ImgWidth, ImgHeight;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_double_ex(argv[7]);
   convert_to_double_ex(argv[8]);
   Buffer      = Z_STRVAL_PP(argv[0]);
   BitsPerPixel= Z_LVAL_PP(argv[1]);
   ColorCount  = Z_LVAL_PP(argv[2]);
   ImgWidth    = Z_LVAL_PP(argv[3]);
   ImgHeight   = Z_LVAL_PP(argv[4]);
   PosX        = Z_DVAL_PP(argv[5]);
   PosY        = Z_DVAL_PP(argv[6]);
   ScaleWidth  = Z_DVAL_PP(argv[7]);
   ScaleHeight = Z_DVAL_PP(argv[8]);

   rc = pdfInsertRawImage(pdf, Buffer, BitsPerPixel, ColorCount, ImgWidth, ImgHeight, PosX, PosY, ScaleWidth, ScaleHeight);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfInsertRawImageEx(const void* IPDF, double PosX, double PosY, double ScaleWidth, double ScaleHeight, struct TPDFRawImage* Image);
PHP_METHOD(dynapdf, InsertRawImageEx)
{
   zval **argv[5];
   PPDF* pdf;
   HashTable* array;
   struct TPDFRawImage Image;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_array_ex(argv[4]);

   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   ScaleWidth  = Z_DVAL_PP(argv[2]);
   ScaleHeight = Z_DVAL_PP(argv[3]);
   array       = Z_ARRVAL_PP(argv[4]);

   GetTPDFRawImageStruct(array, &Image);
   rc = pdfInsertRawImageEx(pdf, PosX, PosY, ScaleWidth, ScaleHeight, &Image);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfIsBidiText()
   const void* IPDF, const unsigned short* AText); */
PHP_METHOD(dynapdf, IsBidiText)
{
   zval **arg1;
   PPDF* pdf;
   unsigned short *AText;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   AText  = (unsigned short *)Z_STRVAL_PP(arg1);

   rc = pdfIsBidiText(pdf, AText);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfIsColorPage()
   const void* IPDF, LBOOL GrayIsColor); */
PHP_METHOD(dynapdf, IsColorPage)
{
   zval **arg1;
   PPDF* pdf;
   SI32 GrayIsColor;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   GrayIsColor  = Z_LVAL_PP(arg1);

   rc = pdfIsColorPage(pdf, GrayIsColor);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfIsEmptyPage()
   const void* IPDF); */
PHP_METHOD(dynapdf, IsEmptyPage)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfIsEmptyPage(pdf);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfLineAnnotA(const void* IPDF, double x1, double y1, double x2, double y2, double LineWidth, TLineEndStyle Start, TLineEndStyle End, UI32 FillColor, UI32 StrokeColor, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Content);
PHP_METHOD(dynapdf, LineAnnot)
{
   zval **argv[13];
   PPDF* pdf;
   double x1, y1, x2, y2, LineWidth;
   char* Author, *Subject, *Content;
   SI32 rc, Start, End, FillColor, StrokeColor, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 13 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_long_ex(argv[8]);
   convert_to_long_ex(argv[9]);
   convert_to_string_ex(argv[10]);
   convert_to_string_ex(argv[11]);
   convert_to_string_ex(argv[12]);

   x1          = Z_DVAL_PP(argv[0]);
   y1          = Z_DVAL_PP(argv[1]);
   x2          = Z_DVAL_PP(argv[2]);
   y2          = Z_DVAL_PP(argv[3]);
   LineWidth   = Z_DVAL_PP(argv[4]);
   Start       = Z_LVAL_PP(argv[5]);
   End         = Z_LVAL_PP(argv[6]);
   FillColor   = Z_LVAL_PP(argv[7]);
   StrokeColor = Z_LVAL_PP(argv[8]);
   CS          = Z_LVAL_PP(argv[9]);
   Author      = Z_STRVAL_PP(argv[10]);
   Subject     = Z_STRVAL_PP(argv[11]);
   Content     = Z_STRVAL_PP(argv[12]);

   rc = pdfLineAnnotA(pdf, x1, y1, x2, y2, LineWidth, Start, End, FillColor, StrokeColor, CS, Author, Subject, Content);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfLineTo()
   const void* IPDF, double PosX, double PosY); */
PHP_METHOD(dynapdf, LineTo)
{
   zval **argv[2];
   PPDF* pdf;
   double PosX, PosY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   PosX = Z_DVAL_PP(argv[0]);
   PosY = Z_DVAL_PP(argv[1]);

   rc = pdfLineTo(pdf, PosX, PosY);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfLoadCMap(const void* IPDF, const char* CMapName, LBOOL Embed);
PHP_METHOD(dynapdf, LoadCMap)
{
   zval **argv[2];
   PPDF* pdf;
   char* CMapName;
   SI32 rc, Embed;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   CMapName = Z_STRVAL_PP(argv[0]);
   Embed    = Z_LVAL_PP(argv[1]);

   rc = pdfLoadCMap(pdf, CMapName, Embed);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfLoadFDFDataA(const void* IPDF, const char* FileName, const char* Password, UI32 Flags);
PHP_METHOD(dynapdf, LoadFDFData)
{
   zval **argv[3];
   PPDF* pdf;
   char* FileName, *Password;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   FileName = Z_STRVAL_PP(argv[0]);
   Password = Z_STRVAL_PP(argv[1]);
   Flags    = Z_LVAL_PP(argv[2]);

   rc = pdfLoadFDFDataA(pdf, FileName, Password, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfLoadFDFDataEx(const void* IPDF, const void* Buffer, UI32 BufSize, const char* Password, UI32 Flags);
PHP_METHOD(dynapdf, LoadFDFDataEx)
{
   zval **argv[3];
   PPDF* pdf;
   char* Buffer, *Password;
   SI32 rc, BufSize, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Buffer   = Z_STRVAL_PP(argv[0]);
   BufSize  = Z_STRLEN_PP(argv[0]);
   Password = Z_STRVAL_PP(argv[1]);
   Flags    = Z_LVAL_PP(argv[2]);

   rc = pdfLoadFDFDataEx(pdf, Buffer, BufSize, Password, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfLoadFont(const void* IPDF, const void* Buffer, UI32 BufSize, TFStyle Style, double Size, LBOOL Embed, TCodepage CP);
PHP_METHOD(dynapdf, LoadFont)
{
   zval **argv[5];
   PPDF* pdf;
   char* Buffer;
   double Size;
   SI32 rc, BufSize, Style, Embed, CP;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   Buffer  = Z_STRVAL_PP(argv[0]);
   BufSize = Z_STRLEN_PP(argv[0]);
   Style   = Z_LVAL_PP(argv[1]);
   Size    = Z_DVAL_PP(argv[2]);
   Embed   = Z_LVAL_PP(argv[3]);
   CP      = Z_LVAL_PP(argv[4]);

   rc = pdfLoadFont(pdf, Buffer, BufSize, Style, Size, Embed, CP);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfLoadFontExA(const void* IPDF, const char* FontFile, UI32 Index, TFStyle Style, double Size, LBOOL Embed, TCodepage CP);
PHP_METHOD(dynapdf, LoadFontEx)
{
   zval **argv[6];
   PPDF* pdf;
   char* FontFile;
   double Size;
   SI32 rc, Index, Style, Embed, CP;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   FontFile = Z_STRVAL_PP(argv[0]);
   Index    = Z_LVAL_PP(argv[1]);
   Style    = Z_LVAL_PP(argv[2]);
   Size     = Z_DVAL_PP(argv[3]);
   Embed    = Z_LVAL_PP(argv[4]);
   CP       = Z_LVAL_PP(argv[5]);

   rc = pdfLoadFontExA(pdf, FontFile, Index, Style, Size, Embed, CP);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfLoadHeaderFooterSettings(const PPDF* IPDF, struct TPDFHeaderFooter* Init, struct TPDFHdrFtr* HFArray);
PHP_METHOD(dynapdf, LoadHeaderFooterSettings)
{
   SI32 cnt;
   PPDF* pdf;
   zval **arg1;
   SI32 searchRun = 0;
   struct TPDFHeaderFooter Init;
   struct TPDFHdrFtr HFArray[6];
   int argc = ZEND_NUM_ARGS();

   DYNAPDF_FETCH_PDFDOC();
   
   Init.StructSize = sizeof(Init);
   pdfInitHeaderFooter(&Init, HFArray, 6);
   Init.Flags |= hffLoadUTF8;
   
   
   if(argc > 0 && (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      argc = 0;
   }
   
   if (argc > 0)
   {
      convert_to_long_ex(arg1);
      if ((searchRun = (SI32)Z_LVAL_PP(arg1)) != 0)
         Init.Flags |= hffSearchRun;
   }
   cnt = pdfLoadHeaderFooterSettings(pdf, &Init, HFArray);
   if (searchRun)
   {
      RETURN_LONG(cnt);
   }else if (cnt > 0)
   {
      SI32 i;
      zval* node, *subNode;
      struct TPDFHdrFtr* v;
      array_init(return_value);

      ALLOC_INIT_ZVAL(node);
      array_init(node);
      add_assoc_long(node,        "FirstPage",      Init.FirstPage);
      add_assoc_long(node,        "Flags",          Init.Flags);
      add_assoc_long(node,        "InitColor",      Init.InitColor);
      add_assoc_long(node,        "InitCS",         Init.InitCS);
      add_assoc_long(node,        "InitCSHandle",   Init.InitCSHandle);
      ADD_ASSOC_UTF8_STRING(node, "InitDate",       (char*)Init.InitDate);
      add_assoc_long(node,        "InitCodepage",   Init.InitCodepage);
      add_assoc_long(node,        "InitEmbed",      Init.InitEmbed);
      ADD_ASSOC_UTF8_STRING(node, "InitFont",       (char*)Init.InitFontA);
      add_assoc_double(node,      "InitFontSize",   Init.InitFontSize);
      ADD_ASSOC_UTF8_STRING(node, "InitFontType",   (char*)Init.InitFontType);
      ADD_ASSOC_UTF8_STRING(node, "InitPageFormat", (char*)Init.InitPageFormatA);
      add_assoc_long(node,        "LastPage",       Init.LastPage);
      AddFltRect(node,            "Margin",        &Init.Margin);
      add_assoc_zval(return_value, "Init", node);

      ALLOC_INIT_ZVAL(node);
      array_init(node);
      for (i = 0; i < cnt; i++)
      {
         ALLOC_INIT_ZVAL(subNode);
         array_init(subNode);
         
         v = &HFArray[i];
         if (!v->TextLen) continue;
         if (v->CS != esInitSpace)
         {
            add_assoc_long(subNode, "Color", v->Color);
            add_assoc_long(subNode, "CS",    v->CS);
            if (v->CSHandle > -1) add_assoc_long(subNode, "CSHandle", v->CSHandle);
         }
         if (v->FontA)
         {
            add_assoc_long(subNode,        "Codepage", v->Codepage);
            add_assoc_bool(subNode,        "Embed",    v->Embed);
            ADD_ASSOC_UTF8_STRING(subNode, "Font",     (char*)v->FontA);
            add_assoc_double(subNode,      "FontSize", v->FontSize);
            ADD_ASSOC_UTF8_STRING(subNode, "FontType", (char*)v->FontType);
         }
         add_assoc_bool(subNode,         "IsHeader", v->IsHeader);
         add_assoc_long(subNode,         "Position", v->Position);
         add_assoc_double(subNode,       "ShiftX",   v->ShiftX);
         add_assoc_double(subNode,       "ShiftY",   v->ShiftY);
         ADD_ASSOC_UTF8_STRINGL(subNode, "Text",     (char*)v->TextA, v->TextLen);
         add_next_index_zval(node, subNode);
      }
      add_assoc_zval(return_value, "HFArray", node);
   }else
      RETURN_NULL();
}

//PDF_EXP LBOOL  PDF_CALL pdfLoadLayerConfig(const PPDF* IPDF, SI32 Index);
PHP_METHOD(dynapdf, LoadLayerConfig)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Index;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Index = Z_LVAL_PP(arg1);

   rc = pdfLoadLayerConfig(pdf, Index);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfLockLayer(const void* IPDF, UI32 Layer);
PHP_METHOD(dynapdf, LockLayer)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Layer;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Layer = Z_LVAL_PP(arg1);

   rc = pdfLockLayer(pdf, Layer);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfMovePage(const void* IPDF, UI32 Source, UI32 Dest);
PHP_METHOD(dynapdf, MovePage)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Source, Dest;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Source = Z_LVAL_PP(argv[0]);
   Dest   = Z_LVAL_PP(argv[1]);

   rc = pdfMovePage(pdf, Source, Dest);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfMoveTo()
   const void* IPDF, double PosX, double PosY); */
PHP_METHOD(dynapdf, MoveTo)
{
   zval **argv[2];
   PPDF* pdf;
   double PosX, PosY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   PosX = Z_DVAL_PP(argv[0]);
   PosY = Z_DVAL_PP(argv[1]);

   rc = pdfMoveTo(pdf, PosX, PosY);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfMultiplyMatrix(struct TCTM* M1, struct TCTM* M2, struct TCTM* NewMatrix);
PHP_METHOD(dynapdf, MultiplyMatrix)
{
   zval **argv[2];
   HashTable *arr1, *arr2;
   struct TCTM M1, M2, NewMatrix;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   convert_to_array_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   arr1 = Z_ARRVAL_PP(argv[0]);
   arr2 = Z_ARRVAL_PP(argv[1]);

   if (GetCTMFromArray(arr1, &M1))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"M1\" must be an array of six numbers (a, b, c, d, x, y)!");
      RETURN_NULL();
   }
   if (GetCTMFromArray(arr2, &M2))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"M2\" must be an array of six numbers (a, b, c, d, x, y)!");
      RETURN_NULL();
   }
   pdfMultiplyMatrix(&M1, &M2, &NewMatrix);
   array_init(return_value);
   add_assoc_double(return_value, "a", NewMatrix.a);
   add_assoc_double(return_value, "b", NewMatrix.b);
   add_assoc_double(return_value, "c", NewMatrix.c);
   add_assoc_double(return_value, "d", NewMatrix.d);
   add_assoc_double(return_value, "x", NewMatrix.x);
   add_assoc_double(return_value, "y", NewMatrix.y);
}

/* {{{ proto SI32 pdfOpenImportBuffer()
   const void* IPDF, const void* Buffer, UI32 BufSize, TPwdType PwdType, const char* Password); */
PHP_METHOD(dynapdf, OpenImportBuffer)
{
   zval **argv[3];
   PPDF* pdf;
   char *Buffer;
   SI32 BufSize;
   TPwdType PwdType;
   char *Password;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   Buffer  = Z_STRVAL_PP(argv[0]);
   BufSize = Z_STRLEN_PP(argv[0]);
   PwdType = Z_LVAL_PP(argv[1]);
   Password= Z_STRVAL_PP(argv[2]);

   rc = pdfOpenImportBuffer(pdf, Buffer, BufSize, PwdType, Password);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfOpenImportFile()
   const void* IPDF, const char* FileName, TPwdType PwdType, const char* Password); */
PHP_METHOD(dynapdf, OpenImportFile)
{
   zval **argv[3];
   PPDF* pdf;
   char *FileName;
   TPwdType PwdType;
   char *Password;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   FileName = Z_STRVAL_PP(argv[0]);
   PwdType  = Z_LVAL_PP(argv[1]);
   Password = Z_STRVAL_PP(argv[2]);

   rc = pdfOpenImportFileA(pdf, FileName, PwdType, Password);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfOpenOutputFileA(const PPDF* IPDF, const char* OutPDF);
PHP_METHOD(dynapdf, OpenOutputFile)
{
   zval **arg1;
   PPDF* pdf;
   char* OutPDF;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   OutPDF = Z_STRVAL_PP(arg1);

   rc = pdfOpenOutputFileA(pdf, OutPDF);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfOpenOutputFileEncryptedA(const PPDF* IPDF, const char* OutPDF, const char* OpenPwd, const char* OwnerPwd, TKeyLen KeyLen, TRestrictions Restrict);
PHP_METHOD(dynapdf, OpenOutputFileEncrypted)
{
   zval **argv[5];
   PPDF* pdf;
   char *OutPDF, *OpenPwd, *OwnerPwd;
   SI32 rc, KeyLen, Restrict;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   OutPDF   = Z_STRVAL_PP(argv[0]);
   OpenPwd  = Z_STRVAL_PP(argv[1]);
   OwnerPwd = Z_STRVAL_PP(argv[2]);
   KeyLen   = Z_LVAL_PP(argv[3]);
   Restrict = Z_LVAL_PP(argv[4]);

   rc = pdfOpenOutputFileEncryptedA(pdf, OutPDF, OpenPwd, OwnerPwd, KeyLen, Restrict);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfOpenTagA(const void* IPDF, TPDFBaseTag Tag, const char* Lang, const char* AltText, const char* Expansion);
PHP_METHOD(dynapdf, OpenTag)
{
   zval **argv[4];
   PPDF* pdf;
   char *Lang, *AltText, *Expansion;
   SI32 rc, Tag;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);

   Tag       = Z_LVAL_PP(argv[0]);
   Lang      = Z_STRVAL_PP(argv[1]);
   AltText   = Z_STRVAL_PP(argv[2]);
   Expansion = Z_STRVAL_PP(argv[3]);

   rc = pdfOpenTagA(pdf, Tag, Lang, AltText, Expansion);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfOpenTagBBoxA(const PPDF* IPDF, TPDFBaseTag Tag, const char* Lang, const char* AltText, const char* Expansion, struct TBBox* BBox);
PHP_METHOD(dynapdf, OpenTagBBox)
{
   zval **argv[4];
   PPDF* pdf;
   char *Lang, *AltText, *Expansion;
   struct TBBox bbox, *BBox;
   SI32 rc, Tag;
   int argc = ZEND_NUM_ARGS();
   BBox = NULL;
   if(argc < 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   if (argc > 4)
   {
      HashTable *arr;
      convert_to_array_ex(argv[4]);
      arr = Z_ARRVAL_PP(argv[4]);
      if (GetTBBoxFromArray(arr, &bbox) == 0)
         BBox = &bbox;
   }
   Tag       = Z_LVAL_PP(argv[0]);
   Lang      = Z_STRVAL_PP(argv[1]);
   AltText   = Z_STRVAL_PP(argv[2]);
   Expansion = Z_STRVAL_PP(argv[3]);

   rc = pdfOpenTagBBoxA(pdf, Tag, Lang, AltText, Expansion, BBox);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfOptimize(const PPDF* IPDF, TOptimizeFlags Flags, struct TOptimizeParams* Parms);
PHP_METHOD(dynapdf, Optimize)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr;
   SI32 rc, Flags;
   UI32 excludeCS[512];
   struct TOptimizeParams parms;
   struct TOptimizeParams* pparms = NULL;
   int argc = ZEND_NUM_ARGS();

   if(argc < 1 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();

   convert_to_long_ex(argv[0]);
   Flags = Z_LVAL_PP(argv[0]);
   if (argc > 1)
   {
      convert_to_array_ex(argv[1]);
      arr = Z_ARRVAL_PP(argv[1]);
      memset(&parms, 0, sizeof(parms));
      GetOptimizeParams(arr, &parms, excludeCS, 512);
      pparms = &parms;
   }
   rc = pdfOptimize(pdf, Flags, pparms);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfPageLink()
   const void* IPDF, double PosX, double PosY, double Width, double Height, UI32 DestPage); */
PHP_METHOD(dynapdf, PageLink)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 DestPage;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   PosX     = Z_DVAL_PP(argv[0]);
   PosY     = Z_DVAL_PP(argv[1]);
   Width    = Z_DVAL_PP(argv[2]);
   Height   = Z_DVAL_PP(argv[3]);
   DestPage = Z_LVAL_PP(argv[4]);

   rc = pdfPageLink(pdf, PosX, PosY, Width, Height, DestPage);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfPageLink2(const void* IPDF, double PosX, double PosY, double Width, double Height, UI32 NamedDest);
PHP_METHOD(dynapdf, PageLink2)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 NamedDest;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   PosX      = Z_DVAL_PP(argv[0]);
   PosY      = Z_DVAL_PP(argv[1]);
   Width     = Z_DVAL_PP(argv[2]);
   Height    = Z_DVAL_PP(argv[3]);
   NamedDest = Z_LVAL_PP(argv[4]);

   rc = pdfPageLink2(pdf, PosX, PosY, Width, Height, NamedDest);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfPageLink3A(const void* IPDF, double PosX, double PosY, double Width, double Height, const char* NamedDest);
PHP_METHOD(dynapdf, PageLink3)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char* NamedDest;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);

   PosX      = Z_DVAL_PP(argv[0]);
   PosY      = Z_DVAL_PP(argv[1]);
   Width     = Z_DVAL_PP(argv[2]);
   Height    = Z_DVAL_PP(argv[3]);
   NamedDest = Z_STRVAL_PP(argv[4]);

   rc = pdfPageLink3A(pdf, PosX, PosY, Width, Height, NamedDest);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfPageLinkEx()
   const void* IPDF, double PosX, double PosY, double Width, double Height, TDestType DestType, UI32 DestPage, double a, double b, double c, double d); */
PHP_METHOD(dynapdf, PageLinkEx)
{
   zval **argv[10];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   TDestType DestType;
   SI32 DestPage;
   double a, b, c, d;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 10 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_double_ex(argv[7]);
   convert_to_double_ex(argv[8]);
   convert_to_double_ex(argv[9]);

   PosX     = Z_DVAL_PP(argv[0]);
   PosY     = Z_DVAL_PP(argv[1]);
   Width    = Z_DVAL_PP(argv[2]);
   Height   = Z_DVAL_PP(argv[3]);
   DestType = Z_LVAL_PP(argv[4]);
   DestPage = Z_LVAL_PP(argv[5]);
   a        = Z_DVAL_PP(argv[6]);
   b        = Z_DVAL_PP(argv[7]);
   c        = Z_DVAL_PP(argv[8]);
   d        = Z_DVAL_PP(argv[9]);

   rc = pdfPageLinkEx(pdf, PosX, PosY, Width, Height, DestType, DestPage, a, b, c, d);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfPlaceImage()
   const void* IPDF, SI32 ImgHandle, double PosX, double PosY, double ScaleWidth, double ScaleHeight); */
PHP_METHOD(dynapdf, PlaceImage)
{
   zval **argv[5];
   PPDF* pdf;
   SI32 ImgHandle;
   double PosX, PosY, ScaleWidth, ScaleHeight;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   ImgHandle   = Z_LVAL_PP(argv[0]);
   PosX        = Z_DVAL_PP(argv[1]);
   PosY        = Z_DVAL_PP(argv[2]);
   ScaleWidth  = Z_DVAL_PP(argv[3]);
   ScaleHeight = Z_DVAL_PP(argv[4]);

   rc = pdfPlaceImage(pdf, ImgHandle, PosX, PosY, ScaleWidth, ScaleHeight);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfPlaceSigFieldValidateIcon(const void* IPDF, UI32 SigField, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, PlaceSigFieldValidateIcon)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 rc, SigField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   SigField = Z_LVAL_PP(argv[0]);
   PosX     = Z_DVAL_PP(argv[1]);
   PosY     = Z_DVAL_PP(argv[2]);
   Width    = Z_DVAL_PP(argv[3]);
   Height   = Z_DVAL_PP(argv[4]);

   rc = pdfPlaceSigFieldValidateIcon(pdf, SigField, PosX, PosY, Width, Height);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfPlaceTemplate(const void* IPDF, SI32 TmplHandle, double PosX, double PosY, double ScaleWidth, double ScaleHeight);
PHP_METHOD(dynapdf, PlaceTemplate)
{
   zval **argv[5];
   PPDF* pdf;
   SI32 TmplHandle;
   double PosX, PosY;
   double ScaleWidth, ScaleHeight;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   TmplHandle  = Z_LVAL_PP(argv[0]);
   PosX        = Z_DVAL_PP(argv[1]);
   PosY        = Z_DVAL_PP(argv[2]);
   ScaleWidth  = Z_DVAL_PP(argv[3]);
   ScaleHeight = Z_DVAL_PP(argv[4]);

   rc = pdfPlaceTemplate(pdf, TmplHandle, PosX, PosY, ScaleWidth, ScaleHeight);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfPlaceTemplateEx(const void* IPDF, SI32 TmplHandle, double PosX, double PosY, double ScaleWidth, double ScaleHeight);
PHP_METHOD(dynapdf, PlaceTemplateEx)
{
   zval **argv[5];
   PPDF* pdf;
   SI32 TmplHandle;
   double PosX, PosY;
   double ScaleWidth, ScaleHeight;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   TmplHandle  = Z_LVAL_PP(argv[0]);
   PosX        = Z_DVAL_PP(argv[1]);
   PosY        = Z_DVAL_PP(argv[2]);
   ScaleWidth  = Z_DVAL_PP(argv[3]);
   ScaleHeight = Z_DVAL_PP(argv[4]);

   rc = pdfPlaceTemplateEx(pdf, TmplHandle, PosX, PosY, ScaleWidth, ScaleHeight);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfPlaceTemplByMatrix(const void* IPDF, SI32 TmplHandle);
PHP_METHOD(dynapdf, PlaceTemplByMatrix)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, TmplHandle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   TmplHandle = Z_LVAL_PP(arg1);

   rc = pdfPlaceTemplByMatrix(pdf, TmplHandle);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfPolygonAnnotA(const void* IPDF, struct TFltPoint* Vertices, UI32 NumVertices, double LineWidth, UI32 FillColor, UI32 StrokeColor, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Content);
PHP_METHOD(dynapdf, PolygonAnnot)
{
   zval **argv[8];
   PPDF* pdf;
   HashTable *arr;
   double LineWidth;
   struct TFltPoint* Vertices;
   char* Author, *Subject, *Content;
   SI32 rc, NumVertices, FillColor, StrokeColor, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_string_ex(argv[7]);

   arr         = Z_ARRVAL_PP(argv[0]);
   LineWidth   = Z_DVAL_PP(argv[1]);
   FillColor   = Z_LVAL_PP(argv[2]);
   StrokeColor = Z_LVAL_PP(argv[3]);
   CS          = Z_LVAL_PP(argv[4]);
   Author      = Z_STRVAL_PP(argv[5]);
   Subject     = Z_STRVAL_PP(argv[6]);
   Content     = Z_STRVAL_PP(argv[7]);

   NumVertices = zend_hash_num_elements(arr) >> 1;
   if (NumVertices < 2)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "At least two x/y coordinate pairs must be provided!");
      RETURN_LONG(-1);
   }
   Vertices = emalloc(NumVertices * sizeof(struct TFltPoint));
   NumVertices = GetFloatArray(arr, (float*)Vertices, NumVertices << 1) >> 1;

   rc = pdfPolygonAnnotA(pdf, Vertices, NumVertices, LineWidth, FillColor, StrokeColor, CS, Author, Subject, Content);
   efree(Vertices);

   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfPolyLineAnnotA(const void* IPDF, struct TFltPoint* Vertices, UI32 NumVertices, double LineWidth, TLineEndStyle Start, TLineEndStyle End, UI32 FillColor, UI32 StrokeColor, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Content);
PHP_METHOD(dynapdf, PolyLineAnnot)
{
   zval **argv[10];
   PPDF* pdf;
   HashTable *arr;
   double LineWidth;
   struct TFltPoint* Vertices;
   char* Author, *Subject, *Content;
   SI32 rc, NumVertices, Start, End, FillColor, StrokeColor, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 10 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_string_ex(argv[7]);
   convert_to_string_ex(argv[8]);
   convert_to_string_ex(argv[9]);

   arr         = Z_ARRVAL_PP(argv[0]);
   LineWidth   = Z_DVAL_PP(argv[1]);
   Start       = Z_LVAL_PP(argv[2]);
   End         = Z_LVAL_PP(argv[3]);
   FillColor   = Z_LVAL_PP(argv[4]);
   StrokeColor = Z_LVAL_PP(argv[5]);
   CS          = Z_LVAL_PP(argv[6]);
   Author      = Z_STRVAL_PP(argv[7]);
   Subject     = Z_STRVAL_PP(argv[8]);
   Content     = Z_STRVAL_PP(argv[9]);

   NumVertices = zend_hash_num_elements(arr) >> 1;
   if (NumVertices < 2)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "At least two x/y coordinate pairs must be provided!");
      RETURN_LONG(-1);
   }
   Vertices = emalloc(NumVertices * sizeof(struct TFltPoint));
   GetFloatArray(arr, (float*)Vertices, NumVertices << 1);

   rc = pdfPolyLineAnnotA(pdf, Vertices, NumVertices, LineWidth, Start, End, FillColor, StrokeColor, CS, Author, Subject, Content);
   efree(Vertices);

   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfPrintPageA(const PPDF* IPDF, UI32 PageNum, const char* DocName, const void* DC, TPDFPrintFlags Flags, struct TRectL* Margin, struct TPDFPrintParams* Parms);
PHP_METHOD(dynapdf, PrintPage)
{
   php_error_docref(NULL TSRMLS_CC, E_ERROR, "This function is not implemented!");
   RETURN_BOOL(0);
}

// PDF_EXP LBOOL  PDF_CALL pdfPrintPDFFileA(const PPDF* IPDF, const char* TmpDir, const char* DocName, const void* DC, TPDFPrintFlags Flags, struct TRectL* Margin, const void* Parms);
PHP_METHOD(dynapdf, PrintPDFFile)
{
   php_error_docref(NULL TSRMLS_CC, E_ERROR, "This function is not implemented!");
   RETURN_BOOL(0);
}

/* {{{ proto SI32 pdfReadImageFormat()
   const void* IPDF, const char* FileName, UI32 * Width, UI32 * Height, SI32 * BitsPerPixel, SI32 * UseZip); */
PHP_METHOD(dynapdf, ReadImageFormat)
{
   zval **arg1;
   PPDF* pdf;
   char *FileName;
   UI32 Width=0, Height=0;
   SI32 rc, BitsPerPixel=0, UseZip=0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   FileName = Z_STRVAL_PP(arg1);

   if ((rc = pdfReadImageFormat(pdf, FileName, &Width, &Height, &BitsPerPixel, &UseZip)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Width", Width);
      add_assoc_long(return_value, "Height", Height);
      add_assoc_long(return_value, "BitsPerPixel", BitsPerPixel);
      add_assoc_bool(return_value, "UseZip", UseZip);
   }
}
/* }}} */

/* {{{ proto SI32 pdfReadImageFormat2()
   const void* IPDF, const char* FileName, UI32 Index, UI32 * Width, UI32 * Height, SI32 * BitsPerPixel, SI32 * UseZip); */
PHP_METHOD(dynapdf, ReadImageFormat2)
{
   zval **argv[2];
   PPDF* pdf;
   char *FileName;
   UI32 Index, Width=0, Height=0;
   SI32 rc, BitsPerPixel=0, UseZip=0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   FileName = Z_STRVAL_PP(argv[0]);
   Index    = Z_LVAL_PP(argv[1]);

   if ((rc = pdfReadImageFormat2(pdf, FileName, Index, &Width, &Height, &BitsPerPixel, &UseZip)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Width", Width);
      add_assoc_long(return_value, "Height", Height);
      add_assoc_long(return_value, "BitsPerPixel", BitsPerPixel);
      add_assoc_bool(return_value, "UseZip", UseZip);
   }
}
/* }}} */


/* {{{ proto SI32 pdfReadImageFormatEx()
   const void* IPDF, void* hBitmap, UI32 * Width, UI32 * Height, SI32 * BitsPerPixel, SI32 * UseZip); */
PHP_METHOD(dynapdf, ReadImageFormatEx)
{
   zval **arg1;
   PPDF* pdf;
   void *hBitmap;
   UI32 Width=0, Height=0;
   SI32 rc, BitsPerPixel=0, UseZip=0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   hBitmap = (void*)Z_LVAL_PP(arg1);

   if ((rc = pdfReadImageFormatEx(pdf, hBitmap, &Width, &Height, &BitsPerPixel, &UseZip)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Width", Width);
      add_assoc_long(return_value, "Height", Height);
      add_assoc_long(return_value, "BitsPerPixel", BitsPerPixel);
      add_assoc_bool(return_value, "UseZip", UseZip);
   }
}
/* }}} */

/* {{{ proto SI32 pdfReadImageFormatFromBuffer()
   const void* IPDF, const void* Buffer, UI32 BufSize, UI32 Index, UI32 * Width, UI32 * Height, SI32 * BitsPerPixel, SI32 * UseZip); */
PHP_METHOD(dynapdf, ReadImageFormatFromBuffer)
{
   zval **argv[2];
   PPDF* pdf;
   char *Buffer;
   UI32 BufSize, Index, Width=0, Height=0;
   SI32 rc, BitsPerPixel=0, UseZip=0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Buffer   = Z_STRVAL_PP(argv[0]);
   BufSize  = Z_STRLEN_PP(argv[0]);
   Index    = Z_LVAL_PP(argv[1]);

   if ((rc = pdfReadImageFormatFromBuffer(pdf, Buffer, BufSize, Index, &Width, &Height, &BitsPerPixel, &UseZip)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "Width", Width);
      add_assoc_long(return_value, "Height", Height);
      add_assoc_long(return_value, "BitsPerPixel", BitsPerPixel);
      add_assoc_bool(return_value, "UseZip", UseZip);
   }
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfReadImageResolutionA(const void* IPDF, const char* FileName, UI32 Index, UI32 ADDR ResX, UI32 ADDR ResY);
PHP_METHOD(dynapdf, ReadImageResolution)
{
   zval **argv[2];
   PPDF* pdf;
   char *FileName;
   UI32 Index, ResX=0, ResY=0;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   FileName = Z_STRVAL_PP(argv[0]);
   Index    = Z_LVAL_PP(argv[1]);

   if ((rc = pdfReadImageResolutionA(pdf, FileName, Index, &ResX, &ResY)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "ResX", ResX);
      add_assoc_long(return_value, "ResY", ResY);
   }
}

//PDF_EXP LBOOL  PDF_CALL pdfReadImageResolutionEx(const void* IPDF, const void* Buffer, UI32 BufSize, UI32 Index, UI32 ADDR ResX, UI32 ADDR ResY);
PHP_METHOD(dynapdf, ReadImageResolutionEx)
{
   zval **argv[2];
   PPDF* pdf;
   char *Buffer;
   UI32 BufSize, Index, ResX=0, ResY=0;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Buffer  = Z_STRVAL_PP(argv[0]);
   BufSize = Z_STRLEN_PP(argv[0]);
   Index   = Z_LVAL_PP(argv[1]);

   if ((rc = pdfReadImageResolutionEx(pdf, Buffer, BufSize, Index, &ResX, &ResY)) != 0)
   {
      array_init(return_value);
      add_assoc_long(return_value, "ResX", ResX);
      add_assoc_long(return_value, "ResY", ResY);
   }
}

/* {{{ proto SI32 pdfRectangle()
   const void* IPDF, double PosX, double PosY, double Width, double Height, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, Rectangle)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   FillMode = Z_LVAL_PP(argv[4]);

   rc = pdfRectangle(pdf, PosX, PosY, Width, Height, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfReEncryptPDF()
   const void* IPDF, const char* FileName, TPwdType PwdType, const char* InPwd, const char* NewOpenPwd, const char* NewOwnerPwd, TKeyLen NewKeyLen, TRestrictions Restrict); */
PHP_METHOD(dynapdf, ReEncryptPDF)
{
   zval **argv[7];
   PPDF* pdf;
   char *FileName;
   TPwdType PwdType;
   char *InPwd, *NewOpenPwd, *NewOwnerPwd;
   TKeyLen NewKeyLen;
   TRestrictions Restrict;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   FileName    = Z_STRVAL_PP(argv[0]);
   PwdType     = Z_LVAL_PP(argv[1]);
   InPwd       = Z_STRVAL_PP(argv[2]);
   NewOpenPwd  = Z_STRVAL_PP(argv[3]);
   NewOwnerPwd = Z_STRVAL_PP(argv[4]);
   NewKeyLen   = Z_LVAL_PP(argv[5]);
   Restrict    = Z_LVAL_PP(argv[6]);

   rc = pdfReEncryptPDFA(pdf, FileName, PwdType, InPwd, NewOpenPwd, NewOwnerPwd, NewKeyLen, Restrict);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfRenameSpotColor(const void* IPDF, const char* Colorant, const char* NewName);
PHP_METHOD(dynapdf, RenameSpotColor)
{
   zval **argv[2];
   PPDF* pdf;
   char *Colorant, *NewName;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Colorant = Z_STRVAL_PP(argv[0]);
   NewName  = Z_STRVAL_PP(argv[1]);

   rc = pdfRenameSpotColor(pdf, Colorant, NewName);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfRenderPageToImageA(const void* IPDF, UI32 PageNum, const char* OutFile, UI32 Resolution, SI32 Width, SI32 Height, TRasterFlags Flags, TPDFPixFormat PixFmt, TCompressionFilter Filter, TImageFormat Format);
PHP_METHOD(dynapdf, RenderPageToImage)
{
   zval **argv[9];
   PPDF* pdf;
   char* OutFile;
   SI32 rc, PageNum, Resolution, Width, Height, Flags, PixFmt, Filter, Format;

   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_long_ex(argv[8]);

   PageNum    = Z_LVAL_PP(argv[0]);
   OutFile    = Z_STRVAL_PP(argv[1]);
   Resolution = Z_LVAL_PP(argv[2]);
   Width      = Z_LVAL_PP(argv[3]);
   Height     = Z_LVAL_PP(argv[4]);
   Flags      = Z_LVAL_PP(argv[5]);
   PixFmt     = Z_LVAL_PP(argv[6]);
   Filter     = Z_LVAL_PP(argv[7]);
   Format     = Z_LVAL_PP(argv[8]);

   rc = pdfRenderPageToImageA(pdf, PageNum, OutFile, Resolution, Width, Height, Flags, PixFmt, Filter, Format);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfRenderPDFFileA(const void* IPDF, const char* OutFile, UI32 Resolution, TRasterFlags Flags, TPDFPixFormat PixFmt, TCompressionFilter Filter, TImageFormat Format);
PHP_METHOD(dynapdf, RenderPDFFile)
{
   zval **argv[6];
   PPDF* pdf;
   char* OutFile;
   SI32 rc, Resolution, Flags, PixFmt, Filter, Format;

   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   OutFile    = Z_STRVAL_PP(argv[0]);
   Resolution = Z_LVAL_PP(argv[1]);
   Flags      = Z_LVAL_PP(argv[2]);
   PixFmt     = Z_LVAL_PP(argv[3]);
   Filter     = Z_LVAL_PP(argv[4]);
   Format     = Z_LVAL_PP(argv[5]);

   rc = pdfRenderPDFFileA(pdf, OutFile, Resolution, Flags, PixFmt, Filter, Format);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfRenderPDFFileExA(const PPDF* IPDF, const char* OutFile, UI32 Resolution, SI32 Width, SI32 Height, TRasterFlags Flags, TPDFPixFormat PixFmt, TCompressionFilter Filter, TImageFormat Format);
PHP_METHOD(dynapdf, RenderPDFFileEx)
{
   zval **argv[8];
   PPDF* pdf;
   char* OutFile;
   SI32 rc, Resolution, Width, Height, Flags, PixFmt, Filter, Format;

   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   OutFile    = Z_STRVAL_PP(argv[0]);
   Resolution = Z_LVAL_PP(argv[1]);
   Width      = Z_LVAL_PP(argv[2]);
   Height     = Z_LVAL_PP(argv[3]);
   Flags      = Z_LVAL_PP(argv[4]);
   PixFmt     = Z_LVAL_PP(argv[5]);
   Filter     = Z_LVAL_PP(argv[6]);
   Format     = Z_LVAL_PP(argv[7]);

   rc = pdfRenderPDFFileExA(pdf, OutFile, Resolution, Width, Height, Flags, PixFmt, Filter, Format);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfReOpenImportFile(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, ReOpenImportFile)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfReOpenImportFile(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfReplaceFontA(const void* IPDF, const PFNT* PDFFont, const char* Name, TFStyle Style, LBOOL NameIsFamilyName);
PHP_METHOD(dynapdf, ReplaceFont)
{
   zval **argv[4];
   struct dynapdf_object *obj;
   UI32 hash;
   PFNT* PDFFont;
   char* Name;
   SI32 rc, Style, NameIsFamilyName;

   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   hash             = Z_LVAL_PP(argv[0]);
   Name             = Z_STRVAL_PP(argv[1]);
   Style            = Z_LVAL_PP(argv[2]);
   NameIsFamilyName = Z_LVAL_PP(argv[3]);

   if ((PDFFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter PDFFont contains an invalid value!");
      RETURN_LONG(-1);
   }

   rc = pdfReplaceFontA(obj->pdf, PDFFont, Name, Style, NameIsFamilyName);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfReplaceFontExA(const void* IPDF, const PFNT* PDFFont, const char* FontFile, LBOOL Embed);
PHP_METHOD(dynapdf, ReplaceFontEx)
{
   zval **argv[3];
   struct dynapdf_object *obj;
   UI32 hash;
   PFNT* PDFFont;
   char* FontFile;
   SI32 rc, Embed;

   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   hash     = Z_LVAL_PP(argv[0]);
   FontFile = Z_STRVAL_PP(argv[1]);
   Embed    = Z_LVAL_PP(argv[2]);

   if ((PDFFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter PDFFont contains an invalid value!");
      RETURN_LONG(-1);
   }

   rc = pdfReplaceFontExA(obj->pdf, PDFFont, FontFile, Embed);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfReplaceICCProfileA(const void* IPDF, UI32 ColorSpace, const char* ICCFile);
PHP_METHOD(dynapdf, ReplaceICCProfile)
{
   zval **argv[2];
   void* pdf;
   char* ICCFile;
   SI32 rc, ColorSpace;

   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   ColorSpace = Z_LVAL_PP(argv[0]);
   ICCFile    = Z_STRVAL_PP(argv[1]);

   rc = pdfReplaceICCProfileA(pdf, ColorSpace, ICCFile);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfReplaceICCProfileEx(const PPDF* IPDF, UI32 ColorSpace, const void* Buffer, UI32 BufSize);
PHP_METHOD(dynapdf, ReplaceICCProfileEx)
{
   zval **argv[2];
   void* pdf;
   char* Buffer;
   SI32 rc, ColorSpace, BufSize;

   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   ColorSpace = Z_LVAL_PP(argv[0]);
   Buffer     = Z_STRVAL_PP(argv[1]);
   BufSize    = Z_STRLEN_PP(argv[1]);

   rc = pdfReplaceICCProfileEx(pdf, ColorSpace, Buffer, BufSize);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfReplaceImageA(const PPDF* IPDF, const PIMG* Source, const char* Image, UI32 Index, TExtColorSpace CS, SI32 CSHandle, TReplaceImageFlags Flags);
PHP_METHOD(dynapdf, ReplaceImage)
{
   zval **argv[6];
   struct dynapdf_object *obj;
   char* Image;
   UI32 Index, hash;
   const PIMG* Source;
   SI32 rc, CS, CSHandle, Flags;

   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   hash     = Z_LVAL_PP(argv[0]);
   Image    = Z_STRVAL_PP(argv[1]);
   Index    = Z_LVAL_PP(argv[2]);
   CS       = Z_LVAL_PP(argv[3]);
   CSHandle = Z_LVAL_PP(argv[4]);
   Flags    = Z_LVAL_PP(argv[5]);

   if ((Source = (PIMG*)GetPointer(obj, hash, ptImage)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Source contains an invalid value!");
      RETURN_BOOL(0);
   }

   rc = pdfReplaceImageA(obj->pdf, Source, Image, Index, CS, CSHandle, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfReplaceImageEx(const PPDF* IPDF, const PIMG* Source, const void* Buffer, UI32 BufSize, UI32 Index, TExtColorSpace CS, SI32 CSHandle, TReplaceImageFlags Flags);
PHP_METHOD(dynapdf, ReplaceImageEx)
{
   zval **argv[6];
   struct dynapdf_object *obj;
   char* Buffer;
   UI32 Index, hash, BufSize;
   const PIMG* Source;
   SI32 rc, CS, CSHandle, Flags;

   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   hash     = Z_LVAL_PP(argv[0]);
   Buffer   = Z_STRVAL_PP(argv[1]);
   BufSize  = Z_STRLEN_PP(argv[1]);
   Index    = Z_LVAL_PP(argv[2]);
   CS       = Z_LVAL_PP(argv[3]);
   CSHandle = Z_LVAL_PP(argv[4]);
   Flags    = Z_LVAL_PP(argv[5]);

   if ((Source = (PIMG*)GetPointer(obj, hash, ptImage)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Source contains an invalid value!");
      RETURN_BOOL(0);
   }

   rc = pdfReplaceImageEx(obj->pdf, Source, Buffer, BufSize, Index, CS, CSHandle, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfReplacePageTextA(const void* IPDF, const char* NewText, struct TPDFStack* Stack);
PHP_METHOD(dynapdf, ReplacePageText)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   char* NewText;
   SI32 rc;

   int argc = ZEND_NUM_ARGS();

   if(argc < 1 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   if (!obj->pdfStack)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Stack is not initialized! You must call InitStack() and GetPageText() before this function can be called!");
      RETURN_BOOL(0);
   }

   convert_to_string_ex(argv[0]);
   if (argc > 1)
   {
      convert_to_long_ex(argv[1]);
      obj->pdfStack->DeleteKerningAt = Z_LVAL_PP(argv[1]);
   }

   NewText = Z_STRVAL_PP(argv[0]);

   rc = pdfReplacePageTextA(obj->pdf, NewText, obj->pdfStack);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfReplacePageTextExA(const void* IPDF, const char* NewText, struct TPDFStack* Stack);
PHP_METHOD(dynapdf, ReplacePageTextEx)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   char* NewText;
   SI32 rc;

   int argc = ZEND_NUM_ARGS();

   if(argc < 1 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   if (!obj->pdfStack)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Stack is not initialized! You must call InitStack() and GetPageText() before this function can be called!");
      RETURN_BOOL(0);
   }

   convert_to_string_ex(argv[0]);
   if (argc > 1)
   {
      convert_to_long_ex(argv[1]);
      obj->pdfStack->DeleteKerningAt = Z_LVAL_PP(argv[1]);
   }

   NewText = Z_STRVAL_PP(argv[0]);

   rc = pdfReplacePageTextExA(obj->pdf, NewText, obj->pdfStack);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfResetAnnotAP(const PPDF* IPDF, SI32 Handle);
PHP_METHOD(dynapdf, ResetAnnotAP)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);

   Handle = Z_LVAL_PP(arg1);

   rc = pdfResetAnnotAP(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfResetEncryptionSettings(const PPDF* IPDF);
PHP_METHOD(dynapdf, ResetEncryptionSettings)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfResetEncryptionSettings(pdf);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfResetLineDashPattern()
   const void* IPDF); */
PHP_METHOD(dynapdf, ResetLineDashPattern)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfResetLineDashPattern(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfRestoreGraphicState()
   const void* IPDF); */
PHP_METHOD(dynapdf, RestoreGraphicState)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfRestoreGraphicState(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

// RGB -> PHP specific
PHP_METHOD(dynapdf, RGB)
{
   zval **argv[3];
   PPDF* pdf;
   BYTE r, g, b;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   r = (BYTE)Z_LVAL_PP(argv[0]);
   g = (BYTE)Z_LVAL_PP(argv[1]);
   b = (BYTE)Z_LVAL_PP(argv[2]);

   rc = PDF_RGB(r, g, b);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfRotateCoords()
   const void* IPDF, double alpha, double OriginX, double OriginY); */
PHP_METHOD(dynapdf, RotateCoords)
{
   zval **argv[3];
   PPDF* pdf;
   double alpha, OriginX, OriginY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   alpha   = Z_DVAL_PP(argv[0]);
   OriginX = Z_DVAL_PP(argv[1]);
   OriginY = Z_DVAL_PP(argv[2]);

   rc = pdfRotateCoords(pdf, alpha, OriginX, OriginY);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfRoundRect()
   const void* IPDF, double PosX, double PosY, double Width, double Height, double Radius, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, RoundRect)
{
   zval **argv[6];
   PPDF* pdf;
   double PosX, PosY, Width, Height, Radius;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   Radius = Z_DVAL_PP(argv[4]);
   FillMode = Z_LVAL_PP(argv[5]);

   rc = pdfRoundRect(pdf, PosX, PosY, Width, Height, Radius, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfRoundRectEx()
   const void* IPDF, double PosX, double PosY, double Width, double Height, double rWidth, double rHeight, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, RoundRectEx)
{
   zval **argv[7];
   PPDF* pdf;
   double PosX, PosY, Width, Height, rWidth, rHeight;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   rWidth = Z_DVAL_PP(argv[4]);
   rHeight = Z_DVAL_PP(argv[5]);
   FillMode = Z_LVAL_PP(argv[6]);

   rc = pdfRoundRectEx(pdf, PosX, PosY, Width, Height, rWidth, rHeight, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSaveGraphicState()
   const void* IPDF); */
PHP_METHOD(dynapdf, SaveGraphicState)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfSaveGraphicState(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfScaleCoords()
   const void* IPDF, double sx, double sy); */
PHP_METHOD(dynapdf, ScaleCoords)
{
   zval **argv[2];
   PPDF* pdf;
   double sx, sy;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   sx = Z_DVAL_PP(argv[0]);
   sy = Z_DVAL_PP(argv[1]);

   rc = pdfScaleCoords(pdf, sx, sy);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSelfTest()
   const void* IPDF); */
PHP_METHOD(dynapdf, SelfTest)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfSelfTest(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSet3DAnnotProps(const void* IPDF, UI32 Handle, T3DActivationType ActType, T3DDeActivateType DeActType, T3DInstanceType InstType, T3DDeActInstance DeInstType, LBOOL DisplToolbar, LBOOL DisplModelTree);
PHP_METHOD(dynapdf, Set3DAnnotProps)
{
   zval **argv[7];
   PPDF* pdf;
   SI32 rc, Handle, ActType, DeActType, InstType, DeInstType, DisplToolbar, DisplModelTree;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);

   Handle         = Z_LVAL_PP(argv[0]);
   ActType        = Z_LVAL_PP(argv[1]);
   DeActType      = Z_LVAL_PP(argv[2]);
   InstType       = Z_LVAL_PP(argv[3]);
   DeInstType     = Z_LVAL_PP(argv[4]);
   DisplToolbar   = Z_LVAL_PP(argv[5]);
   DisplModelTree = Z_LVAL_PP(argv[6]);

   rc = pdfSet3DAnnotProps(pdf, Handle, ActType, DeActType, InstType, DeInstType, DisplToolbar, DisplModelTree);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSet3DAnnotScriptA(const void* IPDF, UI32 Annot, const char* Value, UI32 Len);
PHP_METHOD(dynapdf, Set3DAnnotScript)
{
   zval **argv[2];
   PPDF* pdf;
   char* Value;
   SI32 rc, Annot, Len;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Annot = Z_LVAL_PP(argv[0]);
   Value = Z_STRVAL_PP(argv[1]);
   Len   = Z_STRLEN_PP(argv[1]);

   rc = pdfSet3DAnnotScriptA(pdf, Annot, Value, Len);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetAllocBy()
   const void* IPDF, SI32 Value); */
PHP_METHOD(dynapdf, SetAllocBy)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetAllocBy(pdf, Value);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetAltFontsA(const PPDF* IPDF, UI32 ListHandle, const char** List, UI32 Count);
PHP_METHOD(dynapdf, SetAltFonts)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr_hash;
   SI32 rc, ListHandle, cnt;
   char* list[512];
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   ListHandle = Z_LVAL_PP(argv[0]);
   arr_hash   = Z_ARRVAL_PP(argv[1]);

   cnt = GetStringArray(arr_hash, list, 512);
   rc  = pdfSetAltFontsA(pdf, ListHandle, list, cnt);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotBorderEffect(const PPDF* IPDF, UI32 Handle, TBorderEffect Value);
PHP_METHOD(dynapdf, SetAnnotBorderEffect)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Style;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Style  = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotBorderEffect(pdf, Handle, Style);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotBorderStyle(const void* IPDF, UI32 Handle, TBorderStyle Style);
PHP_METHOD(dynapdf, SetAnnotBorderStyle)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Style;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Style  = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotBorderStyle(pdf, Handle, Style);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotBorderWidth(const void* IPDF, UI32 Handle, double LineWidth);
PHP_METHOD(dynapdf, SetAnnotBorderWidth)
{
   zval **argv[2];
   PPDF* pdf;
   double LineWidth;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   Handle    = Z_LVAL_PP(argv[0]);
   LineWidth = Z_DVAL_PP(argv[1]);

   rc = pdfSetAnnotBorderWidth(pdf, Handle, LineWidth);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotColor(const void* IPDF, UI32 Handle, TAnnotColor ColorType, TPDFColorSpace CS, UI32 Color);
PHP_METHOD(dynapdf, SetAnnotColor)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, Handle, ColorType, CS, Color;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Handle    = Z_LVAL_PP(argv[0]);
   ColorType = Z_LVAL_PP(argv[1]);
   CS        = Z_LVAL_PP(argv[1]);
   Color     = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotColor(pdf, Handle, ColorType, CS, Color);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetAnnotFlags()
   const void* IPDF, TAnnotFlags Flags); */
PHP_METHOD(dynapdf, SetAnnotFlags)
{
   zval **arg1;
   PPDF* pdf;
   TAnnotFlags Flags;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags  = Z_LVAL_PP(arg1);

   rc = pdfSetAnnotFlags(pdf, Flags);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotFlagsEx(const void* IPDF, UI32 Handle, SI32 Flags);
PHP_METHOD(dynapdf, SetAnnotFlagsEx)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Flags  = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotFlagsEx(pdf, Handle, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotHighlightMode(const void* IPDF, UI32 Handle, THighlightMode Mode);
PHP_METHOD(dynapdf, SetAnnotHighlightMode)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Mode;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Mode   = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotHighlightMode(pdf, Handle, Mode);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotIcon(const void* IPDF, UI32 Handle, TAnnotIcon Icon);
PHP_METHOD(dynapdf, SetAnnotIcon)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Icon;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Icon   = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotIcon(pdf, Handle, Icon);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotLineDashPattern(const PPDF* IPDF, UI32 Handle, const float* Dash, UI32 NumValues);
PHP_METHOD(dynapdf, SetAnnotLineDashPattern)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr_hash;
   float Dash[64];
   UI32 Handle;
   SI32 rc, NumValues;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   Handle    = Z_LVAL_PP(argv[0]);
   arr_hash  = Z_ARRVAL_PP(argv[1]);
   NumValues = GetFloatArray(arr_hash, Dash, 64);

   rc = pdfSetAnnotLineDashPattern(pdf, Handle, Dash, NumValues);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotLineEndStyle(const void* IPDF, UI32 Handle, TLineEndStyle Start, TLineEndStyle End);
PHP_METHOD(dynapdf, SetAnnotLineEndStyle)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 rc, Handle, Start, End;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Handle = Z_LVAL_PP(argv[0]);
   Start  = Z_LVAL_PP(argv[1]);
   End    = Z_LVAL_PP(argv[2]);

   rc = pdfSetAnnotLineEndStyle(pdf, Handle, Start, End);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfSetAnnotMigrationStateA(const PPDF* IPDF, UI32 Annot, TAnnotState State, const char* User);
PHP_METHOD(dynapdf, SetAnnotMigrationState)
{
   zval **argv[3];
   PPDF* pdf;
   char* User;
   SI32 rc, Annot, State;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   Annot = Z_LVAL_PP(argv[0]);
   State = Z_LVAL_PP(argv[1]);
   User  = Z_STRVAL_PP(argv[2]);

   rc = pdfSetAnnotMigrationStateA(pdf, Annot, State, User);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotOpacity(const PPDF* IPDF, UI32 Handle, double Value);
PHP_METHOD(dynapdf, SetAnnotOpacity)
{
   zval **argv[2];
   PPDF* pdf;
   double Value;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Value  = Z_DVAL_PP(argv[1]);

   rc = pdfSetAnnotOpacity(pdf, Handle, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotOpenState(const void* IPDF, UI32 Handle, LBOOL Open);
PHP_METHOD(dynapdf, SetAnnotOpenState)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Open;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Open   = Z_LVAL_PP(argv[1]);

   rc = pdfSetAnnotOpenState(pdf, Handle, Open);
   RETURN_BOOL(rc);
}

// PDF_EXP LBOOL  PDF_CALL pdfSetAnnotOrFieldDate(const void* IPDF, UI32 Handle, LBOOL IsField, TDateType Type, UI32 DateTime);
PHP_METHOD(dynapdf, SetAnnotOrFieldDate)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, Handle, IsField, Type, DateTime;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Handle   = Z_LVAL_PP(argv[0]);
   IsField  = Z_LVAL_PP(argv[1]);
   Type     = Z_LVAL_PP(argv[2]);
   DateTime = Z_LVAL_PP(argv[3]);

   rc = pdfSetAnnotOrFieldDate(pdf, Handle, IsField, Type, DateTime);
   RETURN_BOOL(rc);
}

// PDF_EXP LBOOL  PDF_CALL pdfSetAnnotQuadPoints(const PPDF* IPDF, UI32 Handle, struct TFltPoint* Value, UI32 Count);
PHP_METHOD(dynapdf, SetAnnotQuadPoints)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr_hash;
   struct TFltPoint* Value = NULL;
   SI32 rc, Handle, Count;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   Handle   = Z_LVAL_PP(argv[0]);
   arr_hash = Z_ARRVAL_PP(argv[1]);
   Count    = zend_hash_num_elements(arr_hash) >> 1;

   if (Count > 0)
   {
      Value = emalloc(Count * sizeof(struct TFltPoint));
      Count = GetFloatArray(arr_hash, (float*)Value, Count << 1) >> 1;
   }
   rc = pdfSetAnnotQuadPoints(pdf, Handle, Value, Count);

   efree(Value);

   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotStringA(const void* IPDF, UI32 Handle, TAnnotString StringType, const char* Value);
PHP_METHOD(dynapdf, SetAnnotString)
{
   zval **argv[3];
   PPDF* pdf;
   char* Value;
   SI32 rc, Handle, StringType;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   Handle     = Z_LVAL_PP(argv[0]);
   StringType = Z_LVAL_PP(argv[1]);
   Value      = Z_STRVAL_PP(argv[2]);

   rc = pdfSetAnnotStringA(pdf, Handle, StringType, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetAnnotSubjectA(const void* IPDF, UI32 Handle, const char* Value);
PHP_METHOD(dynapdf, SetAnnotSubject)
{
   zval **argv[2];
   PPDF* pdf;
   char* Value;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Value  = Z_STRVAL_PP(argv[1]);

   rc = pdfSetAnnotSubjectA(pdf, Handle, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetBidiMode(const void* IPDF, TPDFBidiMode Mode);
PHP_METHOD(dynapdf, SetBidiMode)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Mode;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Mode = Z_LVAL_PP(arg1);

   rc = pdfSetBidiMode(pdf, Mode);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetBBox()
   const void* IPDF, TPageBoundary Boundary, double LeftX, double LeftY, double RightX, double RightY); */
PHP_METHOD(dynapdf, SetBBox)
{
   zval **argv[5];
   PPDF* pdf;
   TPageBoundary Boundary;
   double LeftX, LeftY, RightX, RightY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   Boundary = Z_LVAL_PP(argv[0]);
   LeftX    = Z_DVAL_PP(argv[1]);
   LeftY    = Z_DVAL_PP(argv[2]);
   RightX   = Z_DVAL_PP(argv[3]);
   RightY   = Z_DVAL_PP(argv[4]);

   rc = pdfSetBBox(pdf, Boundary, LeftX, LeftY, RightX, RightY);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetBookmarkDest()
   const void* IPDF, SI32 ABmk, TDestType DestType, double a, double b, double c, double d); */
PHP_METHOD(dynapdf, SetBookmarkDest)
{
   zval **argv[6];
   PPDF* pdf;
   SI32 ABmk, DestType;
   double a, b, c, d;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   ABmk = Z_LVAL_PP(argv[0]);
   DestType = Z_LVAL_PP(argv[1]);
   a        = Z_DVAL_PP(argv[2]);
   b        = Z_DVAL_PP(argv[3]);
   c        = Z_DVAL_PP(argv[4]);
   d        = Z_DVAL_PP(argv[5]);

   rc = pdfSetBookmarkDest(pdf, ABmk, DestType, a, b, c, d);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetBookmarkStyle()
   const void* IPDF, SI32 ABmk, TFStyle Style, UI32 RGBColor); */
PHP_METHOD(dynapdf, SetBookmarkStyle)
{
   zval **argv[3];

   PPDF* pdf;
   SI32 ABmk, Style, RGBColor;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   ABmk     = Z_LVAL_PP(argv[0]);
   Style    = Z_LVAL_PP(argv[1]);
   RGBColor = Z_LVAL_PP(argv[2]);

   rc = pdfSetBookmarkStyle(pdf, ABmk, Style, RGBColor);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetBorderStyle()
   const void* IPDF, TBorderStyle Style); */
PHP_METHOD(dynapdf, SetBorderStyle)
{
   zval **arg1;
   PPDF* pdf;
   TBorderStyle Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Style  = Z_LVAL_PP(arg1);

   rc = pdfSetBorderStyle(pdf, Style);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetCharacterSpacing()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetCharacterSpacing)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetCharacterSpacing(pdf, Value);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetCheckBoxChar()
   const void* IPDF, TCheckBoxChar CheckBoxChar); */
PHP_METHOD(dynapdf, SetCheckBoxChar)
{
   zval **arg1;
   PPDF* pdf;
   TCheckBoxChar CheckBoxChar;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   CheckBoxChar  = Z_LVAL_PP(arg1);

   rc = pdfSetCheckBoxChar(pdf, CheckBoxChar);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetCheckBoxDefState(const void* IPDF, UI32 AField, LBOOL Checked);
PHP_METHOD(dynapdf, SetCheckBoxDefState)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Checked;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField  = Z_LVAL_PP(argv[0]);
   Checked = Z_LVAL_PP(argv[1]);

   rc = pdfSetCheckBoxDefState(pdf, AField, Checked);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetCheckBoxState()
   const void* IPDF, UI32 AField, LBOOL Checked); */
PHP_METHOD(dynapdf, SetCheckBoxState)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Checked;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField  = Z_LVAL_PP(argv[0]);
   Checked = Z_LVAL_PP(argv[1]);

   rc = pdfSetCheckBoxState(pdf, AField, Checked);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfSetCIDFontA(const void* IPDF, UI32 CMapHandle, const char* Name, TFStyle Style, double Size, LBOOL Embed);
PHP_METHOD(dynapdf, SetCIDFont)
{
   zval **argv[5];
   PPDF* pdf;
   char* Name;
   double Size;
   SI32 rc, CMapHandle, Style, Embed;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   CMapHandle = Z_LVAL_PP(argv[0]);
   Name       = Z_STRVAL_PP(argv[1]);
   Style      = Z_LVAL_PP(argv[2]);
   Size       = Z_DVAL_PP(argv[3]);
   Embed      = Z_LVAL_PP(argv[4]);

   rc = pdfSetCIDFontA(pdf, CMapHandle, Name, Style, Size, Embed);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfSetCMapDirA(const void* IPDF, const char* Path, TLoadCMapFlags Flags);
PHP_METHOD(dynapdf, SetCMapDir)
{
   zval **argv[2];
   PPDF* pdf;
   char* Path;
   SI32 rc, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Path  = Z_STRVAL_PP(argv[0]);
   Flags = Z_LVAL_PP(argv[1]);

   rc = pdfSetCMapDirA(pdf, Path, Flags);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetColDefFile(const void* IPDF, UI32 EmbFile);
PHP_METHOD(dynapdf, SetColDefFile)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, EmbFile;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   EmbFile = Z_LVAL_PP(arg1);

   rc = pdfSetColDefFile(pdf, EmbFile);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetColorMask(const void* IPDF, UI32 ImageHandle, SI32* Mask, UI32 Count);
PHP_METHOD(dynapdf, SetColorMask)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Mask[64];
   HashTable *array;
   SI32 rc, ImageHandle, Count = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);

   ImageHandle = Z_LVAL_PP(argv[0]);
   if (Z_TYPE_PP(argv[1]) != IS_NULL)
   {
      convert_to_array_ex(argv[1]);
      array = Z_ARRVAL_PP(argv[1]);
      Count = GetUI32Array(array, (UI32*)Mask, 64);
   }
   rc = pdfSetColorMask(pdf, ImageHandle, Mask, Count);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetColors()
   const void* IPDF, UI32 Color); */
PHP_METHOD(dynapdf, SetColors)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Color  = Z_LVAL_PP(arg1);

   rc = pdfSetColors(pdf, Color);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetColorSpace()
   const void* IPDF, TPDFColorSpace ColorSpace); */
PHP_METHOD(dynapdf, SetColorSpace)
{
   zval **arg1;
   PPDF* pdf;
   TPDFColorSpace ColorSpace;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   ColorSpace = Z_LVAL_PP(arg1);

   rc = pdfSetColorSpace(pdf, ColorSpace);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetColSortField(const void* IPDF, UI32 ColField, LBOOL AscendingOrder);
PHP_METHOD(dynapdf, SetColSortField)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, ColField, AscendingOrder;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   ColField       = Z_LVAL_PP(argv[0]);
   AscendingOrder = Z_LVAL_PP(argv[1]);

   rc = pdfSetColSortField(pdf, ColField, AscendingOrder);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetCompressionFilter()
   const void* IPDF, TCompressionFilter Filter); */
PHP_METHOD(dynapdf, SetCompressionFilter)
{
   zval **arg1;
   PPDF* pdf;
   TCompressionFilter Filter;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Filter  = Z_LVAL_PP(arg1);

   rc = pdfSetCompressionFilter(pdf, Filter);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetCompressionLevel()
   const void* IPDF, TCompressionLevel CompressLevel); */
PHP_METHOD(dynapdf, SetCompressionLevel)
{
   zval **arg1;
   PPDF* pdf;
   TCompressionLevel CompressLevel;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   CompressLevel = Z_LVAL_PP(arg1);

   rc = pdfSetCompressionLevel(pdf, CompressLevel);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetContent()
   const void* IPDF, const char* Buffer, UI32 BufSize); */
PHP_METHOD(dynapdf, SetContent)
{
   zval **arg1;
   PPDF* pdf;
   char* Buffer;
   SI32 BufSize;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Buffer  = Z_STRVAL_PP(arg1);
   BufSize = Z_STRLEN_PP(arg1);

   rc = pdfSetContent(pdf, Buffer, BufSize);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetDateTimeFormat()
   const void* IPDF, UI32 TxtField, TPDFDateTime Fmt); */
PHP_METHOD(dynapdf, SetDateTimeFormat)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 TxtField;
   TPDFDateTime Fmt;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   TxtField = Z_LVAL_PP(argv[0]);
   Fmt      = Z_LVAL_PP(argv[1]);

   rc = pdfSetDateTimeFormat(pdf, TxtField, Fmt);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetDefBitsPerPixel()
   const void* IPDF, SI32 Value); */
PHP_METHOD(dynapdf, SetDefBitsPerPixel)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetDefBitsPerPixel(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetDocInfo()
   const void* IPDF, TDocumentInfo DInfo, const char* Value); */
PHP_METHOD(dynapdf, SetDocInfo)
{
   zval **argv[2];
   PPDF* pdf;
   TDocumentInfo DInfo;
   char* Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   DInfo = Z_LVAL_PP(argv[0]);
   Value = Z_STRVAL_PP(argv[1]);

   rc = pdfSetDocInfoA(pdf, DInfo, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetDocInfoExA(const void* IPDF, TDocumentInfo DInfo, const char* Key, const char* Value);
PHP_METHOD(dynapdf, SetDocInfoEx)
{
   zval **argv[3];
   PPDF* pdf;
   TDocumentInfo DInfo;
   char* Key, *Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   DInfo = Z_LVAL_PP(argv[0]);
   Key   = Z_STRVAL_PP(argv[1]);
   Value = Z_STRVAL_PP(argv[2]);

   rc = pdfSetDocInfoExA(pdf, DInfo, Key, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetDrawDirection()
   const void* IPDF, TDrawDirection Direction); */
PHP_METHOD(dynapdf, SetDrawDirection)
{
   zval **arg1;
   PPDF* pdf;
   TDrawDirection Direction;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Direction = Z_LVAL_PP(arg1);

   rc = pdfSetDrawDirection(pdf, Direction);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetEMFFrameDPI(const void* IPDF, UI32 DPIX, UI32 DPIY);
PHP_METHOD(dynapdf, SetEMFFrameDPI)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, DPIX, DPIY;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   DPIX = Z_LVAL_PP(argv[0]);
   DPIY = Z_LVAL_PP(argv[1]);

   rc = pdfSetEMFFrameDPI(pdf, DPIX, DPIY);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetEMFPatternDistance()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetEMFPatternDistance)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Value   = Z_DVAL_PP(arg1);

   rc = pdfSetEMFPatternDistance(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetErrorMode()
   const void* IPDF, TErrMode ErrMode); */
PHP_METHOD(dynapdf, SetErrorMode)
{
   zval **arg1;
   PPDF* pdf;
   TErrMode ErrMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   ErrMode  = Z_LVAL_PP(arg1);

   rc = pdfSetErrorMode(pdf, ErrMode);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetExtColorSpace(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, SetExtColorSpace)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfSetExtColorSpace(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetExtFillColorSpace(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, SetExtFillColorSpace)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfSetExtFillColorSpace(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetExtGState(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, SetExtGState)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfSetExtGState(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetExtStrokeColorSpace(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, SetExtStrokeColorSpace)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfSetExtStrokeColorSpace(pdf, Handle);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldBackColor()
   const void* IPDF, UI32 AColor); */
PHP_METHOD(dynapdf, SetFieldBackColor)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AColor;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AColor  = Z_LVAL_PP(arg1);

   rc = pdfSetFieldBackColor(pdf, AColor);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldBBox(const void* IPDF, UI32 AField, struct TPDFRect* BBox);
PHP_METHOD(dynapdf, SetFieldBBox)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *array;
   struct TPDFRect BBox = {0.0, 0.0, 0.0, 0.0};
   SI32 rc, AField;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   AField = Z_LVAL_PP(argv[0]);
   array  = Z_ARRVAL_PP(argv[1]);

   if (GetBBoxFromArray(array, &BBox))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"BBox\" must contain values for Left, Bottom, Right, Top!");
      RETURN_BOOL(0);
   }

   rc = pdfSetFieldBBox(pdf, AField, &BBox);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldBorderColor()
   const void* IPDF, UI32 AColor); */
PHP_METHOD(dynapdf, SetFieldBorderColor)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AColor;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AColor  = Z_LVAL_PP(arg1);

   rc = pdfSetFieldBorderColor(pdf, AColor);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetFieldBorderStyle()
   const void* IPDF, UI32 AField, TBorderStyle Style); */
PHP_METHOD(dynapdf, SetFieldBorderStyle)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField;
   TBorderStyle Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField = Z_LVAL_PP(argv[0]);
   Style  = Z_LVAL_PP(argv[1]);

   rc = pdfSetFieldBorderStyle(pdf, AField, Style);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldBorderWidth(const void* IPDF, UI32 AField, double LineWidth);
PHP_METHOD(dynapdf, SetFieldBorderWidth)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField;
   double LineWidth;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   AField    = Z_LVAL_PP(argv[0]);
   LineWidth = Z_DVAL_PP(argv[1]);

   rc = pdfSetFieldBorderWidth(pdf, AField, LineWidth);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldCalcOrder(const PPDF* IPDF, SI32 CurrIndex, UI32 NewIndex);
PHP_METHOD(dynapdf, SetFieldCalcOrder)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 CurrIndex, NewIndex;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   CurrIndex = Z_LVAL_PP(argv[0]);
   NewIndex  = Z_LVAL_PP(argv[1]);

   rc = pdfSetFieldCalcOrder(pdf, CurrIndex, NewIndex);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldColor()
   const void* IPDF, UI32 AField, TFieldColor ColorType, TPDFColorSpace CS, UI32 Color); */
PHP_METHOD(dynapdf, SetFieldColor)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 AField, ColorType, CS, Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   AField    = Z_LVAL_PP(argv[0]);
   ColorType = Z_LVAL_PP(argv[1]);
   CS        = Z_LVAL_PP(argv[2]);
   Color     = Z_LVAL_PP(argv[3]);

   rc = pdfSetFieldColor(pdf, AField, ColorType, CS, Color);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldExpValue(const void* IPDF, UI32 AField, UI32 ValIndex, const char* Value, const char* ExpValue, LBOOL Selected);
PHP_METHOD(dynapdf, SetFieldExpValue)
{
   zval **argv[5];
   PPDF* pdf;
   char* Value, *ExpValue;
   SI32 rc, AField, ValIndex, Selected;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_string_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   AField   = Z_LVAL_PP(argv[0]);
   ValIndex = Z_LVAL_PP(argv[1]);
   Value    = Z_STRVAL_PP(argv[2]);
   ExpValue = Z_STRVAL_PP(argv[3]);
   Selected = Z_LVAL_PP(argv[4]);

   rc = pdfSetFieldExpValue(pdf, AField, ValIndex, Value, ExpValue, Selected);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldExpValueEx(const void* IPDF, UI32 AField, UI32 ValIndex, LBOOL Selected, LBOOL DefSelected);
PHP_METHOD(dynapdf, SetFieldExpValueEx)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 rc, AField, ValIndex, Selected, DefSelected;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   AField      = Z_LVAL_PP(argv[0]);
   ValIndex    = Z_LVAL_PP(argv[1]);
   Selected    = Z_LVAL_PP(argv[2]);
   DefSelected = Z_LVAL_PP(argv[3]);

   rc = pdfSetFieldExpValueEx(pdf, AField, ValIndex, Selected, DefSelected);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldFlags()
   const void* IPDF, UI32 AField, TFieldFlags Flags, LBOOL Reset); */
PHP_METHOD(dynapdf, SetFieldFlags)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 AField, Flags, Reset;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   AField = Z_LVAL_PP(argv[0]);
   Flags  = Z_LVAL_PP(argv[1]);
   Reset  = Z_LVAL_PP(argv[2]);

   rc = pdfSetFieldFlags(pdf, AField, Flags, Reset);
   RETURN_BOOL(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfSetFieldFontA(const PPDF* IPDF, UI32 Field, const char* Name, TFStyle Style, double Size, LBOOL Embed, TCodepage CP)
PHP_METHOD(dynapdf, SetFieldFont)
{
   zval **argv[6];
   PPDF* pdf;
   UI32 Field;
   char *Name;
   TFStyle Style;
   double Size;
   SI32 Embed;
   TCodepage CP;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   Field = Z_LVAL_PP(argv[0]);
   Name  = Z_STRVAL_PP(argv[1]);
   Style = Z_LVAL_PP(argv[2]);
   Size  = Z_DVAL_PP(argv[3]);
   Embed = Z_LVAL_PP(argv[4]);
   CP    = Z_LVAL_PP(argv[5]);

   rc = pdfSetFieldFontA(pdf, Field, Name, Style, Size, Embed, CP);
   RETURN_LONG(rc);
}

// PDF_EXP LBOOL  PDF_CALL pdfSetFieldFontEx(const PPDF* IPDF, UI32 Field, UI32 Handle, double FontSize);
PHP_METHOD(dynapdf, SetFieldFontEx)
{
   zval **argv[3];
   PPDF* pdf;
   UI32 Field, Handle;
   double Size;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);

   Field  = Z_LVAL_PP(argv[0]);
   Handle = Z_LVAL_PP(argv[1]);
   Size   = Z_DVAL_PP(argv[2]);

   rc = pdfSetFieldFontEx(pdf, Field, Handle, Size);
   RETURN_BOOL(rc);
}


//PDF_EXP LBOOL  PDF_CALL pdfSetFieldFontSize(const void* IPDF, UI32 AField, double FontSize);
PHP_METHOD(dynapdf, SetFieldFontSize)
{
   zval **argv[2];
   PPDF* pdf;
   double FontSize;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   AField   = Z_LVAL_PP(argv[0]);
   FontSize = Z_DVAL_PP(argv[1]);

   rc = pdfSetFieldFontSize(pdf, AField, FontSize);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldHighlightMode(const void* IPDF, UI32 AField, THighlightMode Mode);
PHP_METHOD(dynapdf, SetFieldHighlightMode)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Mode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField = Z_LVAL_PP(argv[0]);
   Mode   = Z_LVAL_PP(argv[1]);

   rc = pdfSetFieldHighlightMode(pdf, AField, Mode);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldIndex()
   const void* IPDF, UI32 AField, UI32 Index); */
PHP_METHOD(dynapdf, SetFieldIndex)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Index;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField = Z_LVAL_PP(argv[0]);
   Index  = Z_LVAL_PP(argv[1]);

   rc = pdfSetFieldIndex(pdf, AField, Index);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldMapNameA(const void* IPDF, UI32 AField, const char* Name);
PHP_METHOD(dynapdf, SetFieldMapName)
{
   zval **argv[2];
   PPDF* pdf;
   char* Name;
   SI32 AField;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   AField = Z_LVAL_PP(argv[0]);
   Name   = Z_STRVAL_PP(argv[1]);

   rc = pdfSetFieldMapNameA(pdf, AField, Name);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldName()
   const void* IPDF, UI32 AField, const char* NewName); */
PHP_METHOD(dynapdf, SetFieldName)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField;
   char *NewName;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   AField  = Z_LVAL_PP(argv[0]);
   NewName = Z_STRVAL_PP(argv[1]);

   rc = pdfSetFieldName(pdf, AField, NewName);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetFieldOrientation(const void* IPDF, UI32 AField, SI32 Value);
PHP_METHOD(dynapdf, SetFieldOrientation)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField = Z_LVAL_PP(argv[0]);
   Value  = Z_LVAL_PP(argv[1]);

   rc = pdfSetFieldOrientation(pdf, AField, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFieldTextAlign()
   const void* IPDF, UI32 AField, TTextAlign Align); */
PHP_METHOD(dynapdf, SetFieldTextAlign)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField, Align;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   AField = Z_LVAL_PP(argv[0]);
   Align  = Z_LVAL_PP(argv[1]);

   rc = pdfSetFieldTextAlign(pdf, AField, Align);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetFieldTextColor()
   const void* IPDF, UI32 Color); */
PHP_METHOD(dynapdf, SetFieldTextColor)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Color  = Z_LVAL_PP(arg1);

   rc = pdfSetFieldTextColor(pdf, Color);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetFieldToolTip()
   const void* IPDF, UI32 AField, const char* Value); */
PHP_METHOD(dynapdf, SetFieldToolTip)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 AField;
   char *Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   AField  = Z_LVAL_PP(argv[0]);
   Value   = Z_STRVAL_PP(argv[1]);

   rc = pdfSetFieldToolTipA(pdf, AField, Value);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfSetFillColor()
   const void* IPDF, UI32 Color); */
PHP_METHOD(dynapdf, SetFillColor)
{
   zval **arg1;
   PPDF* pdf;
   UI32 Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Color  = Z_LVAL_PP(arg1);
   rc = pdfSetFillColor(pdf, Color);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetFillColorEx(const void* IPDF, const BYTE* Color, UI32 NumComponents);
PHP_METHOD(dynapdf, SetFillColorEx)
{
   zval **arg1;
   PPDF* pdf;
   BYTE Color[32];
   HashTable *arr_hash;
   SI32 rc, NumComponents;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(arg1);
   arr_hash = Z_ARRVAL_PP(arg1);

   NumComponents = GetBYTEArray(arr_hash, Color, 32);

   rc = pdfSetFillColorEx(pdf, Color, NumComponents);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetFillColorF(const void* IPDF, const float* Color, UI32 NumComponents);
PHP_METHOD(dynapdf, SetFillColorF)
{
   zval **arg1;
   PPDF* pdf;
   float Color[32];
   HashTable *arr_hash;
   SI32 rc, NumComponents;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(arg1);
   arr_hash = Z_ARRVAL_PP(arg1);

   NumComponents = GetFloatArray(arr_hash, Color, 32);

   rc = pdfSetFillColorF(pdf, Color, NumComponents);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetFillColorSpace(const void* IPDF, TPDFColorSpace CS);
PHP_METHOD(dynapdf, SetFillColorSpace)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   CS = Z_LVAL_PP(arg1);
   rc = pdfSetFillColorSpace(pdf, CS);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetFloatPrecision(const PPDF* IPDF, UI32 NumTextDecDigits, UI32 NumVectDecDigits);
PHP_METHOD(dynapdf, SetFloatPrecision)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 NumTextDecDigits, NumVectDecDigits;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   NumTextDecDigits = Z_LVAL_PP(argv[0]);
   NumVectDecDigits = Z_LVAL_PP(argv[1]);

   rc = pdfSetFloatPrecision(pdf, NumTextDecDigits, NumVectDecDigits);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetFont()
   const void* IPDF, const char* Name, TFStyle Style, double Size, LBOOL Embed, TCodepage CP); */
PHP_METHOD(dynapdf, SetFont)
{
   zval **argv[5];
   PPDF* pdf;
   char *Name;
   TFStyle Style;
   double Size;
   SI32 Embed;
   TCodepage CP;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   Name  = Z_STRVAL_PP(argv[0]);
   Style = Z_LVAL_PP(argv[1]);
   Size  = Z_DVAL_PP(argv[2]);
   Embed = Z_LVAL_PP(argv[3]);
   CP    = Z_LVAL_PP(argv[4]);

   rc = pdfSetFontA(pdf, Name, Style, Size, Embed, CP);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP SI32   PDF_CALL pdfSetFontExA(const void* IPDF, const char* Name, TFStyle Style, double Size, LBOOL Embed, TCodepage CP);
PHP_METHOD(dynapdf, SetFontEx)
{
   zval **argv[5];
   PPDF* pdf;
   char *Name;
   TFStyle Style;
   double Size;
   SI32 Embed;
   TCodepage CP;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   Name  = Z_STRVAL_PP(argv[0]);
   Style = Z_LVAL_PP(argv[1]);
   Size  = Z_DVAL_PP(argv[2]);
   Embed = Z_LVAL_PP(argv[3]);
   CP    = Z_LVAL_PP(argv[4]);

   rc = pdfSetFontExA(pdf, Name, Style, Size, Embed, CP);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfSetFontOrigin()
   const void* IPDF, TOrigin Origin); */
PHP_METHOD(dynapdf, SetFontOrigin)
{
   zval **arg1;
   PPDF* pdf;
   TOrigin Origin;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Origin = Z_LVAL_PP(arg1);

   rc = pdfSetFontOrigin(pdf, Origin);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfSetFontSearchOrder(const void* IPDF, TFontBaseType Order[4]);
PHP_METHOD(dynapdf, SetFontSearchOrder)
{
   zval **arg1;
   PPDF* pdf;
   HashTable *arr_hash;
   TFontBaseType Order[4] = {fbtDisabled, fbtDisabled, fbtDisabled, fbtDisabled};
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(arg1);

   arr_hash = Z_ARRVAL_PP(arg1);

   if (GetFontSearchOrder(arr_hash, Order) < 1)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Order must be an array with at least one value!");
      RETURN_NULL();
   }

   pdfSetFontSearchOrder(pdf, Order);
}

//PDF_EXP void   PDF_CALL pdfSetFontSearchOrderEx(const void* IPDF, TFontBaseType S1, TFontBaseType S2, TFontBaseType S3, TFontBaseType S4);
PHP_METHOD(dynapdf, SetFontSearchOrderEx)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 S1, S2, S3, S4;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   S1 = Z_LVAL_PP(argv[0]);
   S2 = Z_LVAL_PP(argv[1]);
   S3 = Z_LVAL_PP(argv[2]);
   S4 = Z_LVAL_PP(argv[3]);

   pdfSetFontSearchOrderEx(pdf, S1, S2, S3, S4);
}

/* {{{ proto SI32 pdfSetFontSelMode()
   const void* IPDF, TFontSelMode Mode); */
PHP_METHOD(dynapdf, SetFontSelMode)
{
   zval **arg1;
   PPDF* pdf;
   TFontSelMode Mode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Mode = Z_LVAL_PP(arg1);

   rc = pdfSetFontSelMode(pdf, Mode);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfSetFontWeight()
   const void* IPDF, SI32 Weight); */
PHP_METHOD(dynapdf, SetFontWeight)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Weight;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Weight  = Z_LVAL_PP(arg1);

   rc = pdfSetFontWeight(pdf, Weight);
   RETURN_LONG(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfSetGStateFlags(const void* IPDF, TGStateFlags Flags, LBOOL Reset);
PHP_METHOD(dynapdf, SetGStateFlags)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Flags, Reset;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Flags = Z_LVAL_PP(argv[0]);
   Reset = Z_LVAL_PP(argv[1]);

   // This flag is set via Enable or DisableUTF8Support()!
   Flags &= ~gfAnsiStringIsUTF8;

   pdfSetGStateFlags(pdf, Flags, Reset);
}

/* {{{ proto SI32 pdfSetIconColor()
   const void* IPDF, UI32 Color); */
PHP_METHOD(dynapdf, SetIconColor)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Color  = Z_LVAL_PP(arg1);

   rc = pdfSetIconColor(pdf, Color);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetImportFlags()
   const void* IPDF, TImportFlags Flags); */
PHP_METHOD(dynapdf, SetImportFlags)
{
   zval **arg1;
   PPDF* pdf;
   TImportFlags Flags;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags = Z_LVAL_PP(arg1);

   rc = pdfSetImportFlags(pdf, Flags);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetImportFlags2(const void* IPDF, TImportFlags2 Flags);
PHP_METHOD(dynapdf, SetImportFlags2)
{
   zval **arg1;
   PPDF* pdf;
   TImportFlags Flags;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags = Z_LVAL_PP(arg1);

   rc = pdfSetImportFlags2(pdf, Flags);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetItalicAngle()
   const void* IPDF, double Angle); */
PHP_METHOD(dynapdf, SetItalicAngle)
{
   zval **arg1;
   PPDF* pdf;
   double Angle;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Angle  = Z_DVAL_PP(arg1);

   rc = pdfSetItalicAngle(pdf, Angle);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetJPEGQuality()
   const void* IPDF, UI32 Value); */
PHP_METHOD(dynapdf, SetJPEGQuality)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetJPEGQuality(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetLanguage()
   const void* IPDF, const char* ISOTag); */
PHP_METHOD(dynapdf, SetLanguage)
{
   zval **arg1;
   PPDF* pdf;
   char *ISOTag;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   ISOTag = Z_STRVAL_PP(arg1);

   rc = pdfSetLanguage(pdf, ISOTag);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetLeading()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetLeading)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetLeading(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetLicenseKey()
   const void* IPDF, const char* Value); */
PHP_METHOD(dynapdf, SetLicenseKey)
{
   zval **arg1;
   PPDF* pdf;
   char *Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(arg1);
   Value = Z_STRVAL_PP(arg1);

   rc = pdfSetLicenseKey(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetLineAnnotParms(const void* IPDF, UI32 Handle, SI32 FontHandle, double FontSize, struct TLineAnnotParms* Parms);
PHP_METHOD(dynapdf, SetLineAnnotParms)
{
   zval **argv[4];
   PPDF* pdf;
   double FontSize;
   SI32 rc, Handle, FontHandle;
   struct TLineAnnotParms Parms, *p = NULL;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);

   Handle     = Z_LVAL_PP(argv[0]);
   FontHandle = Z_LVAL_PP(argv[1]);
   FontSize   = Z_DVAL_PP(argv[2]);

   if (Z_TYPE_PP(argv[3]) != IS_NULL)
   {
      HashTable *arr_hash;
      convert_to_array_ex(argv[3]);
      arr_hash = Z_ARRVAL_PP(argv[3]);
      memset(&Parms, 0, sizeof(Parms));
      GetLineAnnotParms(arr_hash, &Parms);
      p = &Parms;
   }
   rc = pdfSetLineAnnotParms(pdf, Handle, FontHandle, FontSize, p);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetLineCapStyle()
   const void* IPDF, TLineCapStyle Style); */
PHP_METHOD(dynapdf, SetLineCapStyle)
{
   zval **arg1;
   PPDF* pdf;
   TLineCapStyle Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Style  = Z_LVAL_PP(arg1);

   rc = pdfSetLineCapStyle(pdf, Style);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetLineDashPattern()
   const void* IPDF, const char* Dash, SI32 Phase); */
PHP_METHOD(dynapdf, SetLineDashPattern)
{
   zval **argv[2];
   PPDF* pdf;
   char *Dash;
   SI32 Phase;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Dash  = Z_STRVAL_PP(argv[0]);
   Phase = Z_LVAL_PP(argv[1]);

   if(Dash && !strlen(Dash))
     Dash = NULL;
   rc = pdfSetLineDashPattern(pdf, Dash, Phase);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetLineDashPattern2(const PPDF* IPDF, const float* Dash, UI32 NumValues, float Phase);
PHP_METHOD(dynapdf, SetLineDashPattern2)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr_hash;
   float Dash[64];
   float Phase;
   SI32 rc, NumValues;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   arr_hash  = Z_ARRVAL_PP(argv[0]);
   Phase     = (float)Z_DVAL_PP(argv[1]);
   NumValues = GetFloatArray(arr_hash, Dash, 64);

   rc = pdfSetLineDashPattern2(pdf, Dash, NumValues, Phase);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetLineDashPatternEx(const void* IPDF, const double* Dash, UI32 NumValues, SI32 Phase);
PHP_METHOD(dynapdf, SetLineDashPatternEx)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *arr_hash;
   double Dash[64];
   SI32 Phase;
   SI32 rc, NumValues;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   arr_hash  = Z_ARRVAL_PP(argv[0]);
   Phase     = Z_LVAL_PP(argv[1]);
   NumValues = GetDoubleArray(arr_hash, Dash, 64);

   rc = pdfSetLineDashPatternEx(pdf, Dash, NumValues, Phase);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetLineJoinStyle()
   const void* IPDF, TLineJoinStyle Style); */
PHP_METHOD(dynapdf, SetLineJoinStyle)
{
   zval **arg1;
   PPDF* pdf;
   TLineJoinStyle Style;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Style  = Z_LVAL_PP(arg1);

   rc = pdfSetLineJoinStyle(pdf, Style);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetLineWidth()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetLineWidth)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetLineWidth(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetLinkHighlightMode()
   const void* IPDF, THighlightMode Mode); */
PHP_METHOD(dynapdf, SetLinkHighlightMode)
{
   zval **arg1;
   PPDF* pdf;
   THighlightMode Mode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Mode  = Z_LVAL_PP(arg1);

   rc = pdfSetLinkHighlightMode(pdf, Mode);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetListFont(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, SetListFont)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfSetListFont(pdf, Handle);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetMatrix()
   const void* IPDF, struct TCTM* Matrix); */
PHP_METHOD(dynapdf, SetMatrix)
{
   zval **arg1;
   HashTable *arr_hash;
   PPDF* pdf;
   SI32 rc;
   struct TCTM Matrix = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(arg1);

   arr_hash = Z_ARRVAL_PP(arg1);

   if (GetCTMFromArray(arr_hash, &Matrix))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"Matrix\" must be an array of six numbers (a, b, c, d, x, y)!");
      RETURN_LONG(-1);
   }
   rc = pdfSetMatrix(pdf, &Matrix);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfSetMaxErrLogMsgCount(const void* IPDF, UI32 Value);
PHP_METHOD(dynapdf, SetMaxErrLogMsgCount)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value = Z_LVAL_PP(arg1);

   pdfSetMaxErrLogMsgCount(pdf, Value);
}

/* {{{ proto SI32 pdfSetMaxFieldLen()
   const void* IPDF, UI32 TxtField, SI32 MaxLen); */
PHP_METHOD(dynapdf, SetMaxFieldLen)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 TxtField, MaxLen;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   TxtField  = Z_LVAL_PP(argv[0]);
   MaxLen    = Z_LVAL_PP(argv[1]);

   rc = pdfSetMaxFieldLen(pdf, TxtField, MaxLen);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetMetaConvFlags()
   const void* IPDF, TMetaFlags Flags); */
PHP_METHOD(dynapdf, SetMetaConvFlags)
{
   zval **arg1;
   PPDF* pdf;
   TMetaFlags Flags;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Flags  = Z_LVAL_PP(arg1);

   rc = pdfSetMetaConvFlags(pdf, Flags);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetMetadata(const PPDF* IPDF, TMetadataObj ObjType, SI32 Handle, const void* Buffer, UI32 BufSize);
PHP_METHOD(dynapdf, SetMetadata)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 rc, ObjType, Handle, BufSize;
   char *Buffer;

   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   ObjType = Z_LVAL_PP(argv[0]);
   Handle  = Z_LVAL_PP(argv[1]);
   Buffer  = Z_STRVAL_PP(argv[2]);
   BufSize = Z_STRLEN_PP(argv[2]);

   rc = pdfSetMetadata(pdf, ObjType, Handle, Buffer, BufSize);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetMiterLimit()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetMiterLimit)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value = Z_DVAL_PP(arg1);

   rc = pdfSetMiterLimit(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetNeedAppearance(const void* IPDF, LBOOL Value);
PHP_METHOD(dynapdf, SetNeedAppearance)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value = Z_LVAL_PP(arg1);

   rc = pdfSetNeedAppearance(pdf, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetNumberFormat()
   const void* IPDF, UI32 TxtField, TDecSeparator Sep, UI32 DecPlaces, TNegativeStyle NegStyle, const char* CurrStr, LBOOL Prepend); */
PHP_METHOD(dynapdf, SetNumberFormat)
{
   zval **argv[6];
   PPDF* pdf;
   SI32 TxtField, Sep, DecPlaces, NegStyle;
   char *CurrStr;
   SI32 Prepend;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   TxtField  = Z_LVAL_PP(argv[0]);
   Sep       = Z_LVAL_PP(argv[1]);
   DecPlaces = Z_LVAL_PP(argv[2]);
   NegStyle  = Z_LVAL_PP(argv[3]);
   CurrStr   = Z_STRVAL_PP(argv[4]);
   Prepend   = Z_LVAL_PP(argv[5]);

   rc = pdfSetNumberFormat(pdf, TxtField, Sep, DecPlaces, NegStyle, CurrStr, Prepend);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetOCGContUsage(const PPDF* IPDF, UI32 Handle, struct TPDFOCGContUsage* Value);
PHP_METHOD(dynapdf, SetOCGContUsage)
{
   zval **argv[2];
   HashTable* arr_hash;
   struct dynapdf_object *obj;
   struct TPDFOCGContUsage Value;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   Handle   = Z_LVAL_PP(argv[0]);
   arr_hash = Z_ARRVAL_PP(argv[1]);

   Value.StructSize = sizeof(Value);
   pdfInitOCGContUsage(&Value);
   GetContUsage(obj, arr_hash, &Value);

   rc = pdfSetOCGContUsage(obj->pdf, Handle, &Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetOCGState(const PPDF* IPDF, UI32 Handle, LBOOL On, LBOOL SaveState);
PHP_METHOD(dynapdf, SetOCGState)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 rc, Handle, On, SaveState;
   int argc = ZEND_NUM_ARGS();

   if(argc < 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Handle    = Z_LVAL_PP(argv[0]);
   On        = Z_LVAL_PP(argv[1]);
   SaveState = Z_LVAL_PP(argv[2]);

   rc = pdfSetOCGState(pdf, Handle, On, SaveState);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetOnErrorProc(const void* IPDF, const void* Data, TErrorProc* ErrProc);
PHP_METHOD(dynapdf, SetOnErrorProc)
{
   zval **arg1;
   struct dynapdf_object *obj;
   char *name = NULL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

#if PHP_VERSION_ID < 50327
   if (!zend_is_callable(*arg1, 0, &name))
#else
   if (!zend_is_callable(*arg1, 0, &name TSRMLS_CC))
#endif
   {
      php_error_docref1(NULL TSRMLS_CC, name, E_WARNING, "The argument must be a callback function!");
      efree(name);
      RETURN_BOOL(0);
   }

   if (!obj->onErrorProc)
   {
      MAKE_STD_ZVAL(obj->onErrorProc);
   }else
      zval_dtor(obj->onErrorProc);

   // The name was emalloced by zend_is_callable(). So, we don't need to copy it.
   ZVAL_STRING(obj->onErrorProc, name, 0);
   rc = pdfSetOnErrorProc(obj->pdf, obj, UserErrorProc);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetOnPageBreakProc(const void* IPDF, const void* Data, TOnPageBreakProc* OnBreakProc);
PHP_METHOD(dynapdf, SetOnPageBreakProc)
{
   zval **arg1;
   struct dynapdf_object *obj;
   char *name = NULL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

#if PHP_VERSION_ID < 50327
   if (!zend_is_callable(*arg1, 0, &name))
#else
   if (!zend_is_callable(*arg1, 0, &name TSRMLS_CC))
#endif
   {
      php_error_docref1(NULL TSRMLS_CC, name, E_WARNING, "The argument must be a callback function!");
      efree(name);
      RETURN_BOOL(0);
   }

   if (!obj->onPageBreakProc)
   {
      MAKE_STD_ZVAL(obj->onPageBreakProc);
   }else
      zval_dtor(obj->onPageBreakProc);

   // The name was emalloced by zend_is_callable(). So, we don't need to copy it.
   ZVAL_STRING(obj->onPageBreakProc, name, 0);
   rc = pdfSetOnPageBreakProc(obj->pdf, obj, OnPageBreakProc);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetOpacity()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetOpacity)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetOpacity(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetOrientation()
   const void* IPDF, SI32 Value); */
PHP_METHOD(dynapdf, SetOrientation)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetOrientation(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetOrientationEx(const void* IPDF, SI32 Value);
PHP_METHOD(dynapdf, SetOrientationEx)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetOrientationEx(pdf, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetPageBBox(IPGE* PagePtr, TPageBoundary Boundary, struct TFltRect* BBox);
PHP_METHOD(dynapdf, SetPageBBox)
{
   zval **argv[2];
   PPDF* pdf;
   HashTable *array;
   struct TFltRect BBox = {0.0f, 0.0f, 0.0f, 0.0f};
   SI32 rc, Boundary;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_array_ex(argv[1]);

   Boundary = Z_LVAL_PP(argv[0]);
   array    = Z_ARRVAL_PP(argv[1]);

   if (GetFltRect(array, &BBox))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"BBox\" must contain values for Left, Bottom, Right, Top!");
      RETURN_BOOL(0);
   }

   rc = pdfSetPageBBox(pdf, Boundary, &BBox);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetPageCoords()
   const void* IPDF, TPageCoord PageCoords); */
PHP_METHOD(dynapdf, SetPageCoords)
{
   zval **arg1;

   PPDF* pdf;
   TPageCoord PageCoords;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   PageCoords = Z_LVAL_PP(arg1);

   rc = pdfSetPageCoords(pdf, PageCoords);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfSetPageFormat()
   const void* IPDF, TPageFormat Value); */
PHP_METHOD(dynapdf, SetPageFormat)
{
   zval **arg1;
   PPDF* pdf;
   TPageFormat Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetPageFormat(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetPageHeight()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetPageHeight)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetPageHeight(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetPageLayout()
   const void* IPDF, TPageLayout Layout); */
PHP_METHOD(dynapdf, SetPageLayout)
{
   zval **arg1;
   PPDF* pdf;
   TPageLayout Layout;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Layout  = Z_LVAL_PP(arg1);

   rc = pdfSetPageLayout(pdf, Layout);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetPageMode()
   const void* IPDF, TPageMode Mode); */
PHP_METHOD(dynapdf, SetPageMode)
{
   zval **arg1;
   PPDF* pdf;
   TPageMode Mode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Mode = Z_LVAL_PP(arg1);

   rc = pdfSetPageMode(pdf, Mode);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfSetPageWidth()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetPageWidth)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetPageWidth(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetPDFVersion()
   const void* IPDF, TPDFVersion Version); */
PHP_METHOD(dynapdf, SetPDFVersion)
{
   zval **arg1;
   PPDF* pdf;
   TPDFVersion Version;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Version = Z_LVAL_PP(arg1);

   rc = pdfSetPDFVersion(pdf, Version);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetPrintSettings(const void* IPDF, TDuplexMode Mode, SI32 PickTrayByPDFSize, UI32 NumCopies, TPrintScaling PrintScaling, UI32* PrintRanges, UI32 NumRanges);
PHP_METHOD(dynapdf, SetPrintSettings)
{
   zval **argv[5];
   PPDF* pdf;
   UI32 PrintRanges[256], *pRanges = NULL, NumRanges = 0;
   SI32 rc, Mode, PickTrayByPDFSize, NumCopies, PrintScaling;
   int argc = ZEND_NUM_ARGS();

   if(argc < 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Mode              = Z_LVAL_PP(argv[0]);
   PickTrayByPDFSize = Z_LVAL_PP(argv[1]);
   NumCopies         = Z_LVAL_PP(argv[2]);
   PrintScaling      = Z_LVAL_PP(argv[3]);

   if (argc > 4)
   {
      HashTable *arr_hash;
      convert_to_array_ex(argv[4]);
      arr_hash    = Z_ARRVAL_PP(argv[4]);
      NumRanges   = GetUI32Array(arr_hash, PrintRanges, 256);
      NumRanges >>= 1;
      if (NumRanges > 0)
         pRanges = PrintRanges;
   }
   rc = pdfSetPrintSettings(pdf, Mode, PickTrayByPDFSize, NumCopies, PrintScaling, pRanges, NumRanges);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetProgressProc(const void* IPDF, const void* Data, TInitProgressProc* InitProgress, TProgressProc* Progress);
PHP_METHOD(dynapdf, SetProgressProc)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   char *initProgess = NULL, *progress = NULL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

#if PHP_VERSION_ID < 50327
   if (!zend_is_callable(*argv[0], 0, &initProgess))
#else
   if (!zend_is_callable(*argv[0], 0, &initProgess TSRMLS_CC))
#endif
   {
      php_error_docref1(NULL TSRMLS_CC, initProgess, E_WARNING, "The first argument must be a callback function!");
      efree(initProgess);
      RETURN_BOOL(0);
   }

#if PHP_VERSION_ID < 50327
   if (!zend_is_callable(*argv[1], 0, &progress))
#else
   if (!zend_is_callable(*argv[1], 0, &progress TSRMLS_CC))
#endif
   {
      php_error_docref1(NULL TSRMLS_CC, progress, E_WARNING, "The second argument must be a callback function!");
      efree(initProgess);
      efree(progress);
      RETURN_BOOL(0);
   }

   if (!obj->onInitProgressProc)
   {
      MAKE_STD_ZVAL(obj->onInitProgressProc);
   }else
      zval_dtor(obj->onInitProgressProc);

   if (!obj->onProgressProc)
   {
      MAKE_STD_ZVAL(obj->onProgressProc);
   }else
      zval_dtor(obj->onProgressProc);

   // The name was emalloced by zend_is_callable(). So, we don't need to copy it.
   ZVAL_STRING(obj->onInitProgressProc, initProgess, 0);
   ZVAL_STRING(obj->onProgressProc, progress, 0);
   rc = pdfSetProgressProc(obj->pdf, obj, OnInitProgressProc, OnProgressProc);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetRenderingIntent(const PPDF* IPDF, SI32 ImgHandle, TRenderingIntent Value);
PHP_METHOD(dynapdf, SetRenderingIntent)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, ImgHandle, Value;
   int argc = ZEND_NUM_ARGS();

   if(argc < 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   ImgHandle = Z_LVAL_PP(argv[0]);
   Value     = Z_LVAL_PP(argv[1]);

   rc = pdfSetRenderingIntent(pdf, ImgHandle, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetResolution()
   const void* IPDF, UI32 Value); */
PHP_METHOD(dynapdf, SetResolution)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value = Z_LVAL_PP(arg1);

   rc = pdfSetResolution(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */


/* {{{ proto SI32 pdfSetSaveNewImageFormat()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetSaveNewImageFormat)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value = Z_LVAL_PP(arg1);

   rc = pdfSetSaveNewImageFormat(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetSeparationInfo(const void* IPDF, UI32 Handle);
PHP_METHOD(dynapdf, SetSeparationInfo)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Handle;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Handle = Z_LVAL_PP(arg1);

   rc = pdfSetSeparationInfo(pdf, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetSpaceWidthFactor(const PPDF* IPDF, double Value);
PHP_METHOD(dynapdf, SetSpaceWidthFactor)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value = Z_DVAL_PP(arg1);

   rc = pdfSetSpaceWidthFactor(pdf, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetStrokeColor()
   const void* IPDF, UI32 Color); */
PHP_METHOD(dynapdf, SetStrokeColor)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Color;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Color = Z_LVAL_PP(arg1);

   rc = pdfSetStrokeColor(pdf, Color);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetStrokeColorEx(const void* IPDF, const BYTE* Color, UI32 NumComponents);
PHP_METHOD(dynapdf, SetStrokeColorEx)
{
   zval **arg1;
   PPDF* pdf;
   BYTE Color[32];
   HashTable *arr_hash;
   SI32 rc, NumComponents;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(arg1);
   arr_hash = Z_ARRVAL_PP(arg1);

   NumComponents = GetBYTEArray(arr_hash, Color, 32);

   rc = pdfSetStrokeColorEx(pdf, Color, NumComponents);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetStrokeColorF(const void* IPDF, const float* Color, UI32 NumComponents);
PHP_METHOD(dynapdf, SetStrokeColorF)
{
   zval **arg1;
   PPDF* pdf;
   float Color[32];
   HashTable *arr_hash;
   SI32 rc, NumComponents;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(arg1);
   arr_hash = Z_ARRVAL_PP(arg1);

   NumComponents = GetFloatArray(arr_hash, Color, 32);

   rc = pdfSetStrokeColorF(pdf, Color, NumComponents);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetStrokeColorSpace(const void* IPDF, TPDFColorSpace CS);
PHP_METHOD(dynapdf, SetStrokeColorSpace)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   CS = Z_LVAL_PP(arg1);
   rc = pdfSetStrokeColorSpace(pdf, CS);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetTabLen()
   const void* IPDF, SI32 TabLen); */
PHP_METHOD(dynapdf, SetTabLen)
{
   zval **arg1;
   PPDF* pdf;
   SI32 TabLen;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   TabLen = Z_LVAL_PP(arg1);

   rc = pdfSetTabLen(pdf, TabLen);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetTemplBBox(const PPDF* IPDF, UI32 Handle, TPageBoundary Boundary, struct TPDFRect* BBox);
PHP_METHOD(dynapdf, SetTemplBBox)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 Handle, Boundary, rc;
   struct TPDFRect* bbox = NULL, value = {0.0f, 0.0f, 0.0f, 0.0f};

   int argc = ZEND_NUM_ARGS();

   if(argc < 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle   = Z_LVAL_PP(argv[0]);
   Boundary = Z_LVAL_PP(argv[1]);

   if (argc > 2)
   {
      HashTable* hash;
      convert_to_array_ex(argv[2]);
      hash = Z_ARRVAL_PP(argv[2]);
      if (GetBBoxFromArray(hash, &value) == 0)
         bbox = &value;
   }

   rc = pdfSetTemplBBox(pdf, Handle, Boundary, bbox);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetTextDrawMode()
   const void* IPDF, TDrawMode Mode); */
PHP_METHOD(dynapdf, SetTextDrawMode)
{
   zval **arg1;
   PPDF* pdf;
   TDrawMode Mode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Mode = Z_LVAL_PP(arg1);

   rc = pdfSetTextDrawMode(pdf, Mode);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetTextFieldValue()
   const void* IPDF, UI32 Field, const char* Value, const char* DefValue, TTextAlign Align); */
PHP_METHOD(dynapdf, SetTextFieldValue)
{
   zval **argv[4];
   PPDF* pdf;
   SI32 Field;
   char *Value, *DefValue;
   TTextAlign Align;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   Field    = Z_LVAL_PP(argv[0]);
   Value    = Z_STRVAL_PP(argv[1]);
   DefValue = Z_STRVAL_PP(argv[2]);
   Align    = Z_LVAL_PP(argv[3]);

   rc = pdfSetTextFieldValueA(pdf, Field, Value, DefValue, Align);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetTextFieldValueExA(const void* IPDF, UI32 Field, const char* Value);
PHP_METHOD(dynapdf, SetTextFieldValueEx)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 Field;
   char *Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Field = Z_LVAL_PP(argv[0]);
   Value = Z_STRVAL_PP(argv[1]);

   rc = pdfSetTextFieldValueExA(pdf, Field, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetTextRect()
   const void* IPDF, double PosX, double PosY, double Width, double Height); */
PHP_METHOD(dynapdf, SetTextRect)
{
   zval **argv[4];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);

   rc = pdfSetTextRect(pdf, PosX, PosY, Width, Height);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetTextRise()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetTextRise)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = Z_DVAL_PP(arg1);

   rc = pdfSetTextRise(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetTextScaling()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetTextScaling)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value   = Z_DVAL_PP(arg1);

   rc = pdfSetTextScaling(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetTransparentColor()
   const void* IPDF, UI32 AColor); */
PHP_METHOD(dynapdf, SetTransparentColor)
{
   zval **arg1;
   PPDF* pdf;
   SI32 AColor;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   AColor = Z_LVAL_PP(arg1);

   rc = pdfSetTransparentColor(pdf, AColor);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP void   PDF_CALL pdfSetTrapped(const void* IPDF, LBOOL Value);
PHP_METHOD(dynapdf, SetTrapped)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value = Z_LVAL_PP(arg1);

   pdfSetTrapped(pdf, Value);
}

/* {{{ proto SI32 pdfSetUseExactPwd()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetUseExactPwd)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetUseExactPwd(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetUseGlobalImpFiles()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetUseGlobalImpFiles)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetUseGlobalImpFiles(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetUseImageInterpolation(const PPDF* IPDF, UI32 Handle, LBOOL Value);
PHP_METHOD(dynapdf, SetUseImageInterpolation)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, Handle, Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   Handle = Z_LVAL_PP(argv[0]);
   Value  = Z_LVAL_PP(argv[1]);

   rc = pdfSetUseImageInterpolation(pdf, Handle, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetUseImageInterpolationEx(const PIMG* Image, LBOOL Value);
PHP_METHOD(dynapdf, SetUseImageInterpolationEx)
{
   PIMG* Image;
   zval **argv[2];
   struct dynapdf_object* obj;
   SI32 rc, hash, Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);

   hash  = Z_LVAL_PP(argv[0]);
   Value = Z_LVAL_PP(argv[1]);

   if ((Image = (PIMG*)GetPointer(obj, hash, ptImage)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter Image contains an invalid value!");
      RETURN_BOOL(0);
   }
   rc = pdfSetUseImageInterpolationEx(Image, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL pdfSetUserUnit(const void* IPDF, float Value);
PHP_METHOD(dynapdf, SetUserUnit)
{
   zval **arg1;
   PPDF* pdf;
   float Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value  = (float)Z_DVAL_PP(arg1);

   rc = pdfSetUserUnit(pdf, Value);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetUseStdFonts()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetUseStdFonts)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetUseStdFonts(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetUseSwapFile()
   const void* IPDF, LBOOL SwapContents, UI32 SwapLimit); */
PHP_METHOD(dynapdf, SetUseSwapFile)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 rc, SwapContents, SwapLimit;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   SwapContents  = Z_LVAL_PP(argv[0]);
   SwapLimit     = Z_LVAL_PP(argv[1]);

   rc = pdfSetUseSwapFile(pdf, SwapContents, SwapLimit);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetUseSwapFileEx(const void* IPDF, LBOOL SwapContents, UI32 SwapLimit, const char* SwapDir);
PHP_METHOD(dynapdf, SetUseSwapFileEx)
{
   zval **argv[3];
   PPDF* pdf;
   char* SwapDir;
   SI32 rc, SwapContents, SwapLimit;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);

   SwapContents  = Z_LVAL_PP(argv[0]);
   SwapLimit     = Z_LVAL_PP(argv[1]);
   SwapDir       = Z_STRVAL_PP(argv[2]);

   rc = pdfSetUseSwapFileEx(pdf, SwapContents, SwapLimit, SwapDir);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSetUseSystemFonts()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetUseSystemFonts)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetUseSystemFonts(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetUseTransparency()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetUseTransparency)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetUseTransparency(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetUseVisibleCoords()
   const void* IPDF, LBOOL Value); */
PHP_METHOD(dynapdf, SetUseVisibleCoords)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetUseVisibleCoords(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetViewerPreferences()
   const void* IPDF, TViewerPreference Value, TViewPrefAddVal AddVal); */
PHP_METHOD(dynapdf, SetViewerPreferences)
{
   zval **argv[2];
   PPDF* pdf;
   TViewerPreference Value;
   TViewPrefAddVal AddVal;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Value  = Z_LVAL_PP(argv[0]);
   AddVal = Z_LVAL_PP(argv[1]);

   rc = pdfSetViewerPreferences(pdf, Value, AddVal);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetWMFDefExtent()
   const void* IPDF, UI32 Width, UI32 Height); */
PHP_METHOD(dynapdf, SetWMFDefExtent)
{
   zval **argv[2];
   PPDF* pdf;
   UI32 Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Width  = Z_LVAL_PP(argv[0]);
   Height = Z_LVAL_PP(argv[1]);

   rc = pdfSetWMFDefExtent(pdf, Width, Height);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetWMFPixelPerInch()
   const void* IPDF, UI32 Value); */
PHP_METHOD(dynapdf, SetWMFPixelPerInch)
{
   zval **arg1;
   PPDF* pdf;
   SI32 Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Value  = Z_LVAL_PP(arg1);

   rc = pdfSetWMFPixelPerInch(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSetWordSpacing()
   const void* IPDF, double Value); */
PHP_METHOD(dynapdf, SetWordSpacing)
{
   zval **arg1;
   PPDF* pdf;
   double Value;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(arg1);
   Value   = Z_DVAL_PP(arg1);

   rc = pdfSetWordSpacing(pdf, Value);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfSetXFAStream(const PPDF* IPDF, UI32 Index, const void* Buffer, UI32 BufSize);
PHP_METHOD(dynapdf, SetXFAStream)
{
   zval** argv[2];
   PPDF* pdf;
   char* Buffer;
   UI32 Index, BufSize;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if (argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   Index   = Z_LVAL_PP(argv[0]);
   Buffer  = Z_STRVAL_PP(argv[1]);
   BufSize = Z_STRLEN_PP(argv[1]);

   rc = pdfSetXFAStream(pdf, Index, Buffer, BufSize);
   RETURN_BOOL(rc);
}

/* {{{ proto SI32 pdfSkewCoords()
   const void* IPDF, double alpha, double beta, double OriginX, double OriginY); */
PHP_METHOD(dynapdf, SkewCoords)
{
   zval **argv[4];
   PPDF* pdf;
   double alpha, beta, OriginX, OriginY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   alpha   = Z_DVAL_PP(argv[0]);
   beta    = Z_DVAL_PP(argv[1]);
   OriginX = Z_DVAL_PP(argv[2]);
   OriginY = Z_DVAL_PP(argv[3]);

   rc = pdfSkewCoords(pdf, alpha, beta, OriginX, OriginY);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSortFieldsByIndex()
   const void* IPDF); */
PHP_METHOD(dynapdf, SortFieldsByIndex)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfSortFieldsByIndex(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfSortFieldsByName()
   const void* IPDF); */
PHP_METHOD(dynapdf, SortFieldsByName)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfSortFieldsByName(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

// PDF_EXP SI32   PDF_CALL pdfSquareAnnotA(const void* IPDF, double PosX, double PosY, double Width, double Height, double LineWidth, UI32 FillColor, UI32 StrokeColor, TPDFColorSpace CS, const char* Author, const char* Subject, const char* Comment);
PHP_METHOD(dynapdf, SquareAnnot)
{
   zval **argv[11];
   PPDF* pdf;
   char* Author, *Subject, *Comment;
   double PosX, PosY, Width, Height, LineWidth;
   SI32 rc, FillColor, StrokeColor, CS;
   int argc = ZEND_NUM_ARGS();

   if(argc != 11 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   convert_to_string_ex(argv[8]);
   convert_to_string_ex(argv[9]);
   convert_to_string_ex(argv[10]);

   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   Width       = Z_DVAL_PP(argv[2]);
   Height      = Z_DVAL_PP(argv[3]);
   LineWidth   = Z_DVAL_PP(argv[4]);
   FillColor   = Z_LVAL_PP(argv[5]);
   StrokeColor = Z_LVAL_PP(argv[6]);
   CS          = Z_LVAL_PP(argv[7]);
   Author      = Z_STRVAL_PP(argv[8]);
   Subject     = Z_STRVAL_PP(argv[9]);
   Comment     = Z_STRVAL_PP(argv[10]);

   rc = pdfSquareAnnotA(pdf, PosX, PosY, Width, Height, LineWidth, FillColor, StrokeColor, CS, Author, Subject, Comment);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL pdfStampAnnotA(const void* IPDF, TRubberStamp SubType, double PosX, double PosY, double Width, double Height, const char* Author, const char* Subject, const char* Comment);
PHP_METHOD(dynapdf, StampAnnot)
{
   zval **argv[8];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char *Author, *Subject, *Comment;
   SI32 rc, SubType;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_string_ex(argv[6]);
   convert_to_string_ex(argv[7]);

   SubType = Z_LVAL_PP(argv[0]);
   PosX    = Z_DVAL_PP(argv[1]);
   PosY    = Z_DVAL_PP(argv[2]);
   Width   = Z_DVAL_PP(argv[3]);
   Height  = Z_DVAL_PP(argv[4]);
   Author  = Z_STRVAL_PP(argv[5]);
   Subject = Z_STRVAL_PP(argv[6]);
   Comment = Z_STRVAL_PP(argv[7]);

   rc = pdfStampAnnotA(pdf, SubType, PosX, PosY, Width, Height, Author, Subject, Comment);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfStrokePath()
   const void* IPDF); */
PHP_METHOD(dynapdf, StrokePath)
{
   SI32 rc;
   PPDF* pdf;
   DYNAPDF_FETCH_PDFDOC();
   rc = pdfStrokePath(pdf);
   RETURN_BOOL(rc);
}
/* }}} */

//
PHP_METHOD(dynapdf, TestCallback)
{
   zval **arg1;
   struct dynapdf_object *obj;
   SI32 funcs = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc > 0 && (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();

   if (argc > 0)
   {
      convert_to_long_ex(arg1);
      funcs = Z_LVAL_PP(arg1);
   }


   if (funcs == cbfAll)
   {
      if (obj->onErrorProc)             UserErrorProc(obj, 123, "Just a test...", 321);
      if (obj->onFontNotFoundProc)      OnFontNotFoundProc(obj, NULL, "Just a test...", fsRegular, -1, 0);
      if (obj->onInitProgressProc)      OnInitProgressProc(obj, ptImportPage, 1);
      if (obj->onPageBreakProc)         OnPageBreakProc(obj, 50.0, 50.0, 0);
      if (obj->onProgressProc)          OnProgressProc(obj, 1);
      if (obj->onReplaceICCProfileProc) OnReplaceICCProfileProc(obj, ictGray, -1);
   }else
   {
      if ((funcs & cbfOnErrorProc) != 0 && obj->onErrorProc)                         UserErrorProc(obj, 123, "Just a test...", 321);
      if ((funcs & cbfOnFontNotFoundProc) != 0 && obj->onFontNotFoundProc)           OnFontNotFoundProc(obj, NULL, "Just a test...", fsRegular, -1, 0);
      if ((funcs & cbfOnInitProgressProc) != 0 && obj->onInitProgressProc)           OnInitProgressProc(obj, ptImportPage, 1);
      if ((funcs & cbfOnPageBreakProc) != 0 && obj->onPageBreakProc)                 OnPageBreakProc(obj, 50.0, 50.0, 0);
      if ((funcs & cbfOnProgressProc) != 0 && obj->onProgressProc)                   OnProgressProc(obj, 1);
      if ((funcs & cbfonReplaceICCProfileProc) != 0 && obj->onReplaceICCProfileProc) OnReplaceICCProfileProc(obj, ictGray, -1);
   }
   RETURN_BOOL(1);
}

// PDF_EXP SI32   PDF_CALL pdfTestGlyphsA(const PPDF* IPDF, SI32 FontHandle, const char* Text);
PHP_METHOD(dynapdf, TestGlyphs)
{
   zval **argv[2];
   PPDF* pdf;
   SI32 FontHandle, Len;
   char* Text;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   FontHandle = Z_LVAL_PP(argv[0]);
   Text       = Z_STRVAL_PP(argv[1]);
   Len        = Z_STRLEN_PP(argv[1]);

   rc = pdfTestGlyphsExA(pdf, FontHandle, Text, Len);
   RETURN_LONG(rc);
}

// PDF_EXP SI32   PDF_CALL pdfTestGlyphsExA(const PPDF* IPDF, SI32 FontHandle, const char* Text, UI32 Len);
PHP_METHOD(dynapdf, TestGlyphsEx)
{
   zval **argv[3];
   PPDF* pdf;
   SI32 FontHandle, Len, rl;
   char* Text;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   FontHandle = Z_LVAL_PP(argv[0]);
   Text       = Z_STRVAL_PP(argv[1]);
   rl         = Z_STRLEN_PP(argv[1]);
   Len        = Z_LVAL_PP(argv[2]);

   if (Len > rl) Len = rl;

   rc = pdfTestGlyphsExA(pdf, FontHandle, Text, Len);
   RETURN_LONG(rc);
}

/* {{{ proto SI32 pdfTextAnnot()
   const void* IPDF, double PosX, double PosY, double Width, double Height, const char* Author, const char* Text, TAnnotIcon Icon, LBOOL Open); */
PHP_METHOD(dynapdf, TextAnnot)
{
   zval **argv[8];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char *Author, *Text;
   TAnnotIcon Icon;
   SI32 Open;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   convert_to_long_ex(argv[7]);
   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   Width       = Z_DVAL_PP(argv[2]);
   Height      = Z_DVAL_PP(argv[3]);
   Author      = Z_STRVAL_PP(argv[4]);
   Text        = Z_STRVAL_PP(argv[5]);
   Icon        = Z_LVAL_PP(argv[6]);
   Open        = Z_LVAL_PP(argv[7]);

   rc = pdfTextAnnotA(pdf, PosX, PosY, Width, Height, Author, Text, Icon, Open);
   RETURN_LONG(rc);
}
/* }}} */

/* {{{ proto SI32 pdfTranslateCoords()
   const void* IPDF, double OriginX, double OriginY); */
PHP_METHOD(dynapdf, TranslateCoords)
{
   zval **argv[2];
   PPDF* pdf;
   double OriginX, OriginY;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   OriginX = Z_DVAL_PP(argv[0]);
   OriginY = Z_DVAL_PP(argv[1]);

   rc = pdfTranslateCoords(pdf, OriginX, OriginY);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP UI32   PDF_CALL fntTranslateRawCode(const PFNT* IFont, const BYTE* Text, UI32 Len, double ADDR Width, UI16 OutText[32], SI32 ADDR OutLen, LBOOL ADDR Decoded, float CharSpacing, float WordSpacing, float TextScale);
PHP_METHOD(dynapdf, TranslateRawCode)
{
   zval **argv[5];
   struct dynapdf_object *obj;
   BYTE* Text;
   UI16 OutText[32];
   LBOOL Decoded = 0;
   PFNT* IFont;
   double Width = 0.0;
   float CharSpacing, WordSpacing, TextScale;
   SI32 rc, hash, Len, OutLen = 0;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   hash        = Z_LVAL_PP(argv[0]);
   Text        = (BYTE*)Z_STRVAL_PP(argv[1]);
   Len         = Z_STRLEN_PP(argv[1]);
   CharSpacing = (float)Z_DVAL_PP(argv[2]);
   WordSpacing = (float)Z_DVAL_PP(argv[3]);
   TextScale   = (float)Z_DVAL_PP(argv[4]);

   if ((IFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFont contains an invalid value!");
      RETURN_DOUBLE(0.0);
   }
   rc = fntTranslateRawCode(IFont, Text, Len, &Width, OutText, &OutLen, &Decoded, CharSpacing, WordSpacing, TextScale);

   array_init(return_value);

   add_assoc_long(return_value, "result", rc);
   add_assoc_double(return_value, "Width", Width);
   AddStringExW(return_value, "OutText", OutText, OutLen);
   add_assoc_bool(return_value, "Decoded", Decoded);
}

//PDF_EXP SI32   PDF_CALL fntTranslateString2(const PFNT* IFont, const BYTE* Text, UI32 Len, UI16* OutText, UI32 Size, UI32 Flags);
PHP_METHOD(dynapdf, TranslateString2)
{
   zval **argv[2];
   struct dynapdf_object *obj;
   BYTE* Text;
   UI16* outText;
   PFNT* IFont;
   SI32 rc, hash, Len, outLen;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC_EX();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   hash  = Z_LVAL_PP(argv[0]);
   Text  = (BYTE*)Z_STRVAL_PP(argv[1]);
   Len   = Z_STRLEN_PP(argv[1]);

   if ((IFont = (PFNT*)GetPointer(obj, hash, ptFont)) == NULL)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter IFont contains an invalid value!");
      RETURN_DOUBLE(0.0);
   }

   outLen  = Len * 16 / 10 + 32;
   outText = (UI16*)emalloc(outLen << 1);

   rc = fntTranslateString2(IFont, Text, Len, outText, outLen, 0);

   if (rc > 0) ReturnStringExW(return_value, outText, rc);
   efree(outText);
}

/* {{{ proto SI32 pdfTriangle()
   const void* IPDF, double x1, double y1, double x2, double y2, double x3, double y3, TPathFillMode FillMode); */
PHP_METHOD(dynapdf, Triangle)
{
   zval **argv[7];
   PPDF* pdf;
   double x1, y1, x2, y2, x3, y3;
   TPathFillMode FillMode;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_long_ex(argv[6]);
   x1 = Z_DVAL_PP(argv[0]);
   y1 = Z_DVAL_PP(argv[1]);
   x2 = Z_DVAL_PP(argv[2]);
   y2 = Z_DVAL_PP(argv[3]);
   x3 = Z_DVAL_PP(argv[4]);
   y3 = Z_DVAL_PP(argv[5]);
   FillMode = Z_LVAL_PP(argv[6]);

   rc = pdfTriangle(pdf, x1, y1, x2, y2, x3, y3, FillMode);
   RETURN_BOOL(rc);
}
/* }}} */

//PDF_EXP LBOOL  PDF_CALL pdfUnLockLayer(const void* IPDF, UI32 Layer);
PHP_METHOD(dynapdf, UnLockLayer)
{
   zval **arg1;
   PPDF* pdf;
   SI32 rc, Layer;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(arg1);
   Layer = Z_LVAL_PP(arg1);

   rc = pdfUnLockLayer(pdf, Layer);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL pdfWatermarkAnnot(const PPDF* IPDF, double PosX, double PosY, double Width, double Height);
PHP_METHOD(dynapdf, WatermarkAnnot)
{
   zval **argv[4];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);

   rc = pdfWatermarkAnnot(pdf, PosX, PosY, Width, Height);
   RETURN_LONG(rc);
}


/* {{{ proto SI32 pdfWebLink()
   const void* IPDF, double PosX, double PosY, double Width, double Height, const char* URL); */
PHP_METHOD(dynapdf, WebLink)
{
   zval **argv[5];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   char *URL;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_string_ex(argv[4]);
   PosX        = Z_DVAL_PP(argv[0]);
   PosY        = Z_DVAL_PP(argv[1]);
   Width       = Z_DVAL_PP(argv[2]);
   Height      = Z_DVAL_PP(argv[3]);
   URL         = Z_STRVAL_PP(argv[4]);

   rc = pdfWebLinkA(pdf, PosX, PosY, Width, Height, URL);
   RETURN_LONG(rc);
}
/* }}} */

// PHP specific
PHP_METHOD(dynapdf, WriteBuffer)
{
   PPDF* pdf;
   char* buffer;
   UI32 bufSize = 0;
   DYNAPDF_FETCH_PDFDOC();
   buffer = pdfGetBuffer(pdf, &bufSize);
   if (buffer) PHPWRITE(buffer, bufSize);
   pdfFreePDF(pdf);
}

/* {{{ proto SI32 pdfWriteAngleText()
   const void* IPDF, const char* AText, double Angle, double PosX, double PosY, double Radius, double YOrigin); */
PHP_METHOD(dynapdf, WriteAngleText)
{
   zval **argv[6];
   PPDF* pdf;
   char *AText;
   double Angle, PosX, PosY, Radius, YOrigin;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_string_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   AText   = Z_STRVAL_PP(argv[0]);
   Angle   = Z_DVAL_PP(argv[1]);
   PosX    = Z_DVAL_PP(argv[2]);
   PosY    = Z_DVAL_PP(argv[3]);
   Radius  = Z_DVAL_PP(argv[4]);
   YOrigin = Z_DVAL_PP(argv[5]);

   rc = pdfWriteAngleTextA(pdf, AText, Angle, PosX, PosY, Radius, YOrigin);
   RETURN_LONG(rc);
}
/* }}} */


/* {{{ proto SI32 pdfWriteFText()
   const void* IPDF, TTextAlign Align, const char* AText); */
PHP_METHOD(dynapdf, WriteFText)
{
   zval **argv[2];
   PPDF* pdf;
   TTextAlign Align;
   char *AText;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_long_ex(argv[0]);
   convert_to_string_ex(argv[1]);
   Align = Z_LVAL_PP(argv[0]);
   AText = Z_STRVAL_PP(argv[1]);

   rc = pdfWriteFTextA(pdf, Align, AText);
   RETURN_BOOL(rc);
}
/* }}} */



/* {{{ proto SI32 pdfWriteFTextEx()
   const void* IPDF, double PosX, double PosY, double Width, double Height, TTextAlign Align, const char* AText); */
PHP_METHOD(dynapdf, WriteFTextEx)
{
   zval **argv[6];
   PPDF* pdf;
   double PosX, PosY, Width, Height;
   TTextAlign Align;
   char *AText;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_double_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_string_ex(argv[5]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   Width  = Z_DVAL_PP(argv[2]);
   Height = Z_DVAL_PP(argv[3]);
   Align  = Z_LVAL_PP(argv[4]);
   AText  = Z_STRVAL_PP(argv[5]);

   rc = pdfWriteFTextExA(pdf, PosX, PosY, Width, Height, Align, AText);
   RETURN_BOOL(rc);
}
/* }}} */

// WriteImageBuffer() -> PHP specific
PHP_METHOD(dynapdf, WriteImageBuffer)
{
   PPDF* pdf;
   char* buffer;
   UI32 bufSize = 0;
   DYNAPDF_FETCH_PDFDOC();
   buffer = pdfGetImageBuffer(pdf, &bufSize);
   if (buffer) PHPWRITE(buffer, bufSize);
   pdfFreeImageBuffer(pdf);
}

/* {{{ proto SI32 pdfWriteText()
   const void* IPDF, double PosX, double PosY, const char* AText); */
PHP_METHOD(dynapdf, WriteText)
{
   zval **argv[3];
   PPDF* pdf;
   double PosX, PosY;
   char* AText;
   SI32 rc, len;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   PosX   = Z_DVAL_PP(argv[0]);
   PosY   = Z_DVAL_PP(argv[1]);
   AText  = Z_STRVAL_PP(argv[2]);
   len    = Z_STRLEN_PP(argv[2]);

   rc = pdfWriteTextExA(pdf, PosX, PosY, AText, len);
   RETURN_BOOL(rc);
}
/* }}} */

/* {{{ proto SI32 pdfWriteTextMatrix()
   const void* IPDF, struct TCTM* M, const char* AText); */
PHP_METHOD(dynapdf, WriteTextMatrix)
{
   zval **argv[2];
   HashTable *arr_hash;
   PPDF* pdf;
   char* AText;
   SI32 rc, len;
   struct TCTM M = {1.0, 0.0, 0.0, 1.0, 0.0, 0.0};
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_PDFDOC();
   convert_to_array_ex(argv[0]);
   convert_to_string_ex(argv[1]);

   arr_hash = Z_ARRVAL_PP(argv[0]);
   AText    = Z_STRVAL_PP(argv[1]);
   len      = Z_STRLEN_PP(argv[1]);

   if (GetCTMFromArray(arr_hash, &M))
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "The parameter \"M\" must be an array of six numbers (a, b, c, d, x, y)!");
      RETURN_BOOL(0);
   }

   rc = pdfWriteTextMatrixExA(pdf, &M, AText, len);
   RETURN_BOOL(rc);
}
/* }}} */

zend_function_entry dynapdf_functions[] =
{
   PHP_ME(dynapdf, __construct, NULL, ZEND_ACC_PUBLIC | ZEND_ACC_CTOR)

   PHP_ME(dynapdf, ActivateAltFontList, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddActionToObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddAnnotToPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddArticle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddBookmarkEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddBookmarkEx2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddButtonImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddButtonImageEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddButtonImageEx2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddContinueText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddDeviceNProcessColorants, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddDeviceNSeparations, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddFieldToFormAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddFieldToHideAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddFileComment, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddFontSearchPath, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddHeaderFooter, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddInkList, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddJavaScript, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddLayerToDisplTree, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddMaskImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddObjectToLayer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddOCGToAppEvent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddOutputIntent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddPageLabel, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AddValToChoiceField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Append, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ApplyAppEvent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ApplyPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ApplyShading, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AssociateEmbFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AttachFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AttachFileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, AutoTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginClipPath, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginContinueText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginLayer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginPageTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginTemplateEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, BeginTransparencyGroup, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Bezier_1_2_3, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Bezier_1_3, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Bezier_2_3, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CalcPagePixelSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CalcWidthHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CaretAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeAnnotName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeAnnotPos, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeFontEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeFontSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeFontStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeFontStyleEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeJavaScript, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeJavaScriptAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeJavaScriptName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeLinkAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeNamedDest, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeOCGName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ChangeSeparationColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CheckCollection, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CheckConformance, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CheckFieldNames, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CircleAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ClearAutoTemplates, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ClearErrorLog, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ClearHostFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ClipPath, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseAndSignFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseAndSignFileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseAndSignFileExt, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseFileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseImportFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseImportFileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ClosePath, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CloseTag, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CMYK, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ComputeBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ConvColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ConvertColors, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ConvertEMFSpool, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CopyChoiceValues, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Create3DAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Create3DBackground, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Create3DGotoViewAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Create3DProjection, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Create3DView, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateAltFontList, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateAnnotAP, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateArticleThread, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateAxialShading, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateBarcodeField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateButton, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateCheckBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateComboBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateCIEColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateColItemDate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateColItemNumber, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateColItemString, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateCollection, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateCollectionField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateDeviceNColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateExtGState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateGoToAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateGoToActionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateGoToEAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateGoToRAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateGoToRActionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateGroupField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateHideAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateICCBasedColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateICCBasedColorSpaceEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateImpDataAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateIndexedColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateJSAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateLaunchAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateLaunchActionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateListBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateNamedAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateNamedDest, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateNewPDF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateOCG, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateOCMD, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateRadialShading, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateRadioButton, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateRectilinearMeasure, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateResetAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateSeparationCS, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateSetOCGStateAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateSigField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateSigFieldAP, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateSoftMask, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateStdPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateStructureTree, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateStructureTreeEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateSubmitAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateTextField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateURIAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateViewport, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, CreateXFAStream, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DecryptPDF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteAcroForm, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteActionFromObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteActionFromObjEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteAltFontList, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteAnnotation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteAnnotationFromPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteAppEvents, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteEmbeddedFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteFieldEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteJavaScripts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteNamedDest, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteNamedDestByIndex, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteOCGFromAppEvent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteOCGFromDisplayTree, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteOCUINode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteOutputIntent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeletePage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeletePageLabels, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteSeparationInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteTemplateEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DeleteXFAForm, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DisableUTF8Support, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DrawArc, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DrawArcEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DrawChord, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DrawCircle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DrawNGon, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, DrawPie, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EditPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EditTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EditTemplate2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Ellipse, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EnableUTF8Support, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EncryptPDF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EndContinueText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EndLayer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EndPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EndPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EndTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EnumDocFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EnumHostFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, EnumHostFontsEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ExchangeBookmarks, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ExchangePages, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ExtractText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FileAttachAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FileAttachAnnotEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FileLink, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FindBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FindEmbeddedFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FindField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FinishSignature, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FindLinkAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FindNextBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FlattenAnnotOrField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FlattenAnnots, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FlattenForm, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FlushPageContent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FlushPages, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FlushPagesEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FreeImageBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FreeImageObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FreeImageObjEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FreePDF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FreeTextAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, FreeUniBuf, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetActionCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetActionHandle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetActionType, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetActionTypeEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetActiveFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAllocBy, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAnnotBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAnnotCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAnnotEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAnnotFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAnnotLink, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAnnotType, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAPIVersion, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetAscent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBarcodeDict, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBidiMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBookmarkCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBookmarkEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBorderStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetBufSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCapHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCharacterSpacing, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCheckBoxChar, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCheckBoxCharEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCheckBoxDefState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCMap, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCMapCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCollectionInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetColorSpaceCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetColorSpaceObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetColorSpaceObjEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCompressionFilter, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetCompressionLevel, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetContent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDefBitsPerPixel, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDescent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDeviceNAttributes, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDocInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDocInfoCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDocInfoEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDocUsesTransparency, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDrawDirection, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetDynaPDFVersion, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetEmbeddedFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetEmbeddedFileCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetEmbeddedFileNode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFileSpec, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetEMFPatternDistance, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetErrLogMessage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetErrLogMessageCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetErrorMessage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetErrorMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldBackColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldBorderColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldBorderStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldBorderWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldCalcOrder, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldChoiceValue, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldEx2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldExpValCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldExpValue, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldExpValueEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldGroupType, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldHighlightMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldIndex, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldMapName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldTextAlign, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldTextColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldToolTip, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFieldType, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFillColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontInfoEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontMetrics, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontOrigin, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontSearchOrder, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontSelMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontWeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFontSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFTextHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetFTextHeightEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetGlyphIndex, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetGlyphOutline, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetGoToAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetGoToRAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetGStateFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetHideAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetIconColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageBufSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageCountEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageObjCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageObjEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImageWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImportDataAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImportFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetImportFlags2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInDocInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInDocInfoCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInDocInfoEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInEncryptionFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInFieldCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInIsCollection, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInIsEncrypted, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInIsSigned, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInIsTaggedPDF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInIsTrapped, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInIsXFAForm, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInkList, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInMetadata, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInPageCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInPDFVersion, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInPDFVersionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInPrintSettings, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetInRepairMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetIsFixedPitch, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetIsTaggingEnabled, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetItalicAngle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScript, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScriptAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScriptAction2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScriptActionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScriptCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScriptEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJavaScriptName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetJPEGQuality, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLanguage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLastTextPosX, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLastTextPosY, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLayerConfig, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLayerConfigCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLaunchAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLeading, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLineCapStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLineJoinStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLineWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLinkHighlightMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLogMetafileSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetLogMetafileSizeEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMatrix, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMaxFieldLen, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMeasureObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMetaConvFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMetadata, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMissingGlyphs, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMiterLimit, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetMovieAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetNamedAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetNamedDest, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetNamedDestCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetNeedAppearance, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetNumberFormatObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetObjActionCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetObjActions, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetObjEvent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOCG, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOCGContUsage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOCGCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOCGUsageUserName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOCHandle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOCUINode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOpacity, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOutputIntent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetOutputIntentCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageAnnotCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageAnnotEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageFieldCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageFieldEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageLabel, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageLabelCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageLayout, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageNum, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPageWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPDFVersion, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPDFVersionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPrintSettings, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetRelFileNode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetResetAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetResolution, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPtDataArray, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetPtDataObj, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetSaveNewImageFormat, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetSeparationInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetSpaceWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetSigDict, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetStrokeColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetSubmitAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetSysFontInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTabLen, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTemplCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTemplHandle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTemplHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTemplWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextDrawMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextFieldValue, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextRect, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextRise, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextScaling, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTextWidthF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTrapped, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTransparentColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetTypoLeading, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUseExactPwd, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUseGlobalImpFiles, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUserRights, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUserUnit, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUseStdFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUseSystemFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUsesTransparency, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUseTransparency, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetUseVisibleCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetViewerPreferences, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetViewport, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetViewportCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetWMFDefExtent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetWMFPixelPerInch, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetWordSpacing, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetXFAStream, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, GetXFAStreamCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, HaveOpenDoc, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, HaveOpenPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, HighlightAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportBookmarks, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportCatalogObjects, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportDocInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportEncryptionSettings, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportOCProperties, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportPageEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ImportPDFFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InitColorManagement, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InitStack, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InkAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertBarcode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertBMPFromBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertBMPFromHandle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertBookmark, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertBookmarkEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertImageEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertImageFromBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertMetafile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertMetafileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertMetafileExt, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertMetafileExtEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertRawImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, InsertRawImageEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, IsBidiText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, IsColorPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, IsEmptyPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LineAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LineTo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadCMap, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadFDFData, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadFDFDataEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadFontEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadHeaderFooterSettings, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LoadLayerConfig, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, LockLayer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, MovePage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, MoveTo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, MultiplyMatrix, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, OpenImportBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, OpenImportFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, OpenOutputFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, OpenOutputFileEncrypted, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, OpenTag, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, OpenTagBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Optimize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PageLink, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PageLink2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PageLink3, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PageLinkEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PlaceImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PlaceSigFieldValidateIcon, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PlaceTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PlaceTemplateEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PlaceTemplByMatrix, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PolygonAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PolyLineAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PrintPage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, PrintPDFFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReadImageFormat, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReadImageFormat2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReadImageFormatEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReadImageFormatFromBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReadImageResolution, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReadImageResolutionEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Rectangle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReEncryptPDF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RenameSpotColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RenderPageToImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RenderPDFFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RenderPDFFileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReOpenImportFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplaceFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplaceFontEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplaceICCProfile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplaceICCProfileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplaceImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplaceImageEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplacePageText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ReplacePageTextEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ResetAnnotAP, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ResetEncryptionSettings, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ResetLineDashPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RestoreGraphicState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RGB, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RotateCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RoundRect, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, RoundRectEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SaveGraphicState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, ScaleCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SelfTest, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Set3DAnnotProps, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Set3DAnnotScript, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAllocBy, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAltFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotBorderEffect, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotBorderStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotBorderWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotFlagsEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotHighlightMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotIcon, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotLineDashPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotLineEndStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotMigrationState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotOpenState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotOrFieldDate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotQuadPoints, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotString, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetAnnotSubject, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetBidiMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetBookmarkDest, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetBookmarkStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetBorderStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCharacterSpacing, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCheckBoxChar, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCheckBoxDefState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCheckBoxState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCIDFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCMapDir, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetColDefFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetColorMask, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetColors, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetColSortField, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCompressionFilter, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetCompressionLevel, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetContent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetDateTimeFormat, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetDefBitsPerPixel, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetDocInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetDocInfoEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetDrawDirection, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetEMFFrameDPI, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetEMFPatternDistance, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetErrorMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetExtColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetExtFillColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetExtGState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetExtStrokeColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldBackColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldBorderColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldBorderStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldBorderWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldCalcOrder, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldExpValue, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldExpValueEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldFontEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldFontSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldHighlightMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldIndex, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldMapName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldTextAlign, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldTextColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFieldToolTip, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFillColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFillColorEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFillColorF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFillColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFloatPrecision, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFontEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFontOrigin, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFontSearchOrder, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFontSearchOrderEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFontSelMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetFontWeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetGStateFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetIconColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetImportFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetImportFlags2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetItalicAngle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetJPEGQuality, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLanguage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLeading, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLicenseKey, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineAnnotParms, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineCapStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineDashPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineDashPattern2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineDashPatternEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineJoinStyle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLineWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetLinkHighlightMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetListFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetMatrix, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetMaxErrLogMsgCount, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetMaxFieldLen, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetMetaConvFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetMetadata, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetMiterLimit, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetNeedAppearance, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetNumberFormat, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOCGContUsage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOCGState, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOnErrorProc, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOnPageBreakProc, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOpacity, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetOrientationEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageFormat, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageLayout, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPageWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPDFVersion, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetPrintSettings, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetProgressProc, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetRenderingIntent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetResolution, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetSaveNewImageFormat, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetSeparationInfo, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetSpaceWidthFactor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetStrokeColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetStrokeColorEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetStrokeColorF, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetStrokeColorSpace, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTabLen, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTemplBBox, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTextDrawMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTextFieldValue, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTextFieldValueEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTextRect, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTextRise, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTextScaling, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTransparentColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetTrapped, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseExactPwd, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseGlobalImpFiles, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseImageInterpolation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseImageInterpolationEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUserUnit, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseStdFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseSwapFile, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseSwapFileEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseSystemFonts, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseTransparency, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetUseVisibleCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetViewerPreferences, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetWMFDefExtent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetWMFPixelPerInch, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetWordSpacing, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SetXFAStream, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SkewCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SortFieldsByIndex, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SortFieldsByName, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, SquareAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, StampAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, StrokePath, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, TestCallback, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, TestGlyphs, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, TestGlyphsEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, TextAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, TranslateCoords, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, TranslateString2, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, Triangle, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, UnLockLayer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WatermarkAnnot, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WebLink, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteAngleText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteFText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteFTextEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteImageBuffer, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynapdf, WriteTextMatrix, NULL, ZEND_ACC_PUBLIC)
   {NULL, NULL, NULL}
};

zend_module_entry dynapdf_module_entry =
{
#if ZEND_MODULE_API_NO >= 20010901
    STANDARD_MODULE_HEADER,
#endif
    "dynapdf",
    NULL,
    PHP_MINIT(dynapdf), /* module_startup_func */
    NULL,               /* MSHUTDOWN */
    NULL,               /* RINIT */
    NULL,               /* RSHUTDOWN */
    PHP_MINFO(dynapdf), /* MINFO */
#if ZEND_MODULE_API_NO >= 20010901
    DYNAPDF_VERSIONSTRING, /* version */
#endif
    STANDARD_MODULE_PROPERTIES
};





zend_object_handlers dynapdf_table_object_handlers;
zend_class_entry*    dynapdf_table_ce;

struct dynapdf_table_object
{
   zend_object zo;
   ITBL*       tbl;
};

#define DYNAPDF_FETCH_TABLE() \
{\
   struct dynapdf_table_object *obj = (struct dynapdf_table_object*)zend_object_store_get_object(getThis() TSRMLS_CC); \
   if (obj && obj->tbl) \
      tbl = obj->tbl; \
   else \
   { \
      tbl = NULL; \
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch Table instance!"); \
      return; \
   } \
}

// PDF_EXP ITBL*  PDF_CALL tblCreateTable(const void* IPDF, UI32 AllocRows, UI32 NumCols, float Width, float DefRowHeight);
PHP_METHOD(dynatbl, __construct)
{
   zval **argv[5];
   UI32 AllocRows;
   UI32 NumCols;
   float Width;
   float DefRowHeight;
   struct dynapdf_object *pdfobj;
   struct dynapdf_table_object* obj;

   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   if (Z_TYPE_PP(argv[0]) != IS_OBJECT || Z_OBJCE_PP(argv[0]) != dynapdf_ce)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter 1 must be a valid PDF instance!");
      return;
   }

   obj    = (struct dynapdf_table_object*)zend_object_store_get_object(getThis() TSRMLS_CC);
   pdfobj = (struct dynapdf_object*)zend_object_store_get_object(*argv[0] TSRMLS_CC);
   /*
      PHP increments the reference count if an object is used in another object. So, it is guaranteed that
      the PDF instance stays valid for the entire life time of the table object.
   */
   if (!pdfobj || !pdfobj->pdf)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch PDF instance!");
      return;
   }

   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);

   AllocRows    = Z_LVAL_PP(argv[1]);
   NumCols      = Z_LVAL_PP(argv[2]);
   Width        = (float)Z_DVAL_PP(argv[3]);
   DefRowHeight = (float)Z_DVAL_PP(argv[4]);

   if (obj)
   {
      obj->tbl = tblCreateTable(pdfobj->pdf, AllocRows, NumCols, Width, DefRowHeight);
      if (!obj->tbl) php_error_docref(NULL TSRMLS_CC, E_ERROR, "Out of memory!");
   }else
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Out of memory!");
}

static void dynapdf_table_objects_dtor(void *object, zend_object_handle handle TSRMLS_DC)
{
   zend_objects_destroy_object(object, handle TSRMLS_CC);
}

static void dynapdf_table_free_storage(void* object TSRMLS_DC)
{
   struct dynapdf_table_object *obj = (struct dynapdf_table_object*)object;

   tblDeleteTable(&obj->tbl);
   zend_object_std_dtor(&obj->zo TSRMLS_CC);
   efree(obj);
}

static zend_object_value dynapdf_table_create_handler(zend_class_entry *ce TSRMLS_DC)
{
   struct dynapdf_table_object *obj;
   zend_object_value retval;
   obj = (struct dynapdf_table_object*)ecalloc(1, sizeof(struct dynapdf_table_object));

   zend_object_std_init(&obj->zo, ce TSRMLS_CC);
#if PHP_VERSION_ID < 50399
    zend_hash_copy(obj->zo.properties, &ce->default_properties, (copy_ctor_func_t)zval_add_ref, NULL, sizeof(zval*));
#else
    object_properties_init((zend_object*)obj, ce);
#endif

   retval.handle   = zend_objects_store_put(obj, dynapdf_table_objects_dtor, dynapdf_table_free_storage, NULL TSRMLS_CC);
   retval.handlers = &dynapdf_table_object_handlers;

   return retval;
}

// PDF_EXP LBOOL  PDF_CALL tblAddColumn(ITBL* Table, LBOOL Left, float Width);
PHP_METHOD(dynatbl, AddColumn)
{
   zval **argv[2];
   ITBL* tbl;
   LBOOL Left;
   float Width;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   Left   = Z_LVAL_PP(argv[0]);
   Width  = (float)Z_DVAL_PP(argv[1]);

   rc = tblAddColumn(tbl, Left, Width);
   RETURN_BOOL(rc);
}

//PDF_EXP SI32   PDF_CALL tblAddRow(const ITBL* Table, float Height);
PHP_METHOD(dynatbl, AddRow)
{
   zval **arg1;
   ITBL* tbl;
   float Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_double_ex(arg1);
   Height = (float)Z_DVAL_PP(arg1);

   rc = tblAddRow(tbl, Height);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblAddRows(ITBL* Table, UI32 Count, float Height);
PHP_METHOD(dynatbl, AddRows)
{
   zval **argv[2];
   ITBL* tbl;
   SI32 Count;
   float Height;
   SI32 rc;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   Count   = Z_LVAL_PP(argv[0]);
   Height  = (float)Z_DVAL_PP(argv[1]);

   rc = tblAddRows(tbl, Count, Height);
   RETURN_BOOL(rc);
}

//PDF_EXP void   PDF_CALL tblClearColumn(const ITBL* Table, UI32 Col, TDeleteContent Types);
PHP_METHOD(dynatbl, ClearColumn)
{
   zval **argv[2];
   ITBL* tbl;
   SI32 Col, Types;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Col   = Z_LVAL_PP(argv[0]);
   Types = Z_LVAL_PP(argv[1]);

   tblClearColumn(tbl, Col, Types);
}

//PDF_EXP void   PDF_CALL tblClearContent(ITBL* Table, TDeleteContent Types);
PHP_METHOD(dynatbl, ClearContent)
{
   zval **arg1;
   ITBL* tbl;
   SI32 Types;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(arg1);
   Types = Z_LVAL_PP(arg1);

   tblClearContent(tbl, Types);
}

//PDF_EXP void   PDF_CALL tblClearRow(const ITBL* Table, UI32 Row, TDeleteContent Types);
PHP_METHOD(dynatbl, ClearRow)
{
   zval **argv[2];
   ITBL* tbl;
   SI32 Row, Types;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Row   = Z_LVAL_PP(argv[0]);
   Types = Z_LVAL_PP(argv[1]);

   tblClearRow(tbl, Row, Types);
}

//PDF_EXP void   PDF_CALL tblDeleteCol(ITBL* Table, UI32 Col);
PHP_METHOD(dynatbl, DeleteCol)
{
   zval **arg1;
   ITBL* tbl;
   SI32 Col;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(arg1);
   Col = Z_LVAL_PP(arg1);

   tblDeleteCol(tbl, Col);
}

//PDF_EXP void   PDF_CALL tblDeleteRow(ITBL* Table, UI32 Row);
PHP_METHOD(dynatbl, DeleteRow)
{
   zval **arg1;
   ITBL* tbl;
   SI32 Row;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(arg1);
   Row = Z_LVAL_PP(arg1);

   tblDeleteRow(tbl, Row);
}

//PDF_EXP void   PDF_CALL tblDeleteRows(ITBL* Table);
PHP_METHOD(dynatbl, DeleteRows)
{
   ITBL* tbl;
   DYNAPDF_FETCH_TABLE();
   tblDeleteRows(tbl);
}

//PDF_EXP float  PDF_CALL tblDrawTable(ITBL* Table, float x, float y, float MaxHeight);
PHP_METHOD(dynatbl, DrawTable)
{
   zval **argv[3];
   ITBL* tbl;
   float rc, x, y, maxHeight = 0.0f;
   int argc = ZEND_NUM_ARGS();

   if(argc < 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   x = (float)Z_DVAL_PP(argv[0]);
   y = (float)Z_DVAL_PP(argv[1]);
   if (argc == 3)
   {
      convert_to_double_ex(argv[2]);
      maxHeight = (float)Z_DVAL_PP(argv[2]);
   }
   rc = tblDrawTable(tbl, x, y, maxHeight);
   RETURN_DOUBLE(rc);
}

//PDF_EXP SI32   PDF_CALL tblGetFirstRow(ITBL* Table);
PHP_METHOD(dynatbl, GetFirstRow)
{
   ITBL* tbl;
   SI32 rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblGetFirstRow(tbl);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL tblGetFlags(const ITBL* Table, SI32 Row, SI32 Col);
PHP_METHOD(dynatbl, GetFlags)
{
   zval **argv[2];
   ITBL* tbl;
   SI32 rc, Row, Col;
   int argc = ZEND_NUM_ARGS();

   if(argc < 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   Row = Z_LVAL_PP(argv[0]);
   Col = Z_LVAL_PP(argv[1]);

   DYNAPDF_FETCH_TABLE();
   rc = tblGetFlags(tbl, Row, Col);
   RETURN_LONG(rc);
}

//PDF_EXP float  PDF_CALL tblGetNextHeight(ITBL* Table, float MaxHeight, SI32* NextRow);
PHP_METHOD(dynatbl, GetNextHeight)
{
   ITBL* tbl;
   zval **arg1;
   SI32 NextRow = 0;
   float rc, MaxHeight;
   int argc = ZEND_NUM_ARGS();

   if(argc < 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }
   convert_to_double_ex(arg1);
   MaxHeight = (float)Z_DVAL_PP(arg1);

   DYNAPDF_FETCH_TABLE();
   array_init(return_value);
   rc = tblGetNextHeight(tbl, MaxHeight, &NextRow);

   add_assoc_double(return_value, "result", rc);
   add_assoc_long(return_value, "NextRow", NextRow);
}

//PDF_EXP SI32   PDF_CALL tblGetNextRow(ITBL* Table);
PHP_METHOD(dynatbl, GetNextRow)
{
   ITBL* tbl;
   SI32 rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblGetNextRow(tbl);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL tblGetNumCols(ITBL* Table);
PHP_METHOD(dynatbl, GetNumCols)
{
   ITBL* tbl;
   SI32 rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblGetNumCols(tbl);
   RETURN_LONG(rc);
}

//PDF_EXP SI32   PDF_CALL tblGetNumRows(ITBL* Table);
PHP_METHOD(dynatbl, GetNumRows)
{
   ITBL* tbl;
   SI32 rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblGetNumRows(tbl);
   RETURN_LONG(rc);
}

//PDF_EXP float  PDF_CALL tblGetTableHeight(ITBL* Table);
PHP_METHOD(dynatbl, GetTableHeight)
{
   ITBL* tbl;
   float rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblGetTableHeight(tbl);
   RETURN_DOUBLE(rc);
}

//PDF_EXP float  PDF_CALL tblGetTableWidth(ITBL* Table);
PHP_METHOD(dynatbl, GetTableWidth)
{
   ITBL* tbl;
   float rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblGetTableWidth(tbl);
   RETURN_DOUBLE(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblHaveMore(ITBL* Table);
PHP_METHOD(dynatbl, HaveMore)
{
   ITBL* tbl;
   SI32 rc;
   DYNAPDF_FETCH_TABLE();
   rc = tblHaveMore(tbl);
   RETURN_BOOL(rc != 0);
}

//PDF_EXP LBOOL  PDF_CALL tblSetBoxProperty(ITBL* Table, SI32 Row, SI32 Col, TTableBoxProperty Type, float Left, float Right, float Top, float Bottom);
PHP_METHOD(dynatbl, SetBoxProperty)
{
   zval **argv[7];
   ITBL* tbl;
   SI32 rc, Row, Col, Type;
   float Left, Right, Top, Bottom;
   int argc = ZEND_NUM_ARGS();

   if(argc != 7 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_double_ex(argv[3]);
   convert_to_double_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);

   Row    = Z_LVAL_PP(argv[0]);
   Col    = Z_LVAL_PP(argv[1]);
   Type   = Z_LVAL_PP(argv[2]);
   Left   = (float)Z_DVAL_PP(argv[3]);
   Right  = (float)Z_DVAL_PP(argv[4]);
   Top    = (float)Z_DVAL_PP(argv[5]);
   Bottom = (float)Z_DVAL_PP(argv[6]);

   rc = tblSetBoxProperty(tbl, Row, Col, Type, Left, Right, Top, Bottom);
   RETURN_LONG(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellAction(const ITBL* Table, SI32 Row, SI32 Col, UI32 ActionHandle, THighlightMode Mode);
PHP_METHOD(dynatbl, SetCellAction)
{
   zval **argv[4];
   ITBL* tbl;
   SI32 rc, Row, Col, ActionHandle, Mode;

   int argc = ZEND_NUM_ARGS();

   if(argc != 4 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);

   Row          = Z_LVAL_PP(argv[0]);
   Col          = Z_LVAL_PP(argv[1]);
   ActionHandle = Z_LVAL_PP(argv[2]);
   Mode         = Z_LVAL_PP(argv[3]);

   rc = tblSetCellAction(tbl, Row, Col, ActionHandle, Mode);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellDashPattern(const ITBL* Table, SI32 Row, SI32 Col, TCellSide Side, TLineCapStyle Style, const float* Dash, UI32 NumValues, float Phase);
PHP_METHOD(dynatbl, SetCellDashPattern)
{
   zval **argv[6];
   ITBL* tbl;
   HashTable* hash;
   float Dash[64], Phase;
   SI32 rc, Row, Col, Side, Style, NumValues;

   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_array_ex(argv[4]);
   convert_to_double_ex(argv[5]);

   Row   = Z_LVAL_PP(argv[0]);
   Col   = Z_LVAL_PP(argv[1]);
   Side  = Z_LVAL_PP(argv[2]);
   Style = Z_LVAL_PP(argv[3]);
   hash  = Z_ARRVAL_PP(argv[4]);
   Phase = (float)Z_DVAL_PP(argv[5]);

   NumValues = GetFloatArray(hash, Dash, 64);

   rc = tblSetCellDashPattern(tbl, Row, Col, Side, Style, Dash, NumValues, Phase);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellImageA(ITBL* Table, SI32 Row, SI32 Col, LBOOL ForeGround, TCellOrientation HAlign, TCellOrientation VAlign, float Width, float Height, const char* Image, UI32 Index);
PHP_METHOD(dynatbl, SetCellImage)
{
   zval **argv[9];
   ITBL* tbl;
   SI32 rc, Row, Col, ForeGround, HAlign, VAlign, Index;
   float Width, Height;
   const char* Image;
   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_string_ex(argv[7]);
   convert_to_long_ex(argv[8]);

   Row        = Z_LVAL_PP(argv[0]);
   Col        = Z_LVAL_PP(argv[1]);
   ForeGround = Z_LVAL_PP(argv[2]);
   HAlign     = Z_LVAL_PP(argv[3]);
   VAlign     = Z_LVAL_PP(argv[4]);
   Width      = (float)Z_DVAL_PP(argv[5]);
   Height     = (float)Z_DVAL_PP(argv[6]);
   Image      = Z_STRVAL_PP(argv[7]);
   Index      = Z_LVAL_PP(argv[8]);
   rc = tblSetCellImageA(tbl, Row, Col, ForeGround, HAlign, VAlign, Width, Height, Image, Index);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellImageEx(ITBL* Table, SI32 Row, SI32 Col, LBOOL ForeGround, TCellOrientation HAlign, TCellOrientation VAlign, float Width, float Height, const void* Buffer, UI32 BufSize, UI32 Index);
PHP_METHOD(dynatbl, SetCellImageEx)
{
   zval **argv[9];
   ITBL* tbl;
   SI32 rc, Row, Col, ForeGround, HAlign, VAlign, BufSize, Index;
   float Width, Height;
   const char* Buffer;
   int argc = ZEND_NUM_ARGS();

   if(argc != 9 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_double_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_string_ex(argv[7]);
   convert_to_long_ex(argv[8]);

   Row        = Z_LVAL_PP(argv[0]);
   Col        = Z_LVAL_PP(argv[1]);
   ForeGround = Z_LVAL_PP(argv[2]);
   HAlign     = Z_LVAL_PP(argv[3]);
   VAlign     = Z_LVAL_PP(argv[4]);
   Width      = (float)Z_DVAL_PP(argv[5]);
   Height     = (float)Z_DVAL_PP(argv[6]);
   Buffer     = Z_STRVAL_PP(argv[7]);
   BufSize    = Z_STRLEN_PP(argv[7]);
   Index      = Z_LVAL_PP(argv[8]);
   rc = tblSetCellImageEx(tbl, Row, Col, ForeGround, HAlign, VAlign, Width, Height, Buffer, BufSize, Index);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellOrientation(const ITBL* Table, SI32 Row, SI32 Col, SI32 Orientation);
PHP_METHOD(dynatbl, SetCellOrientation)
{
   zval **argv[3];
   ITBL* tbl;
   SI32 rc, Row, Col, Orientation;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Row         = Z_LVAL_PP(argv[0]);
   Col         = Z_LVAL_PP(argv[1]);
   Orientation = Z_LVAL_PP(argv[2]);

   rc = tblSetCellOrientation(tbl, Row, Col, Orientation);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellTable(const ITBL* Table, UI32 Row, UI32 Col, TCellAlign HAlign, TCellAlign VAlign, const ITBL* SubTable);
PHP_METHOD(dynatbl, SetCellTable)
{
   zval **argv[5];
   ITBL* tbl, *SubTable = NULL;
   SI32 rc, Row, Col, HAlign, VAlign;
   struct dynapdf_table_object* obj;

   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   if (Z_TYPE_PP(argv[4]) != IS_NULL)
   {
      if (Z_TYPE_PP(argv[4]) != IS_OBJECT || Z_OBJCE_PP(argv[4]) != dynapdf_table_ce)
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter 1 must be a valid dynatable instance!");
         return;
      }
      obj = (struct dynapdf_table_object*)zend_object_store_get_object(*argv[4] TSRMLS_CC);
      if (!obj || !obj->tbl)
      {
         php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch table instance!");
         return;
      }
      SubTable = obj->tbl;
   }
   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   Row    = Z_LVAL_PP(argv[0]);
   Col    = Z_LVAL_PP(argv[1]);
   HAlign = Z_LVAL_PP(argv[2]);
   VAlign = Z_LVAL_PP(argv[3]);

   rc = tblSetCellTable(tbl, Row, Col, HAlign, VAlign, SubTable);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellTemplate(const ITBL* Table, SI32 Row, SI32 Col, LBOOL ForeGround, TCellAlign HAlign, TCellAlign VAlign, UI32 TmplHandle, float Width, float Height);
PHP_METHOD(dynatbl, SetCellTemplate)
{
   zval **argv[8];
   ITBL* tbl;
   SI32 rc, Row, Col, ForeGround, HAlign, VAlign, TmplHandle;
   float Width, Height;
   int argc = ZEND_NUM_ARGS();

   if(argc != 8 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);
   convert_to_double_ex(argv[6]);
   convert_to_double_ex(argv[7]);

   Row        = Z_LVAL_PP(argv[0]);
   Col        = Z_LVAL_PP(argv[1]);
   ForeGround = Z_LVAL_PP(argv[2]);
   HAlign     = Z_LVAL_PP(argv[3]);
   VAlign     = Z_LVAL_PP(argv[4]);
   TmplHandle = Z_LVAL_PP(argv[5]);
   Width      = (float)Z_DVAL_PP(argv[6]);
   Height     = (float)Z_DVAL_PP(argv[7]);
   rc = tblSetCellTemplate(tbl, Row, Col, ForeGround, HAlign, VAlign, TmplHandle, Width, Height);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetCellTextA(const ITBL* Table, UI32 Row, UI32 Col, TTextAlign HAlign, TCellAlign VAlign, const char* Text, UI32 Len);
PHP_METHOD(dynatbl, SetCellText)
{
   zval **argv[5];
   ITBL* tbl;
   SI32 rc, Row, Col, HAlign, VAlign, Len;
   const char* Text;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_string_ex(argv[4]);

   Row    = Z_LVAL_PP(argv[0]);
   Col    = Z_LVAL_PP(argv[1]);
   HAlign = Z_LVAL_PP(argv[2]);
   VAlign = Z_LVAL_PP(argv[3]);
   Text   = Z_STRVAL_PP(argv[4]);
   Len    = Z_STRLEN_PP(argv[4]);
   rc = tblSetCellTextA(tbl, Row, Col, HAlign, VAlign, Text, Len);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetColor(ITBL* Table, SI32 Row, SI32 Col, TTableProperty Type, TPDFColorSpace CS, UI32 Color);
PHP_METHOD(dynatbl, SetColor)
{
   zval **argv[5];
   ITBL* tbl;
   SI32 rc, Row, Col, Type, CS, Color;
   int argc = ZEND_NUM_ARGS();

   if(argc != 5 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);

   Row   = Z_LVAL_PP(argv[0]);
   Col   = Z_LVAL_PP(argv[1]);
   Type  = Z_LVAL_PP(argv[2]);
   CS    = Z_LVAL_PP(argv[3]);
   Color = Z_LVAL_PP(argv[4]);

   rc = tblSetColor(tbl, Row, Col, Type, CS, Color);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetColorEx(const ITBL* Table, SI32 Row, SI32 Col, TTableColor Type, float* Color, UI32 NumComps, TExtColorSpace CS, SI32 Handle);
PHP_METHOD(dynatbl, SetColorEx)
{
   zval **argv[6];
   ITBL* tbl;
   float color[32];
   HashTable *array;
   SI32 rc, Row, Col, Type, CS, Handle, NumComps;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);
   convert_to_array_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   Row    = Z_LVAL_PP(argv[0]);
   Col    = Z_LVAL_PP(argv[1]);
   Type   = Z_LVAL_PP(argv[2]);
   array  = Z_ARRVAL_PP(argv[3]);
   CS     = Z_LVAL_PP(argv[4]);
   Handle = Z_LVAL_PP(argv[5]);

   NumComps = GetFloatArray(array, color, 32);
   if (NumComps < 1 || NumComps > 32) WRONG_PARAM_COUNT;

   rc = tblSetColorEx(tbl, Row, Col, Type, color, NumComps, CS, Handle);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetColWidth(ITBL* Table, UI32 Col, float Width, LBOOL ExtTable);
PHP_METHOD(dynatbl, SetColWidth)
{
   zval **argv[3];
   ITBL* tbl;
   SI32 rc, Col, ExtTable;
   float Width;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Col      = Z_LVAL_PP(argv[0]);
   Width    = (float)Z_DVAL_PP(argv[1]);
   ExtTable = Z_LVAL_PP(argv[2]);
   rc = tblSetColWidth(tbl, Col, Width, ExtTable);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetFlags(ITBL* Table, SI32 Row, SI32 Col, TTableFlags Flags);
PHP_METHOD(dynatbl, SetFlags)
{
   zval **argv[3];
   ITBL* tbl;
   SI32 rc, Col, Row, Flags;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Row   = Z_LVAL_PP(argv[0]);
   Col   = Z_LVAL_PP(argv[1]);
   Flags = Z_LVAL_PP(argv[2]);
   rc = tblSetFlags(tbl, Row, Col, Flags);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetFontA(ITBL* Table, SI32 Row, SI32 Col, const char* Name, TFStyle Style, LBOOL Embed, TCodepage CP);
PHP_METHOD(dynatbl, SetFont)
{
   zval **argv[6];
   ITBL* tbl;
   SI32 rc, Row, Col, Style, Embed, CP;
   const char* Name;
   int argc = ZEND_NUM_ARGS();

   if(argc != 6 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_string_ex(argv[2]);
   convert_to_long_ex(argv[3]);
   convert_to_long_ex(argv[4]);
   convert_to_long_ex(argv[5]);

   Row   = Z_LVAL_PP(argv[0]);
   Col   = Z_LVAL_PP(argv[1]);
   Name  = Z_STRVAL_PP(argv[2]);
   Style = Z_LVAL_PP(argv[3]);
   Embed = Z_LVAL_PP(argv[4]);
   CP    = Z_LVAL_PP(argv[5]);
   rc = tblSetFontA(tbl, Row, Col, Name, Style, Embed, CP);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetFontSelMode(ITBL* Table, SI32 Row, SI32 Col, TFontSelMode Mode);
PHP_METHOD(dynatbl, SetFontSelMode)
{
   zval **argv[3];
   ITBL* tbl;
   SI32 rc, Col, Row, Mode;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_long_ex(argv[2]);

   Row  = Z_LVAL_PP(argv[0]);
   Col  = Z_LVAL_PP(argv[1]);
   Mode = Z_LVAL_PP(argv[2]);
   rc = tblSetFontSelMode(tbl, Row, Col, Mode);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetFontSize(ITBL* Table, SI32 Row, SI32 Col, float Value);
PHP_METHOD(dynatbl, SetFontSize)
{
   zval **argv[3];
   ITBL* tbl;
   SI32 rc, Col, Row;
   float Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);

   Row   = Z_LVAL_PP(argv[0]);
   Col   = Z_LVAL_PP(argv[1]);
   Value = (float)Z_DVAL_PP(argv[2]);
   rc = tblSetFontSize(tbl, Row, Col, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP LBOOL  PDF_CALL tblSetGridWidth(ITBL* Table, float Horz, float Vert);
PHP_METHOD(dynatbl, SetGridWidth)
{
   zval **argv[2];
   ITBL* tbl;
   SI32 rc;
   float Horz, Vert;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_double_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   Horz = (float)Z_DVAL_PP(argv[0]);
   Vert = (float)Z_DVAL_PP(argv[1]);
   rc = tblSetGridWidth(tbl, Horz, Vert);
   RETURN_BOOL(rc);
}

//PDF_EXP void   PDF_CALL tblSetPDFInstance(const ITBL* Table, const void* IPDF);
PHP_METHOD(dynatbl, SetPDFInstance)
{
   zval **arg1;
   ITBL* tbl;
   struct dynapdf_object *pdfobj;
   int argc = ZEND_NUM_ARGS();

   if(argc != 1 || (zend_get_parameters_array_ex(argc, &arg1) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   if (Z_TYPE_PP(arg1) != IS_OBJECT || Z_OBJCE_PP(arg1) != dynapdf_ce)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Parameter 1 must be a valid PDF instance!");
      return;
   }

   DYNAPDF_FETCH_TABLE();

   pdfobj = (struct dynapdf_object*)zend_object_store_get_object(*arg1 TSRMLS_CC);
   if (!pdfobj || !pdfobj->pdf)
   {
      php_error_docref(NULL TSRMLS_CC, E_ERROR, "Cannot fetch PDF instance!");
      return;
   }
   tblSetPDFInstance(tbl, pdfobj->pdf);
}

//PDF_EXP LBOOL  PDF_CALL tblSetRowHeight(ITBL* Table, SI32 Row, float Value);
PHP_METHOD(dynatbl, SetRowHeight)
{
   zval **argv[2];
   ITBL* tbl;
   SI32 rc, Row;
   float Value;
   int argc = ZEND_NUM_ARGS();

   if(argc != 2 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_long_ex(argv[0]);
   convert_to_double_ex(argv[1]);

   Row   = Z_LVAL_PP(argv[0]);
   Value = (float)Z_DVAL_PP(argv[1]);
   rc = tblSetRowHeight(tbl, Row, Value);
   RETURN_BOOL(rc);
}

//PDF_EXP void   PDF_CALL tblSetTableWidth(const ITBL* Table, float Value, TColumnAdjust AdjustType, float MinColWidth);
PHP_METHOD(dynatbl, SetTableWidth)
{
   zval **argv[3];
   ITBL* tbl;
   SI32 AdjustType;
   float Value, MinColWidth;
   int argc = ZEND_NUM_ARGS();

   if(argc != 3 || (zend_get_parameters_array_ex(argc, argv) == FAILURE))
   {
      WRONG_PARAM_COUNT;
   }

   DYNAPDF_FETCH_TABLE();

   convert_to_double_ex(argv[0]);
   convert_to_long_ex(argv[1]);
   convert_to_double_ex(argv[2]);

   Value       = (float)Z_DVAL_PP(argv[0]);
   AdjustType  = Z_LVAL_PP(argv[1]);
   MinColWidth = (float)Z_DVAL_PP(argv[2]);

   tblSetTableWidth(tbl, Value, AdjustType, MinColWidth);
}

zend_function_entry dynapdf_table_functions[] =
{
   PHP_ME(dynatbl, __construct, NULL, ZEND_ACC_PUBLIC | ZEND_ACC_CTOR)
   PHP_ME(dynatbl, AddColumn, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, AddRow, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, AddRows, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, ClearColumn, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, ClearContent, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, ClearRow, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, DeleteCol, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, DeleteRow, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, DeleteRows, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, DrawTable, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetFirstRow, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetNextHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetNextRow, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetNumCols, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetNumRows, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetTableHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, GetTableWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, HaveMore, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetBoxProperty, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellAction, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellDashPattern, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellImage, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellImageEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellOrientation, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellTable, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellTemplate, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetCellText, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetColor, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetColorEx, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetColWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetFlags, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetFont, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetFontSelMode, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetFontSize, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetGridWidth, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetRowHeight, NULL, ZEND_ACC_PUBLIC)
   PHP_ME(dynatbl, SetTableWidth, NULL, ZEND_ACC_PUBLIC)
   {NULL, NULL, NULL}
};






#ifdef COMPILE_DL_DYNAPDF
ZEND_GET_MODULE(dynapdf)
#endif

#define REGISTER_CLASS_CONST_LONG(const_name, value) zend_declare_class_constant_long(dynapdf_ce, (const_name), sizeof((const_name))-1, (value) TSRMLS_CC)

PHP_MINIT_FUNCTION(dynapdf)
{
   zend_class_entry ce;
   INIT_CLASS_ENTRY(ce, "dynapdf", dynapdf_functions);
   dynapdf_ce = zend_register_internal_class(&ce TSRMLS_CC);
   dynapdf_ce->create_object = dynapdf_create_handler;
   memcpy(&dynapdf_object_handlers, zend_get_std_object_handlers(), sizeof(zend_object_handlers));

   INIT_CLASS_ENTRY(ce, "dynatbl", dynapdf_table_functions);
   dynapdf_table_ce = zend_register_internal_class(&ce TSRMLS_CC);
   dynapdf_table_ce->create_object = dynapdf_table_create_handler;
   memcpy(&dynapdf_table_object_handlers, zend_get_std_object_handlers(), sizeof(zend_object_handlers));

   REGISTER_CLASS_CONST_LONG("cbfAll",                     cbfAll);
   REGISTER_CLASS_CONST_LONG("cbfOnErrorProc",             cbfOnErrorProc);
   REGISTER_CLASS_CONST_LONG("cbfOnFontNotFoundProc",      cbfOnFontNotFoundProc);
   REGISTER_CLASS_CONST_LONG("cbfOnInitProgressProc",      cbfOnInitProgressProc);
   REGISTER_CLASS_CONST_LONG("cbfOnPageBreakProc",         cbfOnPageBreakProc);
   REGISTER_CLASS_CONST_LONG("cbfOnProgressProc",          cbfOnProgressProc);
   REGISTER_CLASS_CONST_LONG("cbfonReplaceICCProfileProc", cbfonReplaceICCProfileProc);

   REGISTER_CLASS_CONST_LONG("TRANSP_3D_ANNOT",            TRANSP_3D_ANNOT);

   REGISTER_CLASS_CONST_LONG("adAnnotation",               adAnnotation);
   REGISTER_CLASS_CONST_LONG("adCatalog",                  adCatalog);
   REGISTER_CLASS_CONST_LONG("adField",                    adField);
   REGISTER_CLASS_CONST_LONG("adImage",                    adImage);
   REGISTER_CLASS_CONST_LONG("adPage",                     adPage);
   REGISTER_CLASS_CONST_LONG("adTemplate",                 adTemplate);

   REGISTER_CLASS_CONST_LONG("aeExport",                   aeExport);
   REGISTER_CLASS_CONST_LONG("aePrint",                    aePrint);
   REGISTER_CLASS_CONST_LONG("aeView",                     aeView);

   REGISTER_CLASS_CONST_LONG("arAlternative",              arAlternative);
   REGISTER_CLASS_CONST_LONG("arAssociated",               arAssociated);
   REGISTER_CLASS_CONST_LONG("arData",                     arData);
   REGISTER_CLASS_CONST_LONG("arSource",                   arSource);
   REGISTER_CLASS_CONST_LONG("arSupplement",               arSupplement);

   REGISTER_CLASS_CONST_LONG("asNone",                     asNone);
   REGISTER_CLASS_CONST_LONG("asAccepted",                 asAccepted);
   REGISTER_CLASS_CONST_LONG("asRejected",                 asRejected);
   REGISTER_CLASS_CONST_LONG("asCancelled",                asCancelled);
   REGISTER_CLASS_CONST_LONG("asCompleted",                asCompleted);
   REGISTER_CLASS_CONST_LONG("asCreateReply",              asCreateReply);

   REGISTER_CLASS_CONST_LONG("at3D_AppDefault",            at3D_AppDefault);
   REGISTER_CLASS_CONST_LONG("at3D_PageOpen",              at3D_PageOpen);
   REGISTER_CLASS_CONST_LONG("at3D_PageVisible",           at3D_PageVisible);
   REGISTER_CLASS_CONST_LONG("at3D_Explicit",              at3D_Explicit);

   REGISTER_CLASS_CONST_LONG("dt3D_AppDefault",            dt3D_AppDefault);
   REGISTER_CLASS_CONST_LONG("dt3D_PageClosed",            dt3D_PageClosed);
   REGISTER_CLASS_CONST_LONG("dt3D_PageInvisible",         dt3D_PageInvisible);
   REGISTER_CLASS_CONST_LONG("dt3D_Explicit",              dt3D_Explicit);

   REGISTER_CLASS_CONST_LONG("di3D_AppDefault",            di3D_AppDefault);
   REGISTER_CLASS_CONST_LONG("di3D_UnInstantiated",        di3D_UnInstantiated);
   REGISTER_CLASS_CONST_LONG("di3D_Instantiated",          di3D_Instantiated);
   REGISTER_CLASS_CONST_LONG("di3D_Live",                  di3D_Live);

   REGISTER_CLASS_CONST_LONG("it3D_AppDefault",            it3D_AppDefault);
   REGISTER_CLASS_CONST_LONG("it3D_Instantiated",          it3D_Instantiated);
   REGISTER_CLASS_CONST_LONG("it3D_Live",                  it3D_Live);

   REGISTER_CLASS_CONST_LONG("lsArtwork",                  lsArtwork);
   REGISTER_CLASS_CONST_LONG("lsBlue",                     lsBlue);
   REGISTER_CLASS_CONST_LONG("lsCAD",                      lsCAD);
   REGISTER_CLASS_CONST_LONG("lsCube",                     lsCube);
   REGISTER_CLASS_CONST_LONG("lsDay",                      lsDay);
   REGISTER_CLASS_CONST_LONG("lsHard",                     lsHard);
   REGISTER_CLASS_CONST_LONG("lsHeadlamp",                 lsHeadlamp);
   REGISTER_CLASS_CONST_LONG("lsNight",                    lsNight);
   REGISTER_CLASS_CONST_LONG("lsNoLights",                 lsNoLights);
   REGISTER_CLASS_CONST_LONG("lsPrimary",                  lsPrimary);
   REGISTER_CLASS_CONST_LONG("lsRed",                      lsRed);
   REGISTER_CLASS_CONST_LONG("lsWhite",                    lsWhite);
   REGISTER_CLASS_CONST_LONG("lsNotSet",                   lsNotSet);

   REGISTER_CLASS_CONST_LONG("miNotTagged",                miNotTagged);
   REGISTER_CLASS_CONST_LONG("miTagged",                   miTagged);
   REGISTER_CLASS_CONST_LONG("miSuspects",                 miSuspects);
   REGISTER_CLASS_CONST_LONG("miUserProperties",           miUserProperties);

   REGISTER_CLASS_CONST_LONG("naDefault",                  naDefault);
   REGISTER_CLASS_CONST_LONG("naFirst",                    naFirst);
   REGISTER_CLASS_CONST_LONG("naLast",                     naLast);
   REGISTER_CLASS_CONST_LONG("naNext",                     naNext);
   REGISTER_CLASS_CONST_LONG("naPrevious",                 naPrevious);

   REGISTER_CLASS_CONST_LONG("pt3DOrthographic",           pt3DOrthographic);
   REGISTER_CLASS_CONST_LONG("pt3DPerspective",            pt3DPerspective);

   REGISTER_CLASS_CONST_LONG("coAllowDeviceSpaces",        coAllowDeviceSpaces);
   REGISTER_CLASS_CONST_LONG("coApplyExportState",         coApplyExportState);
   REGISTER_CLASS_CONST_LONG("coApplyPrintState",          coApplyPrintState);
   REGISTER_CLASS_CONST_LONG("coCheckImages",              coCheckImages);
   REGISTER_CLASS_CONST_LONG("coDefault",                  coDefault);
   REGISTER_CLASS_CONST_LONG("coDefault_Normalize",        coDefault_Normalize);
   REGISTER_CLASS_CONST_LONG("coDefault_PDFA_1",           coDefault_PDFA_1);
   REGISTER_CLASS_CONST_LONG("coDefault_PDFA_2",           coDefault_PDFA_2);
   REGISTER_CLASS_CONST_LONG("coDefault_PDFA_3",           coDefault_PDFA_3);
   REGISTER_CLASS_CONST_LONG("coDeleteActionsAndScripts",  coDeleteActionsAndScripts);
   REGISTER_CLASS_CONST_LONG("coDeleteAlternateImages",    coDeleteAlternateImages);
   REGISTER_CLASS_CONST_LONG("coDeleteAppEvents",          coDeleteAppEvents);
   REGISTER_CLASS_CONST_LONG("coDeleteDamagedImages",      coDeleteDamagedImages);
   REGISTER_CLASS_CONST_LONG("coDeleteEmbeddedFiles",      coDeleteEmbeddedFiles);
   REGISTER_CLASS_CONST_LONG("coDeleteHalftones",          coDeleteHalftones);
   REGISTER_CLASS_CONST_LONG("coDeleteInvRenderingIntent", coDeleteInvRenderingIntent);
   REGISTER_CLASS_CONST_LONG("coDeleteMultiMediaContents", coDeleteMultiMediaContents);
   REGISTER_CLASS_CONST_LONG("coDeleteOPIComments",        coDeleteOPIComments);
   REGISTER_CLASS_CONST_LONG("coDeletePostscript",         coDeletePostscript);
   REGISTER_CLASS_CONST_LONG("coDeletePresentation",       coDeletePresentation);
   REGISTER_CLASS_CONST_LONG("coDeleteReplies",            coDeleteReplies);
   REGISTER_CLASS_CONST_LONG("coDeleteSignatures",         coDeleteSignatures);
   REGISTER_CLASS_CONST_LONG("coDeleteTransferFuncs",      coDeleteTransferFuncs);
   REGISTER_CLASS_CONST_LONG("coEmbedSubsets",             coEmbedSubsets);
   REGISTER_CLASS_CONST_LONG("coFlattenFormFields",        coFlattenFormFields);
   REGISTER_CLASS_CONST_LONG("coFlattenLayers",            coFlattenLayers);
   REGISTER_CLASS_CONST_LONG("coFlushPages",               coFlushPages);
   REGISTER_CLASS_CONST_LONG("coMakeLayerVisible",         coMakeLayerVisible);
   REGISTER_CLASS_CONST_LONG("coNoFontEmbedding",          coNoFontEmbedding);
   REGISTER_CLASS_CONST_LONG("coReComprJPEG2000Images",    coReComprJPEG2000Images);
   REGISTER_CLASS_CONST_LONG("coRepairDamagedImages",      coRepairDamagedImages);
   REGISTER_CLASS_CONST_LONG("coReplaceV4ICCProfiles",     coReplaceV4ICCProfiles);
   REGISTER_CLASS_CONST_LONG("coReplCCITTFaxWithFlate",    coReplCCITTFaxWithFlate);
   REGISTER_CLASS_CONST_LONG("coResetAnnotAppearance",     coResetAnnotAppearance);
   REGISTER_CLASS_CONST_LONG("coResolveOverprint",         coResolveOverprint);
   

   REGISTER_CLASS_CONST_LONG("cisCreationDate",            cisCreationDate);
   REGISTER_CLASS_CONST_LONG("cisDescription",             cisDescription);
   REGISTER_CLASS_CONST_LONG("cisFileName",                cisFileName);
   REGISTER_CLASS_CONST_LONG("cisModDate",                 cisModDate);
   REGISTER_CLASS_CONST_LONG("cisSize",                    cisSize);
   REGISTER_CLASS_CONST_LONG("cisCustomDate",              cisCustomDate);
   REGISTER_CLASS_CONST_LONG("cisCustomNumber",            cisCustomNumber);
   REGISTER_CLASS_CONST_LONG("cisCustomString",            cisCustomString);
   REGISTER_CLASS_CONST_LONG("cisCompressedSize",          cisCompressedSize);

   REGISTER_CLASS_CONST_LONG("civNotSet",                  civNotSet);
   REGISTER_CLASS_CONST_LONG("civDetails",                 civDetails);
   REGISTER_CLASS_CONST_LONG("civTile",                    civTile);
   REGISTER_CLASS_CONST_LONG("civHidden",                  civHidden);
   REGISTER_CLASS_CONST_LONG("civCustom",                  civCustom);

   REGISTER_CLASS_CONST_LONG("cbfNone",                    cbfNone);
   REGISTER_CLASS_CONST_LONG("cbfIgnoreWhiteAreas",        cbfIgnoreWhiteAreas);
   REGISTER_CLASS_CONST_LONG("cbfParse1BitImages",         cbfParse1BitImages);
   REGISTER_CLASS_CONST_LONG("cbfParseGrayImages",         cbfParseGrayImages);
   REGISTER_CLASS_CONST_LONG("cbfParseColorImages",        cbfParseColorImages);
   REGISTER_CLASS_CONST_LONG("cbfParseAllImages",          cbfParseAllImages);

   REGISTER_CLASS_CONST_LONG("cp1250",                     cp1250);
   REGISTER_CLASS_CONST_LONG("cp1251",                     cp1251);
   REGISTER_CLASS_CONST_LONG("cp1252",                     cp1252);
   REGISTER_CLASS_CONST_LONG("cp1253",                     cp1253);
   REGISTER_CLASS_CONST_LONG("cp1254",                     cp1254);
   REGISTER_CLASS_CONST_LONG("cp1255",                     cp1255);
   REGISTER_CLASS_CONST_LONG("cp1256",                     cp1256);
   REGISTER_CLASS_CONST_LONG("cp1257",                     cp1257);
   REGISTER_CLASS_CONST_LONG("cp1258",                     cp1258);
   REGISTER_CLASS_CONST_LONG("cp8859_2",                   cp8859_2);
   REGISTER_CLASS_CONST_LONG("cp8859_3",                   cp8859_3);
   REGISTER_CLASS_CONST_LONG("cp8859_4",                   cp8859_4);
   REGISTER_CLASS_CONST_LONG("cp8859_5",                   cp8859_5);
   REGISTER_CLASS_CONST_LONG("cp8859_6",                   cp8859_6);
   REGISTER_CLASS_CONST_LONG("cp8859_7",                   cp8859_7);
   REGISTER_CLASS_CONST_LONG("cp8859_8",                   cp8859_8);
   REGISTER_CLASS_CONST_LONG("cp8859_9",                   cp8859_9);
   REGISTER_CLASS_CONST_LONG("cp8859_10",                  cp8859_10);
   REGISTER_CLASS_CONST_LONG("cp8859_13",                  cp8859_13);
   REGISTER_CLASS_CONST_LONG("cp8859_14",                  cp8859_14);
   REGISTER_CLASS_CONST_LONG("cp8859_15",                  cp8859_15);
   REGISTER_CLASS_CONST_LONG("cp8859_16",                  cp8859_16);
   REGISTER_CLASS_CONST_LONG("cpSymbol",                   cpSymbol);
   REGISTER_CLASS_CONST_LONG("cp437",                      cp437);
   REGISTER_CLASS_CONST_LONG("cp737",                      cp737);
   REGISTER_CLASS_CONST_LONG("cp775",                      cp775);
   REGISTER_CLASS_CONST_LONG("cp850",                      cp850);
   REGISTER_CLASS_CONST_LONG("cp852",                      cp852);
   REGISTER_CLASS_CONST_LONG("cp855",                      cp855);
   REGISTER_CLASS_CONST_LONG("cp857",                      cp857);
   REGISTER_CLASS_CONST_LONG("cp860",                      cp860);
   REGISTER_CLASS_CONST_LONG("cp861",                      cp861);
   REGISTER_CLASS_CONST_LONG("cp862",                      cp862);
   REGISTER_CLASS_CONST_LONG("cp863",                      cp863);
   REGISTER_CLASS_CONST_LONG("cp864",                      cp864);
   REGISTER_CLASS_CONST_LONG("cp865",                      cp865);
   REGISTER_CLASS_CONST_LONG("cp866",                      cp866);
   REGISTER_CLASS_CONST_LONG("cp869",                      cp869);
   REGISTER_CLASS_CONST_LONG("cp874",                      cp874);
   REGISTER_CLASS_CONST_LONG("cpUnicode",                  cpUnicode);
   REGISTER_CLASS_CONST_LONG("cpCJK_Big5_Uni",             cpCJK_Big5_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_EUC_JP_Uni",           cpCJK_EUC_JP_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_EUC_KR_Uni",           cpCJK_EUC_KR_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_EUC_TW_Uni",           cpCJK_EUC_TW_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_GBK_Uni",              cpCJK_GBK_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_GB12345_Uni",          cpCJK_GB12345_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_HZ_Uni",               cpCJK_HZ_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_2022_CN_Uni",          cpCJK_2022_CN_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_2022_JP_Uni",          cpCJK_2022_JP_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_2022_KR_Uni",          cpCJK_2022_KR_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_646_CN_Uni",           cpCJK_646_CN_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_646_JP_Uni",           cpCJK_646_JP_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_IR_165_Uni",           cpCJK_IR_165_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_932_Uni",              cpCJK_932_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_949_Uni",              cpCJK_949_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_950_Uni",              cpCJK_950_Uni);
   REGISTER_CLASS_CONST_LONG("cpCJK_JOHAB_Uni",            cpCJK_JOHAB_Uni);
   REGISTER_CLASS_CONST_LONG("cpShiftJIS",                 cpShiftJIS);
   REGISTER_CLASS_CONST_LONG("cpBig5",                     cpBig5);
   REGISTER_CLASS_CONST_LONG("cpGB2312",                   cpGB2312);
   REGISTER_CLASS_CONST_LONG("cpWansung",                  cpWansung);
   REGISTER_CLASS_CONST_LONG("cpJohab",                    cpJohab);
   REGISTER_CLASS_CONST_LONG("cpMacRoman",                 cpMacRoman);
   REGISTER_CLASS_CONST_LONG("cpAdobeStd",                 cpAdobeStd);
   REGISTER_CLASS_CONST_LONG("cpGlyphIndexes",             cpGlyphIndexes);
   REGISTER_CLASS_CONST_LONG("cpDingbats",                 cpDingbats);
   REGISTER_CLASS_CONST_LONG("cpMacExpert",                cpMacExpert);
   REGISTER_CLASS_CONST_LONG("cpRoman8",                   cpRoman8);

   REGISTER_CLASS_CONST_LONG("ctPDFA_1b_2005",             ctPDFA_1b_2005);
   REGISTER_CLASS_CONST_LONG("ctNormalize",                ctNormalize);
   REGISTER_CLASS_CONST_LONG("ctPDFA_2b",                  ctPDFA_2b);
   REGISTER_CLASS_CONST_LONG("ctPDFA_3b",                  ctPDFA_3b);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD_Basic",            ctZUGFeRD_Basic);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD_Comfort",          ctZUGFeRD_Comfort);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD_Extended",         ctZUGFeRD_Extended);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD2_Minimum",         ctZUGFeRD2_Minimum);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD2_Basic",           ctZUGFeRD2_Basic);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD2_Basic_WL",        ctZUGFeRD2_Basic_WL);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD2_Comfort",         ctZUGFeRD2_Comfort);
   REGISTER_CLASS_CONST_LONG("ctZUGFeRD2_Extended",        ctZUGFeRD2_Extended);
   REGISTER_CLASS_CONST_LONG("ctFacturX_Minimum",          ctFacturX_Minimum);
   REGISTER_CLASS_CONST_LONG("ctFacturX_Basic",            ctFacturX_Basic);
   REGISTER_CLASS_CONST_LONG("ctFacturX_Basic_WL",         ctFacturX_Basic_WL);
   REGISTER_CLASS_CONST_LONG("ctFacturX_Comfort",          ctFacturX_Comfort);
   REGISTER_CLASS_CONST_LONG("ctFacturX_Extended",         ctFacturX_Extended);
   REGISTER_CLASS_CONST_LONG("ctFacturX_XRechnung",        ctFacturX_XRechnung);

   REGISTER_CLASS_CONST_LONG("vpUseNone",                  vpUseNone);
   REGISTER_CLASS_CONST_LONG("vpHideToolBar",              vpHideToolBar);
   REGISTER_CLASS_CONST_LONG("vpHideMenuBar",              vpHideMenuBar);
   REGISTER_CLASS_CONST_LONG("vpHideWindowUI",             vpHideWindowUI);
   REGISTER_CLASS_CONST_LONG("vpFitWindow",                vpFitWindow);
   REGISTER_CLASS_CONST_LONG("vpCenterWindow",             vpCenterWindow);
   REGISTER_CLASS_CONST_LONG("vpDisplayDocTitle",          vpDisplayDocTitle);
   REGISTER_CLASS_CONST_LONG("vpNonFullScrPageMode",       vpNonFullScrPageMode);
   REGISTER_CLASS_CONST_LONG("vpDirection",                vpDirection);
   REGISTER_CLASS_CONST_LONG("vpViewArea",                 vpViewArea);
   REGISTER_CLASS_CONST_LONG("vpViewClip",                 vpViewClip);
   REGISTER_CLASS_CONST_LONG("vpPrintArea",                vpPrintArea);
   REGISTER_CLASS_CONST_LONG("vpPrintClip",                vpPrintClip);

   REGISTER_CLASS_CONST_LONG("avNone",                     vpUseNone);
   REGISTER_CLASS_CONST_LONG("avNonFullScrUseNone",        vpHideToolBar);
   REGISTER_CLASS_CONST_LONG("avNonFullScrUseOutlines",    vpHideMenuBar);
   REGISTER_CLASS_CONST_LONG("avNonFullScrUseThumbs",      vpHideWindowUI);
   REGISTER_CLASS_CONST_LONG("avDirectionL2R",             vpFitWindow);
   REGISTER_CLASS_CONST_LONG("avDirectionR2L",             vpCenterWindow);
   REGISTER_CLASS_CONST_LONG("avViewPrintArtBox",          vpDisplayDocTitle);
   REGISTER_CLASS_CONST_LONG("avViewPrintBleedBox",        vpNonFullScrPageMode);
   REGISTER_CLASS_CONST_LONG("avViewPrintCropBox",         vpDirection);
   REGISTER_CLASS_CONST_LONG("avViewPrintMediaBox",        vpViewArea);
   REGISTER_CLASS_CONST_LONG("avViewPrintTrimBox",         vpViewClip);

   REGISTER_CLASS_CONST_LONG("atGoTo",                     atGoTo);
   REGISTER_CLASS_CONST_LONG("atGoToR",                    atGoToR);
   REGISTER_CLASS_CONST_LONG("atHide",                     atHide);
   REGISTER_CLASS_CONST_LONG("atImportData",               atImportData);
   REGISTER_CLASS_CONST_LONG("atJavaScript",               atJavaScript);
   REGISTER_CLASS_CONST_LONG("atLaunch",                   atLaunch);
   REGISTER_CLASS_CONST_LONG("atMovie",                    atMovie);
   REGISTER_CLASS_CONST_LONG("atNamed",                    atNamed);
   REGISTER_CLASS_CONST_LONG("atRendition",                atRendition);
   REGISTER_CLASS_CONST_LONG("atReset",                    atReset);
   REGISTER_CLASS_CONST_LONG("atSetOCGState",              atSetOCGState);
   REGISTER_CLASS_CONST_LONG("atSound",                    atSound);
   REGISTER_CLASS_CONST_LONG("atSubmit",                   atSubmit);
   REGISTER_CLASS_CONST_LONG("atThread",                   atThread);
   REGISTER_CLASS_CONST_LONG("atTransition",               atTransition);
   REGISTER_CLASS_CONST_LONG("atURI",                      atURI);
   REGISTER_CLASS_CONST_LONG("atGoTo3DView",               atGoTo3DView);
   REGISTER_CLASS_CONST_LONG("atGoToE",                    atGoToE);
   REGISTER_CLASS_CONST_LONG("atRichMediaExec",            atRichMediaExec);

   REGISTER_CLASS_CONST_LONG("afNone",                     afNone);
   REGISTER_CLASS_CONST_LONG("afInvisible",                afInvisible);
   REGISTER_CLASS_CONST_LONG("afHidden",                   afHidden);
   REGISTER_CLASS_CONST_LONG("afPrint",                    afPrint);
   REGISTER_CLASS_CONST_LONG("afNoZoom",                   afNoZoom);
   REGISTER_CLASS_CONST_LONG("afNoRotate",                 afNoRotate);
   REGISTER_CLASS_CONST_LONG("afNoView",                   afNoView);
   REGISTER_CLASS_CONST_LONG("afReadOnly",                 afReadOnly);
   REGISTER_CLASS_CONST_LONG("afLocked",                   afLocked);
   REGISTER_CLASS_CONST_LONG("afToggleNoView",             afToggleNoView);
   REGISTER_CLASS_CONST_LONG("afLockedContents",           afLockedContents);

   REGISTER_CLASS_CONST_LONG("affNone",                    affNone);
   REGISTER_CLASS_CONST_LONG("affUseViewState",            affUseViewState);
   REGISTER_CLASS_CONST_LONG("affMarkupAnnots",            affMarkupAnnots);
   REGISTER_CLASS_CONST_LONG("affNonPDFA_1",               affNonPDFA_1);
   REGISTER_CLASS_CONST_LONG("affNonPDFA_2",               affNonPDFA_2);
   REGISTER_CLASS_CONST_LONG("affFormFields",              affFormFields);
   REGISTER_CLASS_CONST_LONG("affUseFieldViewState",       affUseFieldViewState);
   REGISTER_CLASS_CONST_LONG("affSigFields",               affSigFields);

   REGISTER_CLASS_CONST_LONG("affKeepLinkAnnots",          affKeepLinkAnnots);
   REGISTER_CLASS_CONST_LONG("affKeepFileAttach",          affKeepFileAttach);
   REGISTER_CLASS_CONST_LONG("affKeepTextAnnots",          affKeepTextAnnots);
   REGISTER_CLASS_CONST_LONG("affKeepEmptySigFields",      affKeepEmptySigFields);
   REGISTER_CLASS_CONST_LONG("affPreserveZOrder",          affPreserveZOrder);

   REGISTER_CLASS_CONST_LONG("asAuthor",                   asAuthor);
   REGISTER_CLASS_CONST_LONG("asContent",                  asContent);
   REGISTER_CLASS_CONST_LONG("asName",                     asName);
   REGISTER_CLASS_CONST_LONG("asSubject",                  asSubject);
   REGISTER_CLASS_CONST_LONG("asRichStyle",                asRichStyle);
   REGISTER_CLASS_CONST_LONG("asRichText",                 asRichText);

   REGISTER_CLASS_CONST_LONG("aiComment",                  aiComment);
   REGISTER_CLASS_CONST_LONG("aiHelp",                     aiHelp);
   REGISTER_CLASS_CONST_LONG("aiInsert",                   aiInsert);
   REGISTER_CLASS_CONST_LONG("aiKey",                      aiKey);
   REGISTER_CLASS_CONST_LONG("aiNewParagraph",             aiNewParagraph);
   REGISTER_CLASS_CONST_LONG("aiNote",                     aiNote);
   REGISTER_CLASS_CONST_LONG("aiParagraph",                aiParagraph);

   REGISTER_CLASS_CONST_LONG("atCaret",                    atCaret);
   REGISTER_CLASS_CONST_LONG("atCircle",                   atCircle);
   REGISTER_CLASS_CONST_LONG("atFileLink",                 atFileLink);
   REGISTER_CLASS_CONST_LONG("atFreeText",                 atFreeText);
   REGISTER_CLASS_CONST_LONG("atHighlight",                atHighlight);
   REGISTER_CLASS_CONST_LONG("atInk",                      atInk);
   REGISTER_CLASS_CONST_LONG("atLine",                     atLine);
   REGISTER_CLASS_CONST_LONG("atPageLink",                 atPageLink);
   REGISTER_CLASS_CONST_LONG("atPolygon",                  atPolygon);
   REGISTER_CLASS_CONST_LONG("atPolyLine",                 atPolyLine);
   REGISTER_CLASS_CONST_LONG("atPopUp",                    atPopUp);
   REGISTER_CLASS_CONST_LONG("atSquare",                   atSquare);
   REGISTER_CLASS_CONST_LONG("atSquiggly",                 atSquiggly);
   REGISTER_CLASS_CONST_LONG("atStamp",                    atStamp);
   REGISTER_CLASS_CONST_LONG("atStrikeOut",                atStrikeOut);
   REGISTER_CLASS_CONST_LONG("atText",                     atText);
   REGISTER_CLASS_CONST_LONG("atUnderline",                atUnderline);
   REGISTER_CLASS_CONST_LONG("atWebLink",                  atWebLink);
   REGISTER_CLASS_CONST_LONG("atWidget",                   atWidget);
   REGISTER_CLASS_CONST_LONG("at3D",                       at3D);
   REGISTER_CLASS_CONST_LONG("atSoundAnnot",               atSoundAnnot);
   REGISTER_CLASS_CONST_LONG("atFileAttach",               atFileAttach);
   REGISTER_CLASS_CONST_LONG("atWatermark",                atWatermark);
   REGISTER_CLASS_CONST_LONG("atUnknown",                  atUnknown);
   REGISTER_CLASS_CONST_LONG("atMovieAnnot",               atMovieAnnot);
   REGISTER_CLASS_CONST_LONG("atPrinterMark",              atPrinterMark);
   REGISTER_CLASS_CONST_LONG("atProjection",               atProjection);
   REGISTER_CLASS_CONST_LONG("atRichMedia",                atRichMedia);
   REGISTER_CLASS_CONST_LONG("atScreen",                   atScreen);
   REGISTER_CLASS_CONST_LONG("atTrapNet",                  atTrapNet);

   REGISTER_CLASS_CONST_LONG("bctNotInitialized",          bctNotInitialized);
   REGISTER_CLASS_CONST_LONG("bctAustraliaPost",           bctAustraliaPost);
   REGISTER_CLASS_CONST_LONG("bctAustraliaRedir",          bctAustraliaRedir);
   REGISTER_CLASS_CONST_LONG("bctAustraliaReply",          bctAustraliaReply);
   REGISTER_CLASS_CONST_LONG("bctAustraliaRout",           bctAustraliaRout);
   REGISTER_CLASS_CONST_LONG("bctAztec",                   bctAztec);
   REGISTER_CLASS_CONST_LONG("bctAztecRunes",              bctAztecRunes);
   REGISTER_CLASS_CONST_LONG("bctC2Of5IATA",               bctC2Of5IATA);
   REGISTER_CLASS_CONST_LONG("bctC2Of5Industrial",         bctC2Of5Industrial);
   REGISTER_CLASS_CONST_LONG("bctC2Of5Interleaved",        bctC2Of5Interleaved);
   REGISTER_CLASS_CONST_LONG("bctC2Of5Logic",              bctC2Of5Logic);
   REGISTER_CLASS_CONST_LONG("bctC2Of5Matrix",             bctC2Of5Matrix);
   REGISTER_CLASS_CONST_LONG("bctChannelCode",             bctChannelCode);
   REGISTER_CLASS_CONST_LONG("bctCodabar",                 bctCodabar);
   REGISTER_CLASS_CONST_LONG("bctCodablockF",              bctCodablockF);
   REGISTER_CLASS_CONST_LONG("bctCode11",                  bctCode11);
   REGISTER_CLASS_CONST_LONG("bctCode128",                 bctCode128);
   REGISTER_CLASS_CONST_LONG("bctCode128B",                bctCode128B);
   REGISTER_CLASS_CONST_LONG("bctCode16K",                 bctCode16K);
   REGISTER_CLASS_CONST_LONG("bctCode32",                  bctCode32);
   REGISTER_CLASS_CONST_LONG("bctCode39",                  bctCode39);
   REGISTER_CLASS_CONST_LONG("bctCode49",                  bctCode49);
   REGISTER_CLASS_CONST_LONG("bctCode93",                  bctCode93);
   REGISTER_CLASS_CONST_LONG("bctCodeOne",                 bctCodeOne);
   REGISTER_CLASS_CONST_LONG("bctDAFT",                    bctDAFT);
   REGISTER_CLASS_CONST_LONG("bctDataBarOmniTrunc",        bctDataBarOmniTrunc);
   REGISTER_CLASS_CONST_LONG("bctDataBarExpStacked",       bctDataBarExpStacked);
   REGISTER_CLASS_CONST_LONG("bctDataBarExpanded",         bctDataBarExpanded);
   REGISTER_CLASS_CONST_LONG("bctDataBarLimited",          bctDataBarLimited);
   REGISTER_CLASS_CONST_LONG("bctDataBarStacked",          bctDataBarStacked);
   REGISTER_CLASS_CONST_LONG("bctDataBarStackedO",         bctDataBarStackedO);
   REGISTER_CLASS_CONST_LONG("bctDataMatrix",              bctDataMatrix);
   REGISTER_CLASS_CONST_LONG("bctDotCode",                 bctDotCode);
   REGISTER_CLASS_CONST_LONG("bctDPD",                     bctDPD);
   REGISTER_CLASS_CONST_LONG("bctDPIdentCode",             bctDPIdentCode);
   REGISTER_CLASS_CONST_LONG("bctDPLeitcode",              bctDPLeitcode);
   REGISTER_CLASS_CONST_LONG("bctEAN128",                  bctEAN128);
   REGISTER_CLASS_CONST_LONG("bctEAN128_CC",               bctEAN128_CC);
   REGISTER_CLASS_CONST_LONG("bctEAN14",                   bctEAN14);
   REGISTER_CLASS_CONST_LONG("bctEANX",                    bctEANX);
   REGISTER_CLASS_CONST_LONG("bctEANX_CC",                 bctEANX_CC);
   REGISTER_CLASS_CONST_LONG("bctEANXCheck",               bctEANXCheck);
   REGISTER_CLASS_CONST_LONG("bctExtCode39",               bctExtCode39);
   REGISTER_CLASS_CONST_LONG("bctFIM",                     bctFIM);
   REGISTER_CLASS_CONST_LONG("bctFlattermarken",           bctFlattermarken);
   REGISTER_CLASS_CONST_LONG("bctHIBC_Aztec",              bctHIBC_Aztec);
   REGISTER_CLASS_CONST_LONG("bctHIBC_CodablockF",         bctHIBC_CodablockF);
   REGISTER_CLASS_CONST_LONG("bctHIBC_Code128",            bctHIBC_Code128);
   REGISTER_CLASS_CONST_LONG("bctHIBC_Code39",             bctHIBC_Code39);
   REGISTER_CLASS_CONST_LONG("bctHIBC_DataMatrix",         bctHIBC_DataMatrix);
   REGISTER_CLASS_CONST_LONG("bctHIBC_MicroPDF417",        bctHIBC_MicroPDF417);
   REGISTER_CLASS_CONST_LONG("bctHIBC_PDF417",             bctHIBC_PDF417);
   REGISTER_CLASS_CONST_LONG("bctHIBC_QR",                 bctHIBC_QR);
   REGISTER_CLASS_CONST_LONG("bctISBNX",                   bctISBNX);
   REGISTER_CLASS_CONST_LONG("bctITF14",                   bctITF14);
   REGISTER_CLASS_CONST_LONG("bctJapanPost",               bctJapanPost);
   REGISTER_CLASS_CONST_LONG("bctKIX",                     bctKIX);
   REGISTER_CLASS_CONST_LONG("bctKoreaPost",               bctKoreaPost);
   REGISTER_CLASS_CONST_LONG("bctLOGMARS",                 bctLOGMARS);
   REGISTER_CLASS_CONST_LONG("bctMailmark",                bctMailmark);
   REGISTER_CLASS_CONST_LONG("bctMaxicode",                bctMaxicode);
   REGISTER_CLASS_CONST_LONG("bctMicroPDF417",             bctMicroPDF417);
   REGISTER_CLASS_CONST_LONG("bctMicroQR",                 bctMicroQR);
   REGISTER_CLASS_CONST_LONG("bctMSIPlessey",              bctMSIPlessey);
   REGISTER_CLASS_CONST_LONG("bctNVE18",                   bctNVE18);
   REGISTER_CLASS_CONST_LONG("bctPDF417",                  bctPDF417);
   REGISTER_CLASS_CONST_LONG("bctPDF417Truncated",         bctPDF417Truncated);
   REGISTER_CLASS_CONST_LONG("bctPharmaOneTrack",          bctPharmaOneTrack);
   REGISTER_CLASS_CONST_LONG("bctPharmaTwoTrack",          bctPharmaTwoTrack);
   REGISTER_CLASS_CONST_LONG("bctPLANET",                  bctPLANET);
   REGISTER_CLASS_CONST_LONG("bctPlessey",                 bctPlessey);
   REGISTER_CLASS_CONST_LONG("bctPostNet",                 bctPostNet);
   REGISTER_CLASS_CONST_LONG("bctPZN",                     bctPZN);
   REGISTER_CLASS_CONST_LONG("bctQRCode",                  bctQRCode);
   REGISTER_CLASS_CONST_LONG("bctRMQR",                    bctRMQR);
   REGISTER_CLASS_CONST_LONG("bctRoyalMail4State",         bctRoyalMail4State);
   REGISTER_CLASS_CONST_LONG("bctRSS_EXP_CC",              bctRSS_EXP_CC);
   REGISTER_CLASS_CONST_LONG("bctRSS_EXPSTACK_CC",         bctRSS_EXPSTACK_CC);
   REGISTER_CLASS_CONST_LONG("bctRSS_LTD_CC",              bctRSS_LTD_CC);
   REGISTER_CLASS_CONST_LONG("bctRSS14_CC",                bctRSS14_CC);
   REGISTER_CLASS_CONST_LONG("bctRSS14Stacked_CC",         bctRSS14Stacked_CC);
   REGISTER_CLASS_CONST_LONG("bctRSS14StackOMNI_CC",       bctRSS14StackOMNI_CC);
   REGISTER_CLASS_CONST_LONG("bctTelepen",                 bctTelepen);
   REGISTER_CLASS_CONST_LONG("bctTelepenNumeric",          bctTelepenNumeric);
   REGISTER_CLASS_CONST_LONG("bctUltracode",               bctUltracode);
   REGISTER_CLASS_CONST_LONG("bctUPCA",                    bctUPCA);
   REGISTER_CLASS_CONST_LONG("bctUPCA_CC",                 bctUPCA_CC);
   REGISTER_CLASS_CONST_LONG("bctUPCACheckDigit",          bctUPCACheckDigit);
   REGISTER_CLASS_CONST_LONG("bctUPCE",                    bctUPCE);
   REGISTER_CLASS_CONST_LONG("bctUPCE_CC",                 bctUPCE_CC);
   REGISTER_CLASS_CONST_LONG("bctUPCECheckDigit",          bctUPCECheckDigit);
   REGISTER_CLASS_CONST_LONG("bctUPNQR",                   bctUPNQR);
   REGISTER_CLASS_CONST_LONG("bctUSPSOneCode",             bctUSPSOneCode);
   REGISTER_CLASS_CONST_LONG("bctVIN",                     bctVIN);

   REGISTER_CLASS_CONST_LONG("bcdtBinary",                 bcdtBinary);
   REGISTER_CLASS_CONST_LONG("bcdtUnicode",                bcdtUnicode);
   REGISTER_CLASS_CONST_LONG("bcdtGS1Mode",                bcdtGS1Mode);
   REGISTER_CLASS_CONST_LONG("bcdtEscapeMode",             bcdtEscapeMode);

   REGISTER_CLASS_CONST_LONG("bcoDefault",                 bcoDefault);
   REGISTER_CLASS_CONST_LONG("bcoNoASCII",                 bcoNoASCII);
   REGISTER_CLASS_CONST_LONG("bcoBind",                    bcoBind);
   REGISTER_CLASS_CONST_LONG("bcoBox",                     bcoBox);
   REGISTER_CLASS_CONST_LONG("bcoReaderInit",              bcoReaderInit);
   REGISTER_CLASS_CONST_LONG("bcoSmallText",               bcoSmallText);
   REGISTER_CLASS_CONST_LONG("bcoBoldText",                bcoBoldText);
   REGISTER_CLASS_CONST_LONG("bcoDottyMode",               bcoDottyMode);
   REGISTER_CLASS_CONST_LONG("bcoGS1_GS_Separator",        bcoGS1_GS_Separator);
   REGISTER_CLASS_CONST_LONG("bcoImageOutput",             bcoImageOutput);
   REGISTER_CLASS_CONST_LONG("bcoUseActiveFont",           bcoUseActiveFont);

   REGISTER_CLASS_CONST_LONG("beWinAnsi",                  beWinAnsi);
   REGISTER_CLASS_CONST_LONG("beMacRoman",                 beMacRoman);
   REGISTER_CLASS_CONST_LONG("beMacExpert",                beMacExpert);
   REGISTER_CLASS_CONST_LONG("beStandard",                 beStandard);

   REGISTER_CLASS_CONST_LONG("beSolid",                    beSolid);
   REGISTER_CLASS_CONST_LONG("beCloudy1",                  beCloudy1);
   REGISTER_CLASS_CONST_LONG("beCloudy2",                  beCloudy2);

   REGISTER_CLASS_CONST_LONG("bsSolid",                    bsSolid);
   REGISTER_CLASS_CONST_LONG("bsBevelled",                 bsBevelled);
   REGISTER_CLASS_CONST_LONG("bsInset",                    bsInset);
   REGISTER_CLASS_CONST_LONG("bsUnderline",                bsUnderline);
   REGISTER_CLASS_CONST_LONG("bsDashed",                   bsDashed);

   REGISTER_CLASS_CONST_LONG("bsUp",                       bsUp);
   REGISTER_CLASS_CONST_LONG("bsDown",                     bsDown);
   REGISTER_CLASS_CONST_LONG("bsRollOver",                 bsRollOver);

   REGISTER_CLASS_CONST_LONG("btArt",                      btArt);
   REGISTER_CLASS_CONST_LONG("btArtifact",                 btArtifact);
   REGISTER_CLASS_CONST_LONG("btAnnot",                    btAnnot);
   REGISTER_CLASS_CONST_LONG("btBibEntry",                 btBibEntry);
   REGISTER_CLASS_CONST_LONG("btBlockQuote",               btBlockQuote);
   REGISTER_CLASS_CONST_LONG("btCaption",                  btCaption);
   REGISTER_CLASS_CONST_LONG("btCode",                     btCode);
   REGISTER_CLASS_CONST_LONG("btDiv",                      btDiv);
   REGISTER_CLASS_CONST_LONG("btDocument",                 btDocument);
   REGISTER_CLASS_CONST_LONG("btFigure",                   btFigure);
   REGISTER_CLASS_CONST_LONG("btForm",                     btForm);
   REGISTER_CLASS_CONST_LONG("btFormula",                  btFormula);
   REGISTER_CLASS_CONST_LONG("btH",                        btH);
   REGISTER_CLASS_CONST_LONG("btH1",                       btH1);
   REGISTER_CLASS_CONST_LONG("btH2",                       btH2);
   REGISTER_CLASS_CONST_LONG("btH3",                       btH3);
   REGISTER_CLASS_CONST_LONG("btH4",                       btH4);
   REGISTER_CLASS_CONST_LONG("btH5",                       btH5);
   REGISTER_CLASS_CONST_LONG("btH6",                       btH6);
   REGISTER_CLASS_CONST_LONG("btIndex",                    btIndex);
   REGISTER_CLASS_CONST_LONG("btLink",                     btLink);
   REGISTER_CLASS_CONST_LONG("btList",                     btList);
   REGISTER_CLASS_CONST_LONG("btListElem",                 btListElem);
   REGISTER_CLASS_CONST_LONG("btListText",                 btListText);
   REGISTER_CLASS_CONST_LONG("btNote",                     btNote);
   REGISTER_CLASS_CONST_LONG("btP",                        btP);
   REGISTER_CLASS_CONST_LONG("btPart",                     btPart);
   REGISTER_CLASS_CONST_LONG("btQuote",                    btQuote);
   REGISTER_CLASS_CONST_LONG("btReference",                btReference);
   REGISTER_CLASS_CONST_LONG("btSection",                  btSection);
   REGISTER_CLASS_CONST_LONG("btSpan",                     btSpan);
   REGISTER_CLASS_CONST_LONG("btTable",                    btTable);
   REGISTER_CLASS_CONST_LONG("btTableDataCell",            btTableDataCell);
   REGISTER_CLASS_CONST_LONG("btTableHeader",              btTableHeader);
   REGISTER_CLASS_CONST_LONG("btTableRow",                 btTableRow);
   REGISTER_CLASS_CONST_LONG("btTOC",                      btTOC);
   REGISTER_CLASS_CONST_LONG("btTOCEntry",                 btTOCEntry);

   REGISTER_CLASS_CONST_LONG("cmFill",                     cmFill);
   REGISTER_CLASS_CONST_LONG("cmStroke",                   cmStroke);
   REGISTER_CLASS_CONST_LONG("cmFillStroke",               cmFillStroke);

   REGISTER_CLASS_CONST_LONG("cfFlate",                    cfFlate);
   REGISTER_CLASS_CONST_LONG("cfJPEG",                     cfJPEG);
   REGISTER_CLASS_CONST_LONG("cfCCITT3",                   cfCCITT3);
   REGISTER_CLASS_CONST_LONG("cfCCITT4",                   cfCCITT4);
   REGISTER_CLASS_CONST_LONG("cfLZW",                      cfLZW);
   REGISTER_CLASS_CONST_LONG("cfFlateBW",                  cfFlateBW);
   REGISTER_CLASS_CONST_LONG("cfJP2K",                     cfJP2K);
   REGISTER_CLASS_CONST_LONG("cfJBIG2",                    cfJBIG2);
   REGISTER_CLASS_CONST_LONG("cfNone",                     cfNone);
   REGISTER_CLASS_CONST_LONG("cfDitherFloydSteinberg",     cfDitherFloydSteinberg);
   REGISTER_CLASS_CONST_LONG("cfConvGrayToOtsu",           cfConvGrayToOtsu);
   REGISTER_CLASS_CONST_LONG("cfOrderedDithering",         cfOrderedDithering);
   REGISTER_CLASS_CONST_LONG("cfIgnoreICCProfile",         cfIgnoreICCProfile);
   REGISTER_CLASS_CONST_LONG("cfPresLosslessFilter",       cfPresLosslessFilter);
   REGISTER_CLASS_CONST_LONG("cfPresLossyFilter",          cfPresLossyFilter);

   REGISTER_CLASS_CONST_LONG("clNone",                     clNone);
   REGISTER_CLASS_CONST_LONG("clDefault",                  clDefault);
   REGISTER_CLASS_CONST_LONG("clFastest",                  clFastest);
   REGISTER_CLASS_CONST_LONG("clMax",                      clMax);

   REGISTER_CLASS_CONST_LONG("diAuthor",                   diAuthor);
   REGISTER_CLASS_CONST_LONG("diCreator",                  diCreator);
   REGISTER_CLASS_CONST_LONG("diKeywords",                 diKeywords);
   REGISTER_CLASS_CONST_LONG("diProducer",                 diProducer);
   REGISTER_CLASS_CONST_LONG("diSubject",                  diSubject);
   REGISTER_CLASS_CONST_LONG("diTitle",                    diTitle);
   REGISTER_CLASS_CONST_LONG("diCompany",                  diCompany);
   REGISTER_CLASS_CONST_LONG("diPDFX_Ver",                 diPDFX_Ver);
   REGISTER_CLASS_CONST_LONG("diCustom",                   diCustom);
   REGISTER_CLASS_CONST_LONG("diPDFX_Conf",                diPDFX_Conf);
   REGISTER_CLASS_CONST_LONG("diCreationDate",             diCreationDate);
   REGISTER_CLASS_CONST_LONG("diModDate",                  diModDate);

   REGISTER_CLASS_CONST_LONG("dmNormal",                   dmNormal);
   REGISTER_CLASS_CONST_LONG("dmStroke",                   dmStroke);
   REGISTER_CLASS_CONST_LONG("dmFillStroke",               dmFillStroke);
   REGISTER_CLASS_CONST_LONG("dmInvisible",                dmInvisible);
   REGISTER_CLASS_CONST_LONG("dmFillClip",                 dmFillClip);
   REGISTER_CLASS_CONST_LONG("dmStrokeClip",               dmStrokeClip);
   REGISTER_CLASS_CONST_LONG("dmFillStrokeClip",           dmFillStrokeClip);
   REGISTER_CLASS_CONST_LONG("dmClipping",                 dmClipping);

   REGISTER_CLASS_CONST_LONG("efpAnsiPath",                efpAnsiPath);
   REGISTER_CLASS_CONST_LONG("efpUnicodePath",             efpUnicodePath);
   REGISTER_CLASS_CONST_LONG("efpEmbeddable",              efpEmbeddable);

   REGISTER_CLASS_CONST_LONG("emIgnoreAll",                emIgnoreAll);
   REGISTER_CLASS_CONST_LONG("emSyntaxError",              emSyntaxError);
   REGISTER_CLASS_CONST_LONG("emValueError",               emValueError);
   REGISTER_CLASS_CONST_LONG("emWarning",                  emWarning);
   REGISTER_CLASS_CONST_LONG("emFileError",                emFileError);
   REGISTER_CLASS_CONST_LONG("emFontError",                emFontError);
   REGISTER_CLASS_CONST_LONG("emAllErrors",                emAllErrors);
   REGISTER_CLASS_CONST_LONG("emNoFuncNames",              emNoFuncNames);
   REGISTER_CLASS_CONST_LONG("emUseErrLog",                emUseErrLog);
   REGISTER_CLASS_CONST_LONG("emLogFonts",                 emLogFonts);
   REGISTER_CLASS_CONST_LONG("emLogFontsVerbose",          emLogFontsVerbose);

   REGISTER_CLASS_CONST_LONG("smFamilyName",               smFamilyName);
   REGISTER_CLASS_CONST_LONG("smFullName",                 smFullName);
   REGISTER_CLASS_CONST_LONG("smPostScriptName",           smPostScriptName);

   REGISTER_CLASS_CONST_LONG("fsNone",                     fsNone);
   REGISTER_CLASS_CONST_LONG("fsItalic",                   fsItalic);
   REGISTER_CLASS_CONST_LONG("fsBold",                     fsBold);
   REGISTER_CLASS_CONST_LONG("fsUnderlined",               fsUnderlined);
   REGISTER_CLASS_CONST_LONG("fsStriked",                  fsStriked);
   REGISTER_CLASS_CONST_LONG("fsVerticalMode",             fsVerticalMode);
   REGISTER_CLASS_CONST_LONG("fsDblUnderline",             fsDblUnderline);
   REGISTER_CLASS_CONST_LONG("fsUltraCondensed",           fsUltraCondensed);
   REGISTER_CLASS_CONST_LONG("fsExtraCondensed",           fsExtraCondensed);
   REGISTER_CLASS_CONST_LONG("fsCondensed",                fsCondensed);
   REGISTER_CLASS_CONST_LONG("fsNormal",                   fsNormal);
   REGISTER_CLASS_CONST_LONG("fsSemiExpanded",             fsSemiExpanded);
   REGISTER_CLASS_CONST_LONG("fsExpanded",                 fsExpanded);
   REGISTER_CLASS_CONST_LONG("fsExtraExpanded",            fsExtraExpanded);
   REGISTER_CLASS_CONST_LONG("fsUltraExpanded",            fsUltraExpanded);
   REGISTER_CLASS_CONST_LONG("fsThin",                     fsThin);
   REGISTER_CLASS_CONST_LONG("fsExtraLight",               fsExtraLight);
   REGISTER_CLASS_CONST_LONG("fsLight",                    fsLight);
   REGISTER_CLASS_CONST_LONG("fsRegular",                  fsRegular);
   REGISTER_CLASS_CONST_LONG("fsMedium",                   fsMedium);
   REGISTER_CLASS_CONST_LONG("fsDemiBold",                 fsDemiBold);
   REGISTER_CLASS_CONST_LONG("fsExtraBold",                fsExtraBold);
   REGISTER_CLASS_CONST_LONG("fsBlack",                    fsBlack);
   REGISTER_CLASS_CONST_LONG("fsUltraBlack",               fsUltraBlack);

   REGISTER_CLASS_CONST_LONG("gfCompatible",               gfCompatible);
   REGISTER_CLASS_CONST_LONG("gfRestorePageCoords",        gfRestorePageCoords);
   REGISTER_CLASS_CONST_LONG("gfRealTopDownCoords",        gfRealTopDownCoords);
   REGISTER_CLASS_CONST_LONG("gfNativeBlackWhite",         gfNativeBlackWhite);
   REGISTER_CLASS_CONST_LONG("gfUseImageColorSpace",       gfUseImageColorSpace);
   REGISTER_CLASS_CONST_LONG("gfIgnoreICCProfiles",        gfIgnoreICCProfiles);
   REGISTER_CLASS_CONST_LONG("gfAnsiStringIsUTF8",         gfAnsiStringIsUTF8);
   REGISTER_CLASS_CONST_LONG("gfRealPassThrough",          gfRealPassThrough);
   REGISTER_CLASS_CONST_LONG("gfNoBitmapAlpha",            gfNoBitmapAlpha);
   REGISTER_CLASS_CONST_LONG("gfNoImageDuplCheck",         gfNoImageDuplCheck);
   REGISTER_CLASS_CONST_LONG("gfNoObjCompression",         gfNoObjCompression);
   REGISTER_CLASS_CONST_LONG("gfComplexText",              gfComplexText);
   REGISTER_CLASS_CONST_LONG("gfDisableJavascript",        gfDisableJavascript);
   REGISTER_CLASS_CONST_LONG("gfDisableBidiCtrls",         gfDisableBidiCtrls);
   REGISTER_CLASS_CONST_LONG("gfDoNotComprMetadata",       gfDoNotComprMetadata);
   REGISTER_CLASS_CONST_LONG("gfUpdatePDFVTModDate",       gfUpdatePDFVTModDate);

   REGISTER_CLASS_CONST_LONG("hffDefault",                 hffDefault);
   REGISTER_CLASS_CONST_LONG("hffEvenPages",               hffEvenPages);
   REGISTER_CLASS_CONST_LONG("hffOddPages",                hffOddPages);
   REGISTER_CLASS_CONST_LONG("hffFixedPrint",              hffFixedPrint);
   REGISTER_CLASS_CONST_LONG("hffLoadUTF8",                hffLoadUTF8);
   REGISTER_CLASS_CONST_LONG("hffShrink",                  hffShrink);
   REGISTER_CLASS_CONST_LONG("hffLoadBatesN",              hffLoadBatesN);
   REGISTER_CLASS_CONST_LONG("hffLoadHF",                  hffLoadHF);
   REGISTER_CLASS_CONST_LONG("hffSearchRun",               hffSearchRun);

   REGISTER_CLASS_CONST_LONG("ifmTIFF",                    ifmTIFF);
   REGISTER_CLASS_CONST_LONG("ifmJPEG",                    ifmJPEG);
   REGISTER_CLASS_CONST_LONG("ifmPNG",                     ifmPNG);
   REGISTER_CLASS_CONST_LONG("ifmBMP",                     ifmBMP);
   REGISTER_CLASS_CONST_LONG("ifmJPC",                     ifmJPC);

   REGISTER_CLASS_CONST_LONG("kl40bit",                    kl40bit);
   REGISTER_CLASS_CONST_LONG("kl128bit",                   kl128bit);
   REGISTER_CLASS_CONST_LONG("kl128bitEx",                 kl128bitEx);
   REGISTER_CLASS_CONST_LONG("klAES128",                   klAES128);
   REGISTER_CLASS_CONST_LONG("klAES256",                   klAES256);
   REGISTER_CLASS_CONST_LONG("klAESRev6",                  klAESRev6);

   REGISTER_CLASS_CONST_LONG("lcmDefault",                 lcmDefault);
   REGISTER_CLASS_CONST_LONG("lcmRecursive",               lcmRecursive);
   REGISTER_CLASS_CONST_LONG("lcmDelayed",                 lcmDelayed);

   REGISTER_CLASS_CONST_LONG("csButtCap",                  csButtCap);
   REGISTER_CLASS_CONST_LONG("csRoundCap",                 csRoundCap);
   REGISTER_CLASS_CONST_LONG("csSquareCap",                csSquareCap);

   REGISTER_CLASS_CONST_LONG("jsMiterJoin",                jsMiterJoin);
   REGISTER_CLASS_CONST_LONG("jsRoundJoin",                jsRoundJoin);
   REGISTER_CLASS_CONST_LONG("jsBevelJoin",                jsBevelJoin);

   REGISTER_CLASS_CONST_LONG("mfDefault",                  mfDefault);
   REGISTER_CLASS_CONST_LONG("mfDebug",                    mfDebug);
   REGISTER_CLASS_CONST_LONG("mfShowBounds",               mfShowBounds);
   REGISTER_CLASS_CONST_LONG("mfNoTextScaling",            mfNoTextScaling);
   REGISTER_CLASS_CONST_LONG("mfClipView",                 mfClipView);
   REGISTER_CLASS_CONST_LONG("mfUseRclBounds",             mfUseRclBounds);
   REGISTER_CLASS_CONST_LONG("mfNoClippingRgn",            mfNoClippingRgn);
   REGISTER_CLASS_CONST_LONG("mfNoFontEmbedding",          mfNoFontEmbedding);
   REGISTER_CLASS_CONST_LONG("mfNoImages",                 mfNoImages);
   REGISTER_CLASS_CONST_LONG("mfNoStdPatterns",            mfNoStdPatterns);
   REGISTER_CLASS_CONST_LONG("mfNoBmpPatterns",            mfNoBmpPatterns);
   REGISTER_CLASS_CONST_LONG("mfNoText",                   mfNoText);
   REGISTER_CLASS_CONST_LONG("mfUseUnicode",               mfUseUnicode);
   REGISTER_CLASS_CONST_LONG("mfUseTextScaling",           mfUseTextScaling);
   REGISTER_CLASS_CONST_LONG("mfNoUnicode",                mfNoUnicode);
   REGISTER_CLASS_CONST_LONG("mfFullScale",                mfFullScale);
   REGISTER_CLASS_CONST_LONG("mfUseRclFrame",              mfUseRclFrame);
   REGISTER_CLASS_CONST_LONG("mfDefBkModeTransp",          mfDefBkModeTransp);
   REGISTER_CLASS_CONST_LONG("mfApplyBidiAlgo",            mfApplyBidiAlgo);
   REGISTER_CLASS_CONST_LONG("mfGDIFontSelection",         mfGDIFontSelection);
   REGISTER_CLASS_CONST_LONG("mfRclFrameEx",               mfRclFrameEx);
   REGISTER_CLASS_CONST_LONG("mfNoTextClipping",           mfNoTextClipping);
   REGISTER_CLASS_CONST_LONG("mfSrcCopy_Only",             mfSrcCopy_Only);
   REGISTER_CLASS_CONST_LONG("mfClipRclBounds",            mfClipRclBounds);
   REGISTER_CLASS_CONST_LONG("mfDisableRasterEMF",         mfDisableRasterEMF);
   REGISTER_CLASS_CONST_LONG("mfNoBBoxCheck",              mfNoBBoxCheck);
   REGISTER_CLASS_CONST_LONG("mfIgnoreEmbFonts",           mfIgnoreEmbFonts);
   REGISTER_CLASS_CONST_LONG("mfDontSkipROP_DPa",          mfDontSkipROP_DPa);
   REGISTER_CLASS_CONST_LONG("mfCurrComprFilter",          mfCurrComprFilter);
   // Obsolete flags -> these flags are ignored, do no longer use them!
   REGISTER_CLASS_CONST_LONG("mfUseSpacingArray",          mfUseSpacingArray);
   REGISTER_CLASS_CONST_LONG("mfIntersectClipRect",        mfIntersectClipRect);

   REGISTER_CLASS_CONST_LONG("orDownLeft",                 orDownLeft);
   REGISTER_CLASS_CONST_LONG("orTopLeft",                  orTopLeft);

   REGISTER_CLASS_CONST_LONG("pbArtBox",                   pbArtBox);
   REGISTER_CLASS_CONST_LONG("pbBleedBox",                 pbBleedBox);
   REGISTER_CLASS_CONST_LONG("pbCropBox",                  pbCropBox);
   REGISTER_CLASS_CONST_LONG("pbTrimBox",                  pbTrimBox);
   REGISTER_CLASS_CONST_LONG("pbMediaBox",                 pbMediaBox);

   REGISTER_CLASS_CONST_LONG("pcBottomUp",                 pcBottomUp);
   REGISTER_CLASS_CONST_LONG("pcTopDown",                  pcTopDown);

   REGISTER_CLASS_CONST_LONG("pfDIN_A3",                   pfDIN_A3);
   REGISTER_CLASS_CONST_LONG("pfDIN_A4",                   pfDIN_A4);
   REGISTER_CLASS_CONST_LONG("pfDIN_A5",                   pfDIN_A5);
   REGISTER_CLASS_CONST_LONG("pfDIN_B4",                   pfDIN_B4);
   REGISTER_CLASS_CONST_LONG("pfDIN_B5",                   pfDIN_B5);
   REGISTER_CLASS_CONST_LONG("pfDIN_B6",                   pfDIN_B6);
   REGISTER_CLASS_CONST_LONG("pfDIN_C3",                   pfDIN_C3);
   REGISTER_CLASS_CONST_LONG("pfDIN_C4",                   pfDIN_C4);
   REGISTER_CLASS_CONST_LONG("pfDIN_C5",                   pfDIN_C5);
   REGISTER_CLASS_CONST_LONG("pfDIN_C6",                   pfDIN_C6);
   REGISTER_CLASS_CONST_LONG("pfDIN_C65",                  pfDIN_C65);
   REGISTER_CLASS_CONST_LONG("pfDIN_DL",                   pfDIN_DL);
   REGISTER_CLASS_CONST_LONG("pfDIN_E4",                   pfDIN_E4);
   REGISTER_CLASS_CONST_LONG("pfDIN_E5",                   pfDIN_E5);
   REGISTER_CLASS_CONST_LONG("pfDIN_E6",                   pfDIN_E6);
   REGISTER_CLASS_CONST_LONG("pfDIN_E65",                  pfDIN_E65);
   REGISTER_CLASS_CONST_LONG("pfDIN_M5",                   pfDIN_M5);
   REGISTER_CLASS_CONST_LONG("pfDIN_M65",                  pfDIN_M65);
   REGISTER_CLASS_CONST_LONG("pfUS_Legal",                 pfUS_Legal);
   REGISTER_CLASS_CONST_LONG("pfUS_Letter",                pfUS_Letter);
   REGISTER_CLASS_CONST_LONG("pfDIN_A0",                   pfDIN_A0);
   REGISTER_CLASS_CONST_LONG("pfDIN_A1",                   pfDIN_A1);
   REGISTER_CLASS_CONST_LONG("pfDIN_A2",                   pfDIN_A2);
   REGISTER_CLASS_CONST_LONG("pfDIN_A6",                   pfDIN_A6);
   REGISTER_CLASS_CONST_LONG("pfDIN_A7",                   pfDIN_A7);
   REGISTER_CLASS_CONST_LONG("pfDIN_A8",                   pfDIN_A8);
   REGISTER_CLASS_CONST_LONG("pfDIN_A9",                   pfDIN_A9);
   REGISTER_CLASS_CONST_LONG("pfDIN_A10",                  pfDIN_A10);

   REGISTER_CLASS_CONST_LONG("fmFillNoClose",              fmFillNoClose);
   REGISTER_CLASS_CONST_LONG("fmStrokeNoClose",            fmStrokeNoClose);
   REGISTER_CLASS_CONST_LONG("fmFillStrokeNoClose",        fmFillStrokeNoClose);
   REGISTER_CLASS_CONST_LONG("fmFill",                     fmFill);
   REGISTER_CLASS_CONST_LONG("fmStroke",                   fmStroke);
   REGISTER_CLASS_CONST_LONG("fmFillStroke",               fmFillStroke);
   REGISTER_CLASS_CONST_LONG("fmFillEvOdd",                fmFillEvOdd);
   REGISTER_CLASS_CONST_LONG("fmFillStrokeEvOdd",          fmFillStrokeEvOdd);
   REGISTER_CLASS_CONST_LONG("fmFillEvOddNoClose",         fmFillEvOddNoClose);
   REGISTER_CLASS_CONST_LONG("fmFillStrokeEvOddNoClose",   fmFillStrokeEvOddNoClose);
   REGISTER_CLASS_CONST_LONG("fmNoFill",                   fmNoFill);
   REGISTER_CLASS_CONST_LONG("fmClose",                    fmClose);

   REGISTER_CLASS_CONST_LONG("ptColored",                  ptColored);
   REGISTER_CLASS_CONST_LONG("ptUnColored",                ptUnColored);

   REGISTER_CLASS_CONST_LONG("csDeviceRGB",                csDeviceRGB);
   REGISTER_CLASS_CONST_LONG("csDeviceCMYK",               csDeviceCMYK);
   REGISTER_CLASS_CONST_LONG("csDeviceGray",               csDeviceGray);

   REGISTER_CLASS_CONST_LONG("esDeviceRGB",                esDeviceRGB);
   REGISTER_CLASS_CONST_LONG("esDeviceCMYK",               esDeviceCMYK);
   REGISTER_CLASS_CONST_LONG("esDeviceGray",               esDeviceGray);
   REGISTER_CLASS_CONST_LONG("esCalGray",                  esCalGray);
   REGISTER_CLASS_CONST_LONG("esCalRGB",                   esCalRGB);
   REGISTER_CLASS_CONST_LONG("esLab",                      esLab);
   REGISTER_CLASS_CONST_LONG("esICCBased",                 esICCBased);
   REGISTER_CLASS_CONST_LONG("esPattern",                  esPattern);
   REGISTER_CLASS_CONST_LONG("esIndexed",                  esIndexed);
   REGISTER_CLASS_CONST_LONG("esSeparation",               esSeparation);
   REGISTER_CLASS_CONST_LONG("esDeviceN",                  esDeviceN);
   REGISTER_CLASS_CONST_LONG("esNChannel",                 esNChannel);
   REGISTER_CLASS_CONST_LONG("esInitSpace",                esInitSpace);

   REGISTER_CLASS_CONST_LONG("faiGraph",                   faiGraph);
   REGISTER_CLASS_CONST_LONG("faiPaperClip",               faiPaperClip);
   REGISTER_CLASS_CONST_LONG("faiPushPin",                 faiPushPin);
   REGISTER_CLASS_CONST_LONG("faiTag",                     faiTag);
   REGISTER_CLASS_CONST_LONG("faiUserDefined",             faiUserDefined);

   REGISTER_CLASS_CONST_LONG("ofDefault",                  ofDefault);
   REGISTER_CLASS_CONST_LONG("ofInMemory",                 ofInMemory);
   REGISTER_CLASS_CONST_LONG("ofConvertAllColors",         ofConvertAllColors);
   REGISTER_CLASS_CONST_LONG("ofIgnoreICCBased",           ofIgnoreICCBased);
   REGISTER_CLASS_CONST_LONG("ofScaleImages",              ofScaleImages);
   REGISTER_CLASS_CONST_LONG("ofSkipMaskedImages",         ofSkipMaskedImages);
   REGISTER_CLASS_CONST_LONG("ofNewLinkNames",             ofNewLinkNames);
   REGISTER_CLASS_CONST_LONG("ofDeleteInvPaths",           ofDeleteInvPaths);
   REGISTER_CLASS_CONST_LONG("ofFlattenLayers",            ofFlattenLayers);
   REGISTER_CLASS_CONST_LONG("ofDeletePrivateData",        ofDeletePrivateData);
   REGISTER_CLASS_CONST_LONG("ofDeleteThumbnails",         ofDeleteThumbnails);
   REGISTER_CLASS_CONST_LONG("ofDeleteAlternateImages",    ofDeleteAlternateImages);
   REGISTER_CLASS_CONST_LONG("ofNoImageSizeCheck",         ofNoImageSizeCheck);
   REGISTER_CLASS_CONST_LONG("ofIgnoreZeroLineWidth",      ofIgnoreZeroLineWidth);
   REGISTER_CLASS_CONST_LONG("ofAdjZeroLineWidthOnly",     ofAdjZeroLineWidthOnly);
   REGISTER_CLASS_CONST_LONG("ofCompressWithJBIG2",        ofCompressWithJBIG2);
   REGISTER_CLASS_CONST_LONG("ofNoFilterCheck",            ofNoFilterCheck);
   REGISTER_CLASS_CONST_LONG("ofConvertGrayTo1Bit",        ofConvertGrayTo1Bit);
   REGISTER_CLASS_CONST_LONG("ofConvertToGray",            ofConvertToGray);
   REGISTER_CLASS_CONST_LONG("ofConvertToRGB",             ofConvertToRGB);
   REGISTER_CLASS_CONST_LONG("ofConvertToCMYK",            ofConvertToCMYK);
   REGISTER_CLASS_CONST_LONG("ofReplaceJP2KWithJPEG",      ofReplaceJP2KWithJPEG);
   REGISTER_CLASS_CONST_LONG("ofUseOtsuFilter",            ofUseOtsuFilter);
   REGISTER_CLASS_CONST_LONG("ofConvTextToOutlines",       ofConvTextToOutlines);
   REGISTER_CLASS_CONST_LONG("ofConvNonEmbFontsOnly",      ofConvNonEmbFontsOnly);
   REGISTER_CLASS_CONST_LONG("ofIgnoreDeviceN",            ofIgnoreDeviceN);
   REGISTER_CLASS_CONST_LONG("ofIgnoreNChannel",           ofIgnoreNChannel);
   REGISTER_CLASS_CONST_LONG("ofIgnoreSeparation",         ofIgnoreSeparation);
   REGISTER_CLASS_CONST_LONG("ofFailOnOverprinting",       ofFailOnOverprinting);
   REGISTER_CLASS_CONST_LONG("ofSkipOverprintedObj",       ofSkipOverprintedObj);
   REGISTER_CLASS_CONST_LONG("ofRemoveBatesNumbers",       ofRemoveBatesNumbers);
   REGISTER_CLASS_CONST_LONG("ofRemoveHeaderFooter",       ofRemoveHeaderFooter);
   REGISTER_CLASS_CONST_LONG("ofIgnoreSeparationAll",      ofIgnoreSeparationAll);

   REGISTER_CLASS_CONST_LONG("rsDenyNothing",              rsDenyNothing);
   REGISTER_CLASS_CONST_LONG("rsDenyAll",                  rsDenyAll);
   REGISTER_CLASS_CONST_LONG("rsPrint",                    rsPrint);
   REGISTER_CLASS_CONST_LONG("rsModify",                   rsModify);
   REGISTER_CLASS_CONST_LONG("rsCopyObj",                  rsCopyObj);
   REGISTER_CLASS_CONST_LONG("rsAddObj",                   rsAddObj);
   REGISTER_CLASS_CONST_LONG("rsFillInFormFields",         rsFillInFormFields);
   REGISTER_CLASS_CONST_LONG("rsExtractObj",               rsExtractObj);
   REGISTER_CLASS_CONST_LONG("rsAssemble",                 rsAssemble);
   REGISTER_CLASS_CONST_LONG("rsPrintHighRes",             rsPrintHighRes);
   REGISTER_CLASS_CONST_LONG("rsExlMetadata",              rsExlMetadata);
   REGISTER_CLASS_CONST_LONG("rsEmbFilesOnly",             rsEmbFilesOnly);

   REGISTER_CLASS_CONST_LONG("spHorizontal",               spHorizontal);
   REGISTER_CLASS_CONST_LONG("spVertical",                 spVertical);
   REGISTER_CLASS_CONST_LONG("spRDiagonal",                spRDiagonal);
   REGISTER_CLASS_CONST_LONG("spLDiagonal",                spLDiagonal);
   REGISTER_CLASS_CONST_LONG("spCross",                    spCross);
   REGISTER_CLASS_CONST_LONG("spDiaCross",                 spDiaCross);

   REGISTER_CLASS_CONST_LONG("ttConstSpacing",             ttConstSpacing);
   REGISTER_CLASS_CONST_LONG("ttNoDistortion",             ttNoDistortion);
   REGISTER_CLASS_CONST_LONG("ttFastConstSpacing",         ttFastConstSpacing);

   REGISTER_CLASS_CONST_LONG("plfDecimalArabic",           plfDecimalArabic);
   REGISTER_CLASS_CONST_LONG("plfUppercaseRoman",          plfUppercaseRoman);
   REGISTER_CLASS_CONST_LONG("plfLowercaseRoman",          plfLowercaseRoman);
   REGISTER_CLASS_CONST_LONG("plfUppercaseLetters",        plfUppercaseLetters);
   REGISTER_CLASS_CONST_LONG("plfLowercaseLetters",        plfLowercaseLetters);
   REGISTER_CLASS_CONST_LONG("plfNone",                    plfNone);

   REGISTER_CLASS_CONST_LONG("pvPDF_1_0",                  pvPDF_1_0);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_1",                  pvPDF_1_1);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_2",                  pvPDF_1_2);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_3",                  pvPDF_1_3);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_4",                  pvPDF_1_4);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_5",                  pvPDF_1_5);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_6",                  pvPDF_1_6);
   REGISTER_CLASS_CONST_LONG("pvPDF_1_7",                  pvPDF_1_7);
   REGISTER_CLASS_CONST_LONG("pvPDF_2_0",                  pvPDF_2_0);
   REGISTER_CLASS_CONST_LONG("pvPDFX1a_2001",              pvPDFX1a_2001);
   REGISTER_CLASS_CONST_LONG("pvPDFX1a_2003",              pvPDFX1a_2003);
   REGISTER_CLASS_CONST_LONG("pvPDFX3_2002",               pvPDFX3_2002);
   REGISTER_CLASS_CONST_LONG("pvPDFX3_2003",               pvPDFX3_2003);
   REGISTER_CLASS_CONST_LONG("pvPDFA_2005",                pvPDFA_2005);
   REGISTER_CLASS_CONST_LONG("pvPDFX_4",                   pvPDFX_4);
   REGISTER_CLASS_CONST_LONG("pvPDFA_1a",                  pvPDFA_1a);
   REGISTER_CLASS_CONST_LONG("pvPDFA_2a",                  pvPDFA_2a);
   REGISTER_CLASS_CONST_LONG("pvPDFA_2b",                  pvPDFA_2b);
   REGISTER_CLASS_CONST_LONG("pvPDFA_2u",                  pvPDFA_2u);
   REGISTER_CLASS_CONST_LONG("pvPDFA_3a",                  pvPDFA_3a);
   REGISTER_CLASS_CONST_LONG("pvPDFA_3b",                  pvPDFA_3b);
   REGISTER_CLASS_CONST_LONG("pvPDFA_3u",                  pvPDFA_3u);
   REGISTER_CLASS_CONST_LONG("pvZUGFeRD_Basic",            pvZUGFeRD_Basic);
   REGISTER_CLASS_CONST_LONG("pvZUGFeRD_Comfort",          pvZUGFeRD_Comfort);
   REGISTER_CLASS_CONST_LONG("pvZUGFeRD_Extended",         pvZUGFeRD_Extended);
   REGISTER_CLASS_CONST_LONG("pvFacturX_Minimum",          pvFacturX_Minimum);
   REGISTER_CLASS_CONST_LONG("pvFacturX_Basic",            pvFacturX_Basic);
   REGISTER_CLASS_CONST_LONG("pvFacturX_Basic_WL",         pvFacturX_Basic_WL);
   REGISTER_CLASS_CONST_LONG("pvFacturX_Comfort",          pvFacturX_Comfort);
   REGISTER_CLASS_CONST_LONG("pvFacturX_Extended",         pvFacturX_Extended);
   REGISTER_CLASS_CONST_LONG("pvFacturX_XRechnung",        pvFacturX_XRechnung);
   REGISTER_CLASS_CONST_LONG("pvZUGFeRD2_Flag",            pvZUGFeRD2_Flag);
   REGISTER_CLASS_CONST_LONG("pvPDFX5g",                   pvPDFX5g);
   REGISTER_CLASS_CONST_LONG("pvPDFX5n",                   pvPDFX5n);
   REGISTER_CLASS_CONST_LONG("pvPDFX5pg",                  pvPDFX5pg);
   REGISTER_CLASS_CONST_LONG("pvPDFX6",                    pvPDFX6);
   REGISTER_CLASS_CONST_LONG("pvPDFX6n",                   pvPDFX6n);
   REGISTER_CLASS_CONST_LONG("pvPDFX6p",                   pvPDFX6p);
   REGISTER_CLASS_CONST_LONG("pvPDFUA1",                   pvPDFUA1);

   REGISTER_CLASS_CONST_LONG("pvPDFX4p",                   pvPDFX4p);
   REGISTER_CLASS_CONST_LONG("pvPDFVT1",                   pvPDFVT1);
   REGISTER_CLASS_CONST_LONG("pvPDFVT2",                   pvPDFVT2);
   REGISTER_CLASS_CONST_LONG("pvPDFVT3",                   pvPDFVT3);
   REGISTER_CLASS_CONST_LONG("pvPDFVTMask",                pvPDFVTMask);

   REGISTER_CLASS_CONST_LONG("NO_COLOR",                   NO_COLOR);
   REGISTER_CLASS_CONST_LONG("PDF_ANNOT_INDEX",            PDF_ANNOT_INDEX);

   REGISTER_CLASS_CONST_LONG("PDF_AQUA",                   PDF_AQUA);
   REGISTER_CLASS_CONST_LONG("PDF_BLACK",                  PDF_BLACK);
   REGISTER_CLASS_CONST_LONG("PDF_BLUE",                   PDF_BLUE);
   REGISTER_CLASS_CONST_LONG("PDF_CREAM",                  PDF_CREAM);
   REGISTER_CLASS_CONST_LONG("PDF_DKGRAY",                 PDF_DKGRAY);
   REGISTER_CLASS_CONST_LONG("PDF_FUCHSIA",                PDF_FUCHSIA);
   REGISTER_CLASS_CONST_LONG("PDF_GRAY",                   PDF_GRAY);
   REGISTER_CLASS_CONST_LONG("PDF_GREEN",                  PDF_GREEN);
   REGISTER_CLASS_CONST_LONG("PDF_LIME",                   PDF_LIME);
   REGISTER_CLASS_CONST_LONG("PDF_LTGRAY",                 PDF_LTGRAY);
   REGISTER_CLASS_CONST_LONG("PDF_MAROON",                 PDF_MAROON);
   REGISTER_CLASS_CONST_LONG("PDF_MEDGRAY",                PDF_MEDGRAY);
   REGISTER_CLASS_CONST_LONG("PDF_MOGREEN",                PDF_MOGREEN);
   REGISTER_CLASS_CONST_LONG("PDF_NAVY",                   PDF_NAVY);
   REGISTER_CLASS_CONST_LONG("PDF_OLIVE",                  PDF_OLIVE);
   REGISTER_CLASS_CONST_LONG("PDF_PURPLE",                 PDF_PURPLE);
   REGISTER_CLASS_CONST_LONG("PDF_RED",                    PDF_RED);
   REGISTER_CLASS_CONST_LONG("PDF_SILVER",                 PDF_SILVER);
   REGISTER_CLASS_CONST_LONG("PDF_SKYBLUE",                PDF_SKYBLUE);
   REGISTER_CLASS_CONST_LONG("PDF_WHITE",                  PDF_WHITE);
   REGISTER_CLASS_CONST_LONG("PDF_TEAL",                   PDF_TEAL);
   REGISTER_CLASS_CONST_LONG("PDF_YELLOW",                 PDF_YELLOW);

   REGISTER_CLASS_CONST_LONG("dfMM_D",                     dfMM_D);
   REGISTER_CLASS_CONST_LONG("dfM_D_YY",                   dfM_D_YY);
   REGISTER_CLASS_CONST_LONG("dfMM_DD_YY",                 dfMM_DD_YY);
   REGISTER_CLASS_CONST_LONG("dfMM_YY",                    dfMM_YY);
   REGISTER_CLASS_CONST_LONG("dfD_MMM",                    dfD_MMM);
   REGISTER_CLASS_CONST_LONG("dfD_MMM_YY",                 dfD_MMM_YY);
   REGISTER_CLASS_CONST_LONG("dfDD_MMM_YY",                dfDD_MMM_YY);
   REGISTER_CLASS_CONST_LONG("dfYY_MM_DD",                 dfYY_MM_DD);
   REGISTER_CLASS_CONST_LONG("dfMMM_YY",                   dfMMM_YY);
   REGISTER_CLASS_CONST_LONG("dfMMMM_YY",                  dfMMMM_YY);
   REGISTER_CLASS_CONST_LONG("dfMMM_D_YYYY",               dfMMM_D_YYYY);
   REGISTER_CLASS_CONST_LONG("dfMMMM_D_YYYY",              dfMMMM_D_YYYY);
   REGISTER_CLASS_CONST_LONG("dfM_D_YY_H_MM_TT",           dfM_D_YY_H_MM_TT);
   REGISTER_CLASS_CONST_LONG("dfM_D_YY_HH_MM",             dfM_D_YY_HH_MM);
   REGISTER_CLASS_CONST_LONG("df24HR_MM",                  df24HR_MM);
   REGISTER_CLASS_CONST_LONG("df12HR_MM",                  df12HR_MM);
   REGISTER_CLASS_CONST_LONG("df24HR_MM_SS",               df24HR_MM_SS);
   REGISTER_CLASS_CONST_LONG("df12HR_MM_SS",               df12HR_MM_SS);

   REGISTER_CLASS_CONST_LONG("ffReadOnly",                 ffReadOnly);
   REGISTER_CLASS_CONST_LONG("ffRequired",                 ffRequired);
   REGISTER_CLASS_CONST_LONG("ffNoExport",                 ffNoExport);

   REGISTER_CLASS_CONST_LONG("ffInvisible",                ffInvisible);
   REGISTER_CLASS_CONST_LONG("ffHidden",                   ffHidden);
   REGISTER_CLASS_CONST_LONG("ffPrint",                    ffPrint);
   REGISTER_CLASS_CONST_LONG("ffNoZoom",                   ffNoZoom);
   REGISTER_CLASS_CONST_LONG("ffNoRotate",                 ffNoRotate);
   REGISTER_CLASS_CONST_LONG("ffNoView",                   ffNoView);

   REGISTER_CLASS_CONST_LONG("ffMultiline",                ffMultiline);
   REGISTER_CLASS_CONST_LONG("ffPassword",                 ffPassword);
   REGISTER_CLASS_CONST_LONG("ffNoToggleToOff",            ffNoToggleToOff);
   REGISTER_CLASS_CONST_LONG("ffRadioIsUnion",             ffRadioIsUnion);
   REGISTER_CLASS_CONST_LONG("ffCommitOnSelCh",            ffCommitOnSelCh);

   REGISTER_CLASS_CONST_LONG("ffEdit",                     ffEdit);
   REGISTER_CLASS_CONST_LONG("ffSorted",                   ffSorted);
   REGISTER_CLASS_CONST_LONG("ffFileSelect",               ffFileSelect);
   REGISTER_CLASS_CONST_LONG("ffMultiSelect",              ffMultiSelect);
   REGISTER_CLASS_CONST_LONG("ffDoNotSpellCheck",          ffDoNotSpellCheck);
   REGISTER_CLASS_CONST_LONG("ffDoNotScroll",              ffDoNotScroll);
   REGISTER_CLASS_CONST_LONG("ffComb",                     ffComb);

   REGISTER_CLASS_CONST_LONG("taLeft",                     taLeft);
   REGISTER_CLASS_CONST_LONG("taCenter",                   taCenter);
   REGISTER_CLASS_CONST_LONG("taRight",                    taRight);
   REGISTER_CLASS_CONST_LONG("taJustify",                  taJustify);
   REGISTER_CLASS_CONST_LONG("taPlainText",                taPlainText);

   REGISTER_CLASS_CONST_LONG("otAction",                   otAction);
   REGISTER_CLASS_CONST_LONG("otAnnotation",               otAnnotation);
   REGISTER_CLASS_CONST_LONG("otBookmark",                 otBookmark);
   REGISTER_CLASS_CONST_LONG("otCatalog",                  otCatalog);
   REGISTER_CLASS_CONST_LONG("otField",                    otField);
   REGISTER_CLASS_CONST_LONG("otPage",                     otPage);
   REGISTER_CLASS_CONST_LONG("otPageLink",                 otPageLink);

   REGISTER_CLASS_CONST_LONG("ddCounterClockwise",         ddCounterClockwise);
   REGISTER_CLASS_CONST_LONG("ddClockwise",                ddClockwise);

   REGISTER_CLASS_CONST_LONG("cmEvenOdd",                  cmEvenOdd);
   REGISTER_CLASS_CONST_LONG("cmWinding",                  cmWinding);

   REGISTER_CLASS_CONST_LONG("dtXY_Zoom",                  dtXY_Zoom);
   REGISTER_CLASS_CONST_LONG("dtFit",                      dtFit);
   REGISTER_CLASS_CONST_LONG("dtFitH_Top",                 dtFitH_Top);
   REGISTER_CLASS_CONST_LONG("dtFitV_Left",                dtFitV_Left);
   REGISTER_CLASS_CONST_LONG("dtFit_Rect",                 dtFit_Rect);
   REGISTER_CLASS_CONST_LONG("dtFitB",                     dtFitB);
   REGISTER_CLASS_CONST_LONG("dtFitBH_Top",                dtFitBH_Top);
   REGISTER_CLASS_CONST_LONG("dtFitBV_Left",               dtFitBV_Left);

   REGISTER_CLASS_CONST_LONG("ccCheck",                    ccCheck);
   REGISTER_CLASS_CONST_LONG("ccCircle",                   ccCircle);
   REGISTER_CLASS_CONST_LONG("ccCross1",                   ccCross1);
   REGISTER_CLASS_CONST_LONG("ccCross2",                   ccCross2);
   REGISTER_CLASS_CONST_LONG("ccCross3",                   ccCross3);
   REGISTER_CLASS_CONST_LONG("ccCross4",                   ccCross4);
   REGISTER_CLASS_CONST_LONG("ccDiamond",                  ccDiamond);
   REGISTER_CLASS_CONST_LONG("ccSquare",                   ccSquare);
   REGISTER_CLASS_CONST_LONG("ccStar",                     ccStar);

   REGISTER_CLASS_CONST_LONG("dsCommaDot",                 dsCommaDot);
   REGISTER_CLASS_CONST_LONG("dsNoneDot",                  dsNoneDot);
   REGISTER_CLASS_CONST_LONG("dsDotComma",                 dsDotComma);
   REGISTER_CLASS_CONST_LONG("dsNoneComma",                dsNoneComma);
   REGISTER_CLASS_CONST_LONG("dsQuoteDot",                 dsQuoteDot);

   REGISTER_CLASS_CONST_LONG("fcBackColor",                fcBackColor);
   REGISTER_CLASS_CONST_LONG("fcBorderColor",              fcBorderColor);
   REGISTER_CLASS_CONST_LONG("fcTextColor",                fcTextColor);

   REGISTER_CLASS_CONST_LONG("foOpen",                     foOpen);
   REGISTER_CLASS_CONST_LONG("foPrint",                    foPrint);

   REGISTER_CLASS_CONST_LONG("ftButton",                   ftButton);
   REGISTER_CLASS_CONST_LONG("ftCheckBox",                 ftCheckBox);
   REGISTER_CLASS_CONST_LONG("ftRadioBtn",                 ftRadioBtn);
   REGISTER_CLASS_CONST_LONG("ftComboBox",                 ftComboBox);
   REGISTER_CLASS_CONST_LONG("ftListBox",                  ftListBox);
   REGISTER_CLASS_CONST_LONG("ftText",                     ftText);
   REGISTER_CLASS_CONST_LONG("ftSignature",                ftSignature);
   REGISTER_CLASS_CONST_LONG("ftGroup",                    ftGroup);

   REGISTER_CLASS_CONST_LONG("hmNone",                     hmNone);
   REGISTER_CLASS_CONST_LONG("hmInvert",                   hmInvert);
   REGISTER_CLASS_CONST_LONG("hmOutline",                  hmOutline);
   REGISTER_CLASS_CONST_LONG("hmPush",                     hmPush);
   REGISTER_CLASS_CONST_LONG("hmPushUpd",                  hmPushUpd);

   REGISTER_CLASS_CONST_LONG("ifImportAll",                ifImportAll);
   REGISTER_CLASS_CONST_LONG("ifContentOnly",              ifContentOnly);
   REGISTER_CLASS_CONST_LONG("ifNoContent",                ifNoContent);
   REGISTER_CLASS_CONST_LONG("ifImportAsPage",             ifImportAsPage);
   REGISTER_CLASS_CONST_LONG("ifCatalogAction",            ifCatalogAction);
   REGISTER_CLASS_CONST_LONG("ifPageActions",              ifPageActions);
   REGISTER_CLASS_CONST_LONG("ifBookmarks",                ifBookmarks);
   REGISTER_CLASS_CONST_LONG("ifArticles",                 ifArticles);
   REGISTER_CLASS_CONST_LONG("ifPageLabels",               ifPageLabels);
   REGISTER_CLASS_CONST_LONG("ifThumbs",                   ifThumbs);
   REGISTER_CLASS_CONST_LONG("ifTranspGroups",             ifTranspGroups);
   REGISTER_CLASS_CONST_LONG("ifSeparationInfo",           ifSeparationInfo);
   REGISTER_CLASS_CONST_LONG("ifBoxColorInfo",             ifBoxColorInfo);
   REGISTER_CLASS_CONST_LONG("ifStructureTree",            ifStructureTree);
   REGISTER_CLASS_CONST_LONG("ifTransition",               ifTransition);
   REGISTER_CLASS_CONST_LONG("ifSearchIndex",              ifSearchIndex);
   REGISTER_CLASS_CONST_LONG("ifJavaScript",               ifJavaScript);
   REGISTER_CLASS_CONST_LONG("ifJSActions",                ifJSActions);
   REGISTER_CLASS_CONST_LONG("ifDocInfo",                  ifDocInfo);
   REGISTER_CLASS_CONST_LONG("ifEmbeddedFiles",            ifEmbeddedFiles);
   REGISTER_CLASS_CONST_LONG("ifFileCollections",          ifFileCollections);
   REGISTER_CLASS_CONST_LONG("ifAllAnnots",                ifAllAnnots);
   REGISTER_CLASS_CONST_LONG("ifFreeText",                 ifFreeText);
   REGISTER_CLASS_CONST_LONG("ifTextAnnot",                ifTextAnnot);
   REGISTER_CLASS_CONST_LONG("ifLink",                     ifLink);
   REGISTER_CLASS_CONST_LONG("ifStamp",                    ifStamp);
   REGISTER_CLASS_CONST_LONG("if3DAnnot",                  if3DAnnot);
   REGISTER_CLASS_CONST_LONG("ifOtherAnnots",              ifOtherAnnots);
   REGISTER_CLASS_CONST_LONG("ifFormFields",               ifFormFields);
   REGISTER_CLASS_CONST_LONG("ifPieceInfo",                ifPieceInfo);
   REGISTER_CLASS_CONST_LONG("ifPrepareForPDFA",           ifPrepareForPDFA);
   REGISTER_CLASS_CONST_LONG("ifEnumFonts",                ifEnumFonts);
   REGISTER_CLASS_CONST_LONG("ifAllPageObjects",           ifAllPageObjects);

   REGISTER_CLASS_CONST_LONG("if2MergeLayers",             if2MergeLayers);
   REGISTER_CLASS_CONST_LONG("if2Normalize",               if2Normalize);
   REGISTER_CLASS_CONST_LONG("if2UseProxy",                if2UseProxy);
   REGISTER_CLASS_CONST_LONG("if2NoMetadata",              if2NoMetadata);
   REGISTER_CLASS_CONST_LONG("if2DuplicateCheck",          if2DuplicateCheck);
   REGISTER_CLASS_CONST_LONG("if2NoResNameCheck",          if2NoResNameCheck);
   REGISTER_CLASS_CONST_LONG("if2CopyEncryptDict",         if2CopyEncryptDict);

   REGISTER_CLASS_CONST_LONG("naFirstPage",                naFirstPage);
   REGISTER_CLASS_CONST_LONG("naLastPage",                 naLastPage);
   REGISTER_CLASS_CONST_LONG("naNextPage",                 naNextPage);
   REGISTER_CLASS_CONST_LONG("naPrevPage",                 naPrevPage);
   REGISTER_CLASS_CONST_LONG("naGoBack",                   naGoBack);
   REGISTER_CLASS_CONST_LONG("naOpenDlg",                  naOpenDlg);
   REGISTER_CLASS_CONST_LONG("naPrintDlg",                 naPrintDlg);
   REGISTER_CLASS_CONST_LONG("naGeneralInfo",              naGeneralInfo);
   REGISTER_CLASS_CONST_LONG("naFontsInfo",                naFontsInfo);
   REGISTER_CLASS_CONST_LONG("naSaveAs",                   naSaveAs);
   REGISTER_CLASS_CONST_LONG("naSecurityInfo",             naSecurityInfo);
   REGISTER_CLASS_CONST_LONG("naFitPage",                  naFitPage);

   REGISTER_CLASS_CONST_LONG("nsMinusBlack",               nsMinusBlack);
   REGISTER_CLASS_CONST_LONG("nsRed",                      nsRed);
   REGISTER_CLASS_CONST_LONG("nsParensBlack",              nsParensBlack);
   REGISTER_CLASS_CONST_LONG("nsParensRed",                nsParensRed);

   REGISTER_CLASS_CONST_LONG("oeNoEvent",                  oeNoEvent);
   REGISTER_CLASS_CONST_LONG("oeOnOpen",                   oeOnOpen);
   REGISTER_CLASS_CONST_LONG("oeOnClose",                  oeOnClose);
   REGISTER_CLASS_CONST_LONG("oeOnMouseUp",                oeOnMouseUp);
   REGISTER_CLASS_CONST_LONG("oeOnMouseEnter",             oeOnMouseEnter);
   REGISTER_CLASS_CONST_LONG("oeOnMouseExit",              oeOnMouseExit);
   REGISTER_CLASS_CONST_LONG("oeOnMouseDown",              oeOnMouseDown);
   REGISTER_CLASS_CONST_LONG("oeOnFocus",                  oeOnFocus);
   REGISTER_CLASS_CONST_LONG("oeOnBlur",                   oeOnBlur);
   REGISTER_CLASS_CONST_LONG("oeOnKeyStroke",              oeOnKeyStroke);
   REGISTER_CLASS_CONST_LONG("oeOnFormat",                 oeOnFormat);
   REGISTER_CLASS_CONST_LONG("oeOnCalc",                   oeOnCalc);
   REGISTER_CLASS_CONST_LONG("oeOnValidate",               oeOnValidate);
   REGISTER_CLASS_CONST_LONG("oeOnPageVisible",            oeOnPageVisible);
   REGISTER_CLASS_CONST_LONG("oeOnPageInVisible",          oeOnPageInVisible);
   REGISTER_CLASS_CONST_LONG("oeOnPageOpen",               oeOnPageOpen);
   REGISTER_CLASS_CONST_LONG("oeOnPageClose",              oeOnPageClose);
   REGISTER_CLASS_CONST_LONG("oeOnBeforeClosing",          oeOnBeforeClosing);
   REGISTER_CLASS_CONST_LONG("oeOnBeforeSaving",           oeOnBeforeSaving);
   REGISTER_CLASS_CONST_LONG("oeOnAfterSaving",            oeOnAfterSaving);
   REGISTER_CLASS_CONST_LONG("oeOnBeforePrinting",         oeOnBeforePrinting);
   REGISTER_CLASS_CONST_LONG("oeOnAfterPrinting",          oeOnAfterPrinting);

   REGISTER_CLASS_CONST_LONG("oiDesign",                   oiDesign);
   REGISTER_CLASS_CONST_LONG("oiView",                     oiView);
   REGISTER_CLASS_CONST_LONG("oiAll",                      oiAll);
   REGISTER_CLASS_CONST_LONG("oiEmpty",                    oiEmpty);
   REGISTER_CLASS_CONST_LONG("oiVisible",                  oiVisible);

   REGISTER_CLASS_CONST_LONG("peBackgroundImage",          peBackgroundImage);
   REGISTER_CLASS_CONST_LONG("peForegroundImage",          peForegroundImage);
   REGISTER_CLASS_CONST_LONG("peHeaderFooter",             peHeaderFooter);
   REGISTER_CLASS_CONST_LONG("peLogo",                     peLogo);
   REGISTER_CLASS_CONST_LONG("peNone",                     peNone);

   REGISTER_CLASS_CONST_LONG("oucNone",                    oucNone);
   REGISTER_CLASS_CONST_LONG("oucExport",                  oucExport);
   REGISTER_CLASS_CONST_LONG("oucLanguage",                oucLanguage);
   REGISTER_CLASS_CONST_LONG("oucPrint",                   oucPrint);
   REGISTER_CLASS_CONST_LONG("oucUser",                    oucUser);
   REGISTER_CLASS_CONST_LONG("oucView",                    oucView);
   REGISTER_CLASS_CONST_LONG("oucZoom",                    oucZoom);

   REGISTER_CLASS_CONST_LONG("utIndividual",               utIndividual);
   REGISTER_CLASS_CONST_LONG("utOrganization",             utOrganization);
   REGISTER_CLASS_CONST_LONG("utTitle",                    utTitle);
   REGISTER_CLASS_CONST_LONG("utNotSet",                   utNotSet);

   REGISTER_CLASS_CONST_LONG("ooAnnotation",               ooAnnotation);
   REGISTER_CLASS_CONST_LONG("ooField",                    ooField);
   REGISTER_CLASS_CONST_LONG("ooImage",                    ooImage);
   REGISTER_CLASS_CONST_LONG("ooTemplate",                 ooTemplate);

   REGISTER_CLASS_CONST_LONG("ovAllOff",                   ovAllOff);
   REGISTER_CLASS_CONST_LONG("ovAllOn",                    ovAllOn);
   REGISTER_CLASS_CONST_LONG("ovAnyOff",                   ovAnyOff);
   REGISTER_CLASS_CONST_LONG("ovAnyOn",                    ovAnyOn);
   REGISTER_CLASS_CONST_LONG("ovNotSet",                   ovNotSet);

   REGISTER_CLASS_CONST_LONG("pffDefault",                 pffDefault);
   REGISTER_CLASS_CONST_LONG("pff1Bit",                    pff1Bit);
   REGISTER_CLASS_CONST_LONG("pffColor",                   pffColor);
   REGISTER_CLASS_CONST_LONG("pffAutoRotateAndCenter",     pffAutoRotateAndCenter);
   REGISTER_CLASS_CONST_LONG("pffPrintAsImage",            pffPrintAsImage);
   REGISTER_CLASS_CONST_LONG("pffShrinkToPrintArea",       pffShrinkToPrintArea);
   REGISTER_CLASS_CONST_LONG("pffNoStartDoc",              pffNoStartDoc);
   REGISTER_CLASS_CONST_LONG("pffNoStartPage",             pffNoStartPage);
   REGISTER_CLASS_CONST_LONG("pffNoEndDoc",                pffNoEndDoc);

   REGISTER_CLASS_CONST_LONG("plSinglePage",               plSinglePage);
   REGISTER_CLASS_CONST_LONG("plOneColumn",                plOneColumn);
   REGISTER_CLASS_CONST_LONG("plTwoColumnLeft",            plTwoColumnLeft);
   REGISTER_CLASS_CONST_LONG("plTwoColumnRight",           plTwoColumnRight);
   REGISTER_CLASS_CONST_LONG("plTwoPageLeft",              plTwoPageLeft);
   REGISTER_CLASS_CONST_LONG("plTwoPageRight",             plTwoPageRight);
   REGISTER_CLASS_CONST_LONG("plDefault",                  plDefault);

   REGISTER_CLASS_CONST_LONG("pmUseNone",                  pmUseNone);
   REGISTER_CLASS_CONST_LONG("pmUseOutlines",              pmUseOutlines);
   REGISTER_CLASS_CONST_LONG("pmUseThumbs",                pmUseThumbs);
   REGISTER_CLASS_CONST_LONG("pmFullScreen",               pmFullScreen);
   REGISTER_CLASS_CONST_LONG("pmUseOC",                    pmUseOC);
   REGISTER_CLASS_CONST_LONG("pmUseAttachments",           pmUseAttachments);

   REGISTER_CLASS_CONST_LONG("ptOpen",                     ptOpen);
   REGISTER_CLASS_CONST_LONG("ptOwner",                    ptOwner);

   REGISTER_CLASS_CONST_LONG("ptImportPage",               ptImportPage);
   REGISTER_CLASS_CONST_LONG("ptWritePage",                ptWritePage);
   REGISTER_CLASS_CONST_LONG("ptPrintPage",                ptPrintPage);
   REGISTER_CLASS_CONST_LONG("ptConvertPage",              ptConvertPage);

   REGISTER_CLASS_CONST_LONG("sfNone",                     sfNone);
   REGISTER_CLASS_CONST_LONG("sfExlude",                   sfExlude);
   REGISTER_CLASS_CONST_LONG("sfInclNoValFields",          sfInclNoValFields);
   REGISTER_CLASS_CONST_LONG("sfHTML",                     sfHTML);
   REGISTER_CLASS_CONST_LONG("sfGetMethod",                sfGetMethod);
   REGISTER_CLASS_CONST_LONG("sfSubmCoords",               sfSubmCoords);
   REGISTER_CLASS_CONST_LONG("sfXML",                      sfXML);
   REGISTER_CLASS_CONST_LONG("sfInclAppSaves",             sfInclAppSaves);
   REGISTER_CLASS_CONST_LONG("sfInclAnnots",               sfInclAnnots);
   REGISTER_CLASS_CONST_LONG("sfPDF",                      sfPDF);
   REGISTER_CLASS_CONST_LONG("sfCanonicalFormat",          sfCanonicalFormat);
   REGISTER_CLASS_CONST_LONG("sfExlNonUserAnnots",         sfExlNonUserAnnots);
   REGISTER_CLASS_CONST_LONG("sfExlFKey",                  sfExlFKey);
   REGISTER_CLASS_CONST_LONG("sfEmbedForm",                sfEmbedForm);

   REGISTER_CLASS_CONST_LONG("icmDefault",                 icmDefault);
   REGISTER_CLASS_CONST_LONG("icmBPCompensation",          icmBPCompensation);
   REGISTER_CLASS_CONST_LONG("icmCheckBlackPoint",         icmCheckBlackPoint);

   REGISTER_CLASS_CONST_LONG("pxf1Bit",                    pxf1Bit);
   REGISTER_CLASS_CONST_LONG("pxfGray",                    pxfGray);
   REGISTER_CLASS_CONST_LONG("pxfRGB",                     pxfRGB);
   REGISTER_CLASS_CONST_LONG("pxfBGR",                     pxfBGR);
   REGISTER_CLASS_CONST_LONG("pxfRGBA",                    pxfRGBA);
   REGISTER_CLASS_CONST_LONG("pxfBGRA",                    pxfBGRA);
   REGISTER_CLASS_CONST_LONG("pxfARGB",                    pxfARGB);
   REGISTER_CLASS_CONST_LONG("pxfABGR",                    pxfABGR);
   REGISTER_CLASS_CONST_LONG("pxfGrayA",                   pxfGrayA);
   REGISTER_CLASS_CONST_LONG("pxfCMYK",                    pxfCMYK);
   REGISTER_CLASS_CONST_LONG("pxfCMYKA",                   pxfCMYKA);

   REGISTER_CLASS_CONST_LONG("psFitWidth",                 psFitWidth);
   REGISTER_CLASS_CONST_LONG("psFitHeight",                psFitHeight);
   REGISTER_CLASS_CONST_LONG("psFitBest",                  psFitBest);
   REGISTER_CLASS_CONST_LONG("psFitZoom",                  psFitZoom);

   REGISTER_CLASS_CONST_LONG("rfDefault",                  rfDefault);
   REGISTER_CLASS_CONST_LONG("rfScaleToMediaBox",          rfScaleToMediaBox);
   REGISTER_CLASS_CONST_LONG("rfIgnoreCropBox",            rfIgnoreCropBox);
   REGISTER_CLASS_CONST_LONG("rfClipToArtBox",             rfClipToArtBox);
   REGISTER_CLASS_CONST_LONG("rfClipToBleedBox",           rfClipToBleedBox);
   REGISTER_CLASS_CONST_LONG("rfClipToTrimBox",            rfClipToTrimBox);
   REGISTER_CLASS_CONST_LONG("rfExclAnnotations",          rfExclAnnotations);
   REGISTER_CLASS_CONST_LONG("rfExclFormFields",           rfExclFormFields);
   REGISTER_CLASS_CONST_LONG("rfSkipUpdateBG",             rfSkipUpdateBG);
   REGISTER_CLASS_CONST_LONG("rfRotate90",                 rfRotate90);
   REGISTER_CLASS_CONST_LONG("rfRotate180",                rfRotate180);
   REGISTER_CLASS_CONST_LONG("rfRotate270",                rfRotate270);
   REGISTER_CLASS_CONST_LONG("rfInitBlack",                rfInitBlack);
   REGISTER_CLASS_CONST_LONG("rfCompositeWhite",           rfCompositeWhite);

   REGISTER_CLASS_CONST_LONG("rfExclPageContent",          rfExclPageContent);
   REGISTER_CLASS_CONST_LONG("rfExclButtons",              rfExclButtons);
   REGISTER_CLASS_CONST_LONG("rfExclCheckBoxes",           rfExclCheckBoxes);
   REGISTER_CLASS_CONST_LONG("rfExclComboBoxes",           rfExclComboBoxes);
   REGISTER_CLASS_CONST_LONG("rfExclListBoxes",            rfExclListBoxes);
   REGISTER_CLASS_CONST_LONG("rfExclTextFields",           rfExclTextFields);
   REGISTER_CLASS_CONST_LONG("rfExclSigFields",            rfExclSigFields);

   REGISTER_CLASS_CONST_LONG("rfScaleToBBox",              rfScaleToBBox);
   REGISTER_CLASS_CONST_LONG("rfDisableAAClipping",        rfDisableAAClipping);
   REGISTER_CLASS_CONST_LONG("rfDisableAAText",            rfDisableAAText);
   REGISTER_CLASS_CONST_LONG("rfDisableAAVector",          rfDisableAAVector);
   REGISTER_CLASS_CONST_LONG("rfDisableAntiAliasing",      rfDisableAntiAliasing);
   REGISTER_CLASS_CONST_LONG("rfDisableBiLinearFilter",    rfDisableBiLinearFilter);
   REGISTER_CLASS_CONST_LONG("rfRenderInvisibleText",      rfRenderInvisibleText);
   REGISTER_CLASS_CONST_LONG("rfEnableBlendCS",            rfEnableBlendCS);
   REGISTER_CLASS_CONST_LONG("rfRenderPrintState",         rfRenderPrintState);

   REGISTER_CLASS_CONST_LONG("psAppDefault",               psAppDefault);
   REGISTER_CLASS_CONST_LONG("psNone",                     psNone);

   REGISTER_CLASS_CONST_LONG("rifDefault",                 rifDefault);
   REGISTER_CLASS_CONST_LONG("rifDeleteAltImages",         rifDeleteAltImages);
   REGISTER_CLASS_CONST_LONG("rifDeleteMetadata",          rifDeleteMetadata);
   REGISTER_CLASS_CONST_LONG("rifDeleteOCG",               rifDeleteOCG);
   REGISTER_CLASS_CONST_LONG("rifDeleteSoftMask",          rifDeleteSoftMask);

   REGISTER_CLASS_CONST_LONG("rifByteAligned",             rifByteAligned);
   REGISTER_CLASS_CONST_LONG("rifRGBData",                 rifRGBData);
   REGISTER_CLASS_CONST_LONG("rifCMYKData",                rifCMYKData);

   REGISTER_CLASS_CONST_LONG("riAbsoluteColorimetric",     riAbsoluteColorimetric);
   REGISTER_CLASS_CONST_LONG("riPerceptual",               riPerceptual);
   REGISTER_CLASS_CONST_LONG("riRelativeColorimetric",     riRelativeColorimetric);
   REGISTER_CLASS_CONST_LONG("riSaturation",               riSaturation);
   REGISTER_CLASS_CONST_LONG("riNone",                     riNone);

   REGISTER_CLASS_CONST_LONG("rsApproved",                 rsApproved);
   REGISTER_CLASS_CONST_LONG("rsAsIs",                     rsAsIs);
   REGISTER_CLASS_CONST_LONG("rsConfidential",             rsConfidential);
   REGISTER_CLASS_CONST_LONG("rsDepartmental",             rsDepartmental);
   REGISTER_CLASS_CONST_LONG("rsDraft",                    rsDraft);
   REGISTER_CLASS_CONST_LONG("rsExperimental",             rsExperimental);
   REGISTER_CLASS_CONST_LONG("rsExpired",                  rsExpired);
   REGISTER_CLASS_CONST_LONG("rsFinal",                    rsFinal);
   REGISTER_CLASS_CONST_LONG("rsForComment",               rsForComment);
   REGISTER_CLASS_CONST_LONG("rsForPublicRelease",         rsForPublicRelease);
   REGISTER_CLASS_CONST_LONG("rsNotApproved",              rsNotApproved);
   REGISTER_CLASS_CONST_LONG("rsNotForPublicRelease",      rsNotForPublicRelease);
   REGISTER_CLASS_CONST_LONG("rsSold",                     rsSold);
   REGISTER_CLASS_CONST_LONG("rsTopSecret",                rsTopSecret);
   REGISTER_CLASS_CONST_LONG("rsUserDefined",              rsUserDefined);

   REGISTER_CLASS_CONST_LONG("stUnknown",                  stUnknown);
   REGISTER_CLASS_CONST_LONG("stFunctionBased",            stFunctionBased);
   REGISTER_CLASS_CONST_LONG("stAxial",                    stAxial);
   REGISTER_CLASS_CONST_LONG("stRadial",                   stRadial);
   REGISTER_CLASS_CONST_LONG("stFreeFormGouraud",          stFreeFormGouraud);
   REGISTER_CLASS_CONST_LONG("stLatticeFormGouraud",       stLatticeFormGouraud);
   REGISTER_CLASS_CONST_LONG("stCoonsPatch",               stCoonsPatch);
   REGISTER_CLASS_CONST_LONG("stTensorProduct",            stTensorProduct);

   REGISTER_CLASS_CONST_LONG("smtAlpha",                   smtAlpha);
   REGISTER_CLASS_CONST_LONG("smtLuminosity",              smtLuminosity);

   REGISTER_CLASS_CONST_LONG("spcDefault",                 spcDefault);
   REGISTER_CLASS_CONST_LONG("spcIgnorePaperFormat",       spcIgnorePaperFormat);
   REGISTER_CLASS_CONST_LONG("spcDontAddMargins",          spcDontAddMargins);
   REGISTER_CLASS_CONST_LONG("spcLoadSpoolFontsOnly",      spcLoadSpoolFontsOnly);
   REGISTER_CLASS_CONST_LONG("spcFlushPages",              spcFlushPages);

   // Table API
   REGISTER_CLASS_CONST_LONG("cctText",         cctText);
   REGISTER_CLASS_CONST_LONG("cctImage",        cctImage);
   REGISTER_CLASS_CONST_LONG("cctTable",        cctTable);
   REGISTER_CLASS_CONST_LONG("cctTemplate",     cctTemplate);

   REGISTER_CLASS_CONST_LONG("dcText",          dcText);
   REGISTER_CLASS_CONST_LONG("dcImage",         dcImage);
   REGISTER_CLASS_CONST_LONG("dcTemplate",      dcTemplate);
   REGISTER_CLASS_CONST_LONG("dcTable",         dcTable);
   REGISTER_CLASS_CONST_LONG("dcAllCont",       dcAllCont);
   REGISTER_CLASS_CONST_LONG("dcForeGround",    dcForeGround);
   REGISTER_CLASS_CONST_LONG("dcBackGround",    dcBackGround);
   REGISTER_CLASS_CONST_LONG("dcBoth",          dcBoth);

   REGISTER_CLASS_CONST_LONG("coLeft",          coLeft);
   REGISTER_CLASS_CONST_LONG("coTop",           coTop);
   REGISTER_CLASS_CONST_LONG("coRight",         coRight);
   REGISTER_CLASS_CONST_LONG("coBottom",        coBottom);
   REGISTER_CLASS_CONST_LONG("coCenter",        coCenter);

   REGISTER_CLASS_CONST_LONG("coaUniqueWidth",  coaUniqueWidth);
   REGISTER_CLASS_CONST_LONG("coaAdjLeft",      coaAdjLeft);
   REGISTER_CLASS_CONST_LONG("coaAdjRight",     coaAdjRight);

   REGISTER_CLASS_CONST_LONG("tcBackColor",     tcBackColor);
   REGISTER_CLASS_CONST_LONG("tcBorderColor",   tcBorderColor);
   REGISTER_CLASS_CONST_LONG("tcGridHorzColor", tcGridHorzColor);
   REGISTER_CLASS_CONST_LONG("tcGridVertColor", tcGridVertColor);
   REGISTER_CLASS_CONST_LONG("tcImageColor",    tcImageColor);
   REGISTER_CLASS_CONST_LONG("tcTextColor",     tcTextColor);

   REGISTER_CLASS_CONST_LONG("tbpBorderWidth",  tbpBorderWidth);
   REGISTER_CLASS_CONST_LONG("tbpCellSpacing",  tbpCellSpacing);
   REGISTER_CLASS_CONST_LONG("tbpCellPadding",  tbpCellPadding);

   REGISTER_CLASS_CONST_LONG("tefDefault",      tefDefault);
   REGISTER_CLASS_CONST_LONG("tefSortTextX",    tefSortTextX);
   REGISTER_CLASS_CONST_LONG("tefSortTextY",    tefSortTextY);
   REGISTER_CLASS_CONST_LONG("tefSortTextXY",   tefSortTextXY);
   REGISTER_CLASS_CONST_LONG("tefDeleteOverlappingText", tefDeleteOverlappingText);

   REGISTER_CLASS_CONST_LONG("tfDefault",       tfDefault);
   REGISTER_CLASS_CONST_LONG("tfStatic",        tfStatic);
   REGISTER_CLASS_CONST_LONG("tfHeaderRow",     tfHeaderRow);
   REGISTER_CLASS_CONST_LONG("tfNoLineBreak",   tfNoLineBreak);
   REGISTER_CLASS_CONST_LONG("tfScaleToRect",   tfScaleToRect);
   REGISTER_CLASS_CONST_LONG("tfUseImageCS",    tfUseImageCS);
   REGISTER_CLASS_CONST_LONG("tfAddFlags",      tfAddFlags);

   return SUCCESS;
}
