#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "../dynapdf/dynapdf.h"

using namespace DynaPDF;

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   // The error type is a bitmask, see help file for further information.
   if (ErrType & E_WARNING)
      printf("Warning: %s\n", ErrMessage);
   else if (ErrType & E_SYNTAX_ERROR)
      printf("Syntax error!: %s\n", ErrMessage);
   else if (ErrType & E_VALUE_ERROR)
      printf("Value error: %s\n", ErrMessage);
   else if (ErrType & E_FONT_ERROR)
      printf("TrueType font error!: %s\n", ErrMessage);
   else if (ErrType & E_FILE_ERROR)
      printf("File error: %s\n", ErrMessage);
   else if (ErrType & E_FATAL_ERROR)
      printf("Fatal error: %s\n", ErrMessage);
   else
      printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

int main(void)
{
   void* pdf = pdfNewPDF();
   if (!pdf) return 2; // Out of memory?
   
   // On macOS we use UTF-8 encoded source files. String literals are UTF-8 encoded too in this case.
   // We must tell DynaPDF that Ansi strings are UTF-8 encoded.
   pdfSetGStateFlags(pdf, gfAnsiStringIsUTF8, false);
   
   char outFile[] = "personalize.pdf";
   // Error messages and warnings are passed to the callback function.
   pdfSetOnErrorProc(pdf, NULL, PDFError);
   if (!pdfCreateNewPDF(pdf, outFile))
   {
      pdfDeletePDF(pdf);
      return 1;
   }
   pdfSetDocInfo(pdf, diCreator, "C++ Example Project");
   pdfSetDocInfo(pdf, diSubject, "How to Edit an existing pdf file");
   pdfSetDocInfo(pdf, diTitle, "Edit pdf files");
   pdfSetDocInfo(pdf, diAuthor, "Jens Boschulte");
   pdfSetViewerPreferences(pdf, vpDisplayDocTitle, avNone);
   // Conversion of pages to templates is normally not required. Templates are required if
   // the page should be scaled or used multiple times in a document, e.g. as background.
   // See help file for further information.
   pdfSetImportFlags(pdf, ifImportAll + ifImportAsPage);
   pdfOpenImportFile(pdf, "test_files/taxform.pdf", ptOpen, NULL);
   pdfImportPDFFile(pdf, 1, 1.0, 1.0);

   pdfEditPage(pdf, 1);
   // We use the 14 standard fonts only because they are always available. However, in a real
   // project you should use fonts which can be embedded.
   pdfSetFont(pdf, "Courier", fsBold, 14.0, false, cp1252);

   pdfWriteText(pdf, 72.5, 748.5, "X");
   pdfWriteText(pdf, 74.0, 701.0, "Musterstadt");
   pdfWriteText(pdf, 74.0, 677.0, "252/1062/3323");

   pdfBeginContinueText(pdf, 74.0, 628.0);
   pdfSetLeading(pdf, 24.0);
   pdfSetCharacterSpacing(pdf, 5.8);
   pdfAddContinueText(pdf, "Mustermann");
   pdfAddContinueText(pdf, "Hermann");
   pdfAddContinueText(pdf, "22021963keineKaufmann");
   pdfAddContinueText(pdf, "Musterstraße 145");
   pdfAddContinueText(pdf, "12345Musterstadt");
   pdfSetCharacterSpacing(pdf, 0.0);
   pdfSetFont(pdf, "Courier", fsBold, 10.0, false, cp1252);
   pdfSetLeading(pdf, 47.5);
   pdfAddContinueText(pdf, "04.05.1994");
   pdfSetFont(pdf, "Courier", fsBold, 14.0, false, cp1252);
   pdfSetCharacterSpacing(pdf, 5.8);
   pdfAddContinueText(pdf, "Sabine");
   pdfAddContinueText(pdf, "18121966 ev  Hausfrau");
   pdfEndContinueText(pdf);

   pdfWriteText(pdf, 72.5, 365.0, "X");
   pdfWriteText(pdf, 396.0, 365.0, "X");

   pdfBeginContinueText(pdf, 74.0, 316.0);
   pdfSetLeading(pdf, 24.0);
   pdfAddContinueText(pdf, "2346256780     76834560");
   pdfAddContinueText(pdf, "Sparkasse Musterstadt");
   pdfEndContinueText(pdf);

   pdfWriteText(pdf, 72.5, 269.0, "X");
   pdfSetCharacterSpacing(pdf, 0.0);
   pdfSetFont(pdf, "Courier", fsNone, 10.0, false, cp1252);

   time_t t;
   struct tm* now;
   time(&t);
   now = gmtime(&t);
   char buf[64];
   snprintf(buf, 64, "%02d.%02d.%d", now->tm_mday +1, now->tm_mon + 1, now->tm_year + 1900);

   pdfWriteText(pdf, 53.0, 48.0, buf);
   pdfSetFillColor(pdf, PDF_RGB(0xFF, 0x66, 0x66));
   pdfSetFont(pdf, "Helvetica", fsBold, 22.0, false, cp1252);
   pdfWriteText(pdf, 340, 70, "www.dynaforms.de");
   pdfSetLineWidth(pdf, 0.0);
   pdfSetLinkHighlightMode(pdf, hmPush);
   pdfSetAnnotFlags(pdf, afReadOnly);
   pdfWebLink(pdf, 340, 64, 204, 22, "http://www.dynaforms.de");
   pdfEndPage(pdf);
   pdfCloseFile(pdf);

   pdfDeletePDF(pdf);
   printf("PDF file %s successfully created!\n", outFile);
   return 0;
}
