#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

/*
   Note that the dynapdf.dll must be copied into the output directory or into a
   Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
*/

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

// Helper function
void* GetFileBuf(const char* FileName, SI32 &BufSize)
{
   FILE* f = fopen(FileName, "rb");
   if (!f)
   {
      BufSize = 0;
      return NULL;
   }
   fseek(f, 0, SEEK_END);
   BufSize = ftell(f);
   fseek(f, 0, SEEK_SET);
   char* retval = (char*)malloc(BufSize +1);
   if (!retval) return NULL;
   fread(retval, 1, BufSize, f);
   // Not required but debugging is easier if the buffer is null-terminated
   retval[BufSize] = 0;
   fclose(f);
   return retval;
}

bool HaveEInvoice(const PPDF* PDF, const char* InFileName, HANDLE Console)
{
   bool retval = false;
   TPDFVersionInfo info;
   info.StructSize = sizeof(info);

   pdfCreateNewPDFW(PDF, NULL);
   // We need the document info or metadata and embedded files only
   pdfSetImportFlags(PDF, ifDocInfo | ifEmbeddedFiles);
   pdfSetImportFlags2(PDF, if2UseProxy);
   if (pdfOpenImportFile(PDF, InFileName, ptOpen, NULL) < 0) goto finish;

   // Other stuff can be ignored
   pdfImportCatalogObjects(PDF);

   if (!pdfGetPDFVersionEx(PDF, &info)) goto finish;

   if (info.PDFAVersion == 3 && info.FXDocName != NULL)
   {
      SI32 ef;
      TPDFFileSpec fs;
      if ((ef = pdfFindEmbeddedFile(PDF, info.FXDocName)) < 0)
      {
         SetConsoleTextAttribute(Console, 12);
         printf("Invoice %s not found!\n", info.FXDocName);
         goto finish;
      }
      if (ef != 0)
      {
         SetConsoleTextAttribute(Console, 14);
         printf("Warning: The invoice should be the first file attachment. This might cause unnecessary problems.\n");
      }
      retval = (pdfGetEmbeddedFile(PDF, ef, &fs, true) && fs.BufSize > 0);
   }
  finish:
   pdfFreePDF(PDF);
   return retval;
}

bool CreateInvoice(PPDF* PDF, bool FacturX, const char* InvoiceName, const char* OutFile)
{
   SI32 ef;
   bool retval = false;
   // The output file is opened later
   pdfCreateNewPDF(PDF, NULL);

   // Set the license key here if you have one
   // pdfSetLicenseKey(PDF, "");

   // We assume that the pdf invoice is already a valid PDF/A 3 file in this example.

   pdfSetImportFlags(PDF, ifImportAsPage | ifImportAll);
   if (pdfOpenImportFile(PDF, "../../../test_files/test_invoice.pdf", ptOpen, NULL) < 0) goto finish;

   pdfImportPDFFile(PDF, 1, 1.0, 1.0);

   /*
      The test invoice has the file name factur-x.xml. This is the right name for FacturX and ZUGFeRD output.
      However, for XRechnung the file name must be xrechnung.xml. So, we must be able to change the file
      name to get this example working. We could rename the input file but the usage of AttachFileEx() is probably
      more elegant.
      
      It is not of interest here whether the xml file is a valid XRechnung. This example shows how the
      PDF container must be created and how to extract the xml invoice from an arbitrary e-invoice.
   */
   
   SI32 bufSize = 0;
   void* buffer = GetFileBuf("../../../test_files/factur-x.xml", bufSize);
   
   ef = pdfAttachFileEx(PDF, buffer, bufSize, InvoiceName, "EN 19631 compliant invoice", false);
   
   free(buffer);

   pdfAssociateEmbFile(PDF, adCatalog, -1, FacturX ? arAlternative : arSource, ef);

   /*
      Note that ZUGFeRD 2.1 or higher and FacturX is identically defined in PDF. Therefore, both formats share
      the same version constants. Note also that the profiles Minimum, Basic, and Basic WL are not fully EN16931
      compliant, and therefore cannot be used to create e-invoices.
   */
   pdfSetPDFVersion(PDF, FacturX ? pvFacturX_Comfort : pvFacturX_XRechnung);

   // No fatal error occurred?
   if (pdfHaveOpenDoc(PDF))
   {
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(PDF, OutFile)) goto finish;
      retval = pdfCloseFile(PDF) != 0;
   }

  finish:
   pdfFreePDF(PDF);
   return retval;
}

int main(int argc, char* argv[])
{
   char filePath[MAX_PATH+1];
   if (argc < 1) return -1; // This should not occur
   PPDF* pdf = pdfNewPDF();
   if (!pdf) return 2; // Out of memory?

   pdfSetOnErrorProc(pdf, NULL, PDFError);

   GetCurrentDirectory(MAX_PATH, filePath);
   strcat(filePath, "/out.pdf");
   
   HANDLE hConsole = GetStdHandle(STD_OUTPUT_HANDLE);

   // Test cases:
   // - ZUGFeRD or FacturX
   // - XRechnung -> The invoice name must be xrechnung.xml
   if (!CreateInvoice(pdf, true,  "factur-x.xml",  filePath) || !HaveEInvoice(pdf, filePath, hConsole)
   ||  !CreateInvoice(pdf, false, "xrechnung.xml", filePath) || !HaveEInvoice(pdf, filePath, hConsole))
   {
      SetConsoleTextAttribute(hConsole, 12);
      printf("XML Invoice not found!");
   }else
   {
      SetConsoleTextAttribute(hConsole, 10);
      printf("All tests passed!");
   }

   SetConsoleTextAttribute(hConsole, 15);
   pdfDeletePDF(pdf);
   _getch();
   return 0;
}