#ifndef pdf_edit_textH
#define pdf_edit_textH

#define PDF_STATIC // Enable static binding when compiling the project with the workspace dynapdf_static.
#include "../../../include/C_CPP/dynapdf.h"
#include "pdf_text_buffer.h"

#include <stdio.h>
#include <stdlib.h>

/*
   The class CPDFEditText can be used to find, delete, and replace text in a PDF file.

   This algorithm handles correctly space characters, as well as rotated text and text
   lines which consist of multiple text or kerning records. The search text can be found
   arbitrary often in a text line and be replaced or deleted independend of the format in
   which it was found in the content stream. Text lines which consist of multiple text
   records are processed recursively.

   The text behind a replaced or deleted string is realined to avoid overlaping text or
   holes between text records. This reflow is restricted to the line in which the search
   text was found. It is of course possible to develop an algorithm that is able identify
   entire text blocks so that the text reflow can be applied on the entire block but this
   is rather complex...

   The algorithm works correctly if the text records are ordered from left to right and
   from top to bottom. This is usually the case but unordered PDF files exist and you
   create such a file here if you replace a text with a new one.

   So, the new content stream is no longer well ordered because the new text is always
   stored at the end of the content stream.
*/

using namespace DynaPDF;

struct TTextRec
{
   SI32  TmplHandle;
   UI32  First;
   UI32  Last;
   SI32  KernRecord;
   SI32  StrPos;
   bool  NewLine;
};

template<class T>class CTList
{
  public:
   CTList(void) :
      m_Capacity(0),
      m_Count(0),
      m_Items(NULL)
   {}
   ~CTList(void)
   {
      if (m_Items) free(m_Items);
   }
   T* Add(void)
   {
      if (m_Count + 1 > m_Capacity)
      {
         T* tmp = (T*)realloc(m_Items, (m_Capacity + 4196) * sizeof(T));
         if (!tmp) throw "Out of memory!";
         m_Items     = tmp;
         m_Capacity += 4196;
      }
      return &m_Items[m_Count++];
   }
   void Clear(void)               {m_Count = 0;}
   UI32 Count(void)         const {return m_Count;}
   T*   GetItem(UI32 Index) const {return (m_Items + Index);}
  private:
   UI32 m_Capacity;
   UI32 m_Count;
   T*   m_Items;
};

#define MAX_LINE_ERROR 2.0

class CPDFEditText
{
  public:
   CPDFEditText(const void* PDFInst);
  ~CPDFEditText(void);
   UI32 FindPattern(const wchar_t* Text);
   void ReplacePattern(const wchar_t* NewText);
  protected:
   SI32             m_Alpha;
   SI32             m_CurrTmpl;
   SI32             m_First;
   LBOOL            m_HaveMore;
   SI32             m_KernRecord;
   const void*      m_LastFont;
   double           m_LastX;
   double           m_LastY;
   TCTM             m_Matrix1;
   TCTM             m_Matrix2;
   TCTM             m_Matrix3;
   TCTM             m_Matrix4;
   bool             m_NewLine;
   const void*      m_PDFInst;
   UI32             m_RecordNumber;
   CTList<TTextRec> m_Records;
   UI16*            m_SearchPos;
   UI16*            m_SearchText;
   UI32             m_SearchTextLen;
   TPDFStack        m_Stack;
   SI32             m_StrPos;
   CTList<SI32>     m_Templates;

   void   AddKernSpace(float Advance, float SpaceWidth, TCTM &Matrix);
   void   AddKernSpaceEx(float Advance, float SpaceWidth, TCTM &Matrix);
   void   AddRecord(SI32 KernRecord, SI32 StrPos);
   void   AddSpace(double x, double y, double SpaceWidth);
   void   AddSpace(double DistX, double SpaceWidth, TCTM &Matrix);
   void   CalcAlpha(TCTM &M);
   void   CalcDistance(TCTM &M1, TCTM &M2, TCTM &M3, double &DistX, double &DistY, double x, double y);
   double CalcDistance(double x1, double y1, double x2, double y2);
   void   CalcStrPos(TCTM &M, double &x, double &y);
   void   FindAll(void);
   UI32   FindEndPattern(const UI16* NewText, UI32 Len);
   void   FindPattern(void);
   double GetScaleFactor(TCTM& M);
   void   Invert(TCTM &M);
   TCTM   MulMatrix(TCTM &M1, TCTM &M2);
   void   ParseTemplates(void);
   SI32   Pos(const char* Text, BYTE AChar);
   void   SetFont(void);
   void   Transform(TCTM &M, double &x, double &y);
   UI32   WriteLine(UI32 Index, TTextRec* Record, const UI16* NewText, UI32 Len);
   void   WriteRecord(TCTM &M, TTextRec* Record, TTextRecordW &Source, const UI16* Text, UI32 Len);
   UI32   WriteRemaining(UI32 Index, TTextRec* Record, const UI16* NewText, UI32 Len);
   void   WriteText(TCTM &M, const UI16* Text, UI32 Len);
};

#endif
