#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <time.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../win32/dynapdf.lib")  // Multithreaded
#endif

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

int main(int argc, char* argv[])
{
   SI32 sigField, sh;
   char filePath[512];
   PPDF* pdf = pdfNewPDF();
   if (!pdf) return -2; // Out of memory?

   pdfSetOnErrorProc(pdf, NULL, PDFError);
   pdfCreateNewPDF(pdf, NULL); // The output file is created later
   pdfSetDocInfo(pdf, diCreator, "C Test application");
   pdfSetDocInfo(pdf, diTitle, "PDF/A and digital signatures");

   pdfAppend(pdf);
   pdfSetFont(pdf, "Arial", fsNone, 10.0, true, cp1252);
   pdfWriteFText(pdf, taLeft, "This is a PDF/A 1b compatible PDF file that was digitally signed with "
      "a self sign certificate. Because PDF/A requires that all fonts are embedded it is important "
      "to avoid the usage of the 14 Standard fonts.\n\n"

      "When signing a PDF/A compliant PDF file with the default settings (without creation of a user "
      "defined appearance) the font Arial must be available on the system because it is used to print "
      "the certificate properties into the signature field.\n\n"

      "The font Arial must also be available if an empty signature field was added to the file "
      "without signing it when closing the PDF file. Yes, it is still possible to sign a PDF/A "
      "compliant PDF file later with Adobe's Acrobat. The signed PDF file is still compatible "
      "to PDF/A. If you use a third party solution to digitally sign the PDF file then test "
      "whether the signed file is still valid with the PDF/A 1b preflight tool included in Acrobat 8 "
      "Professional.\n\n"

      "Signature fields must be visible and the print flag must be set (default). CheckConformance() "
      "adjusts these flags if necessary and produces a warning if changes were applied. If no changes "
      "should be allowed, just return -1 in the error callback function. If the error callback function "
      "returns 0, DynaPDF assumes that the prior changes were accepted and processing continues.\n\n"

      "\\FC[255]Notice:\\FC[0]\n"
      "It makes no sense to execute CheckConformance() without an error callback function or error event "
      "in VB. If you cannot see what happens during the execution of CheckConformance(), it is "
      "completely useless to use this function!\n\n"

      "CheckConformance() should be used to find the right settings to create PDF/A compatible PDF files. "
      "Once the the settings were found it is usually not longer recommended to execute this function. "
      "However, it is of course possible to use CheckConformance() as a general approach to make sure "
      "that files created with DynaPDF are PDF/A compatible.");

   /* ---------------------- Signature field appearance ---------------------- */

   sigField = pdfCreateSigField(pdf, "Signature", -1, 200.0, 400.0, 200.0, 80.0);
   pdfSetFieldColor(pdf, sigField, fcBorderColor, csDeviceRGB, NO_COLOR);
   // Place the validation icon on the left side of the signature field.
   pdfPlaceSigFieldValidateIcon(pdf, sigField, 0.0, 15.0, 50.0, 50.0);
   /*
      This function creates a template into which you can draw whatever you want. The template
      is already opened when calling the function; it must be closed when finish with EndTemplate().
      An appearance template of a signature field is reserved for this field. It must not be placed
      on another page!

      In addition, it makes no sense to create an appearance template when the file is not digitally
      signed later. Empty signature fields cannot contain a user defined appearance.
   */
   pdfCreateSigFieldAP(pdf, sigField);

   pdfSaveGraphicState(pdf);
      pdfRectangle(pdf, 0.0, 0.0, 200.0, 80.0, fmNoFill);
      pdfClipPath(pdf, cmWinding, fmNoFill);
      sh = pdfCreateAxialShading(pdf, 0.0, 0.0, 200.0, 0.0, 0.5, PDF_RGB(120, 120, 220), PDF_WHITE, true, true);
      pdfApplyShading(pdf, sh);
   pdfRestoreGraphicState(pdf);

   pdfSaveGraphicState(pdf);
      pdfEllipse(pdf, 50.5, 1.0, 148.5, 78.0, fmNoFill);
      pdfClipPath(pdf, cmWinding, fmNoFill);
      sh = pdfCreateAxialShading(pdf, 0.0, 0.0, 0.0, 78.0, 2.0, PDF_WHITE, PDF_RGB(120, 120, 220), true, true);
      pdfApplyShading(pdf, sh);
   pdfRestoreGraphicState(pdf);

   pdfSetFont(pdf, "Arial", fsBold | fsUnderlined, 11.0, true, cp1252);
   pdfSetFillColor(pdf, PDF_RGB(120, 120, 220));
   pdfWriteFTextEx(pdf, 50.0, 60.0, 150.0, -1.0, taCenter, "Digitally signed by:");
   pdfSetFont(pdf, "Arial", fsBold|fsItalic, 18.0, true, cp1252);
   pdfSetFillColor(pdf, PDF_RGB(100, 100, 200));
   pdfWriteFTextEx(pdf, 50.0, 45.0, 150.0, -1.0, taCenter, "DynaPDF");

   pdfEndTemplate(pdf); // Close the appearance template.

   /* ------------------------------------------------------------------------ */

   pdfEndPage(pdf);
   // Check whether the file is compatible to PDF/A 1b
   switch (pdfCheckConformance(pdf, ctPDFA_1b_2005, 0, NULL, NULL, NULL))
   {
      case 1:
      case 3:  pdfAddOutputIntent(pdf, "../../test_files/sRGB.icc");             break; // RGB, Gray
      case 2:  pdfAddOutputIntent(pdf, "../../test_files/ISOcoated_v2_bas.ICC"); break; // CMYK
      default: break;
   }
   // No fatal error occurred?
   if (pdfHaveOpenDoc(pdf))
   {
      // We write the output file into the current directory.
      GetCurrentDirectory(MAX_PATH, filePath);
      strcat(filePath, "\\out.pdf");
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(pdf, filePath))
      {
         pdfDeletePDF(pdf);
         _getch();
         return -1;
      }
      if (!pdfCloseAndSignFile(pdf, "../../test_files/test_cert.pfx", "123456", "Test", NULL))
      {
         // Note: If the certificate cannot be found or when using a wrong password the PDF file is still
         // in memory. This is required so that another password or certificate file can be selected.
         pdfDeletePDF(pdf); // This function frees all used resources. When sharing a PDF instance you must call pdfFreePDF() instead.
         _getch();
         return -2;
      }
      printf("PDF file %s successfully created!\n", filePath);
      ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
   }
   pdfDeletePDF(pdf);
   return 0;
}
