#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

/*
   Note that the dynapdf.dll must be copied into the output directory or into a
   Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
*/

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

/*
   Incremental updates are useful if small changes should be saved quickly, e.g. when adding an annotation or reply to one,
   adding form fields and so on.

   This example is a bit simplified since we know that the file contains only one annotation and a handle of an annotation
   is just an array index from 0 though annotation count -1.
*/

char* CreateTestFile(PPDF* PDF, UI32 &BufSize)
{
   pdfCreateNewPDF(PDF, NULL);

   pdfSetPageCoords(PDF, pcTopDown);

   pdfAppend(PDF);
      pdfSquareAnnot(PDF, 50.0, 50.0, 200.0, 100.0, 3.0, NO_COLOR, 255, csDeviceRGB, "Jim", "Test", "Just a test...");
   pdfEndPage(PDF);

   if (!pdfCloseFile(PDF)) return NULL;
   char* buffer = pdfGetBuffer(PDF, BufSize);

   // Note that the buffer is bound to the PDF instance.
   // We make a copy of it so that we can work with only one PDF instance
   char* retval = (char*)malloc(BufSize);
   if (!retval)
   {
      pdfFreePDF(PDF);
      return NULL; // Out of memory!
   }
   memcpy(retval, buffer, BufSize);
   // Now we can release the original buffer. This also resets the PDF instance,
   // just as CloseFile() would do if the file would be created on a local drive.
   pdfFreePDF(PDF);
   return retval;
}

bool LoadTestFile(PPDF* PDF, const char* Buffer, UI32 BufSize)
{
   pdfCreateNewPDF(PDF, NULL);
   // This flag also sets the flags ifImportAsPage | ifImportAll, and if2UseProxy | if2CopyEncryptDict to make sure that
   // anything is imported and nothing gets changed.
   pdfSetImportFlags2(PDF, if2IncrementalUpd);
   if (pdfOpenImportBuffer(PDF, Buffer, BufSize, ptOpen, NULL) < 0) return false;
   return (pdfImportPDFFile(PDF, 1, 1.0, 1.0) > 0);
}

char* SaveFile(PPDF* PDF, UI32 &BufSize)
{
   if (!pdfCloseFile(PDF)) return NULL;
   char* buffer = pdfGetBuffer(PDF, BufSize);

   // Note that the buffer is bound to the PDF instance.
   // We make a copy of it so that we can work with only one PDF instance
   char* retval = (char*)malloc(BufSize);
   if (!retval)
   {
      pdfFreePDF(PDF);
      return NULL; // Out of memory!
   }
   memcpy(retval, buffer, BufSize);
   // Now we can release the original buffer. This also resets the PDF instance,
   // just as CloseFile() would do if the file would be created on a local drive.
   pdfFreePDF(PDF);
   return retval;
}

int main(int argc, char* argv[])
{
   UI32 bufSize = 0;
   SI32 annot, reply;
   char filePath[MAX_PATH+1], *buffer = NULL;
   PPDF* pdf = pdfNewPDF();
   if (!pdf) return -1; // Out of memory?

   pdfSetOnErrorProc(pdf, NULL, PDFError);

   // We create the base file in memory in this example
   if ((buffer = CreateTestFile(pdf, bufSize)) != NULL)
   {
      // CreateTest() created a copy of the PDF buffer. We are now the owner of the buffer.
      // The PDF instance has now it's inital state with the exception that the font cache, if already loaded, stays in memory.
      if (LoadTestFile(pdf, buffer, bufSize))
      {
         // We don't need the buffer anymore
         free(buffer);
         buffer = NULL;
         // We do not search for the annotation here since we know that the file contains only one annotation
         // and the handle of this annotation is zero since this is just an array index.
         reply = pdfSetAnnotMigrationState(pdf, 0, asCreateReply, "Harry");
         pdfSetAnnotString(pdf, reply, asContent, "Hi Jim, your test annotation looks fine!");
         if ((buffer = SaveFile(pdf, bufSize)) != NULL && LoadTestFile(pdf, buffer, bufSize))
         {
            free(buffer);
            buffer = NULL;

            reply = pdfSetAnnotMigrationState(pdf, reply, asCreateReply, "Tommy");
            pdfSetAnnotString(pdf, reply, asContent, "Just a test whether I can reply to a reply...");
            if ((buffer = SaveFile(pdf, bufSize)) != NULL && LoadTestFile(pdf, buffer, bufSize))
            {
               free(buffer);
               buffer = NULL;

               reply = pdfSetAnnotMigrationState(pdf, reply, asCreateReply, "Jim");
               pdfSetAnnotString(pdf, reply, asContent, "Seems to work very well!");
               if (pdfHaveOpenDoc(pdf))
               {
                  // We write the output file into the current directory.
                  GetCurrentDirectory(MAX_PATH, filePath);
                  strcat(filePath, "\\out.pdf");
                  if (pdfOpenOutputFile(pdf, filePath))
                  {
                     if (pdfCloseFile(pdf))
                     {
                        printf("PDF file \"%s\" successfully created!\n", filePath);
                        ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
                     }
                  }
               }
            }
         }
      }
   }
   if (buffer) free(buffer);
   pdfDeletePDF(pdf);

   _getch();
   return 0;
}
