#include <windows.h>
#include <conio.h>
#include <shellapi.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../../include/C_CPP/dynapdf.h"
#include "pdf_textsearch.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0;
}

SI32 PDF_CALL parseBeginTemplate(const void* Data, const BYTE* Object, SI32 Handle, struct TPDFRect* BBox, struct TCTM* Matrix)
{
   return ((CTextSearch*)Data)->BeginTemplate(BBox, Matrix);
}

void PDF_CALL parseEndTemplate(const void* Data)
{
   ((CTextSearch*)Data)->EndTemplate();
}

void PDF_CALL parseMulMatrix(const void* Data, const BYTE* Object, struct TCTM* M)
{
   ((CTextSearch*)Data)->MulMatrix(M);
}

SI32 PDF_CALL parseRestoreGraphicState(const void* Data)
{
   ((CTextSearch*)Data)->RestoreGState();
   return 0;
}

SI32 PDF_CALL parseSaveGraphicState(const void* Data)
{
   return ((CTextSearch*)Data)->SaveGState();
}

void PDF_CALL parseSetCharSpacing(const void* Data, const BYTE* Object, double Value)
{
   ((CTextSearch*)Data)->SetCharSpacing(Value);
}

void PDF_CALL parseSetFont(const void* Data, const BYTE* Object, TFontType Type, LBOOL Embedded, const char* FontName, TFStyle Style, double FontSize, const void* Font)
{
   ((CTextSearch*)Data)->SetFont(FontSize, Type, Font);
}

void PDF_CALL parseSetTextDrawMode(const void* Data, const BYTE* Object, TDrawMode Mode)
{
   ((CTextSearch*)Data)->SetTextDrawMode(Mode);
}

void PDF_CALL parseSetTextScale(const void* Data, const BYTE* Object, double Value)
{
   ((CTextSearch*)Data)->SetTextScale(Value);
}

void PDF_CALL parseSetWordSpacing(const void* Data, const BYTE* Object, double Value)
{
   ((CTextSearch*)Data)->SetWordSpacing(Value);
}

SI32 PDF_CALL parseShowTextArrayA(const void* Data, const BYTE* Object, struct TCTM* Matrix, const struct TTextRecordA* Source, UI32 Count, double Width)
{
   return ((CTextSearch*)Data)->MarkText(Matrix, Source, Count, Width);
}

int main(int argc, char* argv[])
{
   char filePath[MAX_PATH+1];
   CTextSearch textSearch; // This class tries to find a text in a PDF file.
   UI32 timeStart = GetTickCount();

   PPDF* pdf = pdfNewPDF();
   pdfSetOnErrorProc(pdf, &textSearch, PDFError);
   pdfCreateNewPDF(pdf, NULL); // The output file is opened later

   // External CMaps should always be loaded when processing text from PDF files.
   // See the description of ParseContent() for further information.
   _fullpath(filePath, "../../../../Resource/CMap/", MAX_PATH);
   pdfSetCMapDir(pdf, filePath, (TLoadCMapFlags)(lcmRecursive | lcmDelayed));

   pdfSetImportFlags(pdf, ifImportAll | ifImportAsPage);

   if (pdfOpenImportFile(pdf, "../../../../dynapdf_help.pdf", ptOpen, NULL) < 0)
   {
      pdfDeletePDF(pdf);
      _getch();
      return -1;
   }
   pdfImportPDFFile(pdf, 1, 1.0, 1.0);

   pdfCloseImportFile(pdf);

   // We flatten markup annotations and form fields so that we can search the text in these objects too.
   pdfFlattenAnnots(pdf, affMarkupAnnots);
   pdfFlattenForm(pdf);

   TPDFParseInterface stack;
   memset(&stack, 0, sizeof(stack));

   stack.BeginTemplate       = parseBeginTemplate;
   stack.EndTemplate         = parseEndTemplate;
   stack.MulMatrix           = parseMulMatrix;
   stack.RestoreGraphicState = parseRestoreGraphicState;
   stack.SaveGraphicState    = parseSaveGraphicState;
   stack.SetCharSpacing      = parseSetCharSpacing;
   stack.SetFont             = parseSetFont;
   stack.SetTextDrawMode     = parseSetTextDrawMode;
   stack.SetTextScale        = parseSetTextScale;
   stack.SetWordSpacing      = parseSetWordSpacing;
   stack.ShowTextArrayA      = parseShowTextArrayA;

   textSearch.SetPDFInst(pdf);
   // The search text must be defined in Unicode.
   if (textSearch.SetSearchText(L"PDF") < 0)
   {
      printf("Out of memory!\n");
      pdfDeletePDF(pdf);
      return -2;
   }
   SI32 i, selCount = 0, count = pdfGetPageCount(pdf);

   // We draw rectangles on the position where the search string was found. To make the text
   // in background visible we use the blend mode bmMultiply. Adobes Acrobat rasters a page
   // without anti-aliasing when a blend mode is used. Don't wonder that the rasterizing
   // quality is worse in comparison to normal PDF files.
   TPDFExtGState g;
   pdfInitExtGState(&g);
   g.BlendMode = bmMultiply;

   SI32 gs = pdfCreateExtGState(pdf, &g);

   for (i = 1; i <= count; i++)
   {
      pdfEditPage(pdf, i);

      pdfSetExtGState(pdf, gs);
      pdfSetFillColor(pdf, PDF_YELLOW);

      textSearch.Init();
      pdfParseContent(pdf, &textSearch, &stack, pfNone);
      pdfEndPage(pdf);
      if (textSearch.GetSelCount() > 0)
      {
         selCount += textSearch.GetSelCount();
         printf("Found string on Page: %d %d times!\n", i, textSearch.GetSelCount());
      }
   }
   // No fatal error occurred?
   if (pdfHaveOpenDoc(pdf))
   {
      // We write the output file into the current directory.
      GetCurrentDirectory(MAX_PATH, filePath);
      strcat(filePath, "\\out.pdf");
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(pdf, filePath))
      {
         pdfDeletePDF(pdf);
         _getch();
         return -3;
      }
      if (pdfCloseFile(pdf))
      {
         timeStart = GetTickCount() - timeStart;
         printf("Processing time: %d ms\n", timeStart);
         printf("PDF file \"%s\" successfully created!\n", filePath);
         ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
      }
   }
   pdfDeletePDF(pdf);

   if (!selCount) printf("String not found!\n");
   _getch();
   return 0;
}
