#include <windows.h>
#include <shellapi.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding when compiling the project with the workspace dynapdf_static.
#include "../../../../include/C_CPP/dynapdf.h"
#include "pdf_text_coordinates.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

/*
   Note that the dynapdf.dll must be copied into the output directory or into a
   Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
*/

// Error callback function.
// If the function name should not appear at the beginning of the error message then set
// the flag emNoFuncNames (pdfSetErrorMode(emNoFuncNames);).
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0;
}

SI32 PDF_CALL parseBeginTemplate(const void* Data, const BYTE* Object, SI32 Handle, struct TPDFRect* BBox, struct TCTM* Matrix)
{
   return ((CTextCoordinates*)Data)->BeginTemplate(BBox, Matrix);
}

void PDF_CALL parseEndTemplate(const void* Data)
{
   ((CTextCoordinates*)Data)->EndTemplate();
}

void PDF_CALL parseMulMatrix(const void* Data, const BYTE* Object, struct TCTM* M)
{
   ((CTextCoordinates*)Data)->MulMatrix(M);
}

SI32 PDF_CALL parseRestoreGraphicState(const void* Data)
{
   ((CTextCoordinates*)Data)->RestoreGState();
   return 0;
}

SI32 PDF_CALL parseSaveGraphicState(const void* Data)
{
   return ((CTextCoordinates*)Data)->SaveGState();
}

void PDF_CALL parseSetCharSpacing(const void* Data, const BYTE* Object, double Value)
{
   ((CTextCoordinates*)Data)->SetCharSpacing(Value);
}

void PDF_CALL parseSetFont(const void* Data, const BYTE* Object, TFontType Type, LBOOL Embedded, const char* FontName, TFStyle Style, double FontSize, const void* Font)
{
   ((CTextCoordinates*)Data)->SetFont(FontSize, Type, Font);
}

void PDF_CALL parseSetTextScale(const void* Data, const BYTE* Object, double Value)
{
   ((CTextCoordinates*)Data)->SetTextScale(Value);
}

void PDF_CALL parseSetWordSpacing(const void* Data, const BYTE* Object, double Value)
{
   ((CTextCoordinates*)Data)->SetWordSpacing(Value);
}

SI32 PDF_CALL parseShowTextArrayW(const void* Data, const TTextRecordA* Source, struct TCTM* M, const struct TTextRecordW* Kerning, UI32 Count, double Width, LBOOL Decoded)
{
   return ((CTextCoordinates*)Data)->MarkCoordinates(M, Source, Kerning, Count, Width, Decoded != 0);
}

int main(int argc, char* argv[])
{
   char filePath[MAX_PATH+1];
   UI32 timeStart = GetTickCount();
   CTextCoordinates textCoords; // This class handles the text of a PDF file.
   PPDF* pdf = pdfNewPDF();
   pdfSetOnErrorProc(pdf, &textCoords, PDFError);
   pdfCreateNewPDF(pdf, NULL); // The output file is opened later

   // External CMaps should always be loaded when processing text from PDF files.
   // See the description of ParseContent() for further information.
   _fullpath(filePath, "../../../../Resource/CMap/", MAX_PATH);
   pdfSetCMapDir(pdf, filePath, (TLoadCMapFlags)(lcmRecursive | lcmDelayed));

   pdfSetImportFlags(pdf, ifImportAll | ifImportAsPage);
   if (pdfOpenImportFile(pdf, "../../../../dynapdf_help.pdf", ptOpen, NULL) < 0)
   {
      pdfDeletePDF(pdf);
      _getch();
      return 0;
   }
   pdfImportPDFFile(pdf, 1, 1.0, 1.0);
   pdfCloseImportFile(pdf);

   // We flatten markup annotations and form fields so that we can extract the text from these objects too.
   pdfFlattenAnnots(pdf, affMarkupAnnots);
   pdfFlattenForm(pdf);

   TPDFParseInterface stack;
   memset(&stack, 0, sizeof(stack));

   // More callback functions are not required to extract text
   stack.BeginTemplate       = parseBeginTemplate;
   stack.EndTemplate         = parseEndTemplate;
   stack.MulMatrix           = parseMulMatrix;
   stack.RestoreGraphicState = parseRestoreGraphicState;
   stack.SaveGraphicState    = parseSaveGraphicState;
   stack.SetCharSpacing      = parseSetCharSpacing;
   stack.SetFont             = parseSetFont;
   stack.SetTextScale        = parseSetTextScale;
   stack.SetWordSpacing      = parseSetWordSpacing;
   stack.ShowTextArrayW      = parseShowTextArrayW;

   textCoords.SetPDFInst(pdf);

   SI32 i, count = pdfGetPageCount(pdf);
   for (i = 1; i <= count; i++)
   {
      pdfEditPage(pdf, i);
      pdfSetLineWidth(pdf, 0.5);
      // Initialize the graphics state to the default values
      textCoords.Init();
      pdfParseContent(pdf, &textCoords, &stack, pfNone);
      pdfEndPage(pdf);
   }
   // No fatal error occurred?
   if (pdfHaveOpenDoc(pdf))
   {
      // We write the output file into the current directory.
      GetCurrentDirectory(511, filePath);
      strcat(filePath, "\\out.pdf");
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(pdf, filePath))
      {
         pdfDeletePDF(pdf);
         _getch();
         return -1;
      }
      if (pdfCloseFile(pdf))
      {
         timeStart = GetTickCount() - timeStart;
         printf("Processing time: %d ms\n", timeStart);
         printf("PDF file \"%s\" successfully processed!\n", filePath);
         ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
      }
   }
   pdfDeletePDF(pdf);
   return 0;
}
