#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

/* The fonts in this list are sorted alphabetically because no better order is known. I've chosen
   fonts that MS Word used to display the languages of the test file on my system. It is of course
   not guaranteed that any of these fonts is available on your system.
*/
static const char* ALT_FONTS[5] =
{
   "Malgun Gothic", // Korean
   "Mangal",        // Hindi or Marathi
   "Nyala",         // Amharic
   "Shonar Bangla", // Bengali
   "Shruti"         // Gurjarati
};

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

UI16* GetFileBuffer(const char* FileName, UI32 &BufSize)
{
   FILE* f = fopen(FileName, "rb");
   if (f == NULL) return NULL;
   UI16* buffer;
   fseek(f, 0, SEEK_END);
   BufSize = ftell(f);
   fseek(f, 0, SEEK_SET);
   // allocate one more character for the null-terminator
   if ((buffer = (UI16*)malloc(((BufSize>>1) +1) * sizeof(UI16))) == NULL)
   {
      fclose(f);
      return NULL;
   }
   fread(buffer, 1, BufSize, f);
   fclose(f);
   BufSize >>= 1;
   buffer[BufSize] = 0; // Do NOT forget to add a null-terminator!!!
   return buffer;
}

int main(int argc, char* argv[])
{
   UI32 bufSize;
   UI16* fText = GetFileBuffer("../../../test_files/multi_lang.txt", bufSize);
   if (!fText)
   {
      printf("Cannot open input file!");
      _getch();
      return 1;
   }
   UI32 timeStart = GetTickCount();
   PPDF* pdf = pdfNewPDF(); // Get a new PDF instance
   if (!pdf)   // Out of memory?
   {
      free(fText);
      return 2;
   }
   pdfSetOnErrorProc(pdf, NULL, PDFError);
   pdfCreateNewPDF(pdf, NULL); // The output file is created later

   pdfSetPageCoords(pdf, pcTopDown);

   // Enable complex text layout
   pdfSetGStateFlags(pdf, gfComplexText, false);

   /*
      This is essentially the same example as font_substitution. The only difference is that we provide
      an alternate font list to improve font substitution.
   */
   SI32 altFonts = pdfCreateAltFontList(pdf);
   pdfSetAltFonts(pdf, altFonts, ALT_FONTS, 5);

   pdfAppend(pdf);

   // The font must be loaded with cpUnicode.
   pdfSetFont(pdf, "Arial", fsRegular, 10.0, true, cpUnicode);
   // Activate the alternate font list
   pdfActivateAltFontList(pdf, altFonts, true);

   pdfSetLeading(pdf, pdfGetTypoLeading(pdf));
   pdfWriteFTextExW(pdf, 50.0, 50.0, pdfGetPageWidth(pdf) - 100.0, pdfGetPageHeight(pdf) - 100.0, taJustify, fText);

   free(fText);     // free the text buffer
   pdfEndPage(pdf); // Close the last page

   // No fatal error occurred?
   if (pdfHaveOpenDoc(pdf))
   {
      char filePath[MAX_PATH+1];
      // We write the output file into the current directory.
      GetCurrentDirectory(MAX_PATH, filePath);
      strcat(filePath, "\\out.pdf");
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(pdf, filePath))
      {
         pdfDeletePDF(pdf);
         _getch();
         return -1;
      }
      if (pdfCloseFile(pdf))
      {
         timeStart = GetTickCount() - timeStart;
         printf("Processing time: %d ms\n", timeStart);
         printf("PDF file \"%s\" successfully created!\n", filePath);
         ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
      }
   }
   pdfDeletePDF(pdf); // Do not forget to delete the PDF instance
   _getch();
   return 0;
}
