#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

struct TTestBarcode
{
   TPDFBarcodeType     BarcodeType;
   const char*         BarcodeName;
   TPDFBarcodeDataType DataType;
   const char*         Data;
   const char*         Primary;
};

static const TTestBarcode TEST_CODES[] =
{
   {bctAustraliaPost,      "Australia Post",                   bcdtBinary, "12345678", NULL},
   {bctAustraliaRedir,     "Australia Redirect Code",          bcdtBinary, "12345678", NULL},
   {bctAustraliaReply,     "Australia Reply-Paid",             bcdtBinary, "12345678", NULL},
   {bctAustraliaRout,      "Australia Routing Code",           bcdtBinary, "12345678", NULL},
   {bctAztec,              "Aztec binary mode",                bcdtBinary, "123456789012", NULL},
   {bctAztec,              "Aztec GS1 Mode",                   bcdtGS1Mode, "[01]03453120000011[17]120508[10]ABCD1234[410]9501101020917", NULL},
   {bctAztecRunes,         "Aztec Runes",                      bcdtBinary, "123", NULL},
   {bctC2Of5IATA,          "Code 2 of 5 IATA",                 bcdtBinary, "1234567890", NULL},
   {bctC2Of5Industrial,    "Code 2 of 5 Industrial",           bcdtBinary, "1234567890", NULL},
   {bctC2Of5Interleaved,   "Code 2 of 5 Interleaved",          bcdtBinary, "1234567890", NULL},
   {bctC2Of5Logic,         "Code 2 of 5 Data Logic",           bcdtBinary, "1234567890", NULL},
   {bctC2Of5Matrix,        "Code 2 of 5 Matrix",               bcdtBinary, "1234567890", NULL},
   {bctChannelCode,        "Channel Code",                     bcdtBinary, "1234567", NULL},
   {bctCodabar,            "Codabar",                          bcdtBinary, "A123456789B", NULL},
   {bctCodablockF,         "Codablock-F",                      bcdtBinary, "1234567890abcdefghijklmnopqrstuvwxyz", NULL},
   {bctCode11,             "Code 11",                          bcdtBinary, "1234567890", NULL},
   {bctCode128,            "Code 128",                         bcdtBinary, "1234567890", NULL},
   {bctCode128B,           "Code 128",                         bcdtBinary, "1234567890", NULL},
   {bctCode16K,            "Code 16K binary mode",             bcdtBinary, "[90]A1234567890", NULL},
   {bctCode16K,            "Code 16K GS1 mode",                bcdtGS1Mode, "[90]A1234567890", NULL},
   {bctCode32,             "Code 32",                          bcdtBinary, "12345678", NULL},
   {bctCode39,             "Code 39",                          bcdtBinary, "1234567890", NULL},
   {bctCode49,             "Code 49",                          bcdtBinary, "1234567890", NULL},
   {bctCode93,             "Code 93",                          bcdtBinary, "1234567890", NULL},
   {bctCodeOne,            "Code One",                         bcdtBinary, "1234567890", NULL},
   {bctDAFT,               "DAFT Code",                        bcdtBinary, "aftdaftdftaft", NULL},
   {bctDataBarOmniTrunc,   "GS1 DataBar Omnidirectional",      bcdtBinary, "0123456789012", NULL},
   {bctDataBarExpStacked,  "GS1 DataBar Stacked",              bcdtBinary, "[90]1234567890", NULL},
   {bctDataBarExpanded,    "GS1 DataBar Expanded",             bcdtBinary, "[90]1234567890", NULL},
   {bctDataBarLimited,     "GS1 DataBar Limited",              bcdtBinary, "0123456789012", NULL},
   {bctDataBarStacked,     "GS1 DataBar Stacked",              bcdtBinary, "0123456789012", NULL},
   {bctDataBarStackedO,    "GS1 DataBar Stacked Omni",         bcdtBinary, "0123456789012", NULL},
   {bctDataMatrix,         "Data Matrix ISO 16022",            bcdtBinary, "0123456789012", NULL},
   {bctDotCode,            "DotCode",                          bcdtBinary, "0123456789012", NULL},
   {bctDPD,                "DPD Code",                         bcdtBinary, "1234567890123456789012345678", NULL},
   {bctDPIdentCode,        "Deutsche Post Identcode",          bcdtBinary, "12345678901", NULL},
   {bctDPLeitcode,         "Deutsche Post Leitcode",           bcdtBinary, "1234567890123", NULL},
   {bctEAN128,             "EAN 128",                          bcdtBinary, "[90]0101234567890128TEC-IT", NULL},
   {bctEAN128_CC,          "EAN 128 Composite Code",           bcdtBinary, "[10]1234-1234", "[90]123456"},
   {bctEAN14,              "EAN 14",                           bcdtBinary, "1234567890", NULL},
   {bctEANX,               "EAN X",                            bcdtBinary, "1234567890", NULL},
   {bctEANX_CC,            "EAN Composite Symbol",             bcdtBinary, "[90]12341234", "12345678"},
   {bctEANXCheck,          "EAN + Check Digit",                bcdtBinary, "12345", NULL},
   {bctExtCode39,          "Ext. Code 3 of 9 (Code 39+)",      bcdtBinary, "1234567890", NULL},
   {bctFIM,                "FIM",                              bcdtBinary, "d", NULL},
   {bctFlattermarken,      "Flattermarken",                    bcdtBinary, "11111111111111", NULL},
   {bctHIBC_Aztec,         "HIBC Aztec Code",                  bcdtBinary, "123456789012", NULL},
   {bctHIBC_CodablockF,    "HIBC Codablock-F",                 bcdtBinary, "1234567890abcdefghijklmnopqrstuvwxyz", NULL},
   {bctHIBC_Code128,       "HIBC Code 128",                    bcdtBinary, "1234567890", NULL},
   {bctHIBC_Code39,        "HIBC Code 39",                     bcdtBinary, "1234567890", NULL},
   {bctHIBC_DataMatrix,    "HIBC Data Matrix",                 bcdtBinary, "0123456789012", NULL},
   {bctHIBC_MicroPDF417,   "HIBC Micro PDF417",                bcdtBinary, "01234567890abcde", NULL},
   {bctHIBC_PDF417,        "HIBC PDF417",                      bcdtBinary, "01234567890abcde", NULL},
   {bctHIBC_QR,            "HIBC QR Code",                     bcdtBinary, "01234567890abcde", NULL},
   {bctISBNX,              "ISBN (EAN-13 with validation)",    bcdtBinary, "0123456789", NULL},
   {bctITF14,              "ITF-14",                           bcdtBinary, "0123456789", NULL},
   {bctJapanPost,          "Japanese Postal Code",             bcdtBinary, "0123456789", NULL},
   {bctKIX,                "Dutch Post KIX Code",              bcdtBinary, "0123456789", NULL},
   {bctKoreaPost,          "Korea Post",                       bcdtBinary, "123456", NULL},
   {bctLOGMARS,            "LOGMARS",                          bcdtBinary, "1234567890abcdef", NULL},
   {bctMailmark,           "Royal Mail 4-State Mailmark",      bcdtBinary, "11210012341234567AB19XY1A", NULL},
   {bctMaxicode,           "Maxicode",                         bcdtBinary, "1234567890abcdef", NULL},
   {bctMicroPDF417,        "Micro PDF417",                     bcdtBinary, "1234567890abcdef", NULL},
   {bctMicroQR,            "Micro QR Code",                    bcdtBinary, "1234567890abcdef", NULL},
   {bctMSIPlessey,         "MSI Plessey",                      bcdtBinary, "12345678901", NULL},
   {bctNVE18,              "NVE-18",                           bcdtBinary, "1234567890123456", NULL},
   {bctPDF417,             "PDF417",                           bcdtBinary, "1234567890abcdef", NULL},
   {bctPDF417Truncated,    "PDF417 Truncated",                 bcdtBinary, "1234567890abcdef", NULL},
   {bctPharmaOneTrack,     "Pharmacode One-Track",             bcdtBinary, "123456", NULL},
   {bctPharmaTwoTrack,     "Pharmacode Two-Track",             bcdtBinary, "123456", NULL},
   {bctPLANET,             "PLANET",                           bcdtBinary, "12345678901", NULL},
   {bctPlessey,            "Plessey",                          bcdtBinary, "12345678901", NULL},
   {bctPostNet,            "PostNet",                          bcdtBinary, "12345678901", NULL},
   {bctPZN,                "PZN",                              bcdtBinary, "1234567", NULL},
   {bctQRCode,             "QR Code",                          bcdtBinary, "1234567890abcdef", NULL},
   {bctRMQR,               "Rect. Micro QR Code (rMQR)",       bcdtBinary, "1234567890abcdef", NULL},
   {bctRoyalMail4State,    "Royal Mail 4 State (RM4SCC)",      bcdtBinary, "1234567890abcdef", NULL},
   {bctRSS_EXP_CC,         "CS GS1 DataBar Ext. component",    bcdtBinary, "[90]12341234", "[10]12345678"},
   {bctRSS_EXPSTACK_CC,    "CS GS1 DataBar Exp. Stacked",      bcdtBinary, "[90]12341234", "[10]12345678"},
   {bctRSS_LTD_CC,         "CS GS1 DataBar Limited",           bcdtBinary, "[90]12341234", "1234567"},
   {bctRSS14_CC,           "CS GS1 DataBar-14 Linear",         bcdtBinary, "[90]12341234", "1234567"},
   {bctRSS14Stacked_CC,    "CS GS1 DataBar-14 Stacked",        bcdtBinary, "[90]12341234", "1234567"},
   {bctRSS14StackOMNI_CC,  "CS GS1 DataBar-14 Stacked Omni",   bcdtBinary, "[90]12341234", "1234567"},
   {bctTelepen,            "Telepen Alpha",                    bcdtBinary, "1234567890abcdef", NULL},
   {bctUltracode,          "Ultracode",                        bcdtBinary, "1234567890abcdef", NULL},
   {bctUPCA,               "UPC A",                            bcdtBinary, "1234567890", NULL},
   {bctUPCA_CC,            "CS UPC A linear",                  bcdtBinary, "[90]12341234", "1234567"},
   {bctUPCACheckDigit,     "UPC A + Check Digit",              bcdtBinary, "12345678905", NULL},
   {bctUPCE,               "UCP E",                            bcdtBinary, "1234567", NULL},
   {bctUPCE_CC,            "CS UPC E linear",                  bcdtBinary, "[90]12341234", "1234567"},
   {bctUPCECheckDigit,     "UCP E + Check Digit",              bcdtBinary, "12345670", NULL},
   {bctUPNQR,              "UPNQR (Univ. Placilni Nalog QR)",  bcdtBinary, "1234567890abcdef", NULL},
   {bctUSPSOneCode,        "USPS OneCode",                     bcdtBinary, "01234567094987654321", NULL},
   {bctVIN,                "Vehicle Ident Number (USA)",       bcdtBinary, "01234567094987654", NULL}
};

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

int main(int argc, char* argv[])
{
   double x, y, w, h, pw, ph, incX, incY;
   SI32 i, nx, ny, xx, yy, cnt = sizeof(TEST_CODES) / sizeof(TTestBarcode);

   PPDF* pdf = pdfNewPDF();
   if (!pdf) return 2; // Out of memory?
   // Error messages and warnings are passed to the callback function.
   pdfSetOnErrorProc(pdf, NULL, PDFError);
   pdfCreateNewPDF(pdf, NULL); // The output file is opened later

   pdfSetPageCoords(pdf, pcTopDown);

   TPDFBarcode2 bcd;
   bcd.StructSize = sizeof(bcd);

   pdfInitBarcode2(&bcd);
   bcd.Options = TPDFBarcodeOptions(bcoDefault | bcoUseActiveFont);
   // Use this flag if you want to create image based barcodes
   //bcd.Options = bcoImageOutput;

   pw   = pdfGetPageWidth(pdf)  - 100.0;
   ph   = pdfGetPageHeight(pdf) - 100.0;
   w    = 100.0;
   h    = 120.0;
   nx   = (SI32)(pw / w);
   ny   = (SI32)(ph / h);
   incX = w + (pw - nx * w) / (nx-1);
   incY = h + (ph - ny * h) / (ny-1);
   h    = 100.0;

   for (i = 0; i < cnt; )
   {
      pdfAppend(pdf);

      pdfSetFont(pdf, "Helvetica", fsRegular, 6.5, true, cp1252);
      pdfSetLineWidth(pdf, 0.0);

      y = 50.0;
      for (yy = 0; yy < ny && i < cnt; yy++, y += incY)
      {
         for (xx = 0, x = 50.0; xx < nx && i < cnt; xx++, i++, x += incX)
         {
            const TTestBarcode &b = TEST_CODES[i];
            bcd.BarcodeType = b.BarcodeType;
            bcd.Data        = b.Data;
            bcd.DataType    = b.DataType;
            bcd.Primary     = b.Primary;
            pdfWriteFTextEx(pdf, x, y-10.0, w, -1.0, taCenter, b.BarcodeName);
            pdfRectangle(pdf, x, y, w, h, fmStroke);
            if (pdfInsertBarcode(pdf, x, y, w, h, coCenter, coCenter, &bcd) < 0)
            {
               pdfDeletePDF(pdf);
               return -1;
            }
         }
      }
      pdfEndPage(pdf);
   }

   // No fatal error occurred?
   if (pdfHaveOpenDoc(pdf))
   {
      // We write the output file into the current directory.
      char filePath[MAX_PATH+1];
      GetCurrentDirectory(MAX_PATH, filePath);
      strcat(filePath, "\\out.pdf");
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(pdf, filePath))
      {
         pdfDeletePDF(pdf);
         _getch();
         return -1;
      }
      if (pdfCloseFile(pdf))
      {
         printf("PDF file \"%s\" successfully created!\n", filePath);
         ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
      }
   }
   pdfDeletePDF(pdf); // Do not forget to delete the class instance
   return 0;
}
