#ifndef gpcAPI_H
#define gpcAPI_H

/*
   Project:   Generic Polygon Clipper

              A new algorithm for calculating the difference, intersection,
              exclusive-or or union of arbitrary polygon sets.

   File:      gpc.h (modified version for use with DynaPDF)
   Author:    Alan Murta (email: gpc@cs.man.ac.uk)
   Version:   2.32
   Date:      17th December 2004

   Copyright: (C) 1997-2004, Advanced Interfaces Group,
              University of Manchester.

   DynaPDF contains an optimized version of the General Polygon Clipper. DynaForms has licensed GPC
   for use with DynaPDF. You don't need to obtain a separate license if you use it in conjunction with
   DynaPDF. However, if GPC should be used for something else than PDF creation then please obtain
   a separate license from the University of Manchester.

   See http://www.cs.man.ac.uk/~toby/alan/software/ for further details.
*/

#ifdef __cplusplus
namespace DynaPDF
{
#endif

typedef enum                        /* Set operation type                */
{
  GPC_DIFF,                         /* Difference                        */
  GPC_INT,                          /* Intersection                      */
  GPC_XOR,                          /* Exclusive or                      */
  GPC_UNION                         /* Union                             */
} gpc_op;

typedef struct                      /* Polygon vertex structure          */
{
  double              x;            /* Vertex x component                */
  double              y;            /* vertex y component                */
} gpc_vertex;

typedef struct                      /* Vertex list structure             */
{
  int                 num_vertices; /* Number of vertices in list        */
  gpc_vertex         *vertex;       /* Vertex array pointer              */
} gpc_vertex_list;

typedef struct                      /* Polygon set structure             */
{
  int                 num_contours; /* Number of contours in polygon     */
  gpc_vertex_list    *contour;      /* Contour array pointer             */
} gpc_polygon;

PDF_EXP SI32  PDF_CALL gpcAddVertex(const void* IGPC, gpc_polygon* Poly, gpc_vertex_list* VertexList);
PDF_EXP SI32  PDF_CALL gpcClipPoly(const void* IGPC, gpc_op Operation, gpc_polygon* Poly, gpc_polygon* ClipPoly, gpc_polygon* Result);
PDF_EXP void  PDF_CALL gpcDeleteGPC(const void* IGPC);
PDF_EXP void* PDF_CALL gpcNewGPC(void);
PDF_EXP void  PDF_CALL gpcReset(const void* IGPC);

#ifdef __cplusplus
}
#endif

/*

Usage:

int main(int argc, char* argv[])
{
   void* pdf = pdfNewPDF();
   pdfCreateNewPDF(pdf, "c:/test.pdf");

   void* gpc = gpcNewGPC();

   gpc_polygon poly1, poly2, res;
   gpc_vertex v[4];
   gpc_vertex_list vertex;

   poly1.num_contours  = 0;
   poly2.num_contours  = 0;
   vertex.vertex       = v; // Set the pointer to the vertex array
   vertex.num_vertices = 4; // Number of vertices in the array

   v[0].x = 50.0;
   v[0].y = 50.0;
   v[1].x = 250.0;
   v[1].y = 50.0;
   v[2].x = 250.0;
   v[2].y = 250.0;
   v[3].x = 50.0;
   v[3].y = 250.0;
   gpcAddVertex(gpc, &poly1, &vertex); // Create the first polygon

   v[0].x = 100.0;
   v[0].y = 100.0;
   v[1].x = 350.0;
   v[1].y = 100.0;
   v[2].x = 350.0;
   v[2].y = 350.0;
   v[3].x = 100.0;
   v[3].y = 350.0;
   gpcAddVertex(gpc, &poly2, &vertex); // Create the second polygon

   // Now we can combine or clip the first polygon with the second one.
   if (!gpcClipPoly(gpc, GPC_XOR, &poly1, &poly2, &res)) // gpcClipPoly() returns 0 on success or a negative error code on failure
   {
      int i, j, count;
      gpc_vertex* vertices;

      // Output the result.
      pdfSetPageCoords(pdf, pcTopDown);
      pdfAppend(pdf);

      pdfSetFillColor(pdf, PDF_YELLOW);
      for (i = 0; i < res.num_contours; i++)
      {
         count    = res.contour[i].num_vertices;
         vertices = res.contour[i].vertex;
         if (count < 2) continue;
         if (i > 0) pdfClosePath(pdf, fmClose);

         pdfMoveTo(pdf, vertices[0].x, vertices[0].y);
         for (j = 1; j < count; j++)
         {
            pdfLineTo(pdf, vertices[j].x, vertices[j].y);
         }
      }
      pdfClosePath(pdf, fmFill);
      pdfEndPage(pdf);
   }
   gpcDeleteGPC(gpc); // Do not forget to delete the GPC instance!

   pdfCloseFile(pdf);
   pdfDeletePDF(pdf);
   return 0;
}

*/

#endif
