#include "pdf_edit_text.h"

#include <math.h>

#if defined(_MSC_VER)
   #pragma warning(disable : 4514)
#endif

#define InitCTM(m){m.a = 1.0; m.b = 0.0; m.c = 0.0; m.d = 1.0; m.x = 0.0; m.y = 0.0;}

UI32 StrLen(const char* Text)
{
   UI32 retval = 0;
   if (!Text) return 0;
   while (*Text++) ++retval;
   return retval;
}

UI32 StrLen(const UI16* Text)
{
   UI32 retval = 0;
   if (!Text) return 0;
   while (*Text++) ++retval;
   return retval;
}

// ----------------------------------------- CPDFEditText ----------------------------------------

CPDFEditText::CPDFEditText(const PPDF* PDFInst) :
   m_CurrTmpl(-1),
   m_First(-1),
   m_HaveMore(0),
   m_KernRecord(0),
   m_LastFont(NULL),
   m_LastX(0.0),
   m_LastY(0.0),
   m_NewLine(true),
   m_PDFInst(PDFInst),
   m_RecordNumber(0),
   m_SearchPos(NULL),
   m_SearchText(NULL),
   m_SearchTextLen(0),
   m_StrPos(0)
{
   InitCTM(m_Matrix4);
}

CPDFEditText::~CPDFEditText(void)
{
   if (m_SearchText) free(m_SearchText);
}

void CPDFEditText::AddKernSpace(float Advance, float SpaceWidth, TCTM &Matrix)
{
   if (Advance < SpaceWidth)
   {
      if (m_SearchPos[0] == 32)
      {
         ++m_SearchPos;
         if (!m_SearchPos[0]) m_SearchPos = m_SearchText;
      }else
      {
         m_Matrix4.x = -Advance;
         Matrix      = MulMatrix(Matrix, m_Matrix4);
      }
   }else
   {
      m_Matrix4.x = -Advance;
      Matrix      = MulMatrix(Matrix, m_Matrix4);
   }
}

void CPDFEditText::AddKernSpaceEx(float Advance, float SpaceWidth, TCTM &Matrix)
{
   if (Advance < SpaceWidth)
   {
      if (m_SearchPos[0] == 32)
         ++m_SearchPos;
      else
      {
         m_Matrix4.x = -Advance;
         Matrix      = MulMatrix(Matrix, m_Matrix4);
      }
   }else
   {
      m_Matrix4.x = -Advance;
      Matrix      = MulMatrix(Matrix, m_Matrix4);
   }
}

void CPDFEditText::AddRecord(SI32 KernRecord, SI32 StrPos)
{
   TTextRec*   rec = m_Records.Add();
   rec->TmplHandle = m_CurrTmpl;
   rec->First      = m_First;
   rec->Last       = m_RecordNumber;
   rec->KernRecord = KernRecord;
   rec->StrPos     = StrPos;
   rec->NewLine    = m_NewLine;
   m_KernRecord    = -1;
   m_NewLine       = false;
   m_SearchPos     = m_SearchText;
   m_StrPos        = -1;
}

void CPDFEditText::AddSpace(double x, double y, double SpaceWidth)
{
   double distX;
   if (!m_Alpha)
      distX = x - m_LastX;
   else
      distX = CalcDistance(m_LastX, m_LastY, x, y);
   if (distX > SpaceWidth)
   {
      if (m_SearchPos[0] == 32)
      {
         ++m_SearchPos;
      }
   }
}

void CPDFEditText::AddSpace(double DistX, double SpaceWidth, TCTM &Matrix)
{
   if (DistX > SpaceWidth)
   {
      if (m_SearchPos[0] == 32)
         ++m_SearchPos;
      else
      {
         m_Matrix4.x = DistX;
         Matrix      = MulMatrix(Matrix, m_Matrix4);
      }
   }else
   {
      m_Matrix4.x = DistX;
      Matrix      = MulMatrix(Matrix, m_Matrix4);
   }
}

void CPDFEditText::CalcAlpha(TCTM& M)
{
   double x1 = 0.0;
   double y1 = 0.0;
   double x2 = 1.0;
   double y2 = 0.0;
   Transform(M, x1, y1);
   Transform(M, x2, y2);
   // Get rid of rounding errors...
   m_Alpha = (SI32)((atan2(y2-y1, x2-x1) / 0.017453292519943295769236907684886) * 256.0);
}

void CPDFEditText::CalcDistance(TCTM &M1, TCTM &M2, TCTM &M3, double &DistX, double &DistY, double x, double y)
{
   if (!m_Alpha)
   {
      DistX = x - m_LastX;
      DistY = fabs(y - m_LastY);
   }else
   {
      M3 = MulMatrix(M1, M2);
      /*
         Notice: The distance is always a positive value. If the text is not ordered from left
         to right the result can be incorrect!

         To do:
         Check whether the text occurs behind the previous one. If not, insert the text at the
         beginning of the line. This is recommended if bidirectional text should be processed.
      */
      DistX = CalcDistance(m_LastX, m_LastY, x, y);
      DistY = fabs(M3.y);
   }
}

inline double CPDFEditText::CalcDistance(double x1, double y1, double x2, double y2)
{
   double dx = x2-x1;
   double dy = y2-y1;
   return sqrt(dx*dx + dy*dy);
}

void CPDFEditText::CalcStrPos(TCTM &M, double &x, double &y)
{
   x = 0.0;
   y = 0.0;
   M = MulMatrix(m_Stack.ctm, m_Stack.tm);
   Transform(M, x, y);
   CalcAlpha(M);
}

void CPDFEditText::FindAll(void)
{
   TCTM m;
   SI32 alpha;
   double x, y, spaceWidth, distX, distY;
   m_LastX        = 0.0;
   m_LastY        = 0.0;
   m_RecordNumber = 0;
   m_NewLine      = true;
   m_SearchPos    = m_SearchText;
   // Get the first text record if any
   m_HaveMore = pdfGetPageText(m_PDFInst, &m_Stack);
   // No text found?
   if (!m_Stack.TextLen) return;

   // Calculate the text position and orientation angle
   CalcStrPos(m_Matrix1, x, y);

   FindPattern();

   spaceWidth =  m_Stack.SpaceWidth * GetScaleFactor(m_Matrix1) * 0.5;

   // Calculate the end offset and invert the matrix.
   m_LastX = m_Stack.TextWidth;
   Transform(m_Matrix1, m_LastX, m_LastY);
   Invert(m_Matrix1);

   ++m_RecordNumber;

   if (!m_HaveMore) return;

   alpha = m_Alpha;
   while (pdfGetPageText(m_PDFInst, &m_Stack))
   {
      CalcStrPos(m_Matrix2, x, y);
      if (alpha == m_Alpha)
      {
         CalcDistance(m_Matrix1, m_Matrix2, m, distX, distY, x, y);
         if (distY > MAX_LINE_ERROR)
         {
            // This algorithm does not support line breaks!
            m_SearchPos = m_SearchText;
            m_Matrix1   = m_Matrix2;
            Invert(m_Matrix1);
            m_NewLine = true;
         }else if (distX > spaceWidth)
         {
            if (distX > 6.0 * spaceWidth)
            {
               // The distance is too large. We assume that the text should not be considered
               // as part of the current text line.
               m_SearchPos = m_SearchText;
               m_Matrix1   = m_Matrix2;
               Invert(m_Matrix1);
               m_NewLine = true;
            }else if (*m_SearchPos == 32)
            {
               if (m_SearchPos == m_SearchText)
               {
                  m_First      = m_RecordNumber;
                  m_KernRecord = -1;
                  m_StrPos     = -1;
               }
               ++m_SearchPos;
               if (!m_SearchPos[0])
               {
                  AddRecord(m_KernRecord, m_StrPos);
               }
            }else
            {
               m_SearchPos = m_SearchText;
               if (*m_SearchPos == 32)
               {
                  m_First = m_RecordNumber;
                  ++m_SearchPos;
                  if (!m_SearchPos[0])
                  {
                     AddRecord(-1, -1);
                  }
               }
            }
         }else if (distX < -spaceWidth) // Wrong direction?
         {
            m_SearchPos = m_SearchText;
            m_Matrix1   = m_Matrix2;
            Invert(m_Matrix1);
            m_NewLine = true;
         }
      }else
      {
         m_SearchPos = m_SearchText;
         m_Matrix1   = m_Matrix2;
         Invert(m_Matrix1);
         m_NewLine = true;
      }
      FindPattern();
      m_LastY    = 0.0;
      m_LastX    = m_Stack.TextWidth;
      alpha      = m_Alpha;
      spaceWidth = m_Stack.SpaceWidth * GetScaleFactor(m_Matrix2) * 0.5;
      Transform(m_Matrix2, m_LastX, m_LastY);
      ++m_RecordNumber;
   }
}

UI32 CPDFEditText::FindEndPattern(const UI16* Text, UI32 Len)
{
   UI32 retval = 0;
   while (Len--)
   {
      if (!m_SearchPos[0]) break;
      if (Text[retval] != m_SearchPos[0]) break;
      ++m_SearchPos;
      ++retval;
   }
   return retval;
}

void CPDFEditText::FindPattern(void)
{
   UI32 i, j;
   UI16 c1, c2;
   float spw = -m_Stack.SpaceWidth / 2.0f;
   if (m_SearchPos == m_SearchText)
   {
      m_First      = m_RecordNumber;
      m_KernRecord = -1;
      m_StrPos     = -1;
   }
   // Handle spaces
   for (i = 0; i < m_Stack.KerningCount; i++)
   {
      TTextRecordW &rec = m_Stack.Kerning[i];
      if (rec.Advance < spw)
      {
         if (m_SearchPos[0] == 32)
         {
            if (m_SearchPos == m_SearchText && m_KernRecord < 0)
            {
               m_KernRecord = i;
               m_StrPos     = -1;
            }
            ++m_SearchPos;
            if (!m_SearchPos[0]) AddRecord(m_KernRecord, m_StrPos);
         }else
         {
            m_SearchPos = m_SearchText;
            m_StrPos    = -1;
            if (m_SearchPos[0] == 32)
            {
               ++m_SearchPos;
               if (m_KernRecord < 0) m_KernRecord = i;
               if (!m_SearchPos[0]) AddRecord(m_KernRecord, m_StrPos);
            }
         }
      }
      // Compare the text.
      for (j = 0; j < rec.Length; j++)
      {
         c1 = rec.Text[j];
         c2 = m_SearchPos[0];
         if (c1 != c2)
         {
            switch(c1)
            {
               case 160: // non-break space
               {
                  if (c2 == 32)
                  {
                     if (m_SearchPos == m_SearchText && m_KernRecord < 0)
                     {
                        m_KernRecord = i;
                        m_StrPos     = j;
                     }
                     ++m_SearchPos;
                     continue;
                  }
                  break;
               }
               case 173: // non-break hyphen
               {
                  if (c2 == 45)
                  {
                     if (m_SearchPos == m_SearchText && m_KernRecord < 0)
                     {
                        m_KernRecord = i;
                        m_StrPos     = j;
                     }
                     ++m_SearchPos;
                     continue;
                  }
                  break;
               }
               default:  break;
            }
            m_KernRecord = -1;
            m_SearchPos  = m_SearchText;
            m_StrPos     = -1;
            continue;
         }else
         {
            if (m_SearchPos == m_SearchText && m_KernRecord < 0)
            {
               m_KernRecord = i;
               m_StrPos     = j;
            }
            ++m_SearchPos;
         }
         if (!m_SearchPos[0]) AddRecord(m_KernRecord, m_StrPos);
      }
   }
}

UI32 CPDFEditText::FindPattern(const wchar_t* Text)
{
   const UI16* txt = (const UI16*)ToUTF16(m_PDFInst, Text);
   UI32 len = StrLen(txt);
   if (!len) throw "The search text cannot be an empty string!";
   if (len > m_SearchTextLen)
   {
      UI16* tmp = (UI16*)realloc(m_SearchText, (len + 1) * sizeof(UI16));
      if (!tmp) throw "Out of memory!";
      m_SearchText    = tmp;
      m_SearchTextLen = len;
   }
   memcpy(m_SearchText, txt, len * sizeof(UI16));
   m_SearchText[len] = 0;
   m_LastFont = NULL;
   m_Records.Clear();
   if (!pdfInitStack(m_PDFInst, &m_Stack)) throw -1;
   m_CurrTmpl = -1;
   FindAll();
   ParseTemplates();
   return m_Records.Count();
}

double CPDFEditText::GetScaleFactor(TCTM& M)
{
   #define sin45 0.70710678118654752440084436210485
   double x = sin45 * M.a + sin45 * M.c;
   double y = sin45 * M.b + sin45 * M.d;
   return sqrt(x*x + y*y);
}

inline void CPDFEditText::Invert(TCTM &M)
{
   double  d = 1.0 / (M.a * M.d - M.b * M.c);
   double  a =  M.d * d;
         M.d =  M.a * d;
         M.b = -M.b * d;
         M.c = -M.c * d;
           d = -M.x *   a - M.y * M.c;
         M.y = -M.x * M.b - M.y * M.d;
         M.a =  a;
         M.x =  d;
}

inline TCTM CPDFEditText::MulMatrix(TCTM &M1, TCTM &M2)
{
   TCTM retval;
   retval.a = M2.a * M1.a + M2.b * M1.c;
   retval.b = M2.a * M1.b + M2.b * M1.d;
   retval.c = M2.c * M1.a + M2.d * M1.c;
   retval.d = M2.c * M1.b + M2.d * M1.d;
   retval.x = M2.x * M1.a + M2.y * M1.c + M1.x;
   retval.y = M2.x * M1.b + M2.y * M1.d + M1.y;
   return retval;
}

// Templates are parsed recursively.
void CPDFEditText::ParseTemplates(void)
{
   bool found = false;
   SI32 e, i, j, tmplCount, tmplCount2, *tmpl;
   tmplCount = pdfGetTemplCount(m_PDFInst);
   for (i = 0; i < tmplCount; i++)
   {
      if (!pdfEditTemplate(m_PDFInst, i)) throw -1;
      m_CurrTmpl = pdfGetTemplHandle(m_PDFInst);
      // We must make sure that we don't parse a template twice
      j = 0;
      e = m_Templates.Count() -1;
      while (j <= e)
      {
         tmpl = m_Templates.GetItem(j);
         if (*tmpl == m_CurrTmpl)
         {
            found = true;
            break;
         }
         tmpl = m_Templates.GetItem(e);
         if (*tmpl == m_CurrTmpl)
         {
            found = true;
            break;
         }
         ++j;
         --e;
      }
      if (found)
      {
         pdfEndTemplate(m_PDFInst);
         continue;
      }
      // Add the template handle to the list of templates
      tmpl = m_Templates.Add();
      *tmpl = m_CurrTmpl;

      if (!pdfInitStack(m_PDFInst, &m_Stack)) throw -1;

      FindAll();

      tmplCount2 = pdfGetTemplCount(m_PDFInst);
      for (j = 0; j < tmplCount2; j++)
      {
         ParseTemplates();
      }
      pdfEndTemplate(m_PDFInst);
   }
}

SI32 CPDFEditText::Pos(const char* Text, BYTE AChar)
{
   if (!Text) return -1;
   const char* s = Text;
   for (; *s != 0; ++s)
   {
      if (*s == AChar) return (SI32)(s - (char*)Text);
   }
   return -1;
}

void CPDFEditText::ReplacePattern(const wchar_t* NewText)
{
   UI32 i = 0, textLen;
   SI32 j, lastTmpl = -1;
   const UI16* newText = (const UI16*)ToUTF16(m_PDFInst, NewText);
   textLen = StrLen(newText);
   TTextRec* rec;
   if (!pdfInitStack(m_PDFInst, &m_Stack)) throw -1;
   m_Matrix4.x = 0.0;
   InitCTM(m_Matrix3);
   m_RecordNumber = 0;
   while (i < m_Records.Count())
   {
      rec = m_Records.GetItem(i);
      if (rec->TmplHandle != lastTmpl)
      {
         if (!pdfFlushPageContent(m_PDFInst, &m_Stack)) throw -1;
         if (lastTmpl > -1) pdfEndTemplate(m_PDFInst);
         if (!pdfEditTemplate2(m_PDFInst, rec->TmplHandle)) throw -1;
         if (!pdfInitStack(m_PDFInst, &m_Stack)) throw -1;
         m_RecordNumber = 0;
      }
      while (m_RecordNumber < rec->First)
      {
         pdfGetPageText(m_PDFInst, &m_Stack);
         ++m_RecordNumber;
      }
      if (rec->NewLine)
      {
         m_Matrix4.x = m_Stack.TextWidth;
         m_Matrix1 = MulMatrix(m_Stack.tm, m_Matrix4);
         m_Matrix3 = MulMatrix(m_Stack.ctm, m_Matrix1);
      }
      m_HaveMore = pdfGetPageText(m_PDFInst, &m_Stack);
      if (rec->KernRecord > 0)
      {
         // Delete the string but preserve the kerning records before the string occurred.
         m_Stack.DeleteKerningAt = rec->KernRecord;
         pdfReplacePageTextA(m_PDFInst, NULL, &m_Stack);
         m_Matrix4.x = 0.0;
         for (j = 0; j < rec->KernRecord; j++)
         {
            TTextRecordW &src = m_Stack.Kerning[j];
            m_Matrix4.x -= src.Advance;
            m_Matrix4.x += src.Width;
         }
         TTextRecordW &src = m_Stack.Kerning[rec->KernRecord];
         // If StrPos == -1 the first character is a space character which is emulated with kerning space.
         if (rec->StrPos > -1) m_Matrix4.x -= src.Advance;
         // Compute the string position
         m_Matrix1 = MulMatrix(m_Stack.tm, m_Matrix4);
         m_Matrix2 = MulMatrix(m_Stack.ctm, m_Matrix1);
         i = WriteLine(i, rec, newText, textLen);
      }else
      {
         pdfReplacePageTextA(m_PDFInst, NULL, &m_Stack);
         // Compute the string position
         m_Matrix2 = MulMatrix(m_Stack.ctm, m_Stack.tm);
         i = WriteLine(i, rec, newText, textLen);
      }
      lastTmpl = rec->TmplHandle;
   }
   if (!pdfFlushPageContent(m_PDFInst, &m_Stack)) throw -1;
   if (lastTmpl > -1) pdfEndTemplate(m_PDFInst);
}

void CPDFEditText::SetFont(void)
{
   if (m_LastFont != m_Stack.IFont)
   {
      TFStyle style;
      char familyName[65];
      familyName[0] = 0;
      if (fntBuildFamilyNameAndStyle(m_Stack.IFont, familyName, style))
         pdfSetFontSelMode(m_PDFInst, smFamilyName);
      else
         pdfSetFontSelMode(m_PDFInst, smPostScriptName);

      if (pdfSetFont(m_PDFInst, familyName, style, 10.0, true, cpUnicode) < 0)
      {
         pdfSetFontSelMode(m_PDFInst, smFamilyName);
         if ((m_Stack.FontFlags & 1) != 0) // Fixed pitch
         {
            if (pdfSetFont(m_PDFInst, "Courier New", style, 10.0, true, cpUnicode) < 0)
               throw -1;
         }else if (pdfSetFont(m_PDFInst, "Arial", style, 10.0, true, cpUnicode) < 0)
            throw -1;
      }
      m_LastFont = m_Stack.IFont;
   }
}

inline void CPDFEditText::Transform(TCTM &M, double &x, double &y)
{
   double ox = x;
   x = ox * M.a + y * M.c + M.x;
   y = ox * M.b + y * M.d + M.y;
}

UI32 CPDFEditText::WriteLine(UI32 Index, TTextRec* Record, const UI16* NewText, UI32 Len)
{
   // Process the remaining text in the kerning array
   ++Index;
   TTextRec* rec;
   SI32 j, p, count = 0;
   UI32 i;
   float spaceWidth = -m_Stack.SpaceWidth * 0.5f;
   while (Index < m_Records.Count())
   {
      rec = m_Records.GetItem(Index);
      if (rec->First != Record->First || rec->TmplHandle != Record->TmplHandle) break;
      ++Index;
      ++count;
   }
   /*
      We must always handle the text before and after the search text was found.
      Handling space characters correctly is the most important part if text
      should be replaced or deleted.
   */
   if (count > 0)
   {
      // The text was found multiple times in the text record
      TTextRecordW &src = m_Stack.Kerning[Record->KernRecord];
      WriteRecord(m_Matrix2, Record, src, NewText, Len);
      Index -= count;
      while (count-- > 0)
      {
         rec = m_Records.GetItem(Index);
         for (j = Record->KernRecord + 1; j < rec->KernRecord; j++)
         {
            TTextRecordW &tmp = m_Stack.Kerning[j];
            AddKernSpace(tmp.Advance, spaceWidth, m_Matrix2);
            WriteText(m_Matrix2, tmp.Text, tmp.Length);
         }
         TTextRecordW &src = m_Stack.Kerning[rec->KernRecord];
         if (rec->StrPos > -1 && m_SearchPos[0] != 32)
         {
            m_Matrix4.x = -src.Advance;
            m_Matrix2 = MulMatrix(m_Matrix2, m_Matrix4);
         }
         WriteRecord(m_Matrix2, rec, src, NewText, Len);
         Record = rec;
         ++Index;
      }
      // Handle the remaining records in the kerning array
      for (i = Record->KernRecord + 1; i < m_Stack.KerningCount; i++)
      {
         TTextRecordW &src = m_Stack.Kerning[i];
         AddKernSpace(src.Advance, spaceWidth, m_Matrix2);
         WriteText(m_Matrix2, src.Text, src.Length);
      }
   }else if (Record->KernRecord > -1)
   {
      TTextRecordW &src = m_Stack.Kerning[Record->KernRecord];
      WriteRecord(m_Matrix2, Record, src, NewText, Len);
      for (i = Record->KernRecord + 1; i < m_Stack.KerningCount; i++)
      {
         TTextRecordW &tmp = m_Stack.Kerning[i];
         AddKernSpaceEx(tmp.Advance, spaceWidth, m_Matrix2);
         if (m_SearchPos[0])
         {
            p = FindEndPattern(tmp.Text, tmp.Length);
            WriteText(m_Matrix2, tmp.Text + p, tmp.Length - p);
         }else
            WriteText(m_Matrix2, tmp.Text, tmp.Length);
      }
   }else
   {
      // The text begins at the end of the previous text record.
      TTextRecordW &src = m_Stack.Kerning[0];
      WriteRecord(m_Matrix3, Record, src, NewText, Len);
      for (i = 1; i < m_Stack.KerningCount; i++)
      {
         TTextRecordW &tmp = m_Stack.Kerning[i];
         AddKernSpace(tmp.Advance, spaceWidth, m_Matrix3);
         WriteText(m_Matrix3, tmp.Text, tmp.Length);
      }
      m_Matrix2 = m_Matrix3;
   }
   return WriteRemaining(Index, Record, NewText, Len);
}

void CPDFEditText::WriteRecord(TCTM &M, TTextRec* Record, TTextRecordW &Source, const UI16* NewText, UI32 Len)
{
   SI32 p, len;
   m_SearchPos = m_SearchText;
   if (Record->StrPos < 0)
   {
      ++m_SearchPos;
      p   = FindEndPattern(Source.Text, Source.Length);
      len = Source.Length - p;
      // Write the new text if any
      if (Len)
      {
         WriteText(M, NewText, Len);
      }
      if (len > 0)
      {
         WriteText(M, Source.Text + p, len);
      }
   }else
   {
      p = Record->StrPos;
      if (p > 0)
      {
         WriteText(M, Source.Text, p);
         // Write the new text if any
         if (Len > 0) WriteText(M, NewText, Len);
         p += FindEndPattern(Source.Text + p, Source.Length - p);
         if (p < (SI32)Source.Length)
         {
            WriteText(M, Source.Text + p, Source.Length - p);
         }
      }else
      {
         p   = FindEndPattern(Source.Text, Source.Length);
         len = Source.Length - p;
         // Write the new text if any
         if (Len > 0) WriteText(M, NewText, Len);
         if (len > 0) WriteText(M, Source.Text + p, len);
      }
   }
}

UI32 CPDFEditText::WriteRemaining(UI32 Index, TTextRec* Record, const UI16* NewText, UI32 Len)
{
   /*
    We handle here the rest of the text line if it is stored in separate text records.
    This is rather complex especially if the search text was found multiple times in the
    line and if the line consists in turn of multiple text records. We handle
    this case recursivley because this is much easier in comparison to a non-recursive
    version.
   */
   TCTM m1, m2;
   TTextRec* rec = NULL;
   float spaceWidth1;
   double x, y, spaceWidth2, distX, distY;
   SI32 j, p, len, lastAlpha;
   UI32 i, lastRecord;
   if (!m_HaveMore)
   {
      ++m_RecordNumber;
      return Index;
   }
   if (Index >= m_Records.Count())
      lastRecord = 0xFFFFFFFF;
   else
   {
      rec        = m_Records.GetItem(Index);
      lastRecord = rec->First;
   }
   m_LastX   = 0.0;
   m_LastY   = 0.0;
   m_Matrix3 = m_Matrix2;
   // Get the end offset of the last text record. The current position of the
   // original text is different because the new text was already written to
   // the file.
   m_Matrix4.x = m_Stack.TextWidth;
   m_Matrix2   = MulMatrix(m_Stack.tm, m_Matrix4);
   m_Matrix2   = MulMatrix(m_Stack.ctm, m_Matrix2);
   Transform(m_Matrix2, m_LastX, m_LastY);
   CalcAlpha(m_Matrix2);
   spaceWidth2 = m_Stack.SpaceWidth * GetScaleFactor(m_Matrix2) * 0.5;
   ++m_RecordNumber;
   // Was the search string found in multiple records?
   while (m_RecordNumber < Record->Last)
   {
      if ((m_HaveMore = pdfGetPageText(m_PDFInst, &m_Stack)) == 0) return Index;
      ++m_RecordNumber;
      spaceWidth1 = -m_Stack.SpaceWidth / 2.0f;
      m_Matrix2   = MulMatrix(m_Stack.ctm, m_Stack.tm);
      x = 0.0;
      y = 0.0;
      Transform(m_Matrix2, x, y);
      AddSpace(x, y, spaceWidth2);
      pdfReplacePageTextA(m_PDFInst, NULL, &m_Stack);
      for (i = 0; i < m_Stack.KerningCount; i++)
      {
         TTextRecordW &src = m_Stack.Kerning[i];
         AddKernSpaceEx(src.Advance, spaceWidth1, m_Matrix3);
         if (m_SearchPos[0] != 0)
         {
            p   = FindEndPattern(src.Text, src.Length);
            len = src.Length - p;
            if (len > 0)
            {
               // We are now at the end of the already replaced text. Only the remaining text behind the
               // search text must be written to the file.
               WriteText(m_Matrix3, src.Text + p, len);
            }
         }else
            WriteText(m_Matrix3, src.Text, src.Length);
      }
      spaceWidth2 = m_Stack.SpaceWidth * GetScaleFactor(m_Matrix2) * 0.5;
      m_LastY     = 0.0;
      m_LastX     = m_Stack.TextWidth;
      Transform(m_Matrix2, m_LastX, m_LastY);
   }
   /*
      Now we come to the most complicated part: we must find all records which lie on the same
      text line and move the text behind the replaced one to avoid overlaping text or holes
      between text records. We must consider the spacing between records to preserve the original
      layout. Note that we deal with two text positions here: m_Matrix2 contains the coordinates
      of the original text while m_Matrix3 represents the position of the new text.
   */
   m1 = m_Matrix2;
   Invert(m1);
   CalcAlpha(m_Matrix2);
   lastAlpha = m_Alpha;
   while (m_HaveMore && m_RecordNumber < lastRecord)
   {
      if ((m_HaveMore = pdfGetPageText(m_PDFInst, &m_Stack)) == 0) return Index;
      ++m_RecordNumber;
      spaceWidth1 = -m_Stack.SpaceWidth / 2.0f;
      CalcStrPos(m_Matrix2, x, y);
      if (m_Alpha != lastAlpha)
      {
         rec = NULL;
         break;
      }
      CalcDistance(m1, m_Matrix2, m2, distX, distY, x, y);
      if (distY > MAX_LINE_ERROR)
      {
         // We are on a new text line...
         rec = NULL;
         break;
      }else if (distX < -spaceWidth2 || distX > 6.0 * spaceWidth2)
      {
         // The distance is too large. We assume that the text should not be considered
         // as part of the current text line.
         rec = NULL;
         break;
      }
      pdfReplacePageTextA(m_PDFInst, NULL, &m_Stack);
      AddSpace(distX, spaceWidth2, m_Matrix3);
      for (i = 0; i < m_Stack.KerningCount; i++)
      {
         TTextRecordW &src = m_Stack.Kerning[i];
         AddKernSpaceEx(src.Advance, spaceWidth1, m_Matrix3);
         if (m_SearchPos[0] != 0)
         {
            p   = FindEndPattern(src.Text, src.Length);
            len = src.Length - p;
            if (len > 0)
            {
               // We are now at the end of the already replaced text. Only the remaining text behind the
               // search text must be written to the file.
               WriteText(m_Matrix3, src.Text + p, len);
            }
         }else
            WriteText(m_Matrix3, src.Text, src.Length);
      }
      lastAlpha   = m_Alpha;
      spaceWidth2 = m_Stack.SpaceWidth * GetScaleFactor(m_Matrix2) * 0.5;
      m_LastY     = 0.0;
      m_LastX     = m_Stack.TextWidth;
      Transform(m_Matrix2, m_LastX, m_LastY);
   }
   if (rec && !rec->NewLine)
   {
      ++Index;
      m_HaveMore = pdfGetPageText(m_PDFInst, &m_Stack);
      // We are still in the same text line. We must check whether a space character must be deleted.
      // The rest of the line is now processed recursively.
      CalcStrPos(m_Matrix2, x, y);
      if (!m_Alpha)
         distX = x - m_LastX;
      else
         distX = CalcDistance(m_LastX, m_LastY, x, y);
      AddSpace(distX, spaceWidth2, m_Matrix3);
      if (rec->KernRecord > 0)
      {
         // Delete the string but preserve the kerning records before the string occurred.
         m_Stack.DeleteKerningAt = rec->KernRecord;
         pdfReplacePageTextA(m_PDFInst, NULL, &m_Stack);
         m_Matrix4.x = 0.0;
         for (j = 0; j < rec->KernRecord; j++)
         {
            TTextRecordW &src = m_Stack.Kerning[j];
            m_Matrix4.x -= src.Advance;
            m_Matrix4.x += src.Width;
         }
         TTextRecordW &src = m_Stack.Kerning[rec->KernRecord];
         // If StrPos == -1 the first character is a space character which is emulated with kerning space.
         if (rec->StrPos > -1) m_Matrix4.x -= src.Advance;
         // Compute the string position
         m_Matrix2 = MulMatrix(m_Matrix3, m_Matrix4);
         Index = WriteLine(Index, rec, NewText, Len);
      }else
      {
         pdfReplacePageTextA(m_PDFInst, NULL, &m_Stack);
         m_Matrix2 = m_Matrix3;
         Index = WriteLine(Index, rec, NewText, Len);
      }
   }
   return Index;
}

void CPDFEditText::WriteText(TCTM &M, const UI16* Text, UI32 Len)
{
   // Initialize the graphics state
   SetFont();
   // The new text is written in red color so that you can better find it
   pdfSetColorSpace(m_PDFInst, csDeviceRGB);
   pdfSetFillColor(m_PDFInst, PDF_RED);
   pdfSetStrokeColor(m_PDFInst, PDF_RED);
   pdfSetTextDrawMode(m_PDFInst, m_Stack.DrawMode);

/*
   // This is the normal initalization
   pdfSetFillColorSpace(m_PDFInst, m_Stack.FillCS);
   pdfSetFillColor(m_PDFInst, m_Stack.FillColor);
   pdfSetStrokeColorSpace(m_PDFInst, m_Stack.StrokeCS);
   pdfSetStrokeColor(m_PDFInst, m_Stack.StrokeColor);
   pdfSetTextDrawMode(m_PDFInst, m_Stack.DrawMode);
*/
   pdfChangeFontSize(m_PDFInst, m_Stack.FontSize);
   pdfWriteTextMatrixExW(m_PDFInst, &M, Text, Len);
   // Compute the end offset of the text
   m_Matrix4.x = pdfGetTextWidthExW(m_PDFInst, Text, Len);
   M = MulMatrix(M, m_Matrix4);
}
