#include "render_page.h"

using namespace DynaPDF;

#if defined(_UNICODE) || defined(UNICODE)
   #define FullPath _wfullpath
#else
   #define FullPath _fullpath
#endif

/* ----------------------------------------------------------------------------------------- */

// DynaPDF error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   (void)ErrCode;
   (void)ErrType;
   CApp* app = (CApp*)Data;
   ::MessageBoxA(app->GetHWND(), ErrMessage, "Error", MB_OK);
   return 0; // any other return value breaks processing!
}

// This function is called if the path or image coverage limit was reached.
// It is quite important to set this callback function since it makes sure that
// we can see something if a complex page will be rendered.
SI32 PDF_CALL OnUpdateWindow(const void* Data, struct TIntRect* Area)
{
   ((CApp*)Data)->UpdateArea(Area);
   return 0;
}

/* ----------------------------------------------------------------------------------------- */

DWORD WINAPI RenderPageFunc(LPVOID Param)
{
   ((CApp*)Param)->RenderPage();
   return 0;
}

/* ----------------------------------------------------------------------------------------- */

CApp::CApp(HINSTANCE Instance) :
   CBaseApp(Instance),
   m_AdjWindow(true),
   m_CurrPage(0),
   m_CurrPageObj(NULL),
   m_ImpPages(NULL),
   m_PageCount(0),
   m_PDF(NULL),
   m_RAS(NULL),
   m_RenderThread(0),
   m_Update(false)
{
   // Initialize the TRasterImage structure
   memset(&m_RasImage, 0, sizeof(m_RasImage));
   m_RasImage.StructSize            = sizeof(m_RasImage);
   m_RasImage.DefScale              = psFitBest;
   m_RasImage.InitWhite             = true;
   // We draw the image with SetDIBitsToDevice() and this function does not support alpha transparency.
   // To get a correct result we pre-blend the image with a white background.
   m_RasImage.Flags                 = TRasterFlags(rfDefault | rfCompositeWhite);
   m_RasImage.Matrix.a              = 1.0;
   m_RasImage.Matrix.d              = 1.0;
   // The OnUpdateWindow callback function makes sure that we don't see a gray or white screen for a long time when rendering complex pages.
   m_RasImage.OnUpdateWindow        = OnUpdateWindow;
   m_RasImage.UpdateOnImageCoverage = 0.5f;
   m_RasImage.UpdateOnPathCount     = 1000;
   m_RasImage.UserData              = this;
}

CApp::~CApp(void)
{
   StopRenderThread();
   if (m_ImpPages) free(m_ImpPages);
   if (m_RAS)      rasDeleteRasterizer(&m_RAS);
   if (m_PDF)      pdfDeletePDF(m_PDF);
}

bool CApp::Init(void)
{
   TPDFColorProfiles p;
   TCHAR path[MAX_PATH+1] = {0}, monitor[MAX_PATH+1] = {0};
   if ((m_PDF = pdfNewPDF()) == NULL) return false;
   pdfSetOnErrorProc(m_PDF, this, PDFError);

   // It is important to set an absolute path here since a relative path
   // doesn't work if the working directory will be changed at runtime.
   // The flag lcmDelayed makes sure that the cmaps will only be loaded
   // if necessary.
   FullPath(path, TEXT("../../../../Resource/CMap/"), MAX_PATH);
   pdfSetCMapDir(m_PDF, path, TLoadCMapFlags(lcmRecursive | lcmDelayed));

   // Initialize color management
   GetMonitorProfile(monitor);
   FullPath(path, TEXT("../../../test_files/ISOcoated_v2_bas.ICC"), MAX_PATH);

   memset(&p, 0, sizeof(p));
   p.StructSize = sizeof(p);
  #if defined(_UNICODE) || defined(UNICODE)
   p.DefInCMYKW     = path;
   p.DeviceProfileW = monitor;
  #else
   p.DefInCMYKA     = path;
   p.DeviceProfileA = monitor;
  #endif
   pdfInitColorManagement(m_PDF, &p, csDeviceRGB, TPDFInitCMFlags(icmBPCompensation | icmCheckBlackPoint));
   return true;
}

bool CApp::InitImpPageArray(void)
{
   if (m_ImpPages) free(m_ImpPages);
   if ((m_ImpPages = (BYTE*)calloc(1, (m_PageCount >> 3) +1)) == NULL)
   {
      ::MessageBoxA(m_Wnd, "Out of memory!", "Fatal error", MB_OK);
      return false;
   }
   return true;
}

void CApp::OnKey(UI32 KeyCode, UI32 Ctrl)
{
   switch(Ctrl)
   {
      default:         break;
      case VK_CONTROL: if (KeyCode == 'O') OnShowOpenFileDlg(); break;
      case VK_UP:      RenderNextPage(m_CurrPage-1);            break;
      case VK_DOWN:    RenderNextPage(m_CurrPage+1);            break;
   }
}

void CApp::OnMouseWheel(SI32 Delta, SI32 X, SI32 Y)
{
   (void)X;
   (void)Y;
   if (Delta < 0)
      RenderNextPage(m_CurrPage+1);
   else
      RenderNextPage(m_CurrPage-1);
}

void CApp::OnPaint(void)
{
   if (!m_CurrPage)
      ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &m_ClientRect, NULL, 0, NULL);
   else
   {
      RECT r;
      ::SetBkColor(m_DC, APP_BACK_COLOR);
      // Left
      r.left   = 0;
      r.right  = APP_CLIENT_BORDER2;
      r.bottom = m_ClientRect.bottom;
      r.top    = m_ClientRect.top;
      ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &r, NULL, 0, NULL);
      // Bottom
      r.left   = APP_CLIENT_BORDER2;
      r.right  = m_ClientRect.right;
      r.bottom = m_ClientRect.bottom;
      r.top    = m_ClientRect.bottom - APP_CLIENT_BORDER2;
      ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &r, NULL, 0, NULL);
      // Right
      r.left   = APP_CLIENT_BORDER2;
      r.right  = m_ClientRect.right;
      r.bottom = m_ClientRect.top + APP_CLIENT_BORDER2;
      r.top    = m_ClientRect.top;
      // Top
      ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &r, NULL, 0, NULL);
      r.left   = m_ClientRect.right - APP_CLIENT_BORDER2;
      r.right  = m_ClientRect.right;
      r.bottom = m_ClientRect.bottom;
      r.top    = m_ClientRect.top;
      ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &r, NULL, 0, NULL);
      if (m_Update)
      {
         if (!m_RenderThread) StartRenderThread();
      }else if (m_CurrPage > 0)
         ::SetDIBitsToDevice(m_DC, APP_CLIENT_BORDER2, APP_CLIENT_BORDER2, m_ImgW, m_ImgH, 0, 0, 0, m_ImgH, m_Buffer, m_BMPInfo, 0);
   }
}

void CApp::OnResize(UI32 Width, UI32 Height)
{
   m_WindowW = Width;
   m_WindowH = Height;
   ::GetClientRect(m_Wnd, &m_ClientRect);
   ::GetWindowRect(m_Wnd, &m_WindowRect);

   ::SetBkColor(m_DC, APP_BACK_COLOR);
   ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &m_ClientRect, NULL, 0, NULL);
   m_AdjWindow = true;
   RenderCurrPage();
}

void CApp::OnShowOpenFileDlg(void)
{
   TCHAR in[MAX_PATH+1] = {0};
   m_CurrPage  = 0;
   m_PageCount = 0;
   StopRenderThread();
   ::SetBkColor(m_DC, APP_BACK_COLOR);
   ::ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, &m_ClientRect, NULL, 0, NULL);
   if (OpenFileDlg(m_Wnd, in))
   {
      SetCaption(in);
      if (pdfHaveOpenDoc(m_PDF)) pdfFreePDF(m_PDF);
      pdfCreateNewPDF(m_PDF, NULL); // We create no PDF file in this example

      if (pdfOpenImportFile(m_PDF, in, ptOpen, NULL) < 0) return;

      // We import pages manually in this example and therefore, no global objects will
      // be imported as it would be the case if ImportPDFFile() would be used.
      // However, the one and only thing we need is the output intent for correct
      // color management. Anything else can be discarded.
      pdfSetImportFlags(m_PDF, ifContentOnly);
      pdfImportCatalogObjects(m_PDF);

      pdfSetImportFlags(m_PDF, ifImportAll | ifImportAsPage);     // The flag ifImportAsPage makes sure that pages will not be converted to templates.
      pdfSetImportFlags2(m_PDF, if2UseProxy | if2NoResNameCheck); // The flag if2UseProxy reduces the memory usage and if2NoResNameCheck improves the import speed.

      m_PageCount = pdfGetInPageCount(m_PDF);

      InitImpPageArray();

      m_AdjWindow = true;
      m_CurrPage  = 1;
      RenderCurrPage();
   }else
      ::InvalidateRect(m_Wnd, NULL, TRUE);
}

void CApp::RenderCurrPage(void)
{
   StopRenderThread();
   if (m_CurrPage > 0 && m_WindowW > m_BorderX && m_WindowH > m_BorderY)
   {
      SI32 w = 0, h = 0;
      if (!IsPageAvailable(m_CurrPage))
      {
         // No need to check the return value of ImportPageEx(). Nothing critical happens
         // if the function fails. We get just an empty page in this case.
         pdfEditPage(m_PDF, m_CurrPage);
            pdfImportPageEx(m_PDF, m_CurrPage, 1.0, 1.0);
         pdfEndPage(m_PDF);
         AddPage(m_CurrPage);
      }
      // This check is required to avoid critical errors.
      if ((m_CurrPageObj = pdfGetPageObject(m_PDF, m_CurrPage)) == NULL) return;
      rasCalcPagePixelSize(m_CurrPageObj, psFitBest, 1.0f, m_ScreenW-m_BorderX, m_ScreenH-m_BorderY, m_RasImage.Flags, (UI32*)&w, (UI32*)&h);
      if (w != m_ImgW || h != m_ImgH)
      {
         m_ImgW = w;
         m_ImgH = h;
         if (m_Buffer) free(m_Buffer);
         m_BufSize = (m_ImgW << 2) * m_ImgH;
         if ((m_Buffer = (BYTE*)malloc(m_BufSize)) == NULL) return;
         if (!m_RAS)
         {
            if ((m_RAS = rasCreateRasterizer(m_PDF, NULL, m_Buffer, m_ImgW, m_ImgH, m_ImgW << 2, m_PixFmt)) == NULL) return;
         }else
         {
            if (rasAttachImageBuffer(m_RAS, NULL, m_Buffer, m_ImgW, m_ImgH, m_ImgW << 2) < 0) return;
         }
         if (!UpdateBitmapInfo()) return;
      }
      if (m_AdjWindow)
      {
         w = m_ImgW + m_BorderX;
         h = m_ImgH + m_BorderY;
         if (m_ClientRect.right - m_ClientRect.left - APP_CLIENT_BORDER != m_ImgW || m_ClientRect.bottom - m_ClientRect.top - APP_CLIENT_BORDER != m_ImgH)
         {
            m_AdjWindow = false;
            ::MoveWindow(m_Wnd, (m_ScreenW-w)>>1, (m_ScreenH-h)>>1, w, h, true);
         }
      }
      m_Update = true;
      ::InvalidateRect(m_Wnd, NULL, FALSE);
   }
}

void CApp::RenderNextPage(SI32 PageNum)
{
   if (PageNum < 1 || PageNum > m_PageCount || PageNum == m_CurrPage) return;
   m_CurrPage  = PageNum;
   m_AdjWindow = true;
   RenderCurrPage();
}

void CApp::StartRenderThread(void)
{
   StopRenderThread();
   m_Update = false;
   if ((m_RenderThread = ::CreateThread(NULL, 0, RenderPageFunc, this, CREATE_SUSPENDED, NULL)) == 0)
   {
      ::MessageBox(m_Wnd, TEXT("Failed to create render thread!"), TEXT("Fatal error!"), MB_OK);
      return;
   }
   ::SetThreadPriority(m_RenderThread, THREAD_PRIORITY_LOWEST);
   ::ResumeThread(m_RenderThread);
}

void CApp::StopRenderThread(void)
{
   if (m_RenderThread)
   {
      rasAbort(m_RAS);
      ::WaitForSingleObject(m_RenderThread, INFINITE);
      ::CloseHandle(m_RenderThread);
      m_RenderThread = NULL;
   }
}

/* ----------------------------------------------------------------------------------------- */

int APIENTRY WinMain(HINSTANCE Instance, HINSTANCE PrevInstance, LPSTR CmdLine, int CmdShow)
{
   (void)PrevInstance;
   (void)CmdLine;
   (void)CmdShow;
   CApp app(Instance);
   if (app.InitWindow(TEXT("CApp"), TEXT("RenderPage()"), TEXT("AppMenu")))
   {
      if (!app.Init()) return -1;
      PostMessage(app.GetHWND(), WM_OPEN_FILE, 0, 0);
      return app.Run();
   }else
      return -1;
}
