#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

/*
   Note that the dynapdf.dll must be copied into the output directory or into a
   Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
*/

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

SI32 PDF_CALL FontNotFoundProc(const void* Data, const PFNT* PDFFont, const char* FontName, SI32 Style, SI32 StdFontIndex, LBOOL IsSymbolFont)
{
   if (WeightFromStyle(Style) < 500)
   {
      Style &= 0xF;
      Style |= fsRegular;
   }
   return pdfReplaceFont((PPDF*)Data, PDFFont, "Arial", Style, true);
}

SI32 PDF_CALL ReplaceICCProfileProc(const void* Data, TICCProfileType Type, SI32 ColorSpace)
{
   // The most important ICC profiles are available free of charge from Adobe. Just seach for "Adobe icc profiles".

   // A gray profile is optional (default switch).
   switch(Type)
   {
      case 1:  return pdfReplaceICCProfile((PPDF*)Data, ColorSpace, "../../../test_files/sRGB.icc");
      case 2:  return pdfReplaceICCProfile((PPDF*)Data, ColorSpace, "../../../test_files/ISOcoated_v2_bas.ICC");
      default: return pdfReplaceICCProfile((PPDF*)Data, ColorSpace, "../../../test_files/gray.icc");
   }
}

bool ConvertFile(const PPDF* PDF, TConformanceType Type, const char* InFile, const char* OutFile)
{
   SI32 retval;
   UI32 convFlags;

   pdfCreateNewPDF(PDF, NULL);           // The output file will be created later
   pdfSetDocInfo(PDF, diProducer, NULL); // No need to override the original producer

   switch(Type)
   {
      case ctNormalize:    convFlags = coAllowDeviceSpaces;              break; // For normalization it is not required to convert device spaces to ICC based color spaces.
      case ctPDFA_1b_2005: convFlags = coDefault | coFlattenLayers;      break; // Presentations are not prohibited in PDF/A 1.
      case ctPDFA_2b:
      case ctPDFA_2u:      convFlags = coDefault | coDeletePresentation; break;
      case ctPDFA_3b:
      case ctPDFA_3u:
      case ctPDFA_4:
      case ctPDFA_4e:
      case ctPDFA_4f:
      case ctZUGFeRD_Basic:
      case ctZUGFeRD_Comfort:
      case ctZUGFeRD_Extended:
      case ctZUGFeRD2_Minimum:
      case ctZUGFeRD2_Basic:
      case ctZUGFeRD2_Basic_WL:
      case ctZUGFeRD2_Comfort:
      case ctZUGFeRD2_Extended:
      case ctFacturX_Minimum:
      case ctFacturX_Basic:
      case ctFacturX_Basic_WL:
      case ctFacturX_Comfort:
      case ctFacturX_Extended:
         convFlags  = coDefault | coDeletePresentation;
         convFlags &= ~coDeleteEmbeddedFiles;           // Embedded files are allowed in PDF/A 3.
         break;
   }
   // These flags require some processing time but very useful.
   convFlags |= coCheckImages;
   convFlags |= coRepairDamagedImages;
   if (Type != ctNormalize)
   {
      // The flag ifPrepareForPDFA is required. The flag ifImportAsPage makes sure that pages will not be converted to templates.
      pdfSetImportFlags(PDF, ifImportAll | ifImportAsPage | ifPrepareForPDFA);
      // The flag if2UseProxy reduces the memory usage. The duplicate check is optional but recommended.
      pdfSetImportFlags2(PDF, if2UseProxy | if2DuplicateCheck);
   }else
   {
      pdfSetImportFlags(PDF, ifImportAll | ifImportAsPage);
      pdfSetImportFlags2(PDF, if2UseProxy | if2DuplicateCheck | if2Normalize);
   }
   if ((retval = pdfOpenImportFile(PDF, InFile, ptOpen, NULL)) < 0)
   {
      if (IsWrongPwd(retval))
      {
         printf("File is encrypted!");
      }
      goto err;
   }
   pdfImportPDFFile(PDF, 1, 1.0, 1.0);
   pdfCloseImportFile(PDF);

   // The CMYK profile is just an example profile that can be delivered with DynaPDF.
   // Note that this code requires the PDF/A Extension for DynaPDF.
   switch ((retval = pdfCheckConformance(PDF, Type, (TCheckOptions)convFlags, PDF, FontNotFoundProc, ReplaceICCProfileProc)))
   {
      case 1: pdfAddOutputIntent(PDF, "../../../test_files/sRGB.icc");             break;
      case 2: pdfAddOutputIntent(PDF, "../../../test_files/ISOcoated_v2_bas.ICC"); break;
      case 3: pdfAddOutputIntent(PDF, "../../../test_files/gray.icc");             break;
   }
   // No fatal error occurred?
   if (!pdfHaveOpenDoc(PDF) || !pdfOpenOutputFile(PDF, OutFile)) goto err;
   if (!pdfCloseFile(PDF)) goto err;
   return true;
  err:
   pdfFreePDF(PDF);
   return false;
}

int main(int argc, char* argv[])
{
   char filePath[MAX_PATH+1];
   if (argc < 1) return -1; // This should not occur
   PPDF* pdf = pdfNewPDF();
   if (!pdf) return 2; // Out of memory?

   pdfSetOnErrorProc(pdf, NULL, PDFError);

   // Set the license key here if you have one
   // pdfSetLicenseKey(pdf, "");

   // Non embedded CID fonts depend usually on the availability of external cmaps.
   // So, external cmaps should be loaded if possible.
   _fullpath(filePath, "../../../../Resource/CMap/", MAX_PATH);
   pdfSetCMapDir(pdf, filePath, TLoadCMapFlags(DynaPDF::lcmDelayed | DynaPDF::lcmRecursive));

   GetCurrentDirectory(MAX_PATH, filePath);
   strcat(filePath, "/out.pdf");

   /* To create a ZUGFeRD invoice attach the required XML invoice here and set the conversion type to the ZUGFeRD or Factur-X profile that you need.

      Example (ZUGFeRD 2.1):
      pdfAttachFile(pdf, "c:/invoices/test/factur-x.xml", "ZUGFeRD 2.1 Rechnung", true);
      if (ConvertFile(pdf, ctFacturX_Comfort, "c:/invoices/test/TestInvoice.pdf", filePath))
      {
         printf("PDF file \"%s\" successfully created!\n", filePath);
         ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
      }

      The file name of the XML invoice must be ZUGFeRD-invoice.xml. If the file has another name than rename it or use AttachFileEx() instead.
   */
   if (ConvertFile(pdf, ctPDFA_3b, "../../../../dynapdf_help.pdf", filePath))
   {
      printf("PDF file \"%s\" successfully created!\n", filePath);
      ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
   }

   pdfDeletePDF(pdf);
   return 0;
}