#include <windows.h>
#include <shellapi.h>
#include <stdio.h>
#include <conio.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../include/C_CPP/dynapdf.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

// Error callback function.
SI32 PDF_CALL PDFError(const void* Data, SI32 ErrCode, const char* ErrMessage, SI32 ErrType)
{
   printf("%s\n", ErrMessage);
   return 0; // any other return value breaks processing!
}

SI32 PlaceEMFCentered(const PPDF* PDF, const char* FileName, double Width, double Height)
{
   TRectL r;
   double x, y;
   /*
      We place the EMF file horizontal and vertically centered in this example.
   */
   // We need the picture size of the EMF file
   if (!pdfGetLogMetafileSize(PDF, FileName, &r)) return -1;

   double w = r.Right - r.Left;
   double h = r.Bottom - r.Top;
   /*
      We place the EMF file into a border of 2 units around the page so that we can better see
      the bounding box of the EMF file.
   */
   Width  -= 4.0;
   Height -= 4.0;
   double sx = Width / w;
   /*
      An important note about the bounding rectangle: DynaPDF calculates the zero point
      of the EMF picture automatically so that we don't need to consider the coordinate origin.
      The coordinate origin for our calculation here is always 0.0, 0.0 independent of the real
      origin of the EMF picture.
   */
   if (h * sx <= Height)
   {
      h *= sx;
      x = 2.0;
      // If the file should not be centered vertically then set y to 2.0.
      y = (Height - h) / 2.0;
      pdfInsertMetafile(PDF, FileName, x, y, Width, 0.0);
      // The rectangle represents the real bounding box of the EMF picture.
      pdfSetStrokeColor(PDF, PDF_RED);
      return pdfRectangle(PDF, x, y, Width, h, fmStroke);
   }else
   {
      sx = Height / h;
      w *= sx;
      x = (Width - w) / 2.0;
      y = 2.0;
      pdfInsertMetafile(PDF, FileName, x, y, 0.0, Height);
      // The rectangle represents the real bounding box of the EMF picture.
      pdfSetStrokeColor(PDF, PDF_RED);
      return pdfRectangle(PDF, x, y, w, Height, fmStroke);
   }
}

int main(int argc, char* argv[])
{
   PPDF* pdf = pdfNewPDF();
   if (!pdf) return 2; // Out of memory?
   // Error messages and warnings are passed to the callback function.
   pdfSetOnErrorProc(pdf, NULL, PDFError);
   pdfCreateNewPDF(pdf, NULL); // The output file is created later

   pdfSetPageCoords(pdf, pcTopDown);

   pdfAppend(pdf);
      /*
         We use a landscape paper format in this example. SetOrientationEx() rotates the coordinate
         system according to the orientation and sets the page orientation. You can now work with page
         as if it was not rotated. The real page format is still DIN A4 (this is the default format).
         The difference to SetOrientation() is that this function does not change the page's coordinate
         system.

         It would also be possible to use a user defined paper format without changing the page
         orientation but the disadvantage is that a printer driver must then manually rotate the page
         because landscape paper formats do not exist in most printers. This step requires an
         additional interaction with the user which is simply not required when creating landscape
         paper formats in the right way.
      */
      pdfSetOrientationEx(pdf, 90);

      // This file transforms the coordinate system very often and uses clipping regions. The metafile
      // is scaled to the page width without changing its aspect ratio.
      PlaceEMFCentered(pdf, "../../test_files/coords.emf", pdfGetPageWidth(pdf), pdfGetPageHeight(pdf));
   pdfEndPage(pdf);

   pdfAppend(pdf);
      pdfSetOrientationEx(pdf, 90);
      // Simple test of line and standard patterns
      PlaceEMFCentered(pdf, "../../test_files/fulltest.emf", pdfGetPageWidth(pdf), pdfGetPageHeight(pdf));
   pdfEndPage(pdf);

   pdfAppend(pdf);
      pdfSetOrientationEx(pdf, 90);
      // Outlined text, dash patterns with text. This file requires the font Bookman Old Style. If not available,
      // the result will be wrong!
      PlaceEMFCentered(pdf, "../../test_files/gdi.emf", pdfGetPageWidth(pdf), pdfGetPageHeight(pdf));
   pdfEndPage(pdf);

   // No fatal error occurred?
   if (pdfHaveOpenDoc(pdf))
   {
      // We write the output file into the current directory.
      char filePath[MAX_PATH+1];
      GetCurrentDirectory(MAX_PATH, filePath);
      strcat(filePath, "\\out.pdf");
      // OK, now we can open the output file.
      if (!pdfOpenOutputFile(pdf, filePath))
      {
         pdfDeletePDF(pdf);
         _getch();
         return -1;
      }
      if (pdfCloseFile(pdf))
      {
         printf("PDF file \"%s\" successfully created!\n", filePath);
         ShellExecute(0, "open", filePath, NULL, NULL, SW_SHOWMAXIMIZED);
      }
      // If you want to encrypt the file, use CloseFileEx() instead.
      //pdfCloseFileEx(pdf, NULL, "test", kl128bit, rsDenyAll & ~rsPrint & ~rsPrintHighRes);
   }
   pdfDeletePDF(pdf); // Do not forget to delete the class instance
   return 0;
}
