#include "pdf_signature.h"

#include <stdio.h>
#include <stdlib.h>

namespace DynaPDF
{

// ----------------------------- cryptui.dll ------------------------------

// flags for dwDontUseColumn
#ifndef __CRYPTUIAPI_H__ // Header file cryptuiapi.h was not included?
   #define CRYPTUI_SELECT_ISSUEDTO_COLUMN      0x000000001
   #define CRYPTUI_SELECT_ISSUEDBY_COLUMN      0x000000002
   #define CRYPTUI_SELECT_INTENDEDUSE_COLUMN   0x000000004
   #define CRYPTUI_SELECT_FRIENDLYNAME_COLUMN  0x000000008
   #define CRYPTUI_SELECT_LOCATION_COLUMN      0x000000010
   #define CRYPTUI_SELECT_EXPIRATION_COLUMN    0x000000020
#endif

#ifndef CERT_FIND_HAS_PRIVATE_KEY
   #define CERT_FIND_HAS_PRIVATE_KEY 21<<16
#endif

typedef PCCERT_CONTEXT WINAPI cryptUIDlgSelectCertificateFromStore(
    IN HCERTSTORE hCertStore,
    IN OPTIONAL HWND hwnd,              // Defaults to the desktop window
    IN OPTIONAL LPCWSTR pwszTitle,
    IN OPTIONAL LPCWSTR pwszDisplayString,
    IN DWORD dwDontUseColumn,
    IN DWORD dwFlags,
    IN void *pvReserved);

// ------------------------------------------------------------------------

CWinCrypt::CWinCrypt(void) :
   m_CertCtx(NULL),
   m_PKCS7Obj(NULL),
   m_Store(NULL)
{}

CWinCrypt::~CWinCrypt(void)
{
   CleanUp();
}

void CWinCrypt::CleanUp(void)
{
   if (m_CertCtx)  CertFreeCertificateContext(m_CertCtx);
   if (m_PKCS7Obj) free(m_PKCS7Obj);
   if (m_Store)    CertCloseStore(m_Store, CERT_CLOSE_STORE_CHECK_FLAG);
   m_CertCtx  = NULL;
   m_PKCS7Obj = NULL;
   m_Store    = NULL;
}

bool CWinCrypt::CloseAndSignPDFFile(const PPDF* PDF, const char* OutFile, bool DetachedSignature)
{
   // Dummy string to compute the PKCS#7 object size
   BYTE                       dummy[20] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
   CRYPT_ALGORITHM_IDENTIFIER hashAlgorithm;
   PCCERT_CONTEXT             msgCertArray[1] = {m_CertCtx};
   const BYTE*                msgArray[2]     = {dummy, NULL};
   DWORD                      msgArraySize[2] = {20, 0};
   DWORD                      objLen          = 0;
   TPDFSigParms               pdfParms;
   HRESULT                    result;
   CRYPT_SIGN_MESSAGE_PARA    signParms;

   // Initialize all structures with zero
   memset(&hashAlgorithm, 0, sizeof(hashAlgorithm));
   memset(&pdfParms,      0, sizeof(pdfParms));
   memset(&signParms,     0, sizeof(signParms));

   // The hash algorithm can be changed if necessary. However, make sure that
   // you use the right identifier, e.g. szOID_RSA_MD5 to create a MD5 hash.
   hashAlgorithm.pszObjId = szOID_OIWSEC_sha1;

   // Initialize the signature structure
   signParms.cbSize            = sizeof(signParms);
   signParms.dwMsgEncodingType = CRYPT_ASN_ENCODING | PKCS_7_ASN_ENCODING;
   signParms.pSigningCert      = m_CertCtx;
   signParms.HashAlgorithm     = hashAlgorithm;
   signParms.cMsgCert          = 1;
   signParms.rgpMsgCert        = msgCertArray;

   // Get the size of the signed PKCS#7 object.
   result = CryptSignMessage(
             &signParms,                       // Signature parameters
             DetachedSignature ? TRUE : FALSE, // Detached signature?
             1,                                // Number of messages
             msgArray,                         // Messages to be signed
             msgArraySize,                     // Size of messages
             NULL,                             // Buffer for signed messages
             &objLen);                         // Size of buffer

   if (result == FALSE)
   {
      printf("CryptSignMessage() failed!");
      return false;
   }

   // Allocate memory for the signed PKCS#7 object.
   if ((m_PKCS7Obj = (BYTE*)calloc(1, objLen)) == NULL)
   {
      printf("Out of memory!");
      return false;
   }

   // Initialze the PDF signature structure
   pdfParms.StructSize   = sizeof(pdfParms);                        // Required
   pdfParms.HashType     = DetachedSignature ? htDetached : htSHA1; // Required
   pdfParms.ContactInfoA = "http://www.dynaforms.com";              // Optional
   pdfParms.LocationA    = "Germany";                               // Optional
   pdfParms.PKCS7ObjLen  = objLen;                                  // Required
   pdfParms.ReasonA      = "Test";                                  // Optional

   // Get the file hash or the byte ranges of the PDF buffer. The PDF file stays fully
   // in memory at this point. We open the output file when the PDF file was successfully
   // closed and when we have successfully signed the provided hash or byte ranges.
   if (!pdfCloseAndSignFileExt(PDF, &pdfParms)) return false;

   // Initialize the message array now with the real values
   msgArray[0]     = pdfParms.Range1;
   msgArraySize[0] = pdfParms.Range1Len;
   msgArray[1]     = pdfParms.Range2;
   msgArraySize[1] = pdfParms.Range2Len;

   // Sign the provided hash or byte ranges
   result = CryptSignMessage(
             &signParms,                       // Signature parameters
             DetachedSignature ? TRUE : FALSE, // Detached signature?
             DetachedSignature ? 2 : 1,        // When creating a detached signature we have two byte ranges, otherwise we have a hash
             msgArray,                         // Messages to be signed
             msgArraySize,                     // Size of messages
             m_PKCS7Obj,                       // Buffer for signed messages
             &objLen);                         // Size of buffer

   if (result == FALSE)
   {
      printf("CryptSignMessage() failed to sign the message!");
      return false;
   }

   // Anything seems to be ok. Let's open the output file.
   if (!pdfOpenOutputFile(PDF, OutFile)) return false;

   // Now we can write the PKCS#7 object to the PDF file and finally write the result into the output file.
   bool retval = pdfFinishSignature(PDF, m_PKCS7Obj, pdfParms.PKCS7ObjLen) != 0;
   CleanUp();
   return retval;
}

BYTE* CWinCrypt::GetFileBuf(const char* FileName, long &BufSize)
{
   FILE* f = fopen(FileName, "r+b");
   if (!f)
   {
      BufSize = 0;
      printf("Cannot open PFX file!\n");
      return NULL;
   }
   fseek(f, 0, SEEK_END);
   BufSize = ftell(f);
   if (BufSize <= 0)
   {
      printf("Error reading PFX file!\n");
      fclose(f);
      return NULL;
   }
   fseek(f, 0, SEEK_SET);
   BYTE* retval = (BYTE*)malloc(BufSize);
   if (!retval)
   {
      printf("Out of memory!");
      fclose(f);
      return NULL;
   }
   fread(retval, 1, BufSize, f);
   fclose(f);
   return retval;
}

bool CWinCrypt::LoadCertFromFile(const char* FileName, const wchar_t* Password)
{
   LONG bufSize;
   _CRYPTOAPI_BLOB blob;

   if ((blob.pbData = (BYTE*)GetFileBuf(FileName, bufSize)) == NULL) return false;
   blob.cbData = bufSize;

   m_Store = PFXImportCertStore(&blob, Password, 0);
   free(blob.pbData);

   if (!m_Store)
   {
      printf("Error %x during PFXImportCertStore!\n", GetLastError());
      return false;
   }
   return true;
}

bool CWinCrypt::OpenCert(UI32 Index)
{
   UI32 idx = 0;
   if (m_CertCtx)
   {
      CertFreeCertificateContext(m_CertCtx);
      m_CertCtx = NULL;
   }
   while ((m_CertCtx = CertFindCertificateInStore(m_Store, X509_ASN_ENCODING | PKCS_7_ASN_ENCODING, 0, CERT_FIND_HAS_PRIVATE_KEY, NULL, m_CertCtx)) != NULL)
   {
      if (idx == Index) break;
      ++idx;
   }
   return (m_CertCtx != NULL);
}

bool CWinCrypt::OpenStore(const wchar_t* Name)
{
   if (m_Store) CertCloseStore(m_Store, CERT_CLOSE_STORE_CHECK_FLAG);
   if ((m_Store = CertOpenStore(CERT_STORE_PROV_SYSTEM, 0, NULL, CERT_SYSTEM_STORE_CURRENT_USER | CERT_STORE_READONLY_FLAG, Name)) == NULL)
   {
      printf("Error %x during CertOpenStore!\n", GetLastError());
      return false;
   }
   return true;
}

bool CWinCrypt::SelectCertificateFromStore(void)
{
   HMODULE lib;
   cryptUIDlgSelectCertificateFromStore* selectCertificate;
   if (m_CertCtx)
   {
      CertFreeCertificateContext(m_CertCtx);
      m_CertCtx = NULL;
   }
   if ((lib = LoadLibrary("cryptui.dll")) == NULL)
   {
      printf("Unable to load the cryptui.dll!\nThe DLL is included in the Windows SDK.\n");
      return false;
   }
   if ((selectCertificate = (cryptUIDlgSelectCertificateFromStore*)GetProcAddress(lib, "CryptUIDlgSelectCertificateFromStore")) == NULL)
   {
      FreeLibrary(lib);
      printf("Function CryptUIDlgSelectCertificateFromStore() not found in cryptui.dll!\n");
      return false;
   }
   m_CertCtx = selectCertificate(m_Store,
                                 NULL,
                                 NULL,
                                 NULL,
                                 CRYPTUI_SELECT_LOCATION_COLUMN | CRYPTUI_SELECT_INTENDEDUSE_COLUMN | CRYPTUI_SELECT_FRIENDLYNAME_COLUMN,
                                 0,
                                 NULL);

   FreeLibrary(lib);

   // Some further checkings should normally be applied here, e.g. whether the certificate
   // contains a private key or whether it is already expired. If the certificate contains
   // no private key or if other restrictions avoid the usage CloseAndSignPDFFile() will fail.
   if (!m_CertCtx)
   {
      printf("No certificate selected!\n");
      return false;
   }
   return true;
}

} // end namespace
