#ifndef pdf_textsearchH
#define pdf_textsearchH

#include <stdio.h>
#include <stdlib.h>

#define PDF_STATIC // Enable static binding when compiling the project with the workspace dynapdf_static.
#include "../../../../include/C_CPP/dynapdf.h"

using namespace DynaPDF;

/*
   We assume that a text record lies on the same line if the error on the y-axis is less
   than two units. A little error is always possible especially if the PDF file was created
   by a GDI printer driver because the reference point of text in the GDI is the upper left
   corner of the font's bounding box while it is the baseline in PDF. This difference causes
   often differences if a font is changed, e.g. from a regular style to italic or bold.
*/
#define MAX_LINE_ERROR 4.0 // This must be the square of the allowed error (2 * 2 in this case).
#define MAX_BUF_SIZE   32  // TranslateRawCode() requires a 32 character long conversion buffer

class CTextSearch
{
  public:
   CTextSearch(void);
   ~CTextSearch(void);
   SI32 BeginTemplate(TPDFRect* BBox, TCTM* M);
   void EndTemplate(void)                {RestoreGState();}
   UI32 GetSelCount(void){return m_SelCount;}
   void Init(void);
   SI32 MarkText(TCTM* Matrix, const TTextRecordA* Source, UI32 Count, double Width);
   void MulMatrix(TCTM* M)               {m_GState.Matrix = MulMatrix(m_GState.Matrix, *M);}
   bool RestoreGState(void);
   SI32 SaveGState(void);
   void SetCharSpacing(double Value)     {m_GState.CharSpacing = (float)Value;}
   void SetFont(double FontSize, TFontType Type, const void* Font);
   void SetPDFInst(void* IPDF)           {m_PDF = IPDF;}
   SI32 SetSearchText(const wchar_t* Text);
   void SetTextDrawMode(TDrawMode Mode)  {m_GState.TextDrawMode = Mode;}
   void SetTextScale(double Value)       {m_GState.TextScale    = (float)Value;}
   void SetWordSpacing(double Value)     {m_GState.WordSpacing  = (float)Value;}
  protected:
   typedef enum
   {
      tfLeftToRight    = 0,
      tfRightToLeft    = 1,
      tfTopToBottom    = 2,
      tfBottomToTop    = 4,
      tfNotInitialized = -1
   }TTextDir;
   struct TGState
   {
      const void* ActiveFont;
      TFontType   FontType;
      float       CharSpacing;
      double      FontSize;
      TCTM        Matrix;
      float       SpaceWidth;
      TDrawMode   TextDrawMode;
      float       TextScale;
      float       WordSpacing;
   };
   class CStack
   {
     public:
      CStack(void) :
         m_Capacity(0),
         m_Count(0),
         m_Items(NULL)
      {}
      ~CStack(void)
      {
         if (m_Items) free(m_Items);
      }
      bool Restore(TGState &F);
      SI32 Save(TGState &F);
     private:
      UI32     m_Capacity;
      UI32     m_Count;
      TGState* m_Items;
   };
   friend class CStack;

   double     m_EndX1;
   double     m_EndY1;
   double     m_EndX4;
   double     m_EndY4;
   TGState    m_GState;
   bool       m_HavePos;
   TTextDir   m_LastTextDir;
   double     m_LastTextInfX;
   double     m_LastTextInfY;
   UI16       m_OutBuf[MAX_BUF_SIZE];
   void*      m_PDF;
   UI16*      m_SearchPos;
   UI16*      m_SearchText;
   UI32       m_SearchTextLen;
   UI32       m_SelCount;
   CStack     m_Stack;
   double     m_x1;
   double     m_y1;
   double     m_x4;
   double     m_y4;

   double CalcDistance(double x1, double y1, double x2, double y2);
   bool   Compare(const UI16* Text, UI32 Length);
   LBOOL  DrawRect(TCTM &Matrix, double EndX);
   LBOOL  DrawRectEx(double x2, double y2, double x3, double y3);
   void   InitGState(void);
   bool   IsPointOnLine(double x, double y, double x0, double y0, double x1, double y1);
   LBOOL  MarkSubString(double &x, TCTM &Matrix, const TTextRecordA &Source);
   TCTM   MulMatrix(TCTM &M1, TCTM &M2);
   void   Reset(void);
   void   SetStartCoord(TCTM &Matrix, double x);
   UI32   StrLen(const UI16* AStr)
   {
      if (!AStr) return 0;
      const UI16* p = AStr;
      while (*p++);
      return (UI32)(p - AStr);
   }
   void   Transform(TCTM &M, double &x, double &y);
};

#endif
