#define PDF_STATIC // Enable static binding when compiling the project with the workspace dynapdf_static.
#include <math.h>
#include "../../../../include/C_CPP/dynapdf.h"
#include "pdf_textsearch.h"

#define TRUE 1

CTextSearch::CTextSearch(void) :
   m_EndX1(0.0),
   m_EndY1(0.0),
   m_EndX4(0.0),
   m_EndY4(0.0),
   m_HavePos(false),
   m_LastTextInfX(0.0),
   m_LastTextInfY(0.0),
   m_LastTextDir(tfNotInitialized),
   m_PDF(NULL),
   m_SearchPos(NULL),
   m_SearchText(NULL),
   m_SearchTextLen(0),
   m_SelCount(0),
   m_x1(0.0),
   m_y1(0.0),
   m_x4(0.0),
   m_y4(0.0)
{
   m_GState.ActiveFont   = NULL;
   m_GState.CharSpacing  = 0.0;
   m_GState.FontSize     = 1.0;
   m_GState.FontType     = ftType1;
   m_GState.Matrix.a     = 1.0;
   m_GState.Matrix.b     = 0.0;
   m_GState.Matrix.c     = 0.0;
   m_GState.Matrix.d     = 1.0;
   m_GState.Matrix.x     = 0.0;
   m_GState.Matrix.y     = 0.0;
   m_GState.SpaceWidth   = 0.0;
   m_GState.TextDrawMode = dmNormal;
   m_GState.TextScale    = 100.0;
   m_GState.WordSpacing  = 0.0;
}

CTextSearch::~CTextSearch(void)
{
   if (m_SearchText) free(m_SearchText);
}

SI32 CTextSearch::BeginTemplate(TPDFRect* BBox, TCTM* M)
{
   (void)BBox;
   Reset();
   if (SaveGState() < 0) return -1; // Out of memory?
   if (M) m_GState.Matrix = MulMatrix(m_GState.Matrix, *M);
   return 0;
}

inline double CTextSearch::CalcDistance(double x1, double y1, double x2, double y2)
{
   double dx = x2-x1;
   double dy = y2-y1;
   return sqrt(dx*dx + dy*dy);
}

bool CTextSearch::Compare(const UI16* Text, UI32 Length)
{
   while (Length-- > 0)
   {
      if (*m_SearchPos != *Text++)
      {
         m_HavePos   = false;
         m_SearchPos = m_SearchText;
         return false;
      }
      ++m_SearchPos;
      if (!m_SearchPos[0])
      {
         m_SearchPos = m_SearchText;
         return (Length == 0);
      }
   }
   return true;
}

LBOOL CTextSearch::DrawRect(TCTM &Matrix, double EndX)
{
   // Note that the start and end coordinate can use different transformation matrices
   double x2 = EndX;
   double y2 = 0.0;
   double x3 = EndX;
   double y3 = m_GState.FontSize;
   Transform(Matrix, x2, y2);
   Transform(Matrix, x3, y3);
   return DrawRectEx(x2, y2, x3, y3);
}

LBOOL CTextSearch::DrawRectEx(double x2, double y2, double x3, double y3)
{
   pdfMoveTo(m_PDF, m_x1, m_y1);
   pdfLineTo(m_PDF, x2, y2);
   pdfLineTo(m_PDF, x3, y3);
   pdfLineTo(m_PDF, m_x4, m_y4);
   m_HavePos = false;
   ++m_SelCount;
   return pdfClosePath(m_PDF, fmFill);
}

void CTextSearch::Init(void)
{
   InitGState();
   Reset();
   m_SelCount = 0;
}

void CTextSearch::InitGState(void)
{
   while (RestoreGState());
   m_GState.ActiveFont   = NULL;
   m_GState.CharSpacing  = 0.0;
   m_GState.FontSize     = 1.0;
   m_GState.Matrix.a     = 1.0;
   m_GState.Matrix.b     = 0.0;
   m_GState.Matrix.c     = 0.0;
   m_GState.Matrix.d     = 1.0;
   m_GState.Matrix.x     = 0.0;
   m_GState.Matrix.y     = 0.0;
   m_GState.SpaceWidth   = 0.0;
   m_GState.TextDrawMode = dmNormal;
   m_GState.TextScale    = 100.0;
   m_GState.WordSpacing  = 0.0;

   m_LastTextDir         = tfNotInitialized;
   m_LastTextInfX        = 0.0;
   m_LastTextInfY        = 0.0;
}

bool CTextSearch::IsPointOnLine(double x, double y, double x0, double y0, double x1, double y1)
{
   double dx, dy, di;
   x -= x0;
   y -= y0;
   dx = x1 - x0;
   dy = y1 - y0;
   di = (x*dx + y*dy) / (dx*dx + dy*dy);
   di = (di < 0.0) ? 0.0 : (di > 1.0) ? 1.0 : di;
   dx = x - di * dx;
   dy = y - di * dy;
   di = dx*dx + dy*dy;
   return (di < MAX_LINE_ERROR);
}

LBOOL CTextSearch::MarkSubString(double &x, TCTM &Matrix, const TTextRecordA &Source)
{
   UI32 i = 0;
   SI32 outLen;
   LBOOL decoded;
   float spaceWidth2 = -m_GState.SpaceWidth * 6.0f;
   double width;
   UI32 max = Source.Length;
   BYTE* src = Source.Text;
   if (Source.Advance < -m_GState.SpaceWidth)
   {
      // If the distance is too large then we assume that no space was emulated at this position.
      if (Source.Advance > spaceWidth2 && m_SearchPos[0] == 32)
      {
         if (!m_HavePos)
         {
            SetStartCoord(Matrix, x);
            ++m_SearchPos;
            if (!m_SearchPos[0])
            {
               if (!DrawRect(Matrix, x - Source.Advance)) return -1;
               Reset();
            }
         }else if (!m_SearchPos[0])
         {
            if (!DrawRect(Matrix, 0.0)) return -1;
            Reset();
         }else
            ++m_SearchPos;
      }else
         Reset();
   }
   x -= Source.Advance;
   while (i < max)
   {
      outLen = MAX_BUF_SIZE;
      i += fntTranslateRawCode(m_GState.ActiveFont, src + i, max - i, width, m_OutBuf, outLen, decoded, m_GState.CharSpacing, m_GState.WordSpacing, m_GState.TextScale);
      // We skip this text record if the text cannot be converted to Unicode. The return value must be TRUE,
      // otherwise we would break processing.
      if (!decoded) return TRUE;
      // outLen is always greater zero if decoded is true!
      if (Compare(m_OutBuf, outLen))
      {
         if (!m_HavePos)
         {
            SetStartCoord(Matrix, x);
         }
         x += width;
         if (m_SearchPos == m_SearchText)
         {
            if (!DrawRect(Matrix, x - m_GState.CharSpacing)) return -1;
         }
      }else
         x += width;
   }
   return TRUE;
}

SI32 CTextSearch::MarkText(TCTM* Matrix, const TTextRecordA* Source, UI32 Count, double Width)
{
   /*
      Note that we write rectangles to the page while we parsed it. This is critical because the parser
      doesn't notice when a fatal error occurs, e.g. out of memory. We must make sure that processing
      breaks immediatly in such a case. To archive this the return value of ClosePath() is checked each
      time a rectangle is drawn to the page. The function can only fail if a fatal error occurred.
   */
   UI32 i;
   TTextDir textDir;
   double x;
   double x1 = 0.0;
   double y1 = 0.0;
   double x2 = 0.0;
   double y2 = m_GState.FontSize;
   // Transform the text matrix to user space
   TCTM m = MulMatrix(m_GState.Matrix, *Matrix);
   Transform(m, x1, y1); // Start point of the text record
   Transform(m, x2, y2); // Second point to determine the text direction
   // Determine the text direction
   if (y1 == y2)
      textDir = (TTextDir)(((x1 > x2) + 1) << 1);
   else
      textDir = (TTextDir)(y1 > y2);

   if (textDir != m_LastTextDir || !IsPointOnLine(x1, y1, m_EndX1, m_EndY1, m_LastTextInfX, m_LastTextInfY))
   {
      // Extend the x-coordinate to an infinite point
      m_LastTextInfX = 1000000.0;
      m_LastTextInfY = 0.0;
      Transform(m, m_LastTextInfX, m_LastTextInfY);
      Reset();
   }else
   {
      double distance, spaceWidth;
      double x3 = m_GState.SpaceWidth;
      double y3 = 0.0;
      Transform(m, x3, y3);
      spaceWidth = CalcDistance(x1, y1, x3 ,y3);
      distance   = CalcDistance(m_EndX1, m_EndY1, x1, y1);
      if (distance > spaceWidth)
      {
         // If the distance is too large then we assume that no space was emulated at this position.
         if (distance < spaceWidth * 6.0 && m_SearchPos[0] == 32)
         {
            if (!m_HavePos)
            {
               // The start coordinate is the end coordinate of the last text record.
               m_HavePos = true;
               ++m_SearchPos;
               if (!m_SearchPos[0])
               {
                  m_x1 = m_EndX1;
                  m_y1 = m_EndY1;
                  m_x4 = m_EndX4;
                  m_y4 = m_EndY4;
                  if (!DrawRectEx(x1, y1, x2, y2)) return -1;
                  Reset();
               }
            }else if (!m_SearchPos[0])
            {
               if (!DrawRectEx(x1, y1, x2, y2)) return -1;
               Reset();
            }else
               ++m_SearchPos;
         }else
            Reset();
      }
   }
   x = 0.0;
   for (i = 0; i < Count; i++)
   {
      if (!MarkSubString(x, m, Source[i])) return -1;
   }
   m_LastTextDir = textDir;
   m_EndX1 = Width;
   m_EndY1 = 0.0;
   m_EndX4 = 0.0;
   m_EndY4 = m_GState.FontSize;
   Transform(m, m_EndX1, m_EndY1);
   Transform(m, m_EndX4, m_EndY4);
   return 0;
}

TCTM CTextSearch::MulMatrix(TCTM &M1, TCTM &M2)
{
   TCTM retval;
   retval.a = M2.a * M1.a + M2.b * M1.c;
   retval.b = M2.a * M1.b + M2.b * M1.d;
   retval.c = M2.c * M1.a + M2.d * M1.c;
   retval.d = M2.c * M1.b + M2.d * M1.d;
   retval.x = M2.x * M1.a + M2.y * M1.c + M1.x;
   retval.y = M2.x * M1.b + M2.y * M1.d + M1.y;
   return retval;
}

void CTextSearch::Reset(void)
{
   m_HavePos   = false;
   m_SearchPos = m_SearchText;
}

bool CTextSearch::RestoreGState(void)
{
   return m_Stack.Restore(m_GState);
}

SI32 CTextSearch::SaveGState(void)
{
   return m_Stack.Save(m_GState);
}

void CTextSearch::SetFont(double FontSize, TFontType Type, const void* Font)
{
   m_GState.ActiveFont = Font;
   m_GState.FontSize   = FontSize;
   m_GState.FontType   = Type;
   m_GState.SpaceWidth = (float)(fntGetSpaceWidth(Font, FontSize) * 0.5);
}

SI32 CTextSearch::SetSearchText(const wchar_t* Text)
{
   if (!Text || *Text == 0) return -1;
   m_SearchTextLen = StrLen((UI16*)Text);
   if ((m_SearchText = (UI16*)malloc((m_SearchTextLen +1) * sizeof(UI16))) == NULL) return -2;
   for (UI32 i = 0; i < m_SearchTextLen; i++)
      m_SearchText[i] = Text[i];
   m_SearchText[m_SearchTextLen] = 0;
   m_SearchPos = m_SearchText;
   return 0;
}

void CTextSearch::SetStartCoord(TCTM &Matrix, double x)
{
   m_x1 = x;
   m_y1 = 0.0;
   m_x4 = x;
   m_y4 = m_GState.FontSize;
   Transform(Matrix, m_x1, m_y1);
   Transform(Matrix, m_x4, m_y4);
   m_HavePos = true;
}

inline void CTextSearch::Transform(TCTM &M, double &x, double &y)
{
   double tx = x;
   x = tx * M.a + y * M.c + M.x;
   y = tx * M.b + y * M.d + M.y;
}

/* --------------------------------------- CStack ------------------------------------------- */

bool CTextSearch::CStack::Restore(TGState &F)
{
   if (m_Count > 0)
   {
      --m_Count;
      F = m_Items[m_Count];
      return true;
   }
   return false;
}

SI32 CTextSearch::CStack::Save(TGState &F)
{
   if (m_Count == m_Capacity)
   {
      m_Capacity += 16;
      TGState* tmp = (TGState*)realloc(m_Items, m_Capacity * sizeof(TGState));
      if (!tmp)
      {
         m_Capacity -= 16;
         return -1;
      }
      m_Items = tmp;
   }
   m_Items[m_Count] = F;
   ++m_Count;
   return 0;
}
