#ifndef pdf_text_extractionH
#define pdf_text_extractionH

#include <stdio.h>
#include <stdlib.h>

#define PDF_STATIC // Enable static binding when compiling the project with the workspace dynapdf_static.
#include "../../../../include/C_CPP/dynapdf.h"

using namespace DynaPDF;

/*
   The base class CTextStack creates text lines and words from the output of the
   content parser. The algorithm processes rotated text as well as text lines which
   consist of multiple text records correctly.

   If the logical reading order should be preserved the entire text of a page must
   be stored in an array which must be sorted in x- and y-direction before trying
   to construct text lines.

   Note that all functions of this class are called within a callback function of
   DynaPDF. If you want to use structured exception handling then make sure that
   all exceptions are catched before the callback function returns.
*/

/*
   We assume that a text record lies on the same line if the error on the y-axis is less
   than two units. A little error is always possible especially if the PDF file was created
   by a GDI printer driver because the reference point of text in the GDI is the upper left
   corner of the font's bounding box while it is the baseline in PDF. This difference causes
   often differences if a font is changed, e.g. from a regular style to italic or bold.
*/
#define MAX_LINE_ERROR 4.0 // This must be the square of the allowed error (2 * 2 in this case).

class CTextExtraction
{
  public:
   CTextExtraction(void);
   ~CTextExtraction(void);
   SI32 AddText(TCTM* M, const TTextRecordA* Source, const TTextRecordW* Kerning, UI32 Count, double Width, bool Decoded);
   SI32 BeginTemplate(TPDFRect* BBox, TCTM* M);
   void Close(void)                      {if (m_File){fclose(m_File); m_File = NULL;}}
   void EndTemplate(void)                {RestoreGState();}
   void Init(void);
   void MulMatrix(TCTM* M)               {m_GState.Matrix = MulMatrix(m_GState.Matrix, *M);}
   bool Open(const char* FileName);
   bool RestoreGState(void);
   SI32 SaveGState(void);
   void SetCharSpacing(double Value)     {m_GState.CharSpacing  = (float)Value;}
   void SetFillColor(UI32 Value)         {m_GState.FillColor    = Value;}
   void SetFont(double FontSize, TFontType Type, const void* Font);
   void SetStrokeColor(UI32 Value)       {m_GState.StrokeColor  = Value;}
   void SetTextDrawMode(TDrawMode Mode)  {m_GState.TextDrawMode = Mode;}
   void SetTextScale(double Value)       {m_GState.TextScale    = (float)Value;}
   void SetWordSpacing(double Value)     {m_GState.WordSpacing  = (float)Value;}
   void WritePageIdentifier(SI32 PageNum);
  protected:
   typedef enum
   {
      tfLeftToRight    = 0,
      tfRightToLeft    = 1,
      tfTopToBottom    = 2,
      tfBottomToTop    = 4,
      tfNotInitialized = -1
   }TTextDir;
   struct TGState
   {
      const void* ActiveFont;
      TFontType   FontType;
      float       CharSpacing;
      UI32        FillColor;
      double      FontSize;
      TCTM        Matrix;
      float       SpaceWidth;
      UI32        StrokeColor;
      TDrawMode   TextDrawMode;
      float       TextScale;
      float       WordSpacing;
   };
   class CStack
   {
     public:
      CStack(void) :
         m_Capacity(0),
         m_Count(0),
         m_Items(NULL)
      {}
      ~CStack(void)
      {
         if (m_Items) free(m_Items);
      }
      bool Restore(TGState &F);
      SI32 Save(TGState &F);
     private:
      UI32     m_Capacity;
      UI32     m_Count;
      TGState* m_Items;
   };
   friend class CStack;
   FILE*        m_File;
   TGState      m_GState;
   TTextDir     m_LastTextDir;
   double       m_LastTextEndX;
   double       m_LastTextEndY;
   double       m_LastTextInfX;
   double       m_LastTextInfY;
   CStack       m_Stack;

   double CalcDistance(double x1, double y1, double x2, double y2);
   bool   IsPointOnLine(double x, double y, double x0, double y0, double x1, double y1);
   TCTM   MulMatrix(TCTM &M1, TCTM &M2);
   UI32   StrLen(const UI16* AStr)
   {
      if (!AStr) return 0;
      const UI16* p = AStr;
      while (*p++);
      return (UI32)(p - AStr);
   }
   void Transform(TCTM &M, double &x, double &y);
};

#endif
