#include <math.h>
#define PDF_STATIC // Enable static binding when compiling the project with the workspace dynapdf_static.
#include "../../../../include/C_CPP/dynapdf.h"
#include "pdf_text_extraction.h"

using namespace DynaPDF;

CTextExtraction::CTextExtraction(void) :
   m_File(NULL),
   m_LastTextDir(tfNotInitialized),
   m_LastTextEndX(0.0),
   m_LastTextEndY(0.0),
   m_LastTextInfX(0.0),
   m_LastTextInfY(0.0)
{
   m_GState.ActiveFont   = NULL;
   m_GState.CharSpacing  = 0.0;
   m_GState.FontSize     = 1.0;
   m_GState.FontType     = ftType1;
   m_GState.Matrix.a     = 1.0;
   m_GState.Matrix.b     = 0.0;
   m_GState.Matrix.c     = 0.0;
   m_GState.Matrix.d     = 1.0;
   m_GState.Matrix.x     = 0.0;
   m_GState.Matrix.y     = 0.0;
   m_GState.SpaceWidth   = 0.0;
   m_GState.TextDrawMode = dmNormal;
   m_GState.TextScale    = 100.0;
   m_GState.WordSpacing  = 0.0;
}

CTextExtraction::~CTextExtraction(void)
{
   if (m_File) fclose(m_File);
}

SI32 CTextExtraction::AddText(TCTM* M, const TTextRecordA* Source, const TTextRecordW* Kerning, UI32 Count, double Width, bool Decoded)
{
   if (!Decoded) return 0;
   UI16 space[] = {32};
   UI16 newLine[] = {13, 10};
   TTextDir textDir;
   double x1 = 0.0;
   double y1 = 0.0;
   double x2 = 0.0;
   double y2 = m_GState.FontSize;
   // Transform the text matrix to user space
   TCTM m = MulMatrix(m_GState.Matrix, *M);
   // Start point of the text record
   Transform(m, x1, y1);
   /* The second point to determine the text direction can also be used to calculate
      the visible font size measured in user space:

      double realFontSize = CalcDistance(x1, y1, x2, y2);
   */
   Transform(m, x2, y2);
   // Determine the text direction
   if (y1 == y2)
      textDir = (TTextDir)(((x1 > x2) + 1) << 1);
   else
      textDir = (TTextDir)(y1 > y2);

   // Wrong direction or not on the same text line?
   if (textDir != m_LastTextDir || !IsPointOnLine(x1, y1, m_LastTextEndX, m_LastTextEndY, m_LastTextInfX, m_LastTextInfY))
   {
      // Extend the x-coordinate to an infinite point.
      m_LastTextInfX = 1000000.0;
      m_LastTextInfY = 0.0;
      Transform(m, m_LastTextInfX, m_LastTextInfY);
      if (m_LastTextDir != tfNotInitialized)
      {
         // Add a new line to the output file
         fwrite(newLine, 2, 2, m_File);
      }
   }else
   {
      /*
         The space width is measured in text space but the distance between two text
         records is measured in user space! We must transform the space width to user
         space before we can compare the values.
      */
      double distance, spaceWidth;
      // Note that we use the full space width here because the end position of the last record
      // was set to the record width minus the half space width.
      double x3 = m_GState.SpaceWidth;
      double y3 = 0.0;
      Transform(m, x3, y3);
      spaceWidth = CalcDistance(x1, y1, x3 ,y3);
      distance   = CalcDistance(m_LastTextEndX, m_LastTextEndY, x1, y1);
      if (distance > spaceWidth)
      {
         // Add a space to the output file
         fwrite(space, 2, 1, m_File);
      }
   }
   // We use the half space width to determine whether a space must be inserted at
   // a specific position. This produces better results in most cases.
   float spaceWidth = -m_GState.SpaceWidth * 0.5f;
   for (UI32 i = 0; i < Count; i++)
   {
      const TTextRecordW &rec = Kerning[i];
      if (rec.Advance < spaceWidth)
      {
         // Add a space to the output file
         fwrite(space, 2, 1, m_File);
      }
      fwrite(rec.Text, 2, rec.Length, m_File);
   }
   // We don't set the cursor to the real end of the string because applications like MS Word
   // add often a space to the end of a text record and this space can slightly overlap the next
   // record. IsPointOnLine() would return false if the new record overlaps the previous one.
   m_LastTextEndX = Width + spaceWidth; // spaceWidth is a negative value!
   m_LastTextEndY = 0.0;
   m_LastTextDir  = textDir;
   // Calculate the end coordinate of the text record
   Transform(m, m_LastTextEndX, m_LastTextEndY);
   return 0;
}

SI32 CTextExtraction::BeginTemplate(TPDFRect* BBox, TCTM* M)
{
   if (SaveGState() < 0) return -1; // Out of memory?
   if (M) m_GState.Matrix = MulMatrix(m_GState.Matrix, *M);
   return 0;
}

inline double CTextExtraction::CalcDistance(double x1, double y1, double x2, double y2)
{
   double dx = x2-x1;
   double dy = y2-y1;
   return sqrt(dx*dx + dy*dy);
}

void CTextExtraction::Init(void)
{
   while (RestoreGState());
   m_GState.ActiveFont   = NULL;
   m_GState.CharSpacing  = 0.0;
   m_GState.FontSize     = 1.0;
   m_GState.FontType     = ftType1;
   m_GState.Matrix.a     = 1.0;
   m_GState.Matrix.b     = 0.0;
   m_GState.Matrix.c     = 0.0;
   m_GState.Matrix.d     = 1.0;
   m_GState.Matrix.x     = 0.0;
   m_GState.Matrix.y     = 0.0;
   m_GState.SpaceWidth   = 0.0;
   m_GState.TextDrawMode = dmNormal;
   m_GState.TextScale    = 100.0;
   m_GState.WordSpacing  = 0.0;

   m_LastTextEndX        = 0.0;
   m_LastTextEndY        = 0.0;
   m_LastTextDir         = tfNotInitialized;
   m_LastTextInfX        = 0.0;
   m_LastTextInfY        = 0.0;
}

bool CTextExtraction::IsPointOnLine(double x, double y, double x0, double y0, double x1, double y1)
{
   double dx, dy, di;
   x -= x0;
   y -= y0;
   dx = x1 - x0;
   dy = y1 - y0;
   di = (x*dx + y*dy) / (dx*dx + dy*dy);
   di = (di < 0.0) ? 0.0 : (di > 1.0) ? 1.0 : di;
   dx = x - di * dx;
   dy = y - di * dy;
   di = dx*dx + dy*dy;
   return (di < MAX_LINE_ERROR);
}

TCTM CTextExtraction::MulMatrix(TCTM &M1, TCTM &M2)
{
   TCTM retval;
   retval.a = M2.a * M1.a + M2.b * M1.c;
   retval.b = M2.a * M1.b + M2.b * M1.d;
   retval.c = M2.c * M1.a + M2.d * M1.c;
   retval.d = M2.c * M1.b + M2.d * M1.d;
   retval.x = M2.x * M1.a + M2.y * M1.c + M1.x;
   retval.y = M2.x * M1.b + M2.y * M1.d + M1.y;
   return retval;
}

bool CTextExtraction::Open(const char* FileName)
{
   if ((m_File = fopen(FileName, "wb+")) != NULL)
   {
      // Add a little endian identifier to the file
      fwrite("\377\376", 1, 2, m_File);
      return true;
   }
   return false;
}

bool CTextExtraction::RestoreGState(void)
{
   return m_Stack.Restore(m_GState);
}

SI32 CTextExtraction::SaveGState(void)
{
   return m_Stack.Save(m_GState);
}

void CTextExtraction::SetFont(double FontSize, TFontType Type, const void* Font)
{
   m_GState.ActiveFont = Font;
   m_GState.FontSize   = FontSize;
   m_GState.FontType   = Type;
   m_GState.SpaceWidth = (float)fntGetSpaceWidth(Font, FontSize);
   if (FontSize < 0.0)
      m_GState.SpaceWidth = -m_GState.SpaceWidth;
}

void CTextExtraction::Transform(TCTM &M, double &x, double &y)
{
   double tx = x;
   x = tx * M.a + y * M.c + M.x;
   y = tx * M.b + y * M.d + M.y;
}

void CTextExtraction::WritePageIdentifier(SI32 PageNum)
{
   if (PageNum > 1)
   {
      UI16 newLine[] = {13, 10};
      fwrite(newLine, 2, 2, m_File);
   }
   m_LastTextDir = tfNotInitialized;
   fwprintf(m_File, L"%%----------------------- Page %d -----------------------------\r\n", PageNum);
}

/* --------------------------------------- CStack ------------------------------------------- */

bool CTextExtraction::CStack::Restore(TGState &F)
{
   if (m_Count > 0)
   {
      --m_Count;
      F = m_Items[m_Count];
      return true;
   }
   return false;
}

SI32 CTextExtraction::CStack::Save(TGState &F)
{
   if (m_Count == m_Capacity)
   {
      m_Capacity += 16;
      TGState* tmp = (TGState*)realloc(m_Items, m_Capacity * sizeof(TGState));
      if (!tmp)
      {
         m_Capacity -= 16;
         return -1;
      }
      m_Items = tmp;
   }
   m_Items[m_Count] = F;
   ++m_Count;
   return 0;
}
