#include <math.h>

#define PDF_STATIC // Enable static binding. This define makes nothing when we link dynamically.
#include "../../../../include/C_CPP/dynapdf.h"
#include "pdf_text_coordinates.h"

#if defined(WIN64) || defined(_WIN64)
   #ifdef _DLL
      #pragma comment(lib, "../../../../win64/dynapdfm.lib") // Multithreaded-DLL
   #else
      #pragma comment(lib, "../../../../win64/dynapdf.lib")  // Multithreaded
   #endif
#elif defined(_DLL)
   #pragma comment(lib, "../../../../win32/dynapdfm.lib") // Multithreaded-DLL
#else
   #pragma comment(lib, "../../../../win32/dynapdf.lib")  // Multithreaded
#endif

using namespace DynaPDF;

CTextCoordinates::CTextCoordinates(void) :
   m_Count(0),
   m_PDF(NULL)
{
   m_GState.ActiveFont   = NULL;
   m_GState.CharSpacing  = 0.0;
   m_GState.FontSize     = 1.0;
   m_GState.FontType     = ftType1;
   m_GState.Matrix.a     = 1.0;
   m_GState.Matrix.b     = 0.0;
   m_GState.Matrix.c     = 0.0;
   m_GState.Matrix.d     = 1.0;
   m_GState.Matrix.x     = 0.0;
   m_GState.Matrix.y     = 0.0;
   m_GState.SpaceWidth   = 0.0;
   m_GState.TextDrawMode = dmNormal;
   m_GState.TextScale    = 100.0;
   m_GState.WordSpacing  = 0.0;
}

SI32 CTextCoordinates::BeginTemplate(TPDFRect* BBox, TCTM* M)
{
   if (SaveGState() < 0) return -1; // Out of memory?
   if (M) m_GState.Matrix = MulMatrix(m_GState.Matrix, *M);
   return 0;
}

inline double CTextCoordinates::CalcDistance(double x1, double y1, double x2, double y2)
{
   double dx = x2-x1;
   double dy = y2-y1;
   return sqrt(dx*dx + dy*dy);
}

void CTextCoordinates::Init(void)
{
   while (RestoreGState());
   m_GState.ActiveFont   = NULL;
   m_GState.CharSpacing  = 0.0;
   m_GState.FontSize     = 1.0;
   m_GState.FontType     = ftType1;
   m_GState.Matrix.a     = 1.0;
   m_GState.Matrix.b     = 0.0;
   m_GState.Matrix.c     = 0.0;
   m_GState.Matrix.d     = 1.0;
   m_GState.Matrix.x     = 0.0;
   m_GState.Matrix.y     = 0.0;
   m_GState.SpaceWidth   = 0.0;
   m_GState.TextDrawMode = dmNormal;
   m_GState.TextScale    = 100.0;
   m_GState.WordSpacing  = 0.0;
}

SI32 CTextCoordinates::MarkCoordinates(TCTM* M, const TTextRecordA* Source, const TTextRecordW* Kerning, UI32 Count, double Width, bool Decoded)
{
   /*
      Note that we write lines to the page while we parsed it. This is critical because the parser
      doesn't notice when a fatal error occurs, e.g. out of memory. We must make sure that processing
      breaks immediatly in such a case. To archive this we check the return value of StrokePath() since
      the only reason why this function can fail is out of memory.
   */
   if (!Decoded) return 0;
   UI32 i;
   double x1 = 0.0, y1 = 0.0, x2 = 0.0, y2 = 0.0, textWidth = 0.0;

   // Transform the text matrix to user space
   TCTM m = MulMatrix(m_GState.Matrix, *M);
   Transform(m, x1, y1); // Start point of the text record

   /*
      This code draws lines under each text record of a PDF file to check whether the coordinates are correct.
      It shows also how word spacing must be handled. You need an algorithm like this one if you want to
      develop a text extraction algorithm that tries to preserve the original text layout. Note that word
      spacing must be ignored when a CID font is selected. In addition, word spacing is applied to the space
      character (32) of the non-translated source string only. The Unicode string cannot be used to determine
      whether word spacing must be applied because the character can be encoded to an arbitrary Unicode character.
   */
   if (m_GState.FontType == ftType0)
   {
      // Word spacing must be ignored if a CID font is selected!
      for (i = 0; i < Count; i++)
      {
         if (Kerning[i].Advance != 0.0f)
         {
            textWidth -= Kerning[i].Advance;
            x1 = textWidth;
            y1 = 0.0;
            Transform(m, x1, y1);
         }
         textWidth += Kerning[i].Width;
         x2 = textWidth;
         y2 = 0.0;
         Transform(m, x2, y2);

         pdfMoveTo(m_PDF, x1, y1);
         pdfLineTo(m_PDF, x2, y2);
         if (m_Count++ & 1)
            pdfSetStrokeColor(m_PDF, PDF_RED);
         else
            pdfSetStrokeColor(m_PDF, PDF_BLUE);
         if (!pdfStrokePath(m_PDF)) return -1;
         x1 = x2;
         y1 = y2;
      }
   }else
   {
      UI32 j, last;
      // This code draws lines under line segments which are separated by one or more space character. This is important
      // to handle word spacing correctly. The same code can be used to compute word boundaries of Ansi strings.
      for (i = 0; i < Count; i++)
      {
         j    = 0;
         last = 0;
         if (Source[i].Advance != 0.0f)
         {
            textWidth -= Source[i].Advance;
            x1 = textWidth;
            y1 = 0.0;
            Transform(m, x1, y1);
         }
         while (j < Source[i].Length)
         {
            if (Source[i].Text[j] != 32)
               ++j;
            else
            {
               if (j > last)
               {
                  // Note that the text must be taken from the Source array!
                  textWidth += fntGetTextWidth( m_GState.ActiveFont,
                                                Source[i].Text + last,
                                                j - last,
                                                m_GState.CharSpacing,
                                                m_GState.WordSpacing,
                                                m_GState.TextScale);
                  x2 = textWidth;
                  y2 = 0.0;
                  Transform(m, x2, y2);
                  pdfMoveTo(m_PDF, x1, y1);
                  pdfLineTo(m_PDF, x2, y2);
                  if (m_Count++ & 1)
                     pdfSetStrokeColor(m_PDF, PDF_RED);
                  else
                     pdfSetStrokeColor(m_PDF, PDF_BLUE);
                  if (!pdfStrokePath(m_PDF)) return -1;
               }
               last = j++;
               while (j < Source[i].Length && Source[i].Text[j] == 32)
               {
                  ++j;
               }
               textWidth += fntGetTextWidth( m_GState.ActiveFont,
                                             Source[i].Text + last,
                                             j - last,
                                             m_GState.CharSpacing,
                                             m_GState.WordSpacing,
                                             m_GState.TextScale);
               last = j;
               x1 = textWidth;
               y1 = 0.0;
               Transform(m, x1, y1);
            }
         }
         if (j > last)
         {
            textWidth += fntGetTextWidth( m_GState.ActiveFont,
                                          Source[i].Text + last,
                                          j - last,
                                          m_GState.CharSpacing,
                                          m_GState.WordSpacing,
                                          m_GState.TextScale);
            x2 = textWidth;
            y2 = 0.0;
            Transform(m, x2, y2);
            pdfMoveTo(m_PDF, x1, y1);
            pdfLineTo(m_PDF, x2, y2);
            if (m_Count++ & 1)
               pdfSetStrokeColor(m_PDF, PDF_RED);
            else
               pdfSetStrokeColor(m_PDF, PDF_BLUE);
            if (!pdfStrokePath(m_PDF)) return -1;
         }
         x1 = x2;
         y1 = y2;
      }
   }
   return 0;
}

TCTM CTextCoordinates::MulMatrix(TCTM &M1, TCTM &M2)
{
   TCTM retval;
   retval.a = M2.a * M1.a + M2.b * M1.c;
   retval.b = M2.a * M1.b + M2.b * M1.d;
   retval.c = M2.c * M1.a + M2.d * M1.c;
   retval.d = M2.c * M1.b + M2.d * M1.d;
   retval.x = M2.x * M1.a + M2.y * M1.c + M1.x;
   retval.y = M2.x * M1.b + M2.y * M1.d + M1.y;
   return retval;
}

bool CTextCoordinates::RestoreGState(void)
{
   return m_Stack.Restore(m_GState);
}

SI32 CTextCoordinates::SaveGState(void)
{
   return m_Stack.Save(m_GState);
}

void CTextCoordinates::SetFont(double FontSize, TFontType Type, const void* Font)
{
   m_GState.ActiveFont = Font;
   m_GState.FontSize   = FontSize;
   m_GState.FontType   = Type;
   m_GState.SpaceWidth = (float)(fntGetSpaceWidth(Font, FontSize) * 0.5);
}

void CTextCoordinates::Transform(TCTM &M, double &x, double &y)
{
   double tx = x;
   x = tx * M.a + y * M.c + M.x;
   y = tx * M.b + y * M.d + M.y;
}

/* --------------------------------------- CStack ------------------------------------------- */

bool CTextCoordinates::CStack::Restore(TGState &F)
{
   if (m_Count > 0)
   {
      --m_Count;
      F = m_Items[m_Count];
      return true;
   }
   return false;
}

SI32 CTextCoordinates::CStack::Save(TGState &F)
{
   if (m_Count == m_Capacity)
   {
      m_Capacity += 16;
      TGState* tmp = (TGState*)realloc(m_Items, m_Capacity * sizeof(TGState));
      if (!tmp)
      {
         m_Capacity -= 16;
         return -1;
      }
      m_Items = tmp;
   }
   m_Items[m_Count] = F;
   ++m_Count;
   return 0;
}
